/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2003 Motonobu Ichimura <famao@momonga-linux.org>
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkfunction.c,v 1.1.2.13 2003/04/17 07:25:55 famao Exp $ */

/* vi:set ts=4 sw=4: */

#include <ctype.h>
#include "skkfunction.h"
#include "skkfunction_private.h"

typedef struct _SkkFunctionCallbackPrivate SkkFunctionCallbackPrivate;

struct _SkkFunctionCallbackPrivate {
	SkkFunctionCallback callback;
	gpointer user_data;
};

SkkFunction *
skk_function_new (SkkFunctionMap *map)
{
	SkkFunction *ret;
	ret = g_new0 (SkkFunction, 1);
	ret->ref_count = 1;
	ret->function_table = g_hash_table_new (g_str_hash, g_str_equal);
	ret->map = map;
	skk_function_map_ref (map);
	skk_function_add_default_function (ret);
	return ret;
}

static gboolean
table_clear_func (gpointer key, gpointer value, gpointer user_data)
{
	g_free (key);
	g_free (value);
	return TRUE;
}

void
skk_function_clear (SkkFunction *func)
{
	if (!func)
		return;
	g_hash_table_foreach_remove (func->function_table, table_clear_func, NULL);
}

void
skk_function_destroy (SkkFunction *func)
{
	if (!func)
		return;
	func->ref_count--;
	if (func->ref_count > 0)
		return;

	if (func->map)
		skk_function_map_unref (func->map);
	skk_function_clear (func);
	g_hash_table_destroy (func->function_table);
	g_free (func);
	return;
}

void
skk_function_add_function (SkkFunction *func, const gchar *name, SkkFunctionCallback callback, gpointer user_data)
{
	SkkFunctionCallbackPrivate *p;
	SkkFunctionCallbackPrivate *ret;
	gchar *key;
	if (!func)
		return;
	if (!name || !callback)
		return;
	p = g_new0 (SkkFunctionCallbackPrivate, 1);
	p->callback = callback;
	p->user_data = user_data;
	if (g_hash_table_lookup_extended (func->function_table, name,
				(gpointer *)&key, (gpointer *)&ret)) {
		g_hash_table_remove (func->function_table, name);
		g_free (key);
		g_free (ret);
	}
	g_hash_table_insert (func->function_table, (gpointer)g_strdup (name), p);
	return;
}

gint
skk_function_do_function (SkkFunction *func, SkkBuffer *buf, guint ch, guint mask)
{
	SkkFunctionMapItem *item = NULL;
	SkkFunctionCallbackPrivate *p;
	SkkMajorMode mode = skk_buffer_get_mode (buf);
	gint ret = SKK_NOTHING;

	item = skk_function_map_find_function (func->map, mode,
			skk_buffer_get_query_mode (buf), ch, mask);
	if (!item) {
		if (mode == SKK_LATIN_MODE) {
			return SKK_NOTRANS;
		}
		if (!(mask & (SKK_NONE_MASK | SKK_SHIFT_MASK)))
			return SKK_NOTRANS;

		/* TODO ok ? */
		if (!isascii (ch) || !isprint (ch))
			return SKK_NOTRANS;
		skk_buffer_add_char (buf, ch);
		ret = SKK_NOTHING;
	} else {
		switch (mode) {
			case SKK_J_MODE:
				{
					gchar *buf_check;
					if (isalnum (ch) && 
							(mask & (SKK_NONE_MASK | SKK_SHIFT_MASK)) 
							&& skkbuf_b (buf)) {
						buf_check = g_strdup_printf ("%s%c", skkbuf_b (buf), ch);
						if (skk_conv_is_exist (buf->rule, buf_check)) {
							ret = SKK_INSERT_CHAR;
							g_free (buf_check);
							break;
						}
						g_free (buf_check);
					}
				}
				/* fall through */
			default:
				p = g_hash_table_lookup (func->function_table, item->name);
				if (p && p->callback) {
					if (isprint (item->key)) {
						g_message ("callback %s %c", item->name, item->key);
					} else {
						g_message ("callback %s %d", item->name, item->key);
					}
					ret = p->callback (buf, item, p->user_data);
				}
				if (ret == SKK_INSERT_CHAR) {
					skk_buffer_add_char (buf, ch);
					return SKK_NOTHING;
				}
				break;
		}
	}
#if 1
	/* re-check mode */
	mode = skk_buffer_get_mode (buf);
	if (mode == SKK_J_MODE) {
		if (skk_buffer_can_clear (buf)) {
			skk_buffer_clear (buf);
		}
	}
#endif
	return ret;
}

#ifdef SKKFUNCTION_DEBUG

#include "skkkeysym.h"

int
main (void)
{
	SkkFunction *func;
	SkkFunctionMap *map;
	SkkBuffer *buf;
	buf = skk_buffer_new ();
	map = skk_function_map_new ();
	func = skk_function_new (map);
	skk_buffer_set_mode (buf, SKK_J_MODE);
	skk_buffer_add_char (buf, 'k');
	skk_function_do_function (func, buf, SKK_VK_j, SKK_CONTROL_MASK);
	skk_buffer_add_char (buf, 'A');
	skk_function_do_function (func, buf, SKK_VK_j, SKK_CONTROL_MASK);
	skk_buffer_destroy (buf);
	skk_function_map_destroy (map);
	skk_function_destroy (func);
	return 0;
}
#endif
