/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2003 Motonobu Ichimura <famao@momonga-linux.org>
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkconf.c,v 1.3.2.61 2003/04/17 06:19:22 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <glib.h>
#include <string.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "skkutils.h"
#include "skkconf.h"
#include "skkconf_private.h"

static gboolean table_clear_func (gpointer key, gpointer value, gpointer user_data);
static void clear (SkkConf *conf);

/* Parse Item */
static void parse_item_one (SkkConf *conf, xmlNodePtr node);
static void parse_item (SkkConf *conf, xmlDocPtr doc);

static SkkConfType find_type (gchar *str);

/* Add Item to List */
static void add_defaults (SkkConf* conf);

/* Find Item */

/* Load Configuration */
static void load_rc (SkkConf *conf, const gchar *filename);

typedef struct _SkkConfCallbackPrivate SkkConfCallbackPrivate;

struct _SkkConfCallbackPrivate {
	gchar *key;
	SkkConfCallback callback;
	gpointer user_data;
};

static void
add_defaults (SkkConf* conf)
{
	gint i;
	for (i = 0; i < SKKCONF_DEFAULT_ITEM_SIZE ; i++) {
		skk_conf_add_item (conf, &itemp[i]);
	}
}


static SkkConfType
find_type (gchar *str)
{
	if (!str)
		return SKKCONF_TYPE_INVALID;
	if (!g_strcasecmp (str, "bool")) {
		return SKKCONF_TYPE_BOOL;
	} else if (!g_strcasecmp (str, "string")) {
		return SKKCONF_TYPE_STRING;
	} else if (!g_strcasecmp (str, "num")) {
		return SKKCONF_TYPE_NUM;
	}
	g_assert (0);
	return SKKCONF_TYPE_INVALID;
}

static void
confvalue_free (gpointer data, gpointer user_data)
{
	SkkConfValue *v = (SkkConfValue *)data;
	g_free (v->name);
	if (v->value)
		g_free (v->value);
}

static void
parse_element_one (SkkConf *conf, xmlNodePtr node, SkkConfCallbackPrivate *p)
{
	GSList *l = NULL;
	SkkConfValue *v = NULL;
	xmlNodePtr child;
	xmlChar *type = NULL;
	xmlChar *value = NULL;
	child = node->xmlChildrenNode;
	if (!child)
		return;
	type = xmlGetProp (node, "type");
	for (; child; child = child->next) {
		if (xmlIsBlankNode (node)) {
			continue;
		}
		if (!child->name)
			continue;
		v = g_new0 (SkkConfValue, 1);
		v->name = g_strdup (child->name);
		value = xmlNodeGetContent (child);
		if (value) {
			v->value = skk_utils_utf8_to_eucjp (value);
			g_free (value);
		} else {
			v->value = NULL;
		}
		l = g_slist_append (l, v);
	}
	p->callback (conf, l, type, p->user_data);
	if (type)
		g_free (type);
	skk_utils_slist_free (l, TRUE, confvalue_free, NULL);
	return;

}

static void
parse_item_one (SkkConf *conf, xmlNodePtr node)
{
	xmlNodePtr child;
	xmlChar *name = NULL; /* always ASCII */
	xmlChar *value = NULL;
	xmlChar *info = NULL;
	xmlChar *type = NULL;
	gchar *info_e = NULL;
	gpointer value_e = NULL;
	SkkConfItem *item;
	SkkConfType ctype;
	if (!node)
		return;
	child = node->xmlChildrenNode;
	if (!child)
		return;
	type = xmlGetProp (node, "type");
	if (!type)
		return;
	if ((ctype = find_type (type)) == SKKCONF_TYPE_INVALID) {
		xmlFree (type);
		return;
	}
	for (; child ; child = child->next) {
		if (xmlIsBlankNode (node)) {
			continue;
		}
		if (!xmlStrcmp ("name", child->name)) {
			name = xmlNodeGetContent (child);
		} else if (!xmlStrcmp ("value", child->name)) {
			value = xmlNodeGetContent (child);
		} else if (!xmlStrcmp ("info", child->name)) {
			info = xmlNodeGetContent (child);
		}
	}
	info_e = info ? skk_utils_utf8_to_eucjp (info) : NULL;
	switch (ctype) {
		case SKKCONF_TYPE_BOOL:
			if (!xmlStrcasecmp ("true", value)) {
				value_e = GINT_TO_POINTER (TRUE);
			} else if (!xmlStrcasecmp ("false", value)) {
				value_e = GINT_TO_POINTER (FALSE);
			} else {
				value_e = GINT_TO_POINTER (FALSE);
			}
			break;
		case SKKCONF_TYPE_NUM:
			/* FIXME
			 * need error check ?
			 */
			value_e = GINT_TO_POINTER (atoi (value));
			break;
		case SKKCONF_TYPE_STRING:
			value_e = (gpointer)value ? skk_utils_utf8_to_eucjp (value) : NULL;
			break;
		default:
			g_assert (0); /* not reached */
			break;
	}
	item = skk_conf_item_new_with_value (ctype, name, info_e, value_e);
	skk_conf_add_item (conf, item);
	if (name)
		xmlFree (name);
	if (value)
		xmlFree (value);
	if (type)
		xmlFree (type);
	if (info)
		xmlFree (info);
	if (info_e)
		g_free (info_e);
	if (value_e && ctype == SKKCONF_TYPE_STRING)
		g_free (value_e);
	return;
}

static void
parse_element (SkkConf *conf, xmlDocPtr doc, SkkConfCallbackPrivate *p)
{
	xmlXPathContextPtr context;
	xmlXPathObjectPtr object;
	xmlNodeSetPtr nodeset = NULL;
	xmlNodePtr node = NULL;
	gint i = 0;
	if (!doc)
		return;
	context = xmlXPathNewContext (doc);
	if (!context)
		return;
	object = xmlXPathEval (p->key, context);
	if (!object) {
		xmlXPathFreeContext (context);
		return;
	}
	if (object->type == XPATH_NODESET) {
		nodeset = object->nodesetval;
	}
	/*
	 * NOTICE:
	 * nodesetval is not allocated by libxml2, so don't free after using it
	 */
	if (!nodeset || !nodeset->nodeTab) {
		xmlXPathFreeObject (object);
		return;
	}
	for (node = nodeset->nodeTab[0]; i < nodeset->nodeNr ; node = nodeset->nodeTab[i]) {
		parse_element_one (conf, node, p);
		i++;
	}
	xmlXPathFreeObject (object);
	xmlXPathFreeContext (context);
	return;
}

static void
parse_item (SkkConf *conf, xmlDocPtr doc) 
{
	xmlXPathContextPtr context;
	xmlXPathObjectPtr object;
	xmlNodeSetPtr nodeset = NULL;
	xmlNodePtr node = NULL;
	gint i = 0;
	if (!doc)
		return;
	context = xmlXPathNewContext (doc);
	if (!context)
		return;
	object = xmlXPathEval ("/iiimf-skk/item", context);
	if (!object) {
		xmlXPathFreeContext (context);
		return;
	}
	if (object->type == XPATH_NODESET) {
		nodeset = object->nodesetval;
	}
	/*
	 * NOTICE:
	 * nodesetval is not allocated by libxml2, so don't free after using it
	 */
	if (!nodeset || !nodeset->nodeTab) {
		xmlXPathFreeObject (object);
		return;
	}
	for (node = nodeset->nodeTab[0]; i < nodeset->nodeNr ; node = nodeset->nodeTab[i]) {
		parse_item_one (conf, node);
		i++;
	}
	xmlXPathFreeObject (object);
	xmlXPathFreeContext (context);
	return;
}

static gboolean
table_clear_func (gpointer key, gpointer value, gpointer user_data)
{
	if (value)
		skk_conf_item_destroy ((SkkConfItem *)value);
	return TRUE;
}

static void
clear (SkkConf *conf)
{
	g_hash_table_foreach_remove (conf->conf_table, table_clear_func, NULL);
	return;
}

static void
load_rc (SkkConf *conf, const char *filename)
{
	xmlDocPtr doc = NULL;
	xmlNodePtr root = NULL;
	GSList *callbacks;
	struct stat s;
	if (stat (filename, &s) == -1)
		return;

	/* reset setting */
	if (conf) {
		clear (conf);
		add_defaults (conf);
	}
	doc = xmlParseFile (filename);
	if (!doc)
		return;
	root = xmlDocGetRootElement (doc);
	if (xmlStrcasecmp (root->name, "iiimf-skk") != 0) {
		xmlFreeDoc (doc);
		return;
	}
	for (callbacks = conf->callbacks; callbacks; callbacks = g_slist_next (callbacks)) {
		parse_element (conf, doc, callbacks->data);
	}
	parse_item (conf, doc);
	xmlFreeDoc(doc);
}

void
skk_conf_load_rc (SkkConf *conf, char *user_name)
{
	char fname[FILENAME_MAX];

#ifdef SKKCONF_DEBUG
	strcpy(fname, "sample.config.xml");
#else
	struct passwd *pw;
	if (!user_name)
		strcpy (fname, "sample.config.xml");
	else {
		if ((pw = getpwnam(user_name)) == NULL)
			return;
		snprintf (fname, FILENAME_MAX, "%s/.iiimf-skk/config.xml", pw->pw_dir);
	}
#endif
	load_rc (conf, fname);
}


SkkConf*
skk_conf_new (void)
{
	SkkConf *ret;
	ret = g_new (SkkConf, 1);
	memset (ret, 0, sizeof (SkkConf));
	ret->clear = clear;
	ret->conf_table = g_hash_table_new (g_str_hash, g_str_equal);
	ret->callbacks = NULL;
	/* initialize */
	add_defaults (ret);
	skk_conf_ref (ret);
	return ret;
}

static void
callbacks_free (gpointer data, gpointer user_data)
{
	SkkConfCallbackPrivate *p = (SkkConfCallbackPrivate *)data;
	g_free (p->key);
}

void
skk_conf_destroy (SkkConf *conf)
{
	if (!conf)
		return;
	skk_conf_unref (conf);
	if (conf->ref_count > 0) {
		return;
	}
	skk_conf_clear (conf);
	if (conf->callbacks) {
		skk_utils_slist_free (conf->callbacks, TRUE, callbacks_free, NULL);
	}
	g_hash_table_destroy (conf->conf_table);
	g_free (conf);
}

void
skk_conf_clear (SkkConf *conf)
{
	if (!conf)
		return;
	if (conf->clear)
		conf->clear (conf);
	return;
}

void
skk_conf_ref (SkkConf *conf)
{
	if (!conf)
		return;
	conf->ref_count++;
	return;
}

void
skk_conf_unref (SkkConf *conf)
{
	if (!conf)
		return;
	conf->ref_count--;
	if (!conf->ref_count) {
		skk_conf_destroy (conf);
	}
	return;
}

void
skk_conf_add_item (SkkConf *conf, SkkConfItem *item)
{
	SkkConfItem *removed;
	gchar *key;
	if (!conf || !item || !item->name)
		return;
	if (g_hash_table_lookup_extended (conf->conf_table,
				(gconstpointer)item->name, 
				(gpointer *)&key, (gpointer *)&removed)) {
		g_hash_table_remove (conf->conf_table, item->name);
		skk_conf_item_destroy (removed);
	}
	g_hash_table_insert (conf->conf_table, item->name, item);
	return;
}

void
skk_conf_set_bool (SkkConf *conf, const gchar *name, gboolean value)
{
	SkkConfItem *item;
	if (!name)
		return;
	item = g_hash_table_lookup (conf->conf_table, name);
	if (!item || (item->type != SKKCONF_TYPE_BOOL))
		return;
	item->value = GINT_TO_POINTER (value);
	return;
}

void
skk_conf_set_string (SkkConf *conf, const gchar *name, const gchar *string)
{
	SkkConfItem *item;
	gchar *value = NULL;
	/* TODO How to */
	if (!conf)
		return;
	if (!string)
		return;
	if (!name)
		return;
	item = g_hash_table_lookup (conf->conf_table, name);
	if (!item || item->type != SKKCONF_TYPE_STRING)
		return;
	if (item->value) {
		value = item->value;
	}
	item->value = g_strdup (string);
	if (value)
		g_free (value);
	return;
}

void
skk_conf_set_num (SkkConf *conf, const gchar *name, gint num)
{
	SkkConfItem *item;
	/* TODO How to */
	if (!conf)
		return;
	if (!name)
		return;
	item = g_hash_table_lookup (conf->conf_table, name);
	if (!item || item->type != SKKCONF_TYPE_NUM)
		return;
	item->value = GINT_TO_POINTER (num);
	return;
}

gboolean
skk_conf_get_bool (SkkConf *conf, const gchar *name)
{
	SkkConfItem *item;
	if (!conf)
		return FALSE;
	if (!name)
		return FALSE;

	item = g_hash_table_lookup (conf->conf_table, name);
	if (!item || item->type != SKKCONF_TYPE_BOOL)
		return FALSE;
	else
		return GPOINTER_TO_INT (item->value);
	return FALSE;
}

gchar *
skk_conf_get_string (SkkConf *conf, const gchar *name)
{
	SkkConfItem *item;
	if (!conf)
		return NULL;
	if (!name)
		return NULL;
	item = g_hash_table_lookup (conf->conf_table, name);
	if (!item || item->type != SKKCONF_TYPE_STRING || !item->value)
		return NULL;
	else 
		return g_strdup ((gchar *)item->value);
	return NULL;
}

gint
skk_conf_get_num (SkkConf *conf, const gchar *name)
{
	SkkConfItem *item;
	if (!conf)
		return -1;
	if (!name)
		return -1;
	item = g_hash_table_lookup (conf->conf_table, name);
	if (!item || item->type != SKKCONF_TYPE_NUM)
		return -1;
	else
		return GPOINTER_TO_INT (item->value);
	return -1;
}

void
skk_conf_delete_item (SkkConf *conf, SkkConfItem *item)
{
	return;
}

SkkConfItem*
skk_conf_item_new (void)
{
	SkkConfItem *ret;
	ret = g_new0 (SkkConfItem, 1);
	ret->user_defined = TRUE;
	return ret;
}

SkkConfItem*
skk_conf_item_new_with_value (SkkConfType type, const gchar *name, const gchar *info, gpointer value)
{
	SkkConfItem *ret;
	ret = skk_conf_item_new ();
	ret->type = type;
	ret->name = name ? g_strdup (name) : NULL;
	ret->info = info ? g_strdup (info) : NULL;
	switch (type) {
		case SKKCONF_TYPE_BOOL:
		case SKKCONF_TYPE_NUM:
			ret->value = value;
			break;
		case SKKCONF_TYPE_STRING:
			ret->value = g_strdup ((gchar *)value);
			break;
		default:
			break;
	}
	return ret;
}

void
skk_conf_item_destroy (SkkConfItem *item)
{
	if (!item)
		return;
	if (!item->user_defined) 
		return;
	if (item->name)
		g_free (item->name);
	if (item->info)
		g_free (item->info);
	switch (item->type) {
		case SKKCONF_TYPE_STRING:
			if (item->value)
				g_free (item->value);
			break;
		default:
			break;
	}
	g_free (item);
	return;
}

gboolean
skk_conf_add_callback (SkkConf *conf, const gchar *key, SkkConfCallback callback, gpointer user_data)
{
	SkkConfCallbackPrivate *p;
	if (!conf)
		return FALSE;
	if (!key || !callback)
		return FALSE;
	p = g_new0 (SkkConfCallbackPrivate, 1);
	p->key = g_strdup (key);
	p->callback = callback;
	p->user_data = user_data;
	conf->callbacks = g_slist_append (conf->callbacks, p);
	return TRUE;
}

#ifdef SKKCONF_DEBUG

#include "skkconv_kana.h"

void
callback (SkkConf *conf, GSList *value, const gchar *type, gpointer user_data)
{
	SkkConvRule *rule = (SkkConvRule *)user_data;
	gchar *key = NULL;
	gchar *hira = NULL;
	gchar *kata = NULL;
	gchar *append = NULL;
	SkkConfValue *v;
	GSList *l;
	for (l = value; l ; l = g_slist_next (l)) {
		v = (SkkConfValue *)l->data;
		if (!strcmp (v->name, "key")) {
			key = v->value;
		} else if (!strcmp (v->name, "hiragana")) {
			hira = v->value;
		} else if (!strcmp (v->name, "katakana")) {
			kata = v->value;
		} else if (!strcmp (v->name, "append")) {
			append = v->value;
		}
	}
	skk_conv_add_rule (rule, skk_conv_rule_item_new_with_value (key, hira, kata, append));
	return;
}

int
main (void)
{
	SkkConf *conf;
	SkkConvRule *rule;
	gint i,j;
	conf = skk_conf_new();
	rule = skk_conv_rule_new ();

	if (getenv("SKK_PROF"))
		j = 1000;
	else
		j = 1;

	skk_conf_add_callback (conf, "/iiimf-skk/rule", callback, rule);
	for (i = 0; i < j; i++) {
		printf("'%s' -> '%s'\n",".", skk_conv_get_hiragana(rule, ".", NULL));
		skk_conf_load_rc(conf, NULL);
		printf("skk_egg_like_newline = %s\n",
						skk_conf_get_bool (conf, "skk-egg-like-newline") ? "TRUE" : "FALSE");
		printf("'%s' -> '%s'\n",".", skk_conv_get_hiragana(rule, ".", NULL));
		skk_conf_set_bool (conf, "skk-egg-like-newline", FALSE);
		printf("skk_egg_like_newline = %s\n",
						skk_conf_get_bool (conf, "skk-egg-like-newline") ? "TRUE" : "FALSE");
		printf ("skk_server_host = %s\n",
				skk_conf_get_string (conf, "skk-server-host") ? 
				skk_conf_get_string (conf, "skk-server-host") : "None");
		printf("skk_plugin_use_dict = %s\n",
						skk_conf_get_bool (conf, "skk-plugin-use-dict") ? "TRUE" : "FALSE");
	}
	return 0;
}
#endif
