/*
 * LibSKK, a tiny Library to emulate SKK (Simple Kana Kanji Conversion)
 * 
 * Copyright (C) 2003 Motonobu Ichimura <famao@momonga-linux.org>
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 *
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkbuffer.h,v 1.10 2003/02/19 17:44:51 famao Exp $ */

/* vi:set ts=4 sw=4: */


#ifndef HAVE_SKKBUF_H
#define HAVE_SKKBUF_H

#include <skkmode.h>
#include <skkconf.h>
#include <skkquery.h>
#include <skkcursor.h>
#include <skkconv_kana.h>
#include <skkdictitem.h>

/* in_buf */
#define skkbuf_b(x) (x)->in_buf
#define skkbuf_blen(x) (x)->buf_len
/* preedit_buf */
#define skkbuf_p(x) (x)->preedit_buf
#define skkbuf_plen(x) (x)->preedit_len
/* direction_word */
#define skkbuf_d(x) (x)->direction_word
#define skkbuf_dlen(x) (x)->henkan_len
/* result_buf */
#define skkbuf_r(x) (x)->result_buf
#define skkbuf_rlen(x) (x)->result_len
/* okuri_char */
#define skkbuf_okurip(x) (x)->henkan_okuri_char
#define skkbuf_okuri(x) (x)->henkan_okurigana
/* last kakutei */
#define skkbuf_lasth(x) (x)->last_kakutei_henkan_key
/* number */
#define skkbuf_numb(x) (x)->num_buf

/* candidate */
#define skkbuf_candidate_num(x)		(x)->candidate_count
#define skkbuf_last_candidate(x) (gchar *)(((SkkDictItem*)g_list_last ((x)->candidate_list)->data)->candidate)
#define skkbuf_last_annotate(x)	(gchar *)(((SkkDictItem*)g_list_last ((x)->candidate_list)->data)->annotate)
#define skkbuf_first_candidate(x) (gchar *)(((SkkDictItem*)g_list_first ((x)->candidate_list)->data)->candidate)
#define skkbuf_first_annotate(x) (gchar *)(((SkkDictItem*)g_list_first ((x)->candidate_list)->data)->annotation)
#define skkbuf_nth_candidate(x,y) (gchar *)(((SkkDictItem*)g_list_nth_data ((x)->candidate_list, (y)))->candidate)
#define skkbuf_cur_candidate_list(x) (g_list_nth ((x)->candidate_list, (x)->candidate_count))
#define skkbuf_cur_candidate(x)		(gchar *)(((SkkDictItem*)(g_list_nth_data ((x)->candidate_list, (x)->candidate_count)))->candidate)
#define skkbuf_cur_annotation(x)		(gchar *)(((SkkDictItem*)(g_list_nth_data ((x)->candidate_list, (x)->candidate_count)))->annotation)
#define skkbuf_last_num_list(x) (gchar *)((g_list_last ((x)->number_list))->data)
#define skkbuf_first_num_list(x) (gchar *)((g_list_first ((x)->number_list))->data)

/* config */
#define skkbuf_conf(x) (x)->conf
/* mode */
#define skkbuf_mode(x) (x)->mode
/* query */
#define skkbuf_query(x) (x)->query
/* cursor */
#define skkbuf_cursor(x) (x)->cursor

/* completion */
#define skkbuf_is_completion(x) (x)->in_completion
#define skkbuf_comp_list(x) (x)->completion_list
#define skkbuf_comp_count(x) (x)->completion_count

typedef struct _SkkBuffer SkkBuffer;

typedef void (*SkkBufListener) (SkkBuffer *buf, gchar *value, gpointer user_data);
typedef void (*SkkCodeTableListener) (SkkBuffer *buf, gpointer user_data);
typedef void (*SkkAddDictListener) (SkkBuffer *buf, gchar *query_str, gpointer user_data);
typedef void (*SkkLookupListener) (SkkBuffer *buf, gchar **label, gchar **candidate, gchar **annotate, gint count, gpointer user_data);
struct _SkkBuffer
{
	gchar *in_buf;
	gchar *preedit_buf;
	gchar *direction_word;
	gchar *num_buf;
	/* Ѵ٤겾̾ʬΥץեå */
	gchar *henkan_okuri_char;
	/* ߤѴ겾̾ʬ */
	gchar *henkan_okurigana;
	/* 꼭ˤǸ˳ꤷȤθФ */
	gchar *last_kakutei_henkan_key;
	gchar *result_buf;      
	gint   buf_len;         /* length of buf */
	gint   preedit_len;     /* length of preedit_buf */
	gint   henkan_len;      /* length of direction_word */
	gint   result_len;      /* length of result_buf */
	gint   candidate_count; /* current count of candidate word */
	gint   candidate_max;   /* amount of candidate words */
	gint   completion_count; /* current count of completion list */
	gint   has_num;         /* amount of number which preedit buf has */
	gboolean in_number;     /* TRUE when user is pressing number (0-9)*/
	gboolean in_completion; /* TRUE when in completion mode */
	gboolean check_before;  /* TRUE if we have more than one candidate to translate KANA */ 

	/* List */
	/* number_list */
	GList *number_list;
	GList *candidate_list;
	GList *completion_list;

	SkkMode *mode;
	SkkConf *conf;
	SkkQuery *query;
	SkkCursor *cursor;
	SkkConvRule *rule;

	/* listener */
	GSList *preedit_listener;
	GSList *result_listener;
	GSList *adddict_listener;
	GSList *codetable_listener;
	GSList *lookup_listener;
	
	/* Function */

	/* Clear Buffer (Not Destroy) */
	void           (*clear)(SkkBuffer *buf);
	/* Setter/Getter Status Method */
	void           (*set_status)(SkkBuffer *buf,SkkStatus status);
	void           (*set_query_status)(SkkBuffer *buf,SkkQueryStatus status);
	void           (*set_j_status)(SkkBuffer *buf,SkkJStatus jstatus);
	SkkStatus      (*get_status)(SkkBuffer *buf);
	SkkJStatus     (*get_j_status)(SkkBuffer *buf);
	SkkQueryStatus (*get_query_status)(SkkBuffer *buf);
	/* get Preedit buffer */
	gchar*         (*get_preedit_string) (SkkBuffer *buf);
	/* get Status Buffer */
	gchar*         (*get_status_string) (SkkBuffer *buf);
	/* Delete */
	void           (*delete_backward) (SkkBuffer *buf);
	/* Undo */
	gboolean       (*undo) (SkkBuffer *buf);
	/* Add ch to Buffer */
	gboolean       (*add_char) (SkkBuffer *buf, int ch);
	/* Check Function */
	gboolean       (*can_commit) (SkkBuffer *buf);
	gboolean       (*has_preedit) (SkkBuffer *buf);
	gboolean       (*can_clear) (SkkBuffer *buf);
	gboolean       (*has_buf) (SkkBuffer *buf);
	/* Dict */
	void           (*query_dict) (SkkBuffer *buf);
	/* Change Category */
	void           (*change_category) (SkkBuffer *buf);
	/* Commit String */
	void           (*commit) (SkkBuffer *buf, gboolean regist_dict);
	/* Cursor */
	void           (*move_left) (SkkBuffer *buf);
	void           (*move_right) (SkkBuffer *buf);

};


SkkBuffer* skk_buffer_new (void);
/* Free */
void skk_buffer_destroy (SkkBuffer *buf);
/* Reset SkkBuffer, Not Freed */
void skk_buffer_clear (SkkBuffer *buf);
/* add ch to SkkBuffer */
/* TRUE when successfully added */
gboolean skk_buffer_add_char (SkkBuffer *buf, int ch);
/* set/getter method */
void skk_buffer_set_status (SkkBuffer *buf, SkkStatus status);
SkkStatus skk_buffer_get_status (SkkBuffer *buf);
void skk_buffer_set_query_status (SkkBuffer *buf, SkkQueryStatus mode);
SkkQueryStatus skk_buffer_get_query_status (SkkBuffer *buf);
void skk_buffer_set_j_status (SkkBuffer *buf, SkkJStatus jstatus);
SkkJStatus skk_buffer_get_j_status (SkkBuffer *buf);

gchar* skk_buffer_get_preedit_string (SkkBuffer *buf);
const gchar *skk_buffer_get_status_string (SkkBuffer *buf);
gboolean skk_buffer_get_katakana (SkkBuffer *buf);
void skk_buffer_set_katakana (SkkBuffer *buf, gboolean state);
gboolean skk_buffer_get_jisx0201_katakana (SkkBuffer *buf);
void skk_buffer_set_jisx0201_katakana (SkkBuffer *buf, gboolean state);

/* translate */
void skk_buffer_translate (SkkBuffer *buf);

/* Dict */
void skk_buffer_query_dict (SkkBuffer *buf);
const gchar* skk_buffer_get_first_candidate (SkkBuffer *buf);
gchar* skk_buffer_get_cur_candidate (SkkBuffer *buf);
gchar* skk_buffer_get_cur_annotation (SkkBuffer *buf);
gchar* skk_buffer_get_next_candidate (SkkBuffer *buf);
gchar* skk_buffer_get_prev_candidate (SkkBuffer *buf);
gchar* skk_buffer_get_nth_candidate (SkkBuffer *buf, int num);
const gchar* skk_buffer_get_last_candidate (SkkBuffer *buf);
/* True if we can set next candidate to SkkBuffer */
gboolean skk_buffer_set_next_candidate (SkkBuffer *buf);
/* True if we can set prev candidate to SkkBuffer */
gboolean skk_buffer_set_prev_candidate (SkkBuffer *buf);
/* True if we have next candidate */
gboolean skk_buffer_has_next_candidate (SkkBuffer *buf);

/* Add String to Dict */
void skk_buffer_add_dict (SkkBuffer *buf);
void skk_buffer_add_dict_with_value (SkkBuffer *buf, gchar *value);

/* Completion */

#define skk_buffer_end_completion(x)	((x)->in_completion = FALSE)
void skk_buffer_get_completion (SkkBuffer *buf);
void skk_buffer_set_next_completion (SkkBuffer *buf);
void skk_buffer_set_prev_completion (SkkBuffer *buf);

/* Undo */
gboolean skk_buffer_undo (SkkBuffer *buf);

/* Delete */
void skk_buffer_delete_backward (SkkBuffer *buf);

/* Change category */
void skk_buffer_change_category (SkkBuffer *buf);

/* Check functions */
gboolean skk_buffer_can_clear (SkkBuffer *buf);    /* TRUE when buffer can clear */
gboolean skk_buffer_can_commit (SkkBuffer *buf);   /* TRUE when buffer can commit */
gboolean skk_buffer_has_preedit (SkkBuffer *buf);  /* TRUE when buffer has preedit buf */
gboolean skk_buffer_has_buf (SkkBuffer *buf);      /* TRUE when buffer has in_buf */

/* commit string */
void skk_buffer_commit (SkkBuffer *buf, gboolean regist_dict);

/* set configuration */
void skk_buffer_set_conf (SkkBuffer *buf, SkkConf *conf);

/* set query */
void skk_buffer_set_query (SkkBuffer *buf, SkkQuery *query);

/* set conversion rule */
void skk_buffer_set_rule (SkkBuffer *buf, SkkConvRule *rule);

/* Event Listener */
void skk_buffer_add_preedit_listener (SkkBuffer *buf, SkkBufListener listener, gpointer user_data);
void skk_buffer_add_result_listener (SkkBuffer *buf, SkkBufListener listener, gpointer user_data);
void skk_buffer_add_adddict_listener (SkkBuffer *buf, SkkAddDictListener listener, gpointer user_data);
void skk_buffer_add_codetable_listener (SkkBuffer *buf, SkkCodeTableListener listener, gpointer user_data);
void skk_buffer_add_lookup_listener (SkkBuffer *buf, SkkLookupListener listener, gpointer user_data);
gboolean skk_buffer_remove_preedit_listener (SkkBuffer *buf, SkkBufListener listener);
gboolean skk_buffer_remove_result_listener (SkkBuffer *buf, SkkBufListener listener);
gboolean skk_buffer_remove_adddict_listener (SkkBuffer *buf, SkkAddDictListener listener);
gboolean skk_buffer_remove_codetable_listener (SkkBuffer *buf, SkkCodeTableListener listener);
gboolean skk_buffer_remove_lookup_listener (SkkBuffer *buf, SkkLookupListener listener);

/* Emit Signal */
void skk_buffer_preedit_emit (SkkBuffer *buf);
void skk_buffer_result_emit (SkkBuffer *buf);
void skk_buffer_adddict_emit (SkkBuffer *buf);
void skk_buffer_codetable_emit (SkkBuffer *buf);
void skk_buffer_lookup_emit (SkkBuffer *buf);

/* Set String */
void skk_buffer_set_preedit (SkkBuffer *buf, const gchar *value);
void skk_buffer_add_preedit (SkkBuffer *buf, const gchar *value);
void skk_buffer_set_result (SkkBuffer *buf, const gchar *value);
void skk_buffer_add_result (SkkBuffer *buf, const gchar *value);

/* Cursor */
void skk_buffer_move_left (SkkBuffer *buf);
void skk_buffer_move_right (SkkBuffer *buf);

#endif /* HAVE_SKKBUF_H */
