/*
 * IIIMF-SKK, Japanese Language Engine for 
 *                        IIIMF (Internet/Intranet Input Method Framework)
 * 
 * Copyright (C) 2003 Motonobu Ichimura <famao@momonga-linux.org>
 * Copyright (C) 2002 Motonobu Ichimura <famao@kondara.org>
 * 
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, and/or sell copies of the Software, and to permit persons
 * to whom the Software is furnished to do so, provided that the above
 * copyright notice(s) and this permission notice appear in all copies of
 * the Software and that both the above copyright notice(s) and this
 * permission notice appear in supporting documentation.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT
 * OF THIRD PARTY RIGHTS. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * HOLDERS INCLUDED IN THIS NOTICE BE LIABLE FOR ANY CLAIM, OR ANY SPECIAL
 * INDIRECT OR CONSEQUENTIAL DAMAGES, OR ANY DAMAGES WHATSOEVER RESULTING
 * FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 * Except as contained in this notice, the name of a copyright holder
 * shall not be used in advertising or otherwise to promote the sale, use
 * or other dealings in this Software without prior written authorization
 * of the copyright holder.
 *
 */

/* $Id: skkpalette.c,v 1.4 2003/02/19 17:44:50 famao Exp $ */

/* vi:set ts=4 sw=4: */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gdk/gdkkeysyms.h>
#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <locale.h>
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <X11/Xmd.h>

#include "xaux_common.h"
#include "xaux_ext_common.h"
#include "skk_xaux.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "pixmaps/hira.xpm"
#include "pixmaps/kata.xpm"
#include "pixmaps/jisx0208.xpm"
#include "pixmaps/jisx0201.xpm"
#include "pixmaps/ascii.xpm"
#include "pixmaps/off.xpm"
#include "pixmaps/reload.xpm"
#ifndef notdef
#include "pixmaps/code.xpm"
#endif

#include "wmutil.h"
#include "skkpalette.h"
#include "codetable.h"
#include "support.h"

#include "pixmaputil.h"
#include "menu.h"

xaux_class_t xaux_class = {
	"org.momonga-linux.skk.PaletteAux", 1,
	 NULL, (Window) 0, (Window) 0,
	(Atom) 0, (Atom) 0, (Atom) 0,
	{(Atom) 0,}, 0, {(Atom) 0,} , 
	0, NULL,
};

static SkkPalette *g_palette = NULL;
static xaux_class_t *xc_ = 0;
static int im_ = 0;
static int ic_ = 0;

static gboolean _handle_wm (Palette *p, gint wm_type, gint new_detected);
static gboolean _handle_tray (Palette *p, gint tray_type, gint new_detected);
static void _gnome_hints (Palette *palette, gint wm_type);
static void _ewmh_hints (Palette *palette, gint wm_type);
static void _gnome_hints_tray (TrayWindow *tray);
static void _ewmh_hints_tray (TrayWindow *tray, gint tray_type);

static void _prepare_button (SkkPalette *palette);
/* Position */
static void _palette_reposition (SkkPalette *palette);

/* Event */
static GdkFilterReturn _ext_event_handler (GdkXEvent *gdk_xevent, GdkEvent *event, gpointer user_data);

static void _status_changed (gchar *text, gint type);
static void _set_menu (GtkWidget *menu, gint type, gpointer data);

/* Menu Detach */
static void _menu_detach_func (GtkWidget *attach_widget, GtkMenu *menu);

/* Signal */
static void _reload_button_clicked (GtkWidget *widget, gpointer user_data);
static void _code_button_clicked (GtkWidget *widget, gpointer user_data);
static void _status_button_clicked (GtkWidget *widget, gpointer user_data);
static void _status_menu_activate (GtkMenuItem *menu_item, gpointer user_data);
static void _tray_activate (GtkMenuItem *menu_item, gpointer user_data);

static void _show_codetable (void);
static void _register_xpm (void);

/* Error */
static int palette_x_error (Display *d, XErrorEvent *error);

void skk_aux_palette_send_engine (gint int_count, gint *integers, gint str_count, gchar **strings);

static int
palette_x_error (Display *d, XErrorEvent *error)
{
	if (error->error_code) {
		char buf[64];
		XGetErrorText (d, error->error_code, buf, 63);
		fprintf (stderr, 
				"IIIMF-SKK (PaletteAux) **: %s\n serial %ld error_code %d request_code %d minor_code %d\n",
				buf,
				error->serial,
				error->error_code,
				error->request_code,
				error->minor_code);
	}
	return 0;
}

static void
_menu_detach_func (GtkWidget *attach_widget, GtkMenu *menu)
{
	/* nothing to do */
	return;
}

static void
_register_xpm (void)
{
	pixmap_pool_add_xpm (SKK_MODE_OFF, off_xpm);
	pixmap_pool_add_xpm (SKK_MODE_JMODE_HIRA, hira_xpm);
	pixmap_pool_add_xpm (SKK_MODE_JMODE_KATA, kata_xpm);
	pixmap_pool_add_xpm (SKK_MODE_ABBREV, hira_xpm);
	pixmap_pool_add_xpm (SKK_MODE_LATIN, ascii_xpm);
	pixmap_pool_add_xpm (SKK_MODE_JISX0208_LATIN, jisx0208_xpm);
	pixmap_pool_add_xpm (SKK_MODE_JISX0201_KANA, jisx0201_xpm);
}

static void
_palette_reposition (SkkPalette *palette)
{
	GtkRequisition sizereq;
	gint x, y, width, height, wmflag;
	if (!GTK_WIDGET_REALIZED (SKK_PALETTE_WIDGET (palette)))
		return;
	wmflag = SKK_PALETTE (palette)->running_wm;
	if (wmflag & WM_IS_BLACKBOX) {
		return;
	} else if (wmflag & WM_IS_EWMH) {
		wmutil_window_workarea_ewmh (SKK_PALETTE_WINDOW (palette),
				&x, &y, &width, &height);
		if (x < 0) {
			/* TODO */
			goto fallback;
		}
		width = x + width;
		height = y + height;
	} else {
fallback:
		width = gdk_screen_width ();
		height = gdk_screen_height ();
	}
	gtk_widget_size_request (SKK_PALETTE_WIDGET (palette), &sizereq);
#ifdef USE_GTK2
	gtk_window_move (GTK_WINDOW (SKK_PALETTE_WIDGET (palette)),
			width - sizereq.width,
			height - sizereq.height);
#else
	gtk_window_reposition (GTK_WINDOW (SKK_PALETTE_WIDGET (palette)),
			width - sizereq.width,
			height - sizereq.height);
#endif
	return;
}

static GdkFilterReturn
_ext_event_handler (GdkXEvent *gdk_xevent, GdkEvent *event, gpointer user_data)
{
    XEvent *xevent;
    xevent = (XEvent *) gdk_xevent;
    if (xevent->type == ClientMessage) {
	    if (xaux_ext_process_client_message(GDK_DISPLAY(), (XClientMessageEvent *) xevent))
		    return GDK_FILTER_REMOVE;
	    else
		    return GDK_FILTER_CONTINUE;
    } else if (xevent->type == KeyRelease) {
	 //   key_event_handler(0, 0, xevent, 0);
		    return GDK_FILTER_REMOVE;
    } else if (xevent->type == ReparentNotify) {
	    XReparentEvent *re = (XReparentEvent *)xevent;
	    g_message ("ext reparent %lx %lx -> %lx",
			    re->event,
			    re->window,
			    re->parent
		      );
    } else if (xevent->type == PropertyNotify) {
	    XPropertyEvent *pe = (XPropertyEvent *)xevent;
	    g_message ("ext property notify %lx %d %s",
			    pe->window,
			    pe->state,
			    XGetAtomName (GDK_DISPLAY(), pe->atom)
		      );
    }
    g_message ("ext type %d", xevent->type);
    return GDK_FILTER_CONTINUE;
}

static void
_tray_activate (GtkMenuItem *menu_item, gpointer user_data)
{
	SkkPalette *p = (SkkPalette *)user_data;
	if (GTK_WIDGET_VISIBLE (SKK_PALETTE_WIDGET (p))) {
		gtk_widget_hide (SKK_PALETTE_WIDGET (p));
	} else {
		gtk_widget_show (SKK_PALETTE_WIDGET (p));
	}
}

static void
_status_menu_activate (GtkMenuItem *menu_item, gpointer user_data)
{
	gint send_status[1];
#ifdef USE_GTK2
	if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (menu_item))) {
#else
	if (GTK_CHECK_MENU_ITEM (menu_item)->active) {
#endif
		send_status[0] = GPOINTER_TO_INT (menu_item_get_activate_data (GTK_WIDGET (menu_item)));
		skk_aux_palette_send_engine (1, send_status, 0, NULL);
	}
}

static void
_reload_button_clicked (GtkWidget *widget, gpointer user_data)
{
	gint send_status[1];
	send_status[0] = SKK_AUX_RELOAD_CONFIG;
	skk_aux_palette_send_engine (1, send_status, 0, NULL);
	return;
}

static void
_status_button_clicked (GtkWidget *widget, gpointer user_data)
{
	SkkPalette *palette = (SkkPalette *)user_data;
	gtk_menu_popup (GTK_MENU (SKK_PALETTE_MENU (palette)),
			NULL, NULL, NULL,
			SKK_PALETTE_WIDGET (palette),
			0, 0);
	return;
}

static void
_code_button_clicked (GtkWidget *widget, gpointer user_data)
{
	SkkPalette *palette = (SkkPalette *)user_data;
	gtk_widget_show (palette->codetable);
	return;
}

static void
_set_menu (GtkWidget *menu, gint type, gpointer data)
{
	GtkWidget *menuitem;
	if (!menu)
		return;
	switch (type) {
		case SKK_MODE_JMODE_HIRA:
			menuitem = (GtkWidget *)gtk_object_get_data (GTK_OBJECT (menu), "hiragana");
			break;
		case SKK_MODE_JMODE_KATA:
			menuitem = (GtkWidget *)gtk_object_get_data (GTK_OBJECT (menu), "katakana");
			break;
		case SKK_MODE_ABBREV:
			menuitem = (GtkWidget *)gtk_object_get_data (GTK_OBJECT (menu), "abbrev");
			break;
		case SKK_MODE_LATIN:
			menuitem = (GtkWidget *)gtk_object_get_data (GTK_OBJECT (menu), "latin");
			break;
		case SKK_MODE_JISX0208_LATIN:
			menuitem = (GtkWidget *)gtk_object_get_data (GTK_OBJECT (menu), "jisx0208_latin");
			break;
		case SKK_MODE_JISX0201_KANA:
			menuitem = (GtkWidget *)gtk_object_get_data (GTK_OBJECT (menu), "jisx0201_kana");
			break;
		default:
			menuitem = NULL;
			break;
	}
	if (!menuitem)
		return;
	gtk_signal_handler_block_by_data (GTK_OBJECT (menuitem), data);
	gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menuitem), TRUE);
	gtk_signal_handler_unblock_by_data (GTK_OBJECT (menuitem), data);
	return;

}

static void
_status_changed (gchar *text, gint type)
{
	GdkBitmap *mask;
	GdkPixmap *pix;
	GtkWidget *item;
#ifdef USE_GTK2
	gchar *ustr;
#endif
	if (!text)
		return;

	item = palette_get_element_label_nth (SKK_PALETTE (g_palette), 0);
	if (item) {
#ifdef USE_GTK2
		ustr = g_locale_to_utf8 (text, -1, NULL, NULL, NULL);
		gtk_label_set_text (GTK_LABEL (item), ustr);
		g_free (ustr);
#else
		gtk_label_set_text (GTK_LABEL (item), text);
#endif
	} else {
		g_message ("item not found\n");
	}
	item = palette_get_element_icon_nth (SKK_PALETTE (g_palette), 0);
	if (item) {
		pixmap_get (item, &pix, &mask, type);
#ifndef USE_GTK2
			gtk_pixmap_set (GTK_PIXMAP (item), pix, mask);
#else
			if (GTK_IS_PIXMAP (item)) {
				gtk_pixmap_set (GTK_PIXMAP (item), pix, mask);
			} else if (GTK_IS_IMAGE (item)) {
				gtk_image_set_from_pixmap (GTK_IMAGE (item), pix, mask);
			}
#endif
	}
	/* RePositioning */
	_palette_reposition (g_palette);
	return;
}

static void
_show_codetable (void)
{
	GtkWidget *code;
	code = g_palette->codetable;
	if (!code)
		return;
	/* 
	  in the case of having codetable in DISPLAY,
	  raise window
	 */ 
	if (GTK_WIDGET_VISIBLE (code)) {
		gdk_window_raise (code->window);
		return;
	}
	gtk_widget_show (code);
	/* TODO need it ? */
	gdk_window_raise (code->window);
	return;
}

static void
_reply_initialize (void)
{
	gint status[1];
	status[0] = SKK_AUX_PALETTE_OK;
	skk_aux_palette_send_engine (1, status, 0, NULL);
	return;
}

Bool
xaux_ext_Start (xaux_class_t *xc, aux_ext_data_t *aux_ext_data)
{
	g_message ("ext_Start");
#if 0
	gtk_widget_show (SKK_PALETTE_WIDGET (g_palette));
	gdk_window_raise (SKK_PALETTE_WINDOW (g_palette));
	_palette_reposition (g_palette);
	g_palette = skk_palette_new ();
#endif
	return True;
}

Bool
xaux_ext_Done (xaux_class_t *xc, aux_ext_data_t *aux_ext_data)
{
	g_message ("ext_Done");
	return True;
}

Bool
xaux_ext_Draw (xaux_class_t *xc, aux_ext_data_t *aux_ext_data)
{
	gint i;

	im_ = aux_ext_data->im;
	ic_ = aux_ext_data->ic;
	xc_ = xc;

	if (!GTK_WIDGET_VISIBLE (SKK_PALETTE_WIDGET (g_palette))) {
		g_message ("Draw palette_show");
		palette_show (SKK_PALETTE (g_palette));
	}
#if 0
	g_message ("fuga %d\n", aux_ext_data->integer_count);
	for (i = 0; i < aux_ext_data->string_count; i++) {
		printf ("\tsv[%02d]=%s\n",i, aux_ext_data->string_list[i].ptr);
	}
#endif
	for (i = 0; i < aux_ext_data->integer_count; i++) {
		switch (aux_ext_data->integer_list[i]) {
			/* Initialize Request */
			case SKK_AUX_PALETTE_INIT:
				_reply_initialize ();
				break;
			case SKK_AUX_PALETTE_CODE:
				_show_codetable ();
				break;
			/* Mode */
			case SKK_MODE_JMODE_HIRA:
			case SKK_MODE_JMODE_KATA:
			case SKK_MODE_ABBREV:
			case SKK_MODE_LATIN:
			case SKK_MODE_JISX0201_KANA:
			case SKK_MODE_JISX0208_LATIN:
				_status_changed (aux_ext_data->string_list[i].ptr,
						aux_ext_data->integer_list[i]);
				_set_menu (SKK_PALETTE_MENU (g_palette), aux_ext_data->integer_list[i], 
						(gpointer) g_palette);
				_set_menu (skk_tray_window_get_menu (SKK_TRAY (g_palette)), aux_ext_data->integer_list[i],
						(gpointer) SKK_TRAY (g_palette));
				break;
			case SKK_MODE_OFF:
#if 1
				_status_changed (aux_ext_data->string_list[i].ptr, SKK_MODE_OFF);
#else
				/* TODO use SkkConf */
				gtk_widget_hide (toplevel);
#endif				
				break;
			default:
				break;
		}
	}
	return True;
}

static MenuEntry _status_menu_entry[] = 
{
	{ N_("hiragana"), SKK_MODE_JMODE_HIRA, _status_menu_activate },
	{ N_("katakana"), SKK_MODE_JMODE_KATA, _status_menu_activate },
	{ N_("jisx0201_kana"), SKK_MODE_JISX0201_KANA, _status_menu_activate },
	{ N_("jisx0208_latin"), SKK_MODE_JISX0208_LATIN, _status_menu_activate },
	{ N_("latin"), SKK_MODE_LATIN , _status_menu_activate },
	{ N_("abbrev"), SKK_MODE_ABBREV , _status_menu_activate },
	{ NULL, 0 , NULL }
};

static MenuEntry _tray_additional_entry[] = 
{
	{ "-", -1 , NULL },
	{ N_("Show Palette"), -1, _tray_activate },
	{ NULL, 0, NULL }
};

void
skk_aux_palette_send_engine (gint int_count, gint *integers, gint str_count, gchar **strings) 
{
	gint i;
	aux_ext_data_t *aux_data;

	if (!im_ || !ic_ || !xc_)
		return;
	aux_data = (aux_ext_data_t *) g_new0 (aux_ext_data_t,1);
	aux_data->im = im_;
	aux_data->ic = ic_;
	aux_data->integer_count = int_count;
	aux_data->string_count = str_count;

	if (aux_data->integer_count) {
		aux_data->integer_list = (int *) g_new0 (int, aux_data->integer_count);
		for (i = 0; i < aux_data->integer_count; i++) {
			aux_data->integer_list[i] = integers[i];
		}
	}
	if (aux_data->string_count) {
		aux_data->string_list = (aux_ext_string_t *) g_new0 (aux_ext_string_t, aux_data->string_count);
		for (i = 0; i < aux_data->string_count; i++) {
			aux_ext_string_t *p = &aux_data->string_list[i];
			p->length = strlen (strings[i]);
			p->ptr = (guchar *) g_strdup (strings[i]);
		}
	}
	xaux_ext_SetValue (GDK_DISPLAY(), xc_, aux_data);

	if (aux_data->integer_list) {
		g_free (aux_data->integer_list);
	}
	if (aux_data->string_list) {
		for (i = 0; i < aux_data->string_count; i++) {
			aux_ext_string_t *p = &aux_data->string_list[i];
			if (p && p->ptr)
				g_free (p->ptr);
		}
		g_free (aux_data->string_list);
	}
	g_free (aux_data);
	return;
}

static gboolean
_handle_wm (Palette *palette, gint wm_type, gint new_detected)
{
	g_message ("_handle_wm");
#if 0
	gtk_widget_hide (PALETTE_WIDGET (palette));
#endif
	gdk_window_set_override_redirect (PALETTE_WINDOW (palette), FALSE);
	if (wm_type & WM_IS_BLACKBOX) {
		g_message ("withdraw");
		wmutil_window_hints_icccm (PALETTE_WINDOW (palette), WithdrawnState);
		gtk_widget_show (PALETTE_WIDGET (palette));
		return TRUE;
	}
	if (wm_type & WM_IS_EWMH) {
		gtk_widget_hide (PALETTE_WIDGET (palette));
		_ewmh_hints (palette, wm_type);
	}
	if (wm_type & WM_IS_GNOME) {
		gtk_widget_hide (PALETTE_WIDGET (palette));
		_gnome_hints (palette, wm_type);
	}
	gtk_widget_show (PALETTE_WIDGET (palette));
#if 0
	if (wm_type & WM_IS_EWMH) {
		_ewmh_hints (palette, wm_type);
	}
#endif
	return TRUE;
}

static gboolean
_handle_tray (Palette *palette, gint tray_type, gint new_detected)
{
	g_message ("_handle_tray");
#if 1
	tray_window_set_parent (SKK_TRAY (g_palette), SKK_PALETTE_WIDGET (g_palette));
#endif
	tray_window_realize (SKK_TRAY (g_palette));
	gtk_widget_set_name (SKK_TRAY(g_palette)->window, "skk-tray");
	if (tray_type & WM_IS_EWMH) {
		_ewmh_hints_tray (SKK_TRAY (g_palette), tray_type);
	}
	if (tray_type & WM_IS_GNOME) {
		_gnome_hints_tray (SKK_TRAY (g_palette));
	}
	tray_window_show (SKK_TRAY (g_palette));
#if 1
	if (tray_type & WM_IS_EWMH) {
		_ewmh_hints_tray (SKK_TRAY (g_palette), tray_type);
	}
	if (tray_type & WM_IS_GNOME) {
		_gnome_hints_tray (SKK_TRAY (g_palette));
	}
#endif
	return TRUE;
}

static void
_prepare_button (SkkPalette *palette)
{
	GtkWidget *button;
	GtkWidget *icon;

	icon = pixmap_new (SKK_PALETTE_WIDGET (palette), SKK_MODE_JMODE_HIRA);
#ifndef USE_GTK2
	button = pixmap_button_new ("SKK", icon);
	gtk_container_set_resize_mode (GTK_CONTAINER (button), GTK_RESIZE_PARENT);
	button = palette_add_widget (SKK_PALETTE (palette), button, NULL);
#else
	button = palette_add_button (SKK_PALETTE (palette), icon, "SKK", NULL);
#endif
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (_status_button_clicked), (gpointer)palette);
	gtk_widget_show (button);

#if 1
	icon = pixmap_new_with_xpm (SKK_PALETTE_WIDGET (palette), code_xpm);
	button = pixmap_button_new (NULL, icon);
#else
	icon = gtk_image_new_from_stock (GTK_STOCK_INDEX, GTK_ICON_SIZE_BUTTON);
#endif
	button = palette_add_widget (SKK_PALETTE (palette),  button , N_("Code Table"));
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (_code_button_clicked), palette);
	gtk_widget_show (button);

	/* famao temporary */
	icon = pixmap_new_with_xpm (SKK_PALETTE_WIDGET (palette), reload_xpm);
	button = pixmap_button_new (NULL, icon);

	button = palette_add_widget (SKK_PALETTE (palette), button, N_("Reload Config"));
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (_reload_button_clicked), palette);
	gtk_widget_show (button);
	return;
}

SkkPalette *
skk_palette_new (void)
{
	SkkPalette *ret;
	GtkWidget *tray_menu;
	ret = g_new0 (SkkPalette, 1);
	ret->palette = palette_new ();
	ret->palette->detected_wm = _handle_wm;
	ret->palette->detected_tray = _handle_tray;
	ret->tray = skk_tray_window_new ();
	tray_menu = skk_tray_window_get_menu (ret->tray);
	if (tray_menu)
		menu_add_entry (tray_menu, _tray_additional_entry, (gpointer)ret);
	ret->status_menu = radio_menu_new (_status_menu_entry, ret);
	ret->codetable = skk_aux_code_new ();
	gdk_window_set_transient_for (SKK_PALETTE_WINDOW (ret), ret->codetable->window);
	_prepare_button (ret);
	gtk_menu_attach_to_widget (GTK_MENU (ret->status_menu) , SKK_PALETTE_WIDGET (ret),
			_menu_detach_func);
#if 0
	wmutil_window_input_icccm (SKK_PALETTE_WINDOW (ret), FALSE);
#endif
	return ret;
}

static void
_ewmh_hints_tray (TrayWindow *tray, gint tray_type)
{
	GdkAtom docked_ewmh[1];
	GdkAtom state[3];
	state[0] = gdk_atom_intern ("_NET_WM_STATE_STICKY", FALSE); /* sticky */
	state[1] = gdk_atom_intern ("_NET_WM_STATE_SKIP_TASKBAR", FALSE);
	state[2] = gdk_atom_intern ("_NET_WM_STATE_SKIP_PAGER", FALSE);
	if (GTK_WIDGET_VISIBLE (TRAY_WIDGET (tray))) {
		wmutil_window_state_ewmh_client_message (TRAY_WINDOW (tray),
				state[0], state[1]);
		wmutil_window_state_ewmh_client_message (TRAY_WINDOW (tray),
				state[2], None);
	} else {
		if (!(tray_type & WM_IS_KWIN)) {
			docked_ewmh[0] = gdk_atom_intern ("_NET_WM_WINDOW_TYPE_DOCK", FALSE);
			wmutil_window_type_ewmh (TRAY_WINDOW (tray),
					docked_ewmh, 1, GDK_PROP_MODE_REPLACE);
		}
		wmutil_window_state_ewmh_prop (TRAY_WINDOW (tray),
				state, 3, GDK_PROP_MODE_REPLACE);
		wmutil_window_desktop_ewmh_prop (TRAY_WINDOW (tray),
				0xffffffff);
	}
}

static void
_gnome_hints_tray (TrayWindow *tray)
{
	wmutil_window_hints_gnome (TRAY_WINDOW (tray),
			(WIN_HINTS_SKIP_FOCUS | WIN_HINTS_SKIP_TASKBAR |
			 WIN_HINTS_SKIP_WINLIST));
}

static void
_ewmh_hints (Palette *palette, gint wm_type)
{
	GdkAtom docked_ewmh[1];
	GdkAtom state[4];
	state[0] = gdk_atom_intern ("_NET_WM_STATE_STICKY", FALSE); /* sticky */
	state[1] = gdk_atom_intern ("_NET_WM_STATE_SKIP_TASKBAR", FALSE);
	state[2] = gdk_atom_intern ("_NET_WM_STATE_SKIP_PAGER", FALSE);
	state[3] = gdk_atom_intern ("_NET_WM_STATE_STAYS_ON_TOP", FALSE); /* KDE only ? */
	/* 
	 * when use metacity, we need to set 
	 * _NET_WM_WINDOW_TYPE_DOCK or _NET_WM_WINDOW_TYPE_DESKTOP for sticky option
	 */
	if (!(wm_type & WM_IS_KWIN)) {
		docked_ewmh[0] = gdk_atom_intern ("_NET_WM_WINDOW_TYPE_DOCK", FALSE);
	} else {
		/*
		 * kwin 3.x has trouble with NET_WM_WINDOW_TYPE
		 */
		docked_ewmh[0] = gdk_atom_intern ("_NET_WM_WINDOW_TYPE_BUGGY", FALSE);
	}

	if (GTK_WIDGET_VISIBLE (PALETTE_WIDGET (palette))) {
		wmutil_window_state_ewmh_client_message (PALETTE_WINDOW (palette),
				state[0], None);
		wmutil_window_state_ewmh_client_message (PALETTE_WINDOW (palette),
				state[1], None);
		wmutil_window_state_ewmh_client_message (PALETTE_WINDOW (palette),
				state[2], None);
		wmutil_window_state_ewmh_client_message (PALETTE_WINDOW (palette),
				state[3], None);
		wmutil_window_desktop_ewmh_client_message (PALETTE_WINDOW (palette),
				0xffffffff);
	} else {
		wmutil_window_type_ewmh (PALETTE_WINDOW (palette),
				docked_ewmh, 1, GDK_PROP_MODE_REPLACE);
		wmutil_window_state_ewmh_prop (PALETTE_WINDOW (palette),
				state, 4, GDK_PROP_MODE_REPLACE);
		/* metacity check it */
		wmutil_window_desktop_ewmh_prop (PALETTE_WINDOW (palette),
				0xffffffff);
	}
}

static void
_gnome_hints (Palette *palette, gint wm_type)
{
	wmutil_window_state_gnome (PALETTE_WINDOW (palette),
			WIN_STATE_STICKY);
	wmutil_window_hints_gnome (PALETTE_WINDOW (palette),
			(WIN_HINTS_SKIP_FOCUS | WIN_HINTS_SKIP_TASKBAR |
			 WIN_HINTS_SKIP_WINLIST));
	wmutil_window_layer_gnome (PALETTE_WINDOW (palette),
			WIN_LAYER_ABOVE_DOCK);
}

int
main (int argc, char *argv[])
{
	GtkWidget *managed_win;
	/* force to EUC-JP */
#ifdef HAVE_SETENV
	setenv ("LC_ALL",JA_LOCALE ,True);
#elif HAVE_PUTENV
	putenv ("LC_ALL=" JA_LOCALE);
#endif
	gtk_set_locale ();

#ifdef ENABLE_NLS
	bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
#ifdef USE_GTK2
	bind_textdomain_codeset (PACKAGE, "UTF-8");
#endif
	textdomain (PACKAGE);
#endif

	gtk_init (&argc, &argv);
	_register_xpm ();
	managed_win = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_widget_realize (managed_win);
#if 1
	if (xaux_ext_init_classes (GDK_WINDOW_XDISPLAY (managed_win->window),
				&xaux_class,
				GDK_WINDOW_XWINDOW (managed_win->window)) == False) {
		exit (1);
	}
#else
	if (xaux_ext_init_classes (GDK_WINDOW_XDISPLAY (SKK_PALETTE_WINDOW (g_palette)),
				&xaux_class, 
				GDK_WINDOW_XWINDOW (SKK_PALETTE_WINDOW (g_palette))) == False) {
		exit (1);
	}
#endif
#if 1
	g_palette = skk_palette_new ();
	gtk_widget_set_name (SKK_PALETTE_WIDGET (g_palette), "skk-palette");
#endif
#if 0
	palette_show (SKK_PALETTE (g_palette));
	gdk_window_add_filter (SKK_PALETTE_WINDOW (g_palette), _ext_event_handler, 0);
#endif
	gdk_window_add_filter (managed_win->window, _ext_event_handler, 0);

#if 0
	_palette_reposition (g_palette);
#endif
	/* set Error Handler */
	XSetErrorHandler (palette_x_error);
	gtk_main ();
	return 0;
}
