/*************************************************************************************************/
/*!
   	@file		imagefileloader_png.h
	@author 	Fanzo
 	@date 		2008/3/25
*/
/*************************************************************************************************/
#pragma		once

///////////////////////////////////////////////////////////////////////////////////////////////////
//include files
#include	"png.h"
#include	"iFace/iLoaderImagefile.h"

#pragma pack( push , 8 )		//set align

namespace icubic
{

///////////////////////////////////////////////////////////////////////////////////////////////////
// preprocessor deifne

///////////////////////////////////////////////////////////////////////////////////////////////////
// type define

///////////////////////////////////////////////////////////////////////////////////////////////////
// classes define

/**************************************************************************************************
"LoaderImagefile_png" class 
**************************************************************************************************/
class LoaderImagefile_png : 
	virtual public object_base , 
	public ILoaderImagefile
{
// query
	query_begin();
	iface_hook( ILoaderImagefile , ILoaderImagefile_IID )
	query_end( object_base );

// member class
private:
	struct TABLE
	{
		int		m_readbit;
		bool	m_paletteflag;
		bool	m_transpaletteflag;
	};
	
// variable member
private:
	int									m_bitdepth;
	int									m_colortype;

	bool								m_paletteflag;
	Array< rgba >						m_palette;

	bool								m_transflag;
	Array< uint8 >						m_transpalette;
	rgba								m_transcolor;
	
	Straightdata< Array< uint8 > >		m_image;

	isize								m_size;
	DPI									m_dpi;
	Array< rgba >						m_rgbalist;

// static functions
private:
//=================================================================================================
//!	callback to read data.
//!	@retval			---
//-------------------------------------------------------------------------------------------------
static
void PNGAPI PngReadStreamStatic
		(
		png_struct		*ppng ,
		png_bytep		buf ,
		png_size_t		size
		)
{
	iFileStreamRead	*pfile = ( iFileStreamRead* )png_get_io_ptr( ppng );
	if( size != ( *pfile )->Read( buf , 1 , size , Big_EndianType ) )
	{
		::memset( buf , 0 , size );
	}
}
// private functions
private:

//=================================================================================================
//!	GetBitTable
//!	@retval			---
//-------------------------------------------------------------------------------------------------
int GetBitTable
		(
		int		p
		)const
{
	static 
	const int	bittable[] = 
	{
		-1 , 0 , 1 , -1 , 2 , -1 , -1 , -1 , 3
	};
	return bittable[ p ];
}
//=================================================================================================
//!	GetTable
//!	@retval			---
//-------------------------------------------------------------------------------------------------
TABLE GetTable
		(
		int		p1 , 
		int		p2
		)const
{
	static 
	const TABLE	table[][ 4 ] = 
	{
		{
			1 , true , false , 
			2 , true , false , 
			4 , true , false , 
			8 , true , false , 
		} , 
		{
			1 , true , false , 
			2 , true , false , 
			4 , true , false , 
			8 , true , false , 
		} , 
		{
			-1 , false , false , 
			-1 , false , false , 
			-1 , false , false , 
			8 * 3 , false , false , 
		} , 
		{
			1 , true , false , 
			2 , true , false , 
			4 , true , false , 
			8 , true , false , 
		},
		{
			-1 , false , false , 
			-1 , false , false , 
			-1 , false , false , 
			-1 , false , false , 
		} , 
		{
			1 * 2 , true , true , 
			2 * 2 , true , true , 
			4 * 2 , true , true , 
			8 * 2 , true , true , 
		},
		{
			-1 , false , false , 
			-1 , false , false , 
			-1 , false , false , 
			8 * 4 , false , false , 
		} , 
		{
			1 * 2 , true , true , 
			2 * 2 , true , true , 
			4 * 2 , true , true , 
			8 * 2 , true , true , 
		}
	};
	return table[ p1 ][ p2 ];
}
//=================================================================================================
//!	free
//!	@retval			---
//-------------------------------------------------------------------------------------------------
void Free()
{
	m_bitdepth		= 0;
	m_colortype		= 0;
	m_paletteflag	= false;
	m_palette.Resize( 0 );
	m_transflag		= false;
	m_transpalette.Resize( 0 );
	m_transcolor	= rgba();
	m_image.Resize( 0 );
	m_size			= isize();
	m_rgbalist.Resize( 0 );
}
//=================================================================================================
//!	check whether it's corresponding.
//!	@retval			---
//-------------------------------------------------------------------------------------------------
bool IsOK()const
{
	if( m_bitdepth != 1 && m_bitdepth !=2 && m_bitdepth != 4 && m_bitdepth != 8 )
		return false;
	int	depmode = GetBitTable( m_bitdepth );
	if( depmode == -1 )
		return false;
	int	typemode = m_colortype & 0x7;
	if( GetTable( typemode , depmode ).m_readbit == -1 )
		return false;
	return true;
}
//=================================================================================================
//!	create palette
//!	@retval			---
//-------------------------------------------------------------------------------------------------
void CreatePalette
		(
		png_struct		*ppng,
		png_info		*pinfo , 
		int				bitdepth , 
		int				colortype
		)
{
	int		i;
	m_paletteflag = false;
	if( colortype == PNG_COLOR_TYPE_GRAY ) 
	{		
		if( bitdepth == 4 ) 
		{
			m_palette.Resize( 16 );
			for( i = 0 ; i < 16 ; i++ ) 
			{
				m_palette[i].r	= i * 255 / 15;
				m_palette[i].g	= i * 255 / 15;
				m_palette[i].b	= i * 255 / 15;
			}
			m_paletteflag = true;
		}
		else if( bitdepth == 8 ) 
		{
			m_palette.Resize( 256 );
			for( i = 0 ; i < 256 ; i++ ) 
			{
				m_palette[i].r	= i;
				m_palette[i].g	= i;
				m_palette[i].b	= i;
			}
			m_paletteflag = true;
		}
		else if( bitdepth ==2 ) 
		{
			m_palette.Resize( 4 );
			m_palette[0].r	= 0;
			m_palette[0].g	= 0;
			m_palette[0].b	= 0;
			m_palette[1].r	= 85;
			m_palette[1].g	= 85;
			m_palette[1].b	= 85;
			m_palette[2].r	= 170;
			m_palette[2].g	= 170;
			m_palette[2].b	= 170;
			m_palette[3].r	= 255;
			m_palette[3].g	= 255;
			m_palette[3].b	= 255;
			m_paletteflag = true;
	    }
		else if( bitdepth == 1 ) 
		{
			m_palette.Resize( 2 );
			m_palette[0].r	= 0;
			m_palette[0].g	= 0;
			m_palette[0].b	= 0;
			m_palette[1].r	= 255;
			m_palette[1].g	= 255;
			m_palette[1].b	= 255;
	    }
    }
	int			num_palette;
	png_colorp	palette;
	if ( png_get_PLTE( ppng , pinfo , &palette , &num_palette ) )	
	{
		m_palette.Resize( num_palette );
		for( i = 0 ; i < num_palette ; i++ ) 
		{
			m_palette[i].r	= palette[i].red;
			m_palette[i].g	= palette[i].green;
			m_palette[i].b	= palette[i].blue;
		}
		m_paletteflag = true;
	}
}
//=================================================================================================
//!	create trans palette.
//!	@retval			---
//-------------------------------------------------------------------------------------------------
void CreateTransPalette
		(
		png_struct		*ppng,
		png_info		*pinfo
		)
{
	png_bytep		trans;
	int				transcount;
	png_color_16p	trans16;

	m_transflag = ( 0 == png_get_tRNS( ppng , pinfo , &trans , &transcount , &trans16 ) ) ? false : true;
	if( m_transflag == true )
	{
		m_transpalette.Resize( transcount );
		int			off;
		for( off = 0 ; off < transcount ; off++ )
		{
			m_transpalette[ off ] = trans[ off ];
			m_transcolor.r = ( unsigned char )trans16->red;
			m_transcolor.g = ( unsigned char )trans16->green;
			m_transcolor.b = ( unsigned char )trans16->blue;
		}
	}
	else
		m_transpalette.Resize( 0 );
}
//=================================================================================================
//!	read png image
//!	@retval			---
//-------------------------------------------------------------------------------------------------
void ReadPngImage
		(
		png_struct		*ppng,
		png_info		*pinfo
		)
{
//	png_set_bgr( ppng );
	if( m_size.width == 0 )
		return;
	int		readbit = GetTable( m_colortype & 0x7 , GetBitTable( m_bitdepth ) ).m_readbit;

	// create buffer.
	m_image.Resize( m_size.height );

	for( int y = 0 ; y < m_size.height ; y++ )
	{
		m_image[ y ].Resize( ( m_size.width * readbit + 7 ) / 8 );
		png_read_row( ppng , &m_image[ y ][ 0 ] , NULL );
	}
}
//=================================================================================================
//!	get pixel
//!	@retval			---
//-------------------------------------------------------------------------------------------------
rgba GetPixelFromBit
		(
		int			y ,				//!< [in]  Y
		int			byteoff ,		//!< [in]  byte offset.
		int			bitoff			//!< [in]  bit offset.
		)const
{
	long		filter	= ( 1 << m_bitdepth ) - 1;
	TABLE		state	= GetTable( m_colortype & 0x7 , GetBitTable( m_bitdepth ) );
	rgba		color( 255 , 255 , 255 , 255 );
	if( state.m_paletteflag == true )
	{
		int palid = ( m_image[ y ][ byteoff ] >> ( 8 - bitoff - m_bitdepth ) ) & filter;
		color = GetPaletteColor( palid );
		if( m_transflag == true )
			color.a = GetTransPaletteColor( palid );
		bitoff	+= m_bitdepth;
	}
	if ( state.m_transpaletteflag == true )
	{
		int transpalid = ( m_image[ y ][ byteoff ] >> ( 8 - bitoff - m_bitdepth ) ) & filter;
		color.a = GetTransPaletteColor( transpalid );
	}
	if( state.m_paletteflag == false )
	{
		if( state.m_readbit == 24 )
		{
			color.r = m_image[ y ][ byteoff ];
			color.g = m_image[ y ][ byteoff + 1 ];
			color.b = m_image[ y ][ byteoff + 2 ];
			color.a = 255;
		}
		else if( state.m_readbit == 32 )
		{
			color.r = m_image[ y ][ byteoff ];
			color.g = m_image[ y ][ byteoff + 1 ];
			color.b = m_image[ y ][ byteoff + 2 ];
			color.a = m_image[ y ][ byteoff + 3 ];
		}
	}
	return color;
}
//=================================================================================================
//!	get transpalette.
//!	@retval			---
//-------------------------------------------------------------------------------------------------
unsigned char GetTransPaletteColor
		(
		int		off
		)const
{
	if( m_transflag == false )
		return 255;
	if( off < 0 || off >= m_transpalette.GetDatanum() )
		return 255;
	return m_transpalette[ off ];
}
//=================================================================================================
//!	get palette color.
//!	@retval			---
//-------------------------------------------------------------------------------------------------
rgba GetPaletteColor
		(
		int		off
		)const
{
	rgba	color;
	color.r = 255;
	color.g = 255;
	color.b = 255;
	color.a = 255;
	if( m_paletteflag == false )
		return color;
	if( off < 0 || off >= m_palette.GetDatanum() )
		return color;
	return m_palette[ off ];
}

// "ILoaderImagefile_png" interface functions
public:
//=================================================================================================
//!	Check image format.
//!	@retval			---
//-------------------------------------------------------------------------------------------------
bool cb_call CheckImagefileSign
		(
		iFileStreamRead&	reader
		)
{
	int64		save	= reader->GetPosition();

	// read sig
    png_byte	sig[ 8 ];
	if( 8 != reader->Read( sig , sizeof( png_byte ) , 8 , Big_EndianType ) )
	{
		reader->Seek( save );
		return false;
	}
	reader->Seek( save );

	return ( 0 == png_check_sig( sig , sizeof( sig ) * 8 ) ) ? false : true;
}
//=================================================================================================
//!	load
//!	@retval			---
//-------------------------------------------------------------------------------------------------
bool cb_call Load
		(
		iFileStreamRead&	reader
		)
{
	if( false == CheckImagefileSign( reader ) )
		return false;

	// save file pointer.
	int64	save	= reader->GetPosition();
		
	// free
	Free();

	// create object.
	png_struct	*ppng	= png_create_read_struct( PNG_LIBPNG_VER_STRING , NULL , NULL , NULL );
	if( ppng == 0 )
		return false;

	// create info.
	png_info	*pinfo	= png_create_info_struct( ppng );
	if( pinfo == 0 )
	{
		if( ppng != 0 )
			png_destroy_read_struct( &ppng , ( pinfo == 0 ) ? NULL : &pinfo , NULL );
		return false;
	}
	// set read function.
	png_set_read_fn( ppng , ( png_voidp )&reader , ( png_rw_ptr )PngReadStreamStatic );

	// get info.
	int		bitdepth = 0 , colortype = 0;
	{
		png_read_info( ppng , pinfo );
		png_uint_32		width , height;
		int				interlacetype , compressiontype , filtertype;
		png_get_IHDR( ppng , pinfo , &width , &height , &bitdepth , &colortype , &interlacetype , &compressiontype , &filtertype );
		if( interlacetype != PNG_INTERLACE_NONE )
		{
			if( ppng != 0 )
				png_destroy_read_struct( &ppng , ( pinfo == 0 ) ? NULL : &pinfo , NULL );
			reader->Seek( save );
			Free();
			return false;
		}
		m_size.width		= width;
		m_size.height		= height;
		m_bitdepth	= bitdepth;
		m_colortype	= colortype;
		if( pinfo->phys_unit_type == PNG_RESOLUTION_METER )
			m_dpi	= DPI( (float)pinfo->x_pixels_per_unit * 0.0254f , (float)pinfo->y_pixels_per_unit * 0.0254f );
		else
			m_dpi	= DPI();
	}
	// check whether it's corresponding.
	if( false == IsOK() )
	{
		if( ppng != 0 )
			png_destroy_read_struct( &ppng , ( pinfo == 0 ) ? NULL : &pinfo , NULL );
		reader->Seek( save );
		Free();
		return false;
	}
	// create palette.
	CreatePalette( ppng , pinfo , bitdepth , colortype );

	// create trans palette.
	CreateTransPalette( ppng , pinfo );

	// read image.
	ReadPngImage( ppng , pinfo );

	if( ppng != 0 )
		png_destroy_read_struct( &ppng , ( pinfo == 0 ) ? NULL : &pinfo , NULL );
	ppng	= 0;

	return true;
}
//=================================================================================================
//!	get image num.
//!	@retval			---
//-------------------------------------------------------------------------------------------------
int32 cb_call GetImageNum()
{
	if( m_image.GetDatanum() == 0 )
		return 0;
	return 1;
}
//=================================================================================================
//!	get dpi
//!	@retval			---
//-------------------------------------------------------------------------------------------------
DPI cb_call GetImageDPI
		(
		int32		imageoff
		)const
{
	return m_dpi;
}
//=================================================================================================
//!	get image size.
//!	@retval			---
//-------------------------------------------------------------------------------------------------
isize cb_call GetImageSize
		(
		int32		imageoff
		)
{
	if( imageoff != 0 )
		return isize();
	return m_size;
}
//=================================================================================================
//!	get bitmap
//!	@retval			---
//-------------------------------------------------------------------------------------------------
bool cb_call GetImage
		(
		int32		imageoff , 
		pixelformat	destformat , 
		void*		t_dest , 
		int			pitchbyte
		)
{
	if( m_image.GetDatanum() == 0 )
		return false;
	if( imageoff != 0 )
		return false;

	uint8*		dest	= ( uint8* )t_dest;
	TABLE		state	= GetTable( m_colortype & 0x7 , GetBitTable( m_bitdepth ) );
	int			y;
	for( y = 0 ; y < m_size.height ; y++ )
	{
		int		byteoff = 0;
		int		bitoff	= 0;
		int		x;	
		uint8*	p	= dest;
		for( x = 0 ; x < m_size.width ; x++ )
		{
			// get color.
			rgba	pix = GetPixelFromBit( y , byteoff , bitoff );
			to_pixel( destformat , p , pix );
			p += get_pixel_byte( destformat );

			// update bitoff.
			bitoff += state.m_readbit;
			byteoff += bitoff / 8;
			bitoff	%= 8;
		}
		dest	+= pitchbyte;
	}
	return true;
}

// protect functions
protected:
// public functions
public:
//=================================================================================================
//!	construct
//-------------------------------------------------------------------------------------------------
LoaderImagefile_png() : 
		m_bitdepth( 0 ) , 
		m_colortype( 0 ) , 
		m_paletteflag( false ) , 
		m_transflag( false )
{
}
};

///////////////////////////////////////////////////////////////////////////////////////////////////
// global variable define

///////////////////////////////////////////////////////////////////////////////////////////////////
// global functions define

};	//namespace

//using namespace icubic;		

#pragma pack( pop )			//release align
