#include "htlib2.h"
#include "Receiver.h"
#include "Header.h"
#include "Socket.h"
#include "HtLib_.h"

static void
prepareBuffer(HTLIB* o)
{
	LOGDBG("prepareBuffer");
	/* manage buffer for 100 Continue response */

	int len;
	if ((len=o->_read_index - o->_body_index)>0) {
		LOGDBG("memove %d:{%.20s...} to head",
			   o->_body_index, o->_rec_buffer);
		memmove(o->_rec_buffer, o->_rec_buffer + o->_body_index, len);
	}
	o->_header = NULL;
	o->_read_index = len;
	o->_body_index = 0;
	LOGDBG("read_index=%d", (int)o->_read_index);
}

static int
findBody(char* buffer, int size, BOOL req)
{
	LOGDBG("findBody(%s, %d)", buffer, size);
	BOOL lf = FALSE;
	int spcs = 0;
	BOOL first = TRUE;

	int i;
	for (i=0; i<size; i++) {
		switch (buffer[i]) {
		case ' ':
			spcs++;
			continue;
			
		case '\r':
			continue;

		case '\n':
			if (lf ||
				(req && first && spcs==1 /* HTTP0.9 */)) {
				buffer[i] = 0;
				if (i>0 && buffer[i-1]=='\r') {
					buffer[i-1] = 0;
				}
				return i+1;
			}
			lf = TRUE;
			first = FALSE;
			break;

		default:
			lf = FALSE;
			break;
		}
	}
	return 0;
}

static BOOL
manageHeaders(HTLIB* o, const HTLIB_Header* header_buffer, USHORT blen,
			  BOOL no_clen_means_0, HTLIB_ERROR* err)
{
	/*
	 * Connection:
	 * manage keeping-alive for the next time
	 */
	if ((o->remote_version==HTTP_VER_11 &&
		 M(Header,HasConnectionClose)(header_buffer, blen))||
		(o->remote_version==HTTP_VER_10 &&
		 M(Header,HasConnectionKeepAlive)(header_buffer, blen)==FALSE)) {
		/* "close" in HTTP1.1, or not "Keep-Alive" in HTTP1.0 */
		LOGINFO("CLR KEEP_ALIVE");
		FLAG_CLR(FLAG_KEEP_ALIVE, o->_flags);
	} else {
		/* can't be set TRUE,
		   because it may be already set FALSE when the request. */
	}

	/*
	 * Content-Length:
	 */
	o->_rec_content_transfered = 0;
	FLAG_CLR(FLAG_RECV_CHUNKED, o->_flags);

	int index;
	if ((index=M(Header,Find)(header_buffer, blen,
							  "Transfer-Encoding"))>=0) {
		if (strcasecmp(header_buffer[index].value, "chunked")!=0) {
			/* no other tokens allowed */
			LOGERR("Transfer-Encoding specified, but not chunked");
			*err = HTLIB_E_NOT_IMPLEMENTED;
			return FALSE;
		}
		FLAG_SET(FLAG_RECV_CHUNKED, o->_flags);
		LOGINFO("SET RECV_CHUNKED");

		/* (RFC2616 4.4.2) */
		
		o->_rec_content_length = 0; /* initialized before chunk */

	} else if ((index=M(Header,Find)(header_buffer, blen,
									 "Content-Length"))!=-1 &&
			   header_buffer[index].value != NULL) {
		if (sscanf(header_buffer[index].value, "%lld",
				   &o->_rec_content_length)==0) {
			*err = HTLIB_E_INVALID_SYNTAX;
			LOGERR("Content-Length cannot be recognized");
			return FALSE;
		}
		/* Content-Length was set properly */
		
	} else if (no_clen_means_0) {
		/* GET, HEAD, etc. */
		o->_rec_content_length = 0;
		
	} else if (FLAG_ISSET(FLAG_KEEP_ALIVE, o->_flags)==FALSE &&
			   FLAG_ISSET(FLAG_SERVER_MODE, o->_flags)==FALSE) {
		/* content end will be specified by close (RFC2616 4.4.2) */
		o->_rec_content_length = -1;

	} else {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("Content-Length not found when keep-alive");
		return FALSE;

	}

	return TRUE;
}

static int
parseHeader(HTLIB* o,
			HTLIB_Header* header_buffer, USHORT blen,
			HTLIB_NEEDS_FUNC needs,	 
			HTLIB_ERROR* err)
{
	LOGNOTICE("ParseHeader(%d)", blen);

	if (o->_header == NULL || o->_body_index==0) {
		*err = HTLIB_E_INVALID_STATE;
		LOGERR("didn't receive all headers yet");
		return -1;
	}
	return M(Header,Parse)(o,
						   header_buffer, blen,
						   needs,
						   err);
}

static void
proceedUntilSpace(HTLIB* o)
{
	int index = o->_body_index;
	while (index < o->_read_index && isspace(o->_rec_buffer[index])) {
		/* in RFC2616 4.1, only CRLF is specified... */
		index++;
	}
	if (o->_body_index < index) {
		int mlen = index - o->_body_index;
		memmove(o->_rec_buffer + o->_body_index,
				o->_rec_buffer + index,
				o->_read_index - index);
		o->_read_index -= mlen;
	}
	LOGDBG("read_index=%d", (int)o->_read_index);
}

static BOOL
waitAndRead(HTLIB* o, int timeout_millis, HTLIB_ERROR* err)
{
	if (M(Socket,Wait)(o, timeout_millis, HTLIB_W_READ,
					   M(HTLIB,CheckCanceled), err)==FALSE) {
		if (*err == HTLIB_E_TIMEOUT) {
			LOGINFO("Socket_Wait timeout");
		} else {
			LOGERR("Socket_Wait failed");
		}
		return FALSE;
	}
	
	int len = o->_rec_buffer_len - o->_read_index;
	if (len == 0) {
		*err = HTLIB_E_OUT_OF_MEMORY;
		LOGERR("buffer full");
		return FALSE;
	}
	switch ((len=M(Socket,Read)(o,
								o->_rec_buffer+o->_read_index,
								len,
								err))) {
	case -1:
		LOGERR("Socket_Read failed err=%d errno=%d", *err, o->system_errno);
		return FALSE;

	case 0:
		*err = HTLIB_E_DISCONNECTED;
		LOGERR("Socket disconnected", *err);
		return FALSE;

	default:
		o->_read_index += len;
		break; /* into the while-loop again */
	}
	LOGDBG("read_index=%d", (int)o->_read_index);
	return TRUE;
}

static BOOL
receiveMessage(HTLIB* o,
			   int timeout_millis,
			   HTLIB_Header* header_buffer, USHORT* blen,
			   HTLIB_NEEDS_FUNC needs,
			   HTLIB_ERROR* err)
{
	if (o->_header) {
		/* already receiveMessage returned non-NULL before */
		prepareBuffer(o);
	}
	while (TRUE) {
		/*
		 * First, check if the head-end can be found.
		 * (In case of 1XX, it may be already received.)
		 */

		proceedUntilSpace(o); /* RFC2616 4.1 */
		
		if ((o->_body_index=findBody(o->_rec_buffer, o->_read_index,
									 FLAG_ISSET(FLAG_SERVER_MODE, o->_flags)
									 ==FALSE))>0) {

			/* "CR LF CR LF" found */

			/* skip the first line */
			LOGDBG("check 1st line: (len=%d) %s",
					o->_read_index,  o->_rec_buffer);
			o->_header = strchr(o->_rec_buffer, '\n');
			if (o->_header == NULL) {
				/* HTTP0.9 */
				LOGWARN("it seemds like HTTP0.9...(%s)", o->_rec_buffer);
				*blen = 0;
				return TRUE;
			}
			*o->_header++ = 0;
			char* p;
			if ((p=strchr(o->_rec_buffer, '\r'))==NULL) {
				*p = 0;
			}

			int blen2;
			if ((blen2=parseHeader(o,
								   header_buffer, *blen,
								   needs,
								   err))==-1) {
				LOGERR("parseHeader failed err=%d", *err);
				return FALSE;
			}
			*blen = (USHORT)blen2;

			return TRUE;
		}

		if (o->_rec_buffer_len <= o->_read_index) {
			/* memory cannot be assigned any more */
			break;
		}

		/*
		 * Now try, to receive from network
		 */
		if (waitAndRead(o, timeout_millis, err)==FALSE) {
			return FALSE;
		}
	} /* while */

	*err = HTLIB_E_OUT_OF_MEMORY;
	LOGERR("ReceiveMessage: outof memory(%d >= %d)",
		   o->_read_index, o->_rec_buffer_len);
	return FALSE;
}

static USHORT
getVersion(const char* p, HTLIB_ERROR* err)
{
	ULONG maj;
	ULONG min;

	if (sscanf(p, "%lu.%lu", &maj, &min)!=2) {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("unrecoginzed HTTP version ?/?.");
		return 0;
	}
	
	USHORT ver = (USHORT)((maj << 8)&0xff00) | (min & 0x00ff);
	LOGNOTICE("remote HTTP-Ver: %hx", ver);
	return ver;
}

static BOOL
expectedBody(HTLIB_HANDLE o)
{
	if (FLAG_ISSET(FLAG_RECV_CHUNKED, o->_flags)) {
		LOGDBG("FLAG_RECV_CHUNKED");
		return TRUE;
	}
	
	if (o->_rec_content_length==0) {
		LOGDBG("length == 0");
		return FALSE;
	}
	if (o->_rec_content_length==-1) {
		LOGDBG("length == -1");
		return TRUE;
	}
	if (o->_rec_content_length <= o->_rec_content_transfered) {
		LOGDBG("length <= transfered");
		return FALSE;
	}
	LOGDBG("otherwise");
	return TRUE;
}

static BOOL
readBodyAway(HTLIB* o, int timeout_millis, HTLIB_ERROR* err)
{
	while (expectedBody(o)) {
		char dummy[1000];
		int len;
		if ((len=M(Receiver,ReceiveBody)(o, timeout_millis,
										 dummy, sizeof(dummy), err))==-1) {
			LOGNOTICE("readBodyAway failed err=%d", *err);
			return FALSE;
		}
		LOGINFO("dispose %d bytes of body", len);
		if (len==0) {
			break;
		}
	}
	return TRUE;
}
const char*
M(Receiver,ReceiveRequest)(HTLIB* o,
						   int timeout_millis,
						   const char** method,
						   HTLIB_Header* header_buffer, USHORT* blen,
						   HTLIB_NEEDS_FUNC needs,
						   HTLIB_ERROR* err)
{
	if (o->soc == -1) {
		/* already closed */
		*err = HTLIB_E_DISCONNECTED;
		LOGERR("already closed");
		return NULL;
	}
	if (o->remote_version == HTTP_VER_09) {
		LOGERR("HTTP0.9 is not capable of keep-alive");
		*err = HTLIB_E_HTTP09;
		return NULL;
	}
	if (readBodyAway(o, timeout_millis, err)==FALSE) {
		return NULL;
	}

	FLAG_SET(FLAG_SERVER_MODE, o->_flags);
	
	if (FLAG_ISSET(FLAG_KEEP_ALIVE, o->_flags)==FALSE) {
		LOGNOTICE("no more KEEP_ALIVE, quit.");
		*err = HTLIB_E_DISCONNECTED;
		return NULL;
	}

	if (receiveMessage(o,
					   timeout_millis,
					   header_buffer, blen,
					   needs,
					   err)==FALSE) {
		LOGERR("ReceiveMessage failed err=%d", *err);
		return NULL;
	}

	/* process request-line */

	LOGINFO("request-line: %s", o->_rec_buffer);

	/* method */
	char* saved;
	char* p = strtok_r(o->_rec_buffer, "\r\n\t ", &saved);
	if (p == NULL) {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("no space after method");
		return NULL;
	}
	*method = o->_rec_buffer;

	if (strcmp(*method, "HEAD")==0) {
		FLAG_SET(FLAG_HEAD_REQUEST, o->_flags);
	} else {
		FLAG_CLR(FLAG_HEAD_REQUEST, o->_flags);
	}
	
	/* url */
	char* url;
	if ((url=strtok_r(NULL, "\r\n\t ", &saved))==NULL) {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("no url");
		return NULL;
	}

	char* host = NULL;
	if (strncmp(url, "http://", 7)==0) {
		/* absoluteURI */
		host = url+7;

		char* path = strpbrk(host, " /");
		if (path == NULL) {
			int l = strlen(host);
			path = host+l;
		}
		memmove(url, host, path - host);
		url[path-host] = 0;
		host = url;
		if (*path == ' '||*path == 0) {
			path = "/";
		}
		url = path;
	}
	if (*url != '/') {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("unrecognized URL %s", p);
		return NULL;
	}

	/* p == " HTTP/1.?" */
	p = strtok_r(NULL, "\r\n\t ", &saved);
	if (p == NULL) {
		/* HTTP0.9 ? */
		if (strcmp(*method, "GET")!=0) {
			*err = HTLIB_E_INVALID_SYNTAX;
			LOGERR("HTTP0.9 but not GET (%s)", method);
			return NULL;
		}
		o->remote_version = HTTP_VER_09;
		return url;
	} else {
		if (strncmp(p, "HTTP/", 5)!=0) {
			*err = HTLIB_E_INVALID_SYNTAX;
			LOGERR("no HTTP/ (%.5s)", p);
			return NULL;
		}
		if ((o->remote_version=getVersion(p+5, err))==0) {
			return NULL;
		}
	}

	BOOL no_clen_means_0 = TRUE;
	if (strcmp(*method, "POST")==0 || strcmp(*method, "PUT")==0) {
		no_clen_means_0 = FALSE;
	}
	if (manageHeaders(o, header_buffer, *blen, no_clen_means_0,
					  err)==FALSE) {
		LOGERR("manageHeaders failed err=%d", *err);
		return FALSE;
	}

	if (host != NULL) {
		int i = M(Header,Find)(header_buffer, *blen, "Host");
		if (i != -1) {
			header_buffer[i].value = host;
		}
	}

	return url;
}

int
M(Receiver,ReceiveResponse)(HTLIB* o,
						 int timeout_millis,
						 HTLIB_Header* header_buffer, USHORT* blen,
						 HTLIB_NEEDS_FUNC needs,
						 HTLIB_ERROR* err)
{
	if (o->remote_version == HTTP_VER_09) {
		*err = HTLIB_E_FATAL;
		LOGERR("why HTTP_VER_09? already ReceiveRequest?");
		return -1;
	}

	if (readBodyAway(o, timeout_millis, err)==FALSE) {
		return -1;
	}

	FLAG_CLR(FLAG_SERVER_MODE, o->_flags);

	if (receiveMessage(o,
					   timeout_millis,
					   header_buffer, blen,
					   needs,
					   err)==FALSE) {
		LOGERR("ReceiveMessage failed err=%d", *err);
		return -1;
	}

	int status;

	LOGINFO("response-line: %.20s...", o->_rec_buffer);
	/*
	 * check HTTP version
	 */
	char* saved;
	char* p = strtok_r(o->_rec_buffer, "\r\n\t ", &saved);
	if (p == NULL) {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("no 1st line");
		return -1;
	}
	if (strncmp(p, "HTTP/", 5)!=0) {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("no HTTP/ (%.5s)", p);
		return -1;
	}
	if ((o->remote_version=getVersion(p+5, err))==0) {
		return -1;
	}
	if ((p=strtok_r(NULL, "\r\n\t ", &saved))==NULL) {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("no status");
		return -1;
	}
	if (sscanf(p, "%d", &status)!=1) {
		*err = HTLIB_E_INVALID_SYNTAX;
		LOGERR("unrecognized status: %d", p);
		return -1;
	}

	if ((int)(status/100)==1) {
		prepareBuffer(o);
		return status;
	}

	BOOL no_clen_means_0 = FALSE;
 	if (status == 204 || status == 304) {
		/* (RFC2616 4.4) */
		no_clen_means_0 = TRUE;
	}
	if (manageHeaders(o, header_buffer, *blen, no_clen_means_0,
					  err)==FALSE) {
		LOGERR("manageHeaders failed err=%d", *err);
		return FALSE;
	}
	if (FLAG_ISSET(FLAG_HEAD_REQUEST, o->_flags)||
		status == 204 || status == 304) {
		/* (RFC2616 4.4.1) */
		LOGINFO("HEAD, 204, 304 => length=0");
		o->_rec_content_length = 0;
		FLAG_CLR(FLAG_RECV_CHUNKED, o->_flags);
	}
	return status;
}

static void
memmoveRest(HTLIB* o)
{
	int restlen = o->_read_index - o->_body_index;
	LOGDBG("memmoveRest: %.5s...(len=%d) at %d of %.100s to the head",
		   o->_rec_buffer + o->_body_index, restlen,
		   o->_body_index,
		   o->_rec_buffer);
	
	memmove(o->_rec_buffer,
			o->_rec_buffer + o->_body_index,
			restlen);
	o->_read_index = restlen;
	o->_body_index = 0;
	LOGDBG("read_index=%d", (int)o->_read_index);
}
static void
manageBuffer(HTLIB* o)
{
	if (o->_rec_content_length>=0 &&
		o->_rec_content_length <= o->_rec_content_transfered) {

		LOGINFO("transfering %lld bytes completed.",
				o->_rec_content_transfered);

		/* for the next request */
		memmoveRest(o);
		o->_header = NULL;
	}
}

int
M(Receiver,ReceiveBody)(HTLIB* o,
				  int timeout_millis,
				  char* buffer,
				  USHORT buffer_len,
				  HTLIB_ERROR* err)
{
	/* ignore when HTTP0.9 */
	if (o->remote_version == HTTP_VER_09) {
		if (FLAG_ISSET(FLAG_SERVER_MODE, o->_flags)) {
			LOGINFO("HTTP0.9 request has no body");
			return 0;
		}
	} else if (expectedBody(o)==FALSE) {
		LOGINFO("no more body");
		manageBuffer(o);
		return 0;
	}

	if (M(HTLIB,CheckCanceled)(o, err)==FALSE) {
		return -1;
	}

	/*
	 * chunked consideration
	 */
	while (FLAG_ISSET(FLAG_RECV_CHUNKED, o->_flags)&&
		   o->_rec_content_transfered == o->_rec_content_length) {

		int rest = o->_read_index - o->_body_index;
		if (0 < rest) {
			int i;
			for (i=2; i<rest; i++) {
				if (o->_rec_buffer[o->_body_index + i]=='\n') {
					int clen = 0;
					o->_rec_buffer[o->_body_index + i] = 0;
					sscanf(o->_rec_buffer + o->_body_index, "%x", &clen);
					o->_rec_content_length += clen;
					o->_body_index += i + 1;
					break;
				}
			}
			if (i<rest) {
				/* chunk size was found */
				/* last CRLF will be ignored in the next receive */
				break;
			}
		}
		/* chunk size not found */
		if (waitAndRead(o, timeout_millis, err)==FALSE) {
			return -1;
		}
	}
	
	/*
	 * Connection-Length consideration
	 */
	long long body_rest;
	if (o->_rec_content_length !=-1 &&
		(body_rest = o->_rec_content_length
		 - o->_rec_content_transfered) < buffer_len) {
		LOGINFO("buffer_len shortened (%d => %d)",
				buffer_len, body_rest);
		buffer_len = body_rest;
		if (buffer_len == 0) {
			/* reached the end */
			FLAG_CLR(FLAG_RECV_CHUNKED, o->_flags);
			LOGINFO("CLR RECV_CHUNKED");
			manageBuffer(o);
			return 0;
		}
	}
	/*
	 * First, check if some bytes already read.
	 */
	if (o->_body_index < o->_read_index) {
		
		int len = o->_read_index - o->_body_index;
		if (buffer_len < len) {
			len = buffer_len;
		}
		memcpy(buffer, o->_rec_buffer + o->_body_index, len);
		o->_body_index += len;
		o->_rec_content_transfered += len;
		manageBuffer(o);
		return len;
	}

	/*
	 * Now try to read from the socket.
	 */
	{
		int len;

		if (M(Socket,Wait)(o, timeout_millis, HTLIB_W_READ,
						   M(HTLIB,CheckCanceled),
						   err)==FALSE) {
			return -1;
		}
		
		len = M(Socket,Read)(o, buffer, buffer_len, err);
		if (len>0) {
			o->_rec_content_transfered += len;
		}
		manageBuffer(o);
		return len;
	}
}


