//  -*- buffer-file-coding-system: sjis-unix -*-


#include "Parser.h"
#include <cppunit/extensions/HelperMacros.h>
#include <stdio.h>

namespace Houken {

    class Scan_Test : public CppUnit::TestFixture {
        CPPUNIT_TEST_SUITE(Scan_Test);
        CPPUNIT_TEST(test_Scan);
        CPPUNIT_TEST(test_sjis);
        CPPUNIT_TEST(test_euc);
        CPPUNIT_TEST(test_skip);
        CPPUNIT_TEST_SUITE_END();
    public:
        Scan_Test(void) {}
        
        void setUp(void)
        {
        }

        void tearDown(void)
        {
            Parser::finalize();
        }

        void test_Scan(void)
        {
            StringInputBuffer inp("123");
            gpInp = &inp;
            Parser::initialize();

            CPPUNIT_ASSERT_EQUAL(0, (int) inp.getPos());
            
            Parser* p = new Parser();
            
            CPPUNIT_ASSERT(p->_parse_char(L'1')->isValidTree());
            CPPUNIT_ASSERT_EQUAL(1, (int) inp.getPos());
            CPPUNIT_ASSERT(! p->_parse_char(L'1')->isValidTree());
            CPPUNIT_ASSERT_EQUAL(1, (int) inp.getPos());
            
            CPPUNIT_ASSERT(p->_parse_char(L'2')->isValidTree());
            CPPUNIT_ASSERT_EQUAL(2, (int) inp.getPos());
            CPPUNIT_ASSERT(! p->_parse_char(L'2')->isValidTree());
            CPPUNIT_ASSERT_EQUAL(2, (int) inp.getPos());
            
            inp.setPos(0);
            CPPUNIT_ASSERT_EQUAL(0, (int) inp.getPos());
            CPPUNIT_ASSERT(p->_parse_string("12")->isValidTree());
            CPPUNIT_ASSERT_EQUAL(2, (int) inp.getPos());
            CPPUNIT_ASSERT(! p->_parse_string("12")->isValidTree());
            CPPUNIT_ASSERT_EQUAL(2, (int) inp.getPos());
            
            inp.setPos(0);
            CPPUNIT_ASSERT(! p->_parse_string("13")->isValidTree());
            CPPUNIT_ASSERT_EQUAL(0, (int) inp.getPos());
            
            CPPUNIT_ASSERT(p->_parse_string("12")->isValidTree());
            CPPUNIT_ASSERT_EQUAL(2, (int) inp.getPos());
            CPPUNIT_ASSERT(p->_parse_anyChar()->isValidTree());
            CPPUNIT_ASSERT_EQUAL(3, (int) inp.getPos());
            CPPUNIT_ASSERT(! p->_parse_anyChar()->isValidTree());
            CPPUNIT_ASSERT_EQUAL(3, (int) inp.getPos());
            CPPUNIT_ASSERT(! p->_parse_anyChar()->isValidTree());
            CPPUNIT_ASSERT_EQUAL(3, (int) inp.getPos());
            
            inp.setPos(0);
            CPPUNIT_ASSERT(p->_parse_rangeChar(L'1', L'2')->isValidTree());
            CPPUNIT_ASSERT_EQUAL(1, (int) inp.getPos());
            CPPUNIT_ASSERT(p->_parse_rangeChar(L'1', L'2')->isValidTree());
            CPPUNIT_ASSERT_EQUAL(2, (int) inp.getPos());
            CPPUNIT_ASSERT(! p->_parse_rangeChar(L'1', L'2')->isValidTree());
            CPPUNIT_ASSERT_EQUAL(2, (int) inp.getPos());
        }

        void test_sjis(void)
        {
            StringInputBuffer inp("1\202\1202\202\1213\202\122\\1");
            gpInp = &inp;
            Parser::initialize();
#ifdef __CYGWIN__
            CPPUNIT_ASSERT(InputBuffer::setLocale("C-SJIS") != NULL);
#else
#ifdef WIN32
            CPPUNIT_ASSERT(InputBuffer::setLocale("japanese_japan.932") != NULL);
#else
            CPPUNIT_ASSERT(InputBuffer::setLocale("ja_JP.SJIS") != NULL);
#endif
#endif
            CPPUNIT_ASSERT_EQUAL(0, (int) inp.getPos());
            Parser* p = new Parser();
            wchar_t wc, wc2;
            size_t len;
            mbstate_t mbs;
            memset(&mbs, 0, sizeof(mbstate_t));

            CPPUNIT_ASSERT(p->_parse_char(L'1')->isValidTree());
            CPPUNIT_ASSERT_EQUAL(1, (int) inp.getPos());
            len = mbrtowc(&wc, "\202\120", 2, &mbs);
            CPPUNIT_ASSERT_EQUAL(2, (int)len);
            CPPUNIT_ASSERT(p->_parse_char(wc));
            CPPUNIT_ASSERT_EQUAL(3, (int) inp.getPos());

            CPPUNIT_ASSERT(p->_parse_anyChar()->isValidTree());
            CPPUNIT_ASSERT_EQUAL(4, (int) inp.getPos());
            len = mbrtowc(&wc, "\202\121", 2, &mbs);
            CPPUNIT_ASSERT_EQUAL(2, (int)len);
            CPPUNIT_ASSERT(p->_parse_anyChar()->isValidTree());
            CPPUNIT_ASSERT_EQUAL(6, (int) inp.getPos());

            CPPUNIT_ASSERT(p->_parse_rangeChar(L'1', L'5')->isValidTree());
            CPPUNIT_ASSERT_EQUAL(7, (int) inp.getPos());
            len = mbrtowc(&wc, "\202\120", 2, &mbs);
            CPPUNIT_ASSERT_EQUAL(2, (int)len);
            len = mbrtowc(&wc2, "\202\124", 2, &mbs);
            CPPUNIT_ASSERT_EQUAL(2, (int)len);
            CPPUNIT_ASSERT(p->_parse_rangeChar(wc,wc2)->isValidTree());
            CPPUNIT_ASSERT_EQUAL(9, (int) inp.getPos());

            // Linuxł 92(L'\\')ł͂Ȃ165AĂ邽ߎs 
            //len = mbrtowc(&wc, "\\1", 2, &mbs);
            //CPPUNIT_ASSERT_EQUAL(1, (int)len);
            //CPPUNIT_ASSERT(wc == L'\\');
            

            wc = inp.getChar();
            CPPUNIT_ASSERT(wc == L'\\');
            wc = inp.getChar();
            CPPUNIT_ASSERT(wc == L'1');
        }

        void test_euc(void)
        {
            StringInputBuffer inp("1\243\2612\243\2623\243\263\\1");
            gpInp = &inp;
            Parser::initialize();
#ifdef __CYGWIN__
            CPPUNIT_ASSERT(InputBuffer::setLocale("C-EUCJP") != NULL);
#else
#ifdef WIN32
            CPPUNIT_ASSERT(InputBuffer::setLocale("japanese_japan.20932") != NULL);
#else
            CPPUNIT_ASSERT(InputBuffer::setLocale("ja_JP.eucjp") != NULL);
#endif
#endif
            CPPUNIT_ASSERT_EQUAL(0, (int) inp.getPos());
            Parser* p = new Parser();
            wchar_t wc, wc2;
            size_t len;
            mbstate_t mbs;
            memset(&mbs, 0, sizeof(mbstate_t));

            CPPUNIT_ASSERT(p->_parse_char(L'1')->isValidTree());
            CPPUNIT_ASSERT_EQUAL(1, (int) inp.getPos());
            len = mbrtowc(&wc, "\243\261", 2, &mbs);
            CPPUNIT_ASSERT_EQUAL(2, (int)len);
            CPPUNIT_ASSERT(p->_parse_char(wc)->isValidTree());
            CPPUNIT_ASSERT_EQUAL(3, (int) inp.getPos());

            CPPUNIT_ASSERT(p->_parse_anyChar()->isValidTree());
            CPPUNIT_ASSERT_EQUAL(4, (int) inp.getPos());
            len = mbrtowc(&wc, "\243\262", 2, &mbs);
            CPPUNIT_ASSERT_EQUAL(2, (int)len);
            CPPUNIT_ASSERT(p->_parse_anyChar()->isValidTree());
            CPPUNIT_ASSERT_EQUAL(6, (int) inp.getPos());

            CPPUNIT_ASSERT(p->_parse_rangeChar(L'1', L'5')->isValidTree());
            CPPUNIT_ASSERT_EQUAL(7, (int) inp.getPos());
            len = mbrtowc(&wc, "\243\261", 2, &mbs);
            CPPUNIT_ASSERT_EQUAL(2, (int)len);
            len = mbrtowc(&wc2, "\243\265", 2, &mbs);
            CPPUNIT_ASSERT_EQUAL(2, (int)len);
            CPPUNIT_ASSERT(p->_parse_rangeChar(wc,wc2)->isValidTree());
            CPPUNIT_ASSERT_EQUAL(9, (int) inp.getPos());

            wc = inp.getChar();
            CPPUNIT_ASSERT(wc == L'\\');
            wc = inp.getChar();
            CPPUNIT_ASSERT(wc == L'1');
        }


        void _test_skip_sub(const char* str,
                            int ident,int low,int cap,int up)
        {
            StringInputBuffer inp(str);
            CPPUNIT_ASSERT_EQUAL(0, (int) inp.getPos());
            if (ident > 0) {
                CPPUNIT_ASSERT(inp.skip_ident());
                CPPUNIT_ASSERT_EQUAL(ident, (int) inp.getPos());
            } else {
                CPPUNIT_ASSERT(! inp.skip_ident());
                CPPUNIT_ASSERT_EQUAL(0, (int) inp.getPos());
            }
            inp.setPos(0);
            if (low > 0) {
                CPPUNIT_ASSERT(inp.skip_identL());
                CPPUNIT_ASSERT_EQUAL(low, (int) inp.getPos());
            } else {
                CPPUNIT_ASSERT(! inp.skip_identL());
                CPPUNIT_ASSERT_EQUAL(0, (int) inp.getPos());
            }
            inp.setPos(0);
            if (cap > 0) {
                CPPUNIT_ASSERT(inp.skip_identC());
                CPPUNIT_ASSERT_EQUAL(cap, (int) inp.getPos());
            } else {
                CPPUNIT_ASSERT(! inp.skip_identC());
                CPPUNIT_ASSERT_EQUAL(0, (int) inp.getPos());
            }
            inp.setPos(0);
            if (up > 0) {
                CPPUNIT_ASSERT(inp.skip_identU());
                CPPUNIT_ASSERT_EQUAL(up, (int) inp.getPos());
            } else {
                CPPUNIT_ASSERT(! inp.skip_identU());
                CPPUNIT_ASSERT_EQUAL(0, (int) inp.getPos());
            }                
        }

        void test_skip(void)
        {
            StringInputBuffer inp("");
            gpInp = &inp;
            Parser::initialize();

            //              012345678901234567
            _test_skip_sub("low_abc123_45zzABC+-", 18, 18, 0, 0);
            _test_skip_sub("CapAbc123deF_g4_()", 16, 0, 16, 0);
            _test_skip_sub("UPPER_12_ABC FOO", 12, 0, 0, 12);
            _test_skip_sub("_under_9AB==1", 10, 10, 0, 0);
            _test_skip_sub("_UNDER_9AB==1", 10, 10, 0, 0);
            _test_skip_sub("azAZ0_9$",    7, 7, 0, 0);
            _test_skip_sub("_0Zza#",      5, 5, 0, 0);
            _test_skip_sub("_ABCdef",     7, 7, 0, 0);
            _test_skip_sub("FooBar09",    8, 0, 8, 0);
            _test_skip_sub("BAR_BAZ_9$",  9 ,0, 0, 9);
            _test_skip_sub("BAR_BAZ_9z",  10,0, 10,0);
            _test_skip_sub("Bar_BAZ_0$",  9 ,0, 9, 0);
            _test_skip_sub("1290afAFgh",  0, 0, 0, 0);
        }

    };
    CPPUNIT_TEST_SUITE_REGISTRATION(Scan_Test);
}
