
#ifndef _PTRARRAY_H_
#define _PTRARRAY_H_

#include <string.h>
#include "sysDep.h"

namespace Houken {

    // simple array of T*
    template <typename T> class PtrArray {
    protected:
        int     _capacity;
        int     _size;
        T**     _array;
    public:
        static void* operator new(size_t size) { return sys_alloc((u32)size); }
        static void operator delete(void* p) { sys_free(p); }

        PtrArray(int initCapacity = 0)
            : _size(0)
        {
            _capacity = initCapacity;
            if (initCapacity > 0)
                _array = sys_allocT<T*>(_capacity);
            else
                _array = NULL;
        }

        PtrArray(T* t)
            : _capacity(1), _size(1)
        {
            _array = sys_allocT<T*>(_capacity);
            _array[0] = t;
        }

        ~PtrArray() {
            sys_free(_array);
        }

        int size(void) { return _size; }

        void clean(void) { _size = 0; }

        T* get(int idx) {
            sys_assert((idx >= 0) && (idx < _size));
            return _array[idx];
        }

        PtrArray* add(T* t) {
            if (t == NULL) return this;
            if (_size + 1 >= _capacity)
                resize(_capacity * 2);
            _array[_size++] = t;
            return this;
        }

        void insert(int idx, T* t) {
            if (_size + 1 >= _capacity)
                resize(_capacity * 2);
            for (int i = idx; i < _size; i++)
                _array[i+1] = _array[i];
            _array[idx] = t;
            ++ _size;
        }

        PtrArray* remove(int idx) {
            sys_assert((idx >= 0) && (idx < _size));
            delete _array[idx];
            _array[idx] = NULL;
            return this;
        }
            
        T* extract(int idx) {
            sys_assert((idx >= 0) && (idx < _size));
            T* d = _array[idx];
            _array[idx] = NULL;
            return d;
        }
            
        PtrArray* shrink(void) {
            int n = 0;
            for (int i = 0; i < _size; i++) {
                if (_array[i] != NULL) {
                    if (n != i)
                        _array[n] = _array[i];
                    ++n;
                }
            }
            _size = n;
            return this;
        }

        void resize(int newCapacity) {
            if (newCapacity < 8)
                newCapacity = 8;
            if (newCapacity <= _capacity)
                return;
            _array = sys_reallocT<T*>(_array, newCapacity);
            _capacity = newCapacity;
        }

    };


    template <typename T> class PtrArrayIterator {
    protected:
        PtrArray<T>*    _arr;
        int             _idx;
        int             _size;
    public:
        PtrArrayIterator(PtrArray<T>* arr)
            : _arr(arr)
        {
            _idx = 0;
            _size = arr->size();
        }

        bool hasMore(void)
        {
            return _idx < _size;
        }

        T* get(void)
        {
            if (_idx < _size)
                return _arr->get(_idx);
            return NULL;
        }
            
        T* next(void)
        {
            if (_idx < _size)
                return _arr->get(_idx++);
            return NULL;
        }
            
        T* extract_next(void)
        {
            if (_idx < _size)
                return _arr->extract(_idx++);
            return NULL;
        }
            
        void rewind(void)
        {
            _idx = 0;
        }
    };

}


#endif /* _PTRARRAY_H_ */
