
#ifndef _PARSER_H_
#define _PARSER_H_

#include "InputBuffer.h"
#include "PtrArray.h"
#include "SyntaxTree.h"
#include "Memoize.h"


namespace Houken {

    // ========== base parser ==========
    class Parser {
        friend class Scan_Test;                 // for unittest

    public:
        static void* operator new(size_t size) { return sys_alloc((u32)size); }
        static void operator delete(void* p) { sys_free(p); }

        Parser(void);
        virtual ~Parser() {}

        virtual SyntaxTree* createSyntaxTree(u32 startPos, PtrArray<SyntaxTree>* childs = NULL);
        virtual SyntaxTree* createSyntaxTree(u32 startPos, u32 endPos, PtrArray<SyntaxTree>* childs = NULL);

        virtual SyntaxTree* parse(void) { return _FATAL_PARSER_ERROR; }
        virtual const char* name(void) { return NULL; }
        virtual void removeMemo(SyntaxTree*) {}

        virtual void addSequenceToParent(PtrArray<Parser>* parentSeq);
        virtual void addSyntaxTreeToParent(PtrArray<SyntaxTree>* arr, SyntaxTree* st);


        // initialize static parser settings
        static void initialize(int initHolderCapacity = 100);

        // delete everything
        static void finalize(void);

        // use memoization
        static void setUseMemoize(bool b) { _bUseMemoize = b; }

        // print intermediate parsing result for debug
        // 0 = none, 1 = success only, >2 = detail
        static void setPrintIntermediateLevel(int level) { _printIntermediateLevel = level; }

        static bool cmpStr(SyntaxTree* st, const char* str);


        Parser* seq(Parser* p);         // create Parser: Sequence
        Parser* choice(Parser* p);      // create Parser: Ordered Choice
        Parser* star(void);             // create Parser: Zero or More
        Parser* plus(void);             // create Parser: One or More
        Parser* optional(void);         // create Parser: Zero or One
        Parser* andPred(void);          // create Parser: And-predicate
        Parser* notPred(void);          // create Parser: Not-predicate

        Parser* noTree(void);           // create Parser: not create SyntaxTree

    public:
        static const int        MAX_MANY = 9999; // max limit n of _parse_m2n 

    protected:
        SyntaxTree* _parse_sequence(PtrArray<Parser>* ps);
        SyntaxTree* _parse_orderedChoice(PtrArray<Parser>* ps);
        SyntaxTree* _parse_m2n(Parser* p, int m, int n);
        SyntaxTree* _parse_andPredicate(Parser* p);
        SyntaxTree* _parse_notPredicate(Parser* p);
        PtrArray<SyntaxTree>* _parse_infix_common(Parser* exp, Parser* op, bool rep);
        static PtrArray<SyntaxTree>* const _INFIX_COMMON_FATAL_ERROR;
        SyntaxTree* _parse_op_infixl(Parser* my, Parser* exp, Parser* op);
        SyntaxTree* _parse_op_infixr(Parser* my, Parser* exp, Parser* op);
        SyntaxTree* _parse_op_infixn(Parser* my, Parser* exp, Parser* op);
        SyntaxTree* _parse_op_prefix(Parser* my, Parser* exp, Parser* op, bool allowRepeat = true);
        SyntaxTree* _parse_op_postfix(Parser* my, Parser* exp, Parser* op, bool allowRepeat = true);
        SyntaxTree* _parse_op_ternary(Parser* my, Parser* exp, Parser* op1, Parser* op2);
        SyntaxTree* _parse_noTree(Parser* p);
        SyntaxTree* _parse_string(const char* str);
        SyntaxTree* _parse_char(wchar_t chr);
        SyntaxTree* _parse_anyChar(void);
        SyntaxTree* _parse_rangeChar(wchar_t c1, wchar_t c2);
        SyntaxTree* _parse_EOF(void);

        void _reduceSyntaxTreeToParent(PtrArray<SyntaxTree>* arr, SyntaxTree* st);
        void _reduceOpSyntaxTreeToParent(PtrArray<SyntaxTree>* arr, SyntaxTree* st);

    protected:
        u32 _curPos(void) { return gpInp->getPos(); }
        struct _ErrorPos {
            u32 parsePos;
            int errIdx;
        };
        _ErrorPos _curErrPos(void); // current error position and parse position
        void _back(_ErrorPos& errPos); // back to parse start position
        void _fail(_ErrorPos& errPos); // add error log and _back()
        SyntaxTree* _fatalError(_ErrorPos& errPos, const char* mes) {
            _fail(errPos); sys_printf(mes); return _FATAL_PARSER_ERROR; }

    public:
        struct Error_t {
            Parser*     parser;
            u32         pos;
        };
        static int      numErrors(void) { return _errors->size(); }
        static Error_t  getError(int n) { return _errors->get(n); }

    protected:
        static PtrArray<Parser>*        _holder;
        static HArray<Error_t>*         _errors;
        static bool                     _bUseMemoize;
        static int                      _printIntermediateLevel;

        static const char* _sequenceName;
        static const char* _orderedChoiceName;
    };


    // ========== user defined parser ==========
    class UserParser : public Parser {
    public:
        UserParser(const char* nam) : _memo(), _name(nam) {}
        virtual SyntaxTree* uParse(void) { return _PARSE_FAILED; }

        SyntaxTree* parse(void);
        virtual void errorMessage(int errorCutId, u32 pos);
        virtual void actionAtParse(SyntaxTree*) {}
        const char* name(void) { return (_name == NULL) ? "(noname)" : _name; }

        void removeMemo(SyntaxTree*);
    protected:
        SyntaxTree* _parse1(_ErrorPos startPos);
        virtual SyntaxTree* _parse2(_ErrorPos startPos, SyntaxTree* st);

        Memoize _memo;
        const char* _name;
    };


    // ========== generated parser ==========
    class GenParser : public UserParser {
    public:
        GenParser(const char* nam) : UserParser(nam), _parser(NULL) {}
        SyntaxTree* uParse(void);
        // virtual void construct(void); // setup _parser
    protected:
        SyntaxTree* _parse2(_ErrorPos startPos, SyntaxTree* st);
        Parser* _parser;
    };


    // ========== operator parser ==========
    class OperatorParser : public Parser {
    public:
        OperatorParser(const char* nam) : _parser(NULL), _name(nam) {}
        SyntaxTree* parse(void);
        const char* name(void) { return _name; }
    protected:
        Parser* _parser;
        const char* _name;
    };


    // ========== error cut ==========
    class Parser_ErrorCut : public Parser {
    public:
        Parser_ErrorCut(int errorCutId) { _errorCutId = errorCutId; }
        SyntaxTree* parse(void);
        const char* name(void) { return "(error cut)"; }
    protected:
        int     _errorCutId;
    };

    // ========== sequence ==========
    class Parser_Sequence : public Parser {
    public:
        Parser_Sequence(Parser* ps[], int n);
        Parser_Sequence(Parser* p1, Parser* p2);
        ~Parser_Sequence() { delete _ps; }
        void addSequenceToParent(PtrArray<Parser>* parentSeq);
        void addSyntaxTreeToParent(PtrArray<SyntaxTree>* arr, SyntaxTree* st) {
            _reduceSyntaxTreeToParent(arr, st); }
        SyntaxTree* parse(void) { return _parse_sequence(_ps); }
        const char* name(void) { return _sequenceName; }
        void add(Parser* p) { p->addSequenceToParent(_ps); }
    protected:
        PtrArray<Parser>* _ps;
    };


    // ========== ordered choice ==========
    class Parser_OrderedChoice : public Parser {
    protected:
        static const char* _choiceName;
    public:
        Parser_OrderedChoice(Parser* ps[], int n);
        Parser_OrderedChoice(Parser* p1, Parser* p2);
        ~Parser_OrderedChoice() { delete _ps; }
        SyntaxTree* parse(void) { return _parse_orderedChoice(_ps); }
        const char* name(void) { return _orderedChoiceName; }
        void add(Parser* p) { _ps->add(p); }
    protected:
        PtrArray<Parser>* _ps;
    };


    // ========== many ==========
    template <int m, int n> class Parser_Many : public Parser {
    public:
        Parser_Many(Parser* p) : Parser(), _p(p) {}
        void addSyntaxTreeToParent(PtrArray<SyntaxTree>* arr, SyntaxTree* st) {
            _reduceSyntaxTreeToParent(arr, st); }
        SyntaxTree* parse(void) { return _parse_m2n(_p, m, n); }
        const char* name(void) { return "(many)"; }
    protected:
        Parser* _p;
    };

    typedef Parser_Many<0,Parser::MAX_MANY> Parser_ZeroOrMore;
    typedef Parser_Many<1,Parser::MAX_MANY> Parser_OneOrMore;
    typedef Parser_Many<0,1>    Parser_Optional;


    // ========== predicate ==========
    class Parser_AndPredicate : public Parser {
    public:
        Parser_AndPredicate(Parser* p) : Parser(), _p(p) {}
        SyntaxTree* parse(void) { return _parse_andPredicate(_p); }
        const char* name(void) { return "(andPred)"; }
    protected:
        Parser* _p;
    };

    class Parser_NotPredicate : public Parser {
    public:
        Parser_NotPredicate(Parser* p) : Parser(), _p(p) {}
        SyntaxTree* parse(void) { return _parse_notPredicate(_p); }
        const char* name(void) { return "(notPred)"; }
    protected:
        Parser* _p;
    };	  


    // ========== operator ==========
    class Parser_InfixL : public Parser {
    public:
        Parser_InfixL(Parser* my, Parser* exp, Parser* op) : Parser(), _my(my), _exp(exp), _op(op) {}
        SyntaxTree* parse(void) { return _parse_op_infixl(_my, _exp, _op); }
        const char* name(void) { return "(infixl)"; }
    protected:
        Parser* _my;
        Parser* _exp;
        Parser* _op;
    };

    class Parser_InfixR : public Parser {
    public:
        Parser_InfixR(Parser* my, Parser* exp, Parser* op) : Parser(), _my(my), _exp(exp), _op(op) {}
        SyntaxTree* parse(void) { return _parse_op_infixr(_my, _exp, _op); }
        const char* name(void) { return "(infixr)"; }
    protected:
        Parser* _my;
        Parser* _exp;
        Parser* _op;
    };

    class Parser_InfixN : public Parser {
    public:
        Parser_InfixN(Parser* my, Parser* exp, Parser* op) : Parser(), _my(my), _exp(exp), _op(op) {}
        SyntaxTree* parse(void) { return _parse_op_infixn(_my, _exp, _op); }
        const char* name(void) { return "(infixn)"; }
    protected:
        Parser* _my;
        Parser* _exp;
        Parser* _op;
    };

    class Parser_Prefix : public Parser {
    public:
        Parser_Prefix(Parser* my, Parser* exp, Parser* op) : Parser(), _my(my), _exp(exp), _op(op) {}
        SyntaxTree* parse(void) { return _parse_op_prefix(_my, _exp, _op); }
        const char* name(void) { return "(prefix)"; }
    protected:
        Parser* _my;
        Parser* _exp;
        Parser* _op;
    };

    class Parser_Postfix : public Parser {
    public:
        Parser_Postfix(Parser* my, Parser* exp, Parser* op) : Parser(), _my(my), _exp(exp), _op(op) {}
        SyntaxTree* parse(void) { return _parse_op_postfix(_my, _exp, _op); }
        const char* name(void) { return "(postfix)"; }
    protected:
        Parser* _my;
        Parser* _exp;
        Parser* _op;
    };

    class Parser_Ternary : public Parser {
    public:
        Parser_Ternary(Parser* my, Parser* exp, Parser* op1, Parser* op2) : Parser(), _my(my), _exp(exp), _op1(op1), _op2(op2) {}
        SyntaxTree* parse(void) { return _parse_op_ternary(_my, _exp, _op1, _op2); }
        const char* name(void) { return "(ternary)"; }
    protected:
        Parser* _my;
        Parser* _exp;
        Parser* _op1;
        Parser* _op2;
    };


    // ========== pseudo ==========
    class Parser_NoTree : public Parser {
    public:
        Parser_NoTree(Parser* p) : Parser(), _p(p) {}
        SyntaxTree* parse(void) { return _parse_noTree(_p); }
        const char* name(void) { return "(noTree)"; }
    protected:
        Parser* _p;
    };


    // ========== scanner ==========
    class Parser_String : public Parser {
    public:
    Parser_String(const char* str) : Parser(), _str(str) {}
        SyntaxTree* parse(void) { return _parse_string(_str); }
        const char* name(void) { return "String"; }
    protected:
        const char* _str;
    };

    class Parser_Char : public Parser {
    public:
        Parser_Char(wchar_t chr) : Parser() { _chr = chr; }
        SyntaxTree* parse(void) { return _parse_char(_chr); }
        const char* name(void) { return "Char"; }
    protected:
        wchar_t _chr;
    };

   class Parser_AnyChar : public Parser {
    public:
        Parser_AnyChar(void) : Parser() {}
        SyntaxTree* parse(void) { return _parse_anyChar(); }
        const char* name(void) { return "AnyChar"; }
    };

    class Parser_EOF : public Parser {
    public:
        Parser_EOF(void) : Parser() {}
        SyntaxTree* parse(void) { return _parse_EOF(); }
        const char* name(void) { return "EOF"; }
    };

    class Parser_RangeChar : public Parser {
    public:
        Parser_RangeChar(wchar_t c1, wchar_t c2) : Parser() {
            _c1 = c1; _c2 = c2; }
        SyntaxTree* parse(void) { return _parse_rangeChar(_c1, _c2); }
        const char* name(void) { return "RangeChar"; }
    protected:
        wchar_t _c1, _c2;
    };

}

#endif /* _PARSER_H_ */
