
#ifndef _INPUTBUFFER_H_
#define _INPUTBUFFER_H_

#include "sysDep.h"
#include "HArray.h"
#include "PtrArray.h"
#include "Substr.h"
#include "SourceInfo.h"
#include <stdio.h>


// in case of Japanese Shift-JIS, linux mbrtowc() converts
// backslash '\' to 165, but we need 92 (L'\\') .
// define this if you want to change single byte 165 to 92.
#define AVOID_LINUX_SJIS_BUG


namespace Houken {

    class InputBuffer {
    public:
        InputBuffer(void);
        virtual ~InputBuffer() {}

        void    clean(void);
        u32     getPos(void) { return (u32)(_ptr - _buffer); }
        void    setPos(u32 pos) { _ptr = _prevPtr = addr(pos); }
        u32     remain(void) { return _bufSize - getPos(); }
        void    advance(u32 len);
        const char*     addr(void) { return _ptr; }
        const char*     addr(u32 pos);
        
        virtual wchar_t getCharAt(const char** ptr, mbstate_t* pMbstat = NULL);

        wchar_t getChar(void) {
            _prevPtr = _ptr;
            return getCharAt(&_ptr, &_mbstat);
        }
        wchar_t getCharAt(u32 pos, mbstate_t* pMbstat = NULL) {
            const char* ptr = addr(pos);
            return getCharAt(&ptr, pMbstat); }
        void    ungetChar(bool check = false); // only once after getChar()

        bool    isEOF(void);

        u32     copyStr(char* dest, u32 destSize, Substr_st& ss);
        void    copySummary(char* dest, u32 destSize, Substr_st& ss);
        void    fprintStr(FILE* fp, Substr_st& ss);
        void    fprintSummary(FILE* fp, Substr_st& ss);

        // if match, cmpStr returns true
        bool    cmpStr(const char* str);
        bool    cmpStr(Substr_st& ss, const char* str);
        bool    cmpStr(Substr_st& s1, Substr_st& s2);

        bool    pos2linecol(u32 pos, u32* line, u32* col);
        virtual void buildSourceInfo(SourceInfo* dest, u32 pos);
        void    sprintSourceInfo(char* dest, u32 destSize, u32 pos);

        virtual void    addIncludePath(const char*) = 0;
        virtual bool    include(const char*) = 0;

        static char* setLocale(const char* locale);


        // skip before \r or \n or EOF
        bool skip_beforeEol(void);
        // skip \r and \n
        bool skip_eols(void);

        ///////// in ident:
        // a-z : lower
        // A-Z : upper
        // _ : lower at top, not match identC at second, otherwise any
        // 0-9 : cannot be at top, not match identC at second, otherwise any

        // topCase: 0=lower, 1=any, 2=UPPER
        // followCase: 0=at least 1 lower, 1=any, 2=no lower
        bool skip__ident(int topCase, int followCase);
        bool skip_ident(void) { return skip__ident(1,1); }
        bool skip_identL(void) { return skip__ident(0,1); } // lower
        bool skip_identU(void) { return skip__ident(2,2); } // upper only
        bool skip_identC(void) { return skip__ident(2,0); } // capital
        // skip 0-9
        bool skip_decimalNum(void);
        // skip one char 0-9,a-f,A-F
        bool skip_hexChar(void);



#if 0
        void debugOutLinePos(void);
#endif

    protected:
        const char*     _buffer;
        u32             _bufSize;
        const char*     _ptr;
        const char*     _prevPtr;
        mbstate_t       _mbstat;

        HArray<u32>     _linePos;

#ifdef AVOID_LINUX_SJIS_BUG
        static bool     _sjis;
#endif
    };


    class FileInputBuffer : public InputBuffer {
    public:
        FileInputBuffer(const char* filename);
        ~FileInputBuffer();

        void addIncludePath(const char* path);
        bool include(const char* filename);
        char* expandPath(const char* filename, u32* pSize = NULL);
        /*virtual*/ void buildSourceInfo(SourceInfo* dest, u32 pos);
    protected:
        class _LocInfo {
        public:
            u32 pos;
            const char* fname;
            u32 line;
            u32 col;
            _LocInfo(u32, const char*);
            /* _LocInfo(_LocInfo&); */
            ~_LocInfo();
            static void* operator new(size_t size) { return sys_alloc((u32)size); }
            static void operator delete(void* p) { sys_free(p); }
        };
        int _searchLocate(u32 pos);
        _LocInfo* _getLocInfo(u32 pos);
    protected:
        PtrArray<const char>    _paths;
        PtrArray<const char>    _includePaths;
        PtrArray<_LocInfo> _locate;
    };


    class StringInputBuffer : public InputBuffer {
    public:
        StringInputBuffer(const char* str);
        ~StringInputBuffer();

        void addIncludePath(const char*) {}
        bool include(const char* str);
    };



    // global pointer to InputBuffer
    extern InputBuffer* gpInp;
}

#endif /* _INPUTBUFFER_H_ */
