
#include <locale.h>
#include <stdio.h>
#include "sysDep.h"
#include "InputBuffer.h"

using namespace Houken;


#ifdef AVOID_LINUX_SJIS_BUG
bool InputBuffer::_sjis = false;
#endif


InputBuffer* Houken::gpInp = NULL;


InputBuffer::InputBuffer(void)
    : _linePos(100)
{
    clean();
}

char* InputBuffer::setLocale(const char* locale)
{
#ifdef AVOID_LINUX_SJIS_BUG
    if (strstr(locale, "SJIS") != NULL)
        _sjis = true;
    else if (strstr(locale, "sjis") != NULL)
        _sjis = true;
    else
        _sjis = false;
#endif
    return setlocale(LC_CTYPE, locale);
}

void InputBuffer::clean(void)
{
    _buffer = NULL;
    _bufSize = 0;
    _ptr = _prevPtr = NULL;
    _linePos.clean();
    _linePos.add(0);    // first line
    memset(&_mbstat, 0, sizeof(mbstate_t));
}

void InputBuffer::advance(u32 len)
{
    _ptr += len;
    const char* endaddr = _buffer + _bufSize;
    if (_ptr > endaddr)
        _ptr = endaddr;
    _prevPtr = _ptr;
    memset(&_mbstat, 0, sizeof(mbstate_t));
}

const char* InputBuffer::addr(u32 pos)
{
    if (pos < _bufSize)
        return _buffer + pos;
    else
        return _buffer + _bufSize;
}

u32 InputBuffer::copyStr(char* dest, u32 destSize, Substr_st& ss)
{
    const char* adr = addr(ss.startPos);
    u32 len = ss.len();
    if (ss.startPos > _bufSize)
        len = 0;
    else if (ss.startPos + len > _bufSize)
        len = _bufSize - ss.startPos;
    if (len > destSize) len = destSize - 1;
    sys_strncpy(dest, adr, len+1);
    dest[len] = '\0';
    return len;
}

void InputBuffer::copySummary(char* dest, u32 destSize, Substr_st& ss)
{
    const char* adr = addr(ss.startPos);
    u32 len = ss.len();
    if (ss.startPos > _bufSize)
        len = 0;
    else if (ss.startPos + len > _bufSize)
        len = _bufSize - ss.startPos;
    if (len < destSize) {
        sys_strncpy(dest, adr, len+1);
        dest[len] = '\0';
        return;
    }
    u32 h = (destSize - 6) / 2 ;
    sys_strncpy(dest, adr, h+1);
    sys_strncpy(dest+h, " ... ", 6);
    sys_strncpy(dest+h+5, adr+len-h, h+1);
    dest[h*2+5] = '\0';
}

void InputBuffer::fprintStr(FILE* fp, Substr_st& ss)
{
    const char* adr = addr(ss.startPos);
    u32 len = ss.len();
    if (ss.startPos > _bufSize)
        return;
    if (ss.startPos + len > _bufSize)
        len = _bufSize - ss.startPos;
    while (len-- > 0) {
        fputc(*adr++, fp);
    }
}

void InputBuffer::fprintSummary(FILE* fp, Substr_st& ss)
{
    char buf[80];
    copySummary(buf, 80, ss);
    fprintf(fp, buf);
}


wchar_t InputBuffer::getCharAt(const char** ptr, mbstate_t* pMbstat)
{
    const char* endaddr = _buffer + _bufSize;
    const char* uptr = *ptr;
    if ((uptr < _buffer) || (uptr >= endaddr))
        return WEOF;
    wchar_t wc;
    size_t len;
    if (pMbstat != NULL) {
        len = mbrtowc(&wc, uptr, endaddr-uptr, pMbstat);
    } else {
        mbstate_t mbs;
        memset(&mbs, 0, sizeof(mbstate_t));
        len = mbrtowc(&wc, uptr, endaddr-uptr, &mbs);
    }
    if (len == (size_t)-1) {
        char buf[128];
        u32 pos = getPos();
        sprintSourceInfo(buf, 127, pos);
        sys_printf("%s: invalid char\n", buf);

        len = 1;
        wc = (wchar_t)uptr[0];
    } else if (len == (size_t)-2) {
        sys_printf("InputBuffer::getCharAt(): imcomplete multibyte character\n");
        len = endaddr - uptr;
        wc = L'\0';
    }
    *ptr += len;

#ifdef AVOID_LINUX_SJIS_BUG
    if (_sjis && len == 1 && wc == (wchar_t)165) wc = L'\\';
#endif

    return wc;
}

void InputBuffer::ungetChar(bool check)
{
    // if last getChar() was WEOF, then _ptr == _prevPtr
    if (check)
        sys_assert(_ptr != _prevPtr);
    _ptr = _prevPtr;
}

bool InputBuffer::isEOF(void)
{
    return (_ptr >= _buffer + _bufSize);
}

bool InputBuffer::cmpStr(const char* str)
{
    if (str == NULL)
        return false;
    u32 len = sys_strlen(str);
    if (remain() < len)
        return false;
    return ::strncmp(_ptr, str, len) == 0;
}
bool InputBuffer::cmpStr(Substr_st& ss, const char* str)
{
    if (str == NULL)
        return false;
    return ::strncmp(_buffer + ss.startPos, str, ss.len()) == 0;
}
bool InputBuffer::cmpStr(Substr_st& s1, Substr_st& s2)
{
    if (s1.len() != s2.len())
        return false;
    return ::strncmp(_buffer + s1.startPos, _buffer + s2.startPos, s1.len()) == 0;
}
    

bool InputBuffer::skip_beforeEol(void)
{
    wchar_t c;
    u32 startPos = getPos();
    do {
        c = getChar();
    } while (c != L'\n' && c != L'\r' && c != (wchar_t)WEOF);
    ungetChar();
    return (startPos != getPos());
}
    
bool InputBuffer::skip_eols(void)
{
    wchar_t c;
    u32 startPos = getPos();
    do {
        c = getChar();
    } while (c == L'\n' || c == L'\r');
    ungetChar();

    return (startPos != getPos());
}
    
bool InputBuffer::skip__ident(int topCase, int followCase)
{
    u32 start = getPos();
    wchar_t c = getChar();
    switch (topCase) {
    case 0: // lower case
        if (c != L'_' && (c < L'a' || c > L'z')) {
            ungetChar();
            return false;
        }
        break;
    case 2: // UPPER case
        if (c < L'A' || c > L'Z') {
            ungetChar();
            return false;
        }
        break;
    default:
        if (!(c >= L'a' && c <= L'z') && !(c >= L'A' && c <= L'Z') && c != L'_') {
            ungetChar();
            return false;
        }
        break;
    }


    bool lflag = false;
    do {
    SkipCond:
        c = getChar();
        if (c >= L'a' && c <= L'z') {
            if (followCase == 2) {
                setPos(start);
                return false;
            }
            lflag = true;
            goto SkipCond;
        }
    } while ((c >= L'A' && c <= L'Z') || c == L'_' || (c >= L'0' && c <= L'9'));

    if ((! lflag) && (followCase == 0)) {
        setPos(start);
        return false;
    }

    ungetChar();
    return true;
}

bool InputBuffer::skip_decimalNum(void)
{
    u32 startPos = getPos();
    wchar_t c;
    do {
        c = getChar();
    } while (c >= L'0' && c <= L'9');
    ungetChar();

    return (startPos != getPos());
}

bool InputBuffer::skip_hexChar(void)
{
    wchar_t c = getChar();
    if ((c >= L'0' && c <= L'9')
        || (c >= L'a' && c <= L'f')
        || (c >= L'A' && c <= L'F'))
        return true;
    ungetChar();
    return false;
}
    


bool InputBuffer::pos2linecol(u32 pos, u32* line, u32* col)
{
    if (pos >= _bufSize)
        return false;

    int lastLine = _linePos.size();
    u32 lastPos = _linePos.get(lastLine - 1);
    if (pos < lastPos) {
        // pos is within _linePos, do binary search
        int i = 0;
        int j = lastLine - 1;
        while (i < j - 1) {
            int k = (i + j) / 2;
            u32 kpos = _linePos.get(k);
            if (pos < kpos)
                j = k;
            else {
                i = k;
                if (pos == kpos) break;
            }
        }
        *line = i + 1;
        *col = pos - _linePos.get(i);
        return true;
    }

    // now, lastPos <= pos <= _bufSize

    char c1, c2;
    u32 chkPos = lastPos;
    while (chkPos < pos) {      // chkPos < _bufSize
        c1 = _buffer[chkPos];
        if (chkPos + 1 < _bufSize) {
            c2 = _buffer[chkPos+1];
        } else {
            c2 = '\0';
        }

        if (c1 == '\r' || c1 == '\n') {
            if (c1 == '\r' && c2 == '\n') {
                chkPos += 2;
            } else {
                ++ chkPos;
            }
            lastPos = chkPos;
            _linePos.add(lastPos);
            ++ lastLine;
        } else {
            ++ chkPos;
        }
    }

    if (chkPos == pos)  {
        *line = lastLine;
        *col = pos - lastPos;
    } else {
        *line = lastLine - 1;
        *col = pos - _linePos.get(lastLine-2);
    }
    return true;
}

void InputBuffer::buildSourceInfo(SourceInfo* dest, u32 pos)
{
    pos2linecol(pos, &(dest->line), &(dest->col));
    dest->fname = NULL;
}

void InputBuffer::sprintSourceInfo(char* dest, u32 destSize, u32 pos)
{
    SourceInfo si;
    buildSourceInfo(&si, pos);
    si.printns(dest, destSize);
}



#if 0
void InputBuffer::debugOutLinePos(void)
{
    char buf[80];
    pos2linecol(_bufSize, &line, &col);      // build all _linePos
    u32 prevPos = 0;
    for (int i = 0; i < _linePos.size(); i++) {
        u32 pos = _linePos.get(i);
        sprintSourceInfo(buf, 80, pos);
        sys_printf("%s: ", buf);
        Substr ss(prevPos, pos);
        prevPos = pos;
        copySummary(buf, 80, ss);
        sys_printf("'%s'\n", buf);
    }
}
#endif


FileInputBuffer::FileInputBuffer(const char* filename)
    : InputBuffer(), _paths(0), _includePaths(1), _locate(1)
{
    _bufSize = sys_fileSize(filename);
    _buffer = sys_allocT<char>(_bufSize + 8);
    u32 nread = sys_readFile((char*)_buffer, filename, _bufSize);
    *(char*)(_buffer + nread) = '\0';
    _ptr = _prevPtr = _buffer;
    _LocInfo* li = new _LocInfo(0, filename);
    _locate.add(li);
}

FileInputBuffer::~FileInputBuffer()
{
    if (_buffer != NULL)
        sys_free((void*)_buffer);
    clean();
    PtrArrayIterator<_LocInfo> itr(&_locate);
    while (itr.hasMore())
        delete itr.extract_next();
    PtrArrayIterator<const char> itr2(&_paths);
    while (itr2.hasMore())
        sys_free((void*) itr2.extract_next());
    PtrArrayIterator<const char> itr3(&_includePaths);
    while (itr3.hasMore())
        sys_free((void*) itr3.extract_next());
}

void FileInputBuffer::addIncludePath(const char* path)
{
    u32 psize = sys_strlen(path) + 1;
    char* p = sys_allocT<char>(psize);
    sys_strncpy(p, path, psize);
    _includePaths.add(p);
}

char* FileInputBuffer::expandPath(const char* filename, u32* pSize)
{
    char* path;
    u32 fnlen = sys_strlen(filename);
    u32 pathCapa = fnlen + 1;
    path = sys_allocT<char>(pathCapa);
    sys_strncpy(path, filename, pathCapa);
    u32 size = sys_fileSize(path);
    if (size > 0) {
        if (pSize != NULL) *pSize = size;
        return path;
    }

    PtrArrayIterator<const char> itr(&_includePaths);
    while (itr.hasMore()) {
        const char* inc = itr.next();
        size_t ilen = sys_strlen(inc);
        if (pathCapa < fnlen + ilen + 2) {
            sys_free(path);
            pathCapa = fnlen + ilen + 2;
            path = sys_allocT<char>(pathCapa);
        }
        sys_strncpy(path, inc, pathCapa);
        if (path[ilen-1] != '/') {
            path[ilen] = '/';
            path[ilen+1] = '\0';
        }
        sys_strncat(path, filename, pathCapa-ilen-1);
        size = sys_fileSize(path);
        if (size > 0) {
            if (pSize != NULL) *pSize = size;
            return path;
        }
    }

    sys_free(path);
    if (pSize != NULL) *pSize = 0;
    return NULL;
}


bool FileInputBuffer::include(const char* filename)
{
    char* path;
    u32 size;
    path = expandPath(filename, &size);
    if (path == NULL)
        return false;
    _paths.add(path);

    u32 newBufSize = _bufSize + size + 2;
    u32 curPos = getPos();

    int x = _searchLocate(curPos);
    _LocInfo* li = _getLocInfo(curPos);
    sys_assert(x >= 0);
    sys_assert(li != NULL);

    while (_linePos.get(_linePos.size() - 1) >= curPos)
        _linePos.chop(_linePos.size() - 1);
    // TODO: remove memoized SyntaxTree after curPos

    const char* newBuf = sys_reallocT<const char>(_buffer, newBufSize + 8);
    if (newBuf == NULL)
        return false;
    _ptr = _prevPtr = newBuf + curPos;
    memmove((void*)(_ptr + size + 2), (void*)_ptr, _bufSize - curPos);
    *(char*)(_ptr + size) = '\r';       // insert CRLF after file end
    *(char*)(_ptr + size + 1) = '\n';
    u32 nread = sys_readFile((char*)_ptr, path, size);
    sys_assert(nread == size);
    _buffer = newBuf;
    _bufSize = newBufSize;

    li->pos += size + 2;
    _LocInfo* nf = new _LocInfo(curPos, path);
    _locate.insert(x+1, nf);
    _locate.insert(x+2, li);

    for (x=x+3; x < _locate.size(); ++x) {
        _LocInfo* p = _locate.get(x);
        p->pos += size + 2;
    }

    return true;
}

int FileInputBuffer::_searchLocate(u32 pos)
{
    if (pos > _bufSize)
        return -1;
    int i = 0;
    int j = _locate.size();
    while (i < j - 1) {
        int k = (i + j) / 2;
        if (pos >= _locate.get(k)->pos)
            i = k;
        else
            j = k;
    }
    return i;
}

FileInputBuffer::_LocInfo* FileInputBuffer::_getLocInfo(u32 pos)
{
    int i = _searchLocate(pos);
    if (i < 0)
        return NULL;
    u32 line, col;
    if (! pos2linecol(pos, &line, &col))
        return NULL;

    _LocInfo* p = new _LocInfo(*(_locate.get(i)));
    u32 pl, pc;
    pos2linecol(p->pos, &pl, &pc);
    line -= pl;         // lines from locate(i)
    p->line += line;
    if (line == 0)
        p->col += col - pc;
    else
        p->col = col;
    p->pos = pos;
    return p;
}


FileInputBuffer::_LocInfo::_LocInfo(u32 ipos, const char* filename)
    : pos(ipos), fname(filename)
{
    line = 1;
    col = 0;
}

/*
FileInputBuffer::_LocInfo::_LocInfo(FileInputBuffer::_LocInfo& inf)
{
    pos = inf.pos;
    fname = inf.fname;
    line = inf.line;
    col = inf.col;
}
*/

FileInputBuffer::_LocInfo::~_LocInfo()
{
}

void FileInputBuffer::buildSourceInfo(SourceInfo* dest, u32 pos)
{
    _LocInfo* p = _getLocInfo(pos);
    if (p == NULL) {
        dest->line = dest->col = 0;
        dest->fname = "(out of file)";
        return;
    }
    dest->line = p->line;
    dest->col = p->col;
    dest->fname = p->fname;
    delete p;
}





StringInputBuffer::StringInputBuffer(const char* str)
    : InputBuffer()
{
    _buffer = str;
    _bufSize = sys_strlen(str);
    _ptr = _prevPtr = _buffer;
}

StringInputBuffer::~StringInputBuffer()
{
    clean();
}

bool StringInputBuffer::include(const char* str)
{
    sys_assert(false);      // not implemented yet
    return false;
}
