
#ifndef _TARRAY_H_
#define _TARRAY_H_

#include <string.h>
#include "sysDep.h"

namespace Houken {

    // simple array of <T>
    template <typename T> class HArray {
    protected:
        int     _capacity;
        int     _size;
        T*      _array;
    public:
        static void* operator new(size_t size) { return sys_alloc((u32)size); }
        static void operator delete(void* p) { sys_free(p); }

        HArray(int initCapacity = 1)
            : _size(0)
        {
            if (initCapacity > 0)
                _capacity = initCapacity;
            else
                _capacity = 1;
            _array = sys_allocT<T>(_capacity);
        }

        HArray(const T& t)
            : _capacity(1), _size(1)
        {
            _array = sys_allocT<T>(_capacity);
            _array[0] = t;
        }

        ~HArray() {
            sys_free(_array);
        }

        int size(void) { return _size; }

        int remain(void) { return _capacity - _size; }

        void clean(void) { _size = 0; }

        T& get(int idx) {
            sys_assert((idx >= 0) && (idx < _size));
            return _array[idx];
        }

        void set(int idx, const T& x) {
            sys_assert((idx >= 0) && (idx < _size));
            _array[idx] = x;
        }

        HArray* add(const T& t) {
            expand();
            _array[_size++] = t;
            return this;
        }

        T* keep(void) {
            expand();
            return &_array[_size++];
        }
        
        HArray* remove(int idx) {
            sys_assert((idx >= 0) && (idx < _size));
            for (int i = idx+1; i < _size; ++idx,++i)
                _array[idx] = _array[i];
            -- _size;
            return this;
        }

        HArray* insertAt(int idx, const T& t) {
            expand();
            for (int i = _size; i > idx; --i) {
                _array[i] = _array[i-1];
            }
            ++_size;
            _array[idx] = t;
            return this;
        }

        void expand(void) {
            if (_size >= _capacity)
                resize( ((_capacity * 3 + 3) >> 2) * 2 );
        }

        void resize(int newCapacity) {
            if (newCapacity <= _capacity)
                return;
            _array = sys_reallocT<T>(_array, newCapacity);
            _capacity = newCapacity;
        }

        void chop(int idx) {
            sys_assert((idx >= 0) && (idx <= _size));
            _size = idx;
        }

    };


    template <typename T> class HArrayIterator {
    protected:
        HArray<T>*      _arr;
        int             _idx;
        int             _size;
    public:
        HArrayIterator(HArray<T>* arr)
            : _arr(arr)
        {
            _idx = 0;
            _size = arr->size();
        }

        bool hasMore(void)
        {
            return _idx < _size;
        }

        const T& next(void)
        {
            sys_assert(_idx < _size);
            return _arr->get(_idx++);
        }
            
        void rewind(void)
        {
            _idx = 0;
        }
    };

}


#endif /* _TARRAY_H_ */
