
#include "PegParser.h"
#include "../etc/printSyntaxTree.h"
#include "sysDep.h"
#include <stdio.h>


#define THIS_NAME       "houken"
#define VERSION         "version 0.2"
#if defined(__CYGWIN__) || defined(_WIN32)
#define EXE_FILENAME    "houken.exe"
#else
#define EXE_FILENAME    "houken"
#endif

char* tab_basepath;     // export to PegParserAction.cpp
char* tab_basename;     // export to PegParserAction.cpp

using namespace Houken;


void printError(InputBuffer* inp)
{
    int n = Parser::numErrors();
    for (--n; n >= 0; --n) {
        Parser::Error_t err = Parser::getError(n);
        char pbuf[128];
        inp->sprintSourceInfo(pbuf, 128, err.pos);
        const char* nam = err.parser->name();
        if ((nam != NULL) && (*nam != '('))
            printf("parser=%s, %s\n", nam, pbuf);
    }
}

void errorCutMessage(u32 pos, const char* message)
{
    SourceInfo si;
    gpInp->buildSourceInfo(&si, pos);
    printf("%s:%d:%d: %s\n",
           si.fname, si.line, si.col,
           message);
}

void usage(void)
{
    printf("usage: " EXE_FILENAME " [options] PEGfile\n");
    printf("option:\n");
    printf("  -c <locale>: set LC_CTYPE to <locale>\n");
    printf("  -m : not use memoization\n");
    printf("  -s : print syntax tree\n");
    printf("  -d : print intermediate parsing result\n");
    printf("  -d2 : print detail intermediate parsing result\n");
}

int main(int argc, const char* argv[])
{
    const char* pegFileName = NULL;
    bool bPrintSyntaxTree = false;
    int  printIntermediateLevel = 0;
    bool bUseMemoize = true;
    int exitValue = 0;
    const char* ctype = NULL;

    printf("PEG parser generator " THIS_NAME " " VERSION "\n");
    for (int i = 1; i < argc; i++) {
        if (argv[i][0] == '-')
            for (const char* p = &argv[i][1]; *p != '\0'; ++p) {
                switch (*p) {
                case 'c':
                    if (p[1] != '\0')
                        ctype = &p[1];
                    else if (++i < argc)
                        ctype = argv[i];
                    else {
                        printf("-c option needs <locale>");
                        usage();
                        return 1;
                    }
                    break;
                case 's':
                    bPrintSyntaxTree = true;
                    break;
                case 'd':
                    if (p[1] >= '0' && p[1] <= '9')
                        printIntermediateLevel = *++p - '0';
                    else
                        printIntermediateLevel = 1;
                    break;
                case 'm':
                    bUseMemoize = false;
                    break;
                default:
                    printf("unknown option: %s\n", argv[i]);
                    usage();
                    return 1;
                }
            }
        else {
            if (pegFileName != NULL) {
                printf("error: only 1 PEGfile required\n");
                usage();
                return 1;
            }
            pegFileName = argv[i];
        }
    }

    if (pegFileName == NULL) {
        printf("error: PEGfile required\n");
        usage();
        return 1;
    }

    const char* p = strrchr(pegFileName, '.');
    u32 len = (sys_strcasecmp(p, ".peg") != 0) ? sys_strlen(pegFileName) : (u32)(p - pegFileName);
    tab_basepath = sys_allocT<char>(len+1);
    sys_strncpy(tab_basepath, pegFileName, len+1);
    tab_basepath[len] = '\0';
    p = strrchr(tab_basepath, '/');
    if (p == NULL)
        p = strrchr(tab_basepath, '\\');
    else
        p = strrchr(p, '\\');
    if (p == NULL) {
        len = sys_strlen(tab_basepath);
        p = tab_basepath;
    } else {
        ++p;
        len = (u32)(p - tab_basepath);
    }
    len = (p == NULL) ? sys_strlen(tab_basepath) : (u32)(p - tab_basepath);
    tab_basename = sys_allocT<char>(len+1);
    sys_strncpy(tab_basename, p, len+1);

    printf("generating %s.tab.* from %s\n", tab_basepath, pegFileName);

    FileInputBuffer inp(pegFileName);
    gpInp = &inp;
    if (ctype != NULL) {
#ifdef __CYGWIN__
        if (strcasecmp(ctype, "ja_JP.SJIS") == 0)
            InputBuffer::setLocale("C-SJIS");
        else if (strcasecmp(ctype, "ja_JP.eucjp") == 0)
            InputBuffer::setLocale("C-EUCJP");
        else
#endif
            InputBuffer::setLocale(ctype);
    }
    Parser::initialize(40);
    Parser::setUseMemoize(bUseMemoize);
    Parser::setPrintIntermediateLevel(printIntermediateLevel);
    SyntaxTree* st = p_Start->parse();
    if (st->isValidTree()) {
        if (bPrintSyntaxTree)
            printSyntaxTree(st, &inp, 0);
        st->action();
        printf("done.\n");
        delete st;
    } else {
        printf("parse error\n");
        printError(&inp);
        exitValue = 1;
    }

    sys_free(tab_basepath);

    Parser::finalize();
    SyntaxTree::finalize();

    return exitValue;
 }
