#!/bin/sh

# Copyright (C) since 2013 NEC Corporation
# version 2.0
#
# <License>
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

################################################################################
# Bash preparation
################################################################################
SCRIPT_DIR=$(cd $(dirname $0);pwd)
. ${SCRIPT_DIR}/install.cfg || exit -1

USER=`/usr/bin/whoami`

# load language file
LANGUAGE=$1
if [ ! -f ${SCRIPT_DIR}/lib/${LANGUAGE}_install.lng ]
then
    echo "not supported language : ${LANGUAGE}"
    exit 1
fi
. ${SCRIPT_DIR}/lib/${LANGUAGE}_install.lng || exit -1

# Default AP server
APSERVER_PATH="/opt/jboss-5.1.0.GA"

# Host information
VIRTUAL_HOST="localhost"
CONTEXT_ROOT="hinemos"

# Installed time
INSTALL_TIME=`date +%Y%m%d%H%M%S`

################################################################################
# Function
################################################################################

########################################
# Util
########################################
echoOK() {
    echo "[OK]"
}

echoNG() {
    echo "[NG]"
}

SIGINTExit() {
    echo
    echo
    echo "${MSG_E004}" "(SIGINT)"
    exit
}

backupFile() {
    local TARGET_FILE=$1

    echo
    printf "${MSG_I033}\n" $TARGET_FILE

    mv $TARGET_FILE ${TARGET_FILE}_${INSTALL_TIME}

    if [ $? -eq 0 ] ; then
	echoOK
    else
	echoNG
	printf "${MSG2_E006}" $TARGET_FILE
	echo "${MSG_E004}"; return 1
    fi
}

########################################
# Prepare
########################################
welcome() {
    local version=`cat ${SCRIPT_DIR}/hinemos_web/_version`
    echo
    echo "-------------------------------------------------------"
    echo "   Welcome to the installation for Hinemos Web Client"
    echo "                             Version ${version}"
    echo
    echo "       Copyright (C) NEC Corporation 2012. All Rights Reserved."
    echo "-------------------------------------------------------"
}

checkUser() {
    echo
    echo "${MSG_I002}"

    if [ "x"$USER != "xroot" ] ; then
	echoNG
	echo "${MSG_E002}"
	echo "${MSG_E004}"; return 1
    fi

    echoOK
}

checkJavaEnv() {
    echo
    echo "${MSG_I003}"

    # Check Java command.
    if ! type -P "java" >/dev/null 2>&1 ; then
	echoNG
	echo "${MSG_E003A}" "${MSG_E003}" 
	echo "${MSG_E004}"; return 1
    fi

    # Check Java version.
    JAVA_VER=$(java -version 2>&1 | awk 'NR==1 { print $3 }')
    if ! [[ "${JAVA_VER}" =~ 1\.6\.[0-9]+ ]]; then
	echoNG
	echo "${MSG_E003}" 
	echo "${MSG_E004}"; return 1
    fi

    # Check jar command.
    if ! type -P "jar" >/dev/null 2>&1 ; then
	echoNG
	echo "${MSG_E003B}" "${MSG_E003}" 
	echo "${MSG_E004}"; return 1
    fi

    echoOK
}

checkVersion() {
    if [ -d "${HINEMOS_WEB_HOME}" ] ; then
	echo
	echo "${MSG_W002}"
	echo

	# Can't find version file.
	if [ ! -f ${HINEMOS_WEB_HOME}/_version ] ; then
	    echo "${MSG_I013}"
	    echo "${MSG_E004}"; return 1
	fi
	current_ver=`cat ${HINEMOS_WEB_HOME}/_version`

	# Other than 2.0.x
	if ! [[ "$current_ver" =~ 2\.0\.[0-9]+ ]] ; then
	    if [[ "$current_ver" =~  1\.0\.[0-9]+ ]] ; then
		echo "${MSG2_E001}"
	    else
		echo "${MSG2_E002}"
	    fi

	    echo "${MSG_E004}"; return 1
	fi

	# 2.0.x 
	printf "${MSG_I005}" $current_ver
	echo "${MSG_I027}"
	local input
	while : ; do
	    read -p "${MSG_I028}: " input
	    input=${input:-N}
	    case $input in
		y|Y)
		    break
		    ;;
		n|N)
		    echo "${MSG_E004}"; return 1
		    ;;
		*)
		    echo "${MSG_W001}"
		    ;;
	    esac
	done
	echoOK

	install_ver=`cat ${SCRIPT_DIR}/hinemos_web/_version`
	# Check minior version.
	local current_minior_ver=`echo $current_ver | awk -F. '{print $3}'`
	local install_minior_ver=`echo $install_ver | awk -F. '{print $3}'`

	if [ $current_minior_ver -lt $install_minior_ver ] ; then
	    echo "${MSG2_I008B}"
	elif [ $current_minior_ver -gt $install_minior_ver ] ; then
	    echo "${MSG2_I008A}"
	else
	    echo "${MSG2_I008}"
	fi

    fi
}

########################################
# Interact 
########################################
confirmInstall() {
    echo

    local input
    while : ; do
	read -p "${MSG_I001} : " input
	input=${input:-Y}
	case $input in
	    y|Y)
		break
		;;
	    n|N)
		echo "${MSG_I007}"
		return 1
		;;
	    *)
		echo "${MSG_W001}"
		;;
	esac
    done
}

readAPServerType() {
    echo
    echo "${MSG_I008}"
    echo "    0.Tomcat"
    echo "    1.Jboss"

    local input
    while : ; do
	read -p "${MSG_I009} " input
	input=${input:-1}
	case $input in
	    0)
		APSERVER_TYPE=0
		break;
		;;
	    1)
		APSERVER_TYPE=1
		break;
		;;
	    *)
		echo "${MSG_W001}"
		;;
	esac
    done

    return 0
}

readAPServerDir() {
    echo

    local input
    local promptMsg=`printf "${MSG_I030}" ${APSERVER_PATH}`
    while : ; do
	read -e -p "${promptMsg} : " input
	input=${input:-"$APSERVER_PATH"}

	if [ -d "${input}" ] ; then
	    APSERVER_PATH=$input
	    break
	else
	    printf "${MSG_E005}\n" $input
	fi
    done
}

checkTomcatType() {
    # 0 for community version, 1 for rh6 rpm version.
    if [ -d ${APSERVER_PATH}/bin ] ; then
	TOMCAT_TYPE=0
    else
	TOMCAT_TYPE=1
    fi
}

checkAPServerSubDir() {
    if [ "$APSERVER_TYPE" -eq 1 ] ; then
	if [ ! -d ${APSERVER_PATH}/lib ]  ; then
	    echoNG
	    printf "${MSG2_E003}\n" ${APSERVER_PATH}/lib
	    echo "${MSG_E004}"; return 1
	fi

	if [ ! -d ${APSERVER_PATH}/common/lib ]  ; then
	    echoNG
	    printf "${MSG2_E003}\n" ${APSERVER_PATH}/common/lib 
	    echo "${MSG_E004}"; return 1
	fi
    else
	checkTomcatType

	if [ "$TOMCAT_TYPE" -eq 0 ] ; then
	    if [ ! -d ${APSERVER_PATH}/lib ]  ; then
		echoNG
		printf "${MSG2_E003}\n" ${APSERVER_PATH}/lib
		echo "${MSG_E004}"; return 1
	    fi
	fi

	if [ "$TOMCAT_TYPE" -eq 0 ] ; then
	    HOST_CONF_DIR=${APSERVER_PATH}/conf/Catalina/localhost
	else
	    HOST_CONF_DIR=${APSERVER_PATH}/Catalina/localhost
	fi

	if [ ! -d $HOST_CONF_DIR ]  ; then
	    echoNG
	    printf "${MSG2_E003}\n" ${APSERVER_PATH}/conf/Catalina/localhost 
	    echo "${MSG_E004}"; return 1
	fi
    fi
}

readServerConfiguration() {
    local configList=()
    local config

    # Initialize configuration list.
    for config in ${APSERVER_PATH}/server/* ; do
	if [ -d "$config" ] ; then
	    configList=("${configList[@]}" "$config")
	fi
    done

    # Error! No server configuration found.
    if [ ${#configList[@]} -eq 0 ] ; then
	echo "${MSG2_E201}"
	echo "${MSG_E004}"; return 1
    fi

    # Show menu.
    echo
    echo "${MSG_I015}"
    local index=0
    local default=0
    for config in ${configList[@]} ; do
	echo "    ${index}. ${config}"

	# The "default" configration will be used as default.
	configName=`basename ${config}`
	if [ "x"$configName = "xdefault" ] ; then
	    default=$index
	fi

	index=`expr ${index} + 1`
    done

    # Prompt for select.
    local promptMsg=`printf "${MSG2_I001}" "${default}"`
    while : ; do
	read -p "${promptMsg}: " input
	input=${input:-$default}
	if [ $input -lt $index ] ; then
	    JBOSS_CONFIG_PATH=${configList[$input]}

	    if [ ! -d ${JBOSS_CONFIG_PATH}/deploy ] ; then
		echo "${MSG2_E202}"
	    elif [ ! -d ${JBOSS_CONFIG_PATH}/deploy/jbossws.sar ] ; then
		echo "${MSG2_E203}"
	    else
		return 0
	    fi
	else
	    echo "${MSG_W001}"
	fi
    done
}

confirmModifyListenPort() {
    echo
    echo "${MSG2_I002}"
    echo "${MSG2_I002A}"

    local input
    while : ; do
	read -p "${MSG2_I003} : " input
	input=${input:-Y}
	case $input in
	    y|Y)
		MODIFY_LISTEN_PORT=0
		break
		;;
	    n|N)
		MODIFY_LISTEN_PORT=1
		break
		;;
	    *)
		echo "${MSG_W001}"
		;;
	esac
    done

    return 0
}

checkServerXml() {
    echo
    echo "${MSG2_I007}"

    if [ "$APSERVER_TYPE" -eq 1 ] ; then
	SEVER_XML=${JBOSS_CONFIG_PATH}/deploy/jbossweb.sar/server.xml 
    else
	if [ "$TOMCAT_TYPE" -eq 0 ] ; then
	    SEVER_XML=${APSERVER_PATH}/conf/server.xml 
	else
	    SEVER_XML=${APSERVER_PATH}/server.xml 
	fi
    fi

    if [ ! -f "$SEVER_XML" ] ; then
	echoNG
	printf "${MSG2_E005}" $SEVER_XML
	echo "${MSG_E004}"; return 1
    fi

    echoOK
}

confirmInstallationInfo() {
    echo
    echo "${MSG_I017}"

    echo "${MSG_I018}"
    printf "${MSG_I019}\n" $APSERVER_TYPE
    printf "${MSG_I020}\n" $APSERVER_PATH
    if [ "$APSERVER_TYPE" -eq 1 ] ; then
	printf "${MSG2_I004}\n" `basename ${JBOSS_CONFIG_PATH}`
    else
	printf "${MSG_I023}\n" "$VIRTUAL_HOST"
    fi
    printf "${MSG_I024}\n" "$CONTEXT_ROOT"
    printf "${MSG2_I006}\n" `[ $MODIFY_LISTEN_PORT -eq 0 ] && echo Yes || echo No`

    local input
    while : ; do
	read -p "${MSG_I025}: "  input
	input=${input:-Y}
	case $input in
	    y|Y)
		break
		;;
	    n|N)
		echo "${MSG_E004}"; return 1
		;;
	    *)
		echo "${MSG_W001}"
		;;
	esac
    done
}

########################################
# Install 
########################################
copyTomcatLib() {
    cp ${SCRIPT_DIR}/lib/jstl*.jar ${HINEMOS_WEB_HOME}/war/WEB-INF/lib/ \
	&& cp ${SCRIPT_DIR}/lib/standard*.jar ${HINEMOS_WEB_HOME}/war/WEB-INF/lib/ \
	&& cp ${SCRIPT_DIR}/lib/commons-collections*.jar  ${HINEMOS_WEB_HOME}/war/WEB-INF/lib/ 
}

copyBaseFile() {
    echo
    echo "${MSG_I010}"

    if [ ! -d "${HINEMOS_WEB_HOME}" ] ; then
	mkdir -p $HINEMOS_WEB_HOME
	echo "${MSG_I011}"
    fi

    if [ -d "${HINEMOS_WEB_HOME}/war" ] ; then
	rm -rf ${HINEMOS_WEB_HOME}/war
    fi

    echoOK

    echo
    echo "${MSG_I012}"
    cp -pfr ${SCRIPT_DIR}/hinemos_web/* $HINEMOS_WEB_HOME \
	&& mkdir -p ${HINEMOS_WEB_HOME}/sbin/lib \
	&& cp -pf ${SCRIPT_DIR}/install.cfg ${HINEMOS_WEB_HOME}/sbin/ \
	&& cp -pf ${SCRIPT_DIR}/uninstall_all.sh ${HINEMOS_WEB_HOME}/sbin/ \
	&& cp -pf ${SCRIPT_DIR}/webclient_uninstaller_EN.sh ${HINEMOS_WEB_HOME}/sbin/ \
	&& cp -pf ${SCRIPT_DIR}/webclient_uninstaller_JP.sh ${HINEMOS_WEB_HOME}/sbin/ \
	&& cp -pf ${SCRIPT_DIR}/lib/ja_uninstall.lng ${HINEMOS_WEB_HOME}/sbin/lib/ \
	&& cp -pf ${SCRIPT_DIR}/lib/en_uninstall.lng ${HINEMOS_WEB_HOME}/sbin/lib/ \
	&& cat ${SCRIPT_DIR}/lib/log4j.properties | awk '{sub(/\${hinemos.web.home}/,"'$HINEMOS_WEB_HOME'"); print}' > ${HINEMOS_WEB_HOME}/war/WEB-INF/classes/log4j.properties \
	&& ([ $APSERVER_TYPE -eq 1 ] && : || copyTomcatLib)

    if [ $? -ne 0 ] ; then
	echoNG
	echo "${MSG_E004}"; return 1
    fi

    echoOK
}

copyJBossLib() {
    echo 
    echo "${MSG_I202}"

    local SRC_DIR=${APSERVER_PATH}/common/lib
    local DST_DIR=${APSERVER_PATH}/lib/endorsed
    local lib

    # Check copy desitnation.
    if [ ! -d "$DST_DIR" ] ; then
	if [ -e "$DST_DIR" ] ; then
	    false
	else
	    mkdir $DST_DIR 
	fi

	if [ $? -ne 0 ] ; then
	    echoNG
	    printf "${MSG2_E004}\n" $DST_DIR
	    echo "${MSG_E004}"; return 1
	fi
    fi

    # Execute copy.
    for lib in ${JBOSS_ENDORESED_LIB[@]} ; do
	if [[ -f ${SRC_DIR}/${lib} && ! -f ${DST_DIR}/${lib} ]] ; then
	    cp -pf ${SRC_DIR}/${lib} ${DST_DIR}/${lib}
	    if [ $? -ne 0 ] ; then
		printf "${MSG2_E204}\n" $lib
		echo "${MSG_E004}"; return 1
	    fi
	fi
    done

    echoOK
}

modifyListenPort() {
    # Backup server.xml.
    backupFile $SEVER_XML || return 1

    # Modify server.xml.
    echo
    printf "${MSG_I034}\n" $SEVER_XML

    java -jar "${SCRIPT_DIR}/lib/editXml.jar" ${SEVER_XML}_${INSTALL_TIME} ${SEVER_XML} 

    if [ $? -eq 0 ] ; then 
	local owner=`ls ${SEVER_XML}_${INSTALL_TIME} | awk '{printf \$3":"\$4}'`
	chown $owner $SEVER_XML
	echoOK
    else
	echoNG
	printf "${MSG2_E007}" $SEVER_XML
	mv ${SEVER_XML}_${INSTALL_TIME} $SEVER_XML 
	echo "${MSG_E004}"; return 1
    fi
}

modifyRunConf() {
    local RUN_CONF=${APSERVER_PATH}/bin/run.conf
    . $RUN_CONF || exit -1

    local modifyBindingSet=1
    local modifyResolvWarn=1

    if [[ $MODIFY_LISTEN_PORT -eq 0 && "$JAVA_OPTS" != *"Djboss.service.binding.set="*  ]]; then
	modifyBindingSet=0
    fi

    if [[ "$JAVA_OPTS" != *"Dorg.jboss.resolver.warning=false"* ]] ; then
	modifyResolvWarn=0
    fi

    if [[ $modifyBindingSet -eq 1 && $modifyResolvWarn -eq 1 ]] ; then
	return 0
    fi


    # Backup run.conf.
    backupFile $RUN_CONF || return 1

    echo
    printf "${MSG_I034}\n" $RUN_CONF

    local cfgBindingSet='JAVA_OPTS="$JAVA_OPTS -Djboss.service.binding.set=ports-01"'
    local cfgResolvWarn='JAVA_OPTS="$JAVA_OPTS -Dorg.jboss.resolver.warning=false"'

    cp -pf ${RUN_CONF}_${INSTALL_TIME} $RUN_CONF \
	&& ([ $modifyBindingSet -ne 0 ] && : || echo "$cfgBindingSet" >> $RUN_CONF) \
	&& ([ $modifyResolvWarn -ne 0 ] && : || echo "$cfgResolvWarn" >> $RUN_CONF)

    if [ $? -eq 0 ] ; then 
	echoOK
    else
	echoNG
	printf "${MSG2_E007}" $RUN_CONF
	mv ${RUN_CONF}_${INSTALL_TIME} $RUN_CONF || return 1
	echo "${MSG_E004}"; return 1
    fi
}

jbossDeploy() {
    echo
    echo "${MSG_I200}"

    local HINEMOS_WAR=${JBOSS_CONFIG_PATH}/deploy/hinemos_web.war

    # Backup installed war file.
    echo
    echo "${MSG_I205}"
    if [ -f $HINEMOS_WAR ] ; then
	echo "${MSG_I206}"
	mv $HINEMOS_WAR ${HINEMOS_WAR}_${INSTALL_TIME}
	if [ $? -ne 0 ] ; then
	    echoNG
	    echo "${MSG_E202}"
	    echo "${MSG_E004}"; return 1
	fi

	echoOK
    fi

    # Execute deploy.
    echo
    echo "${MSG_I207}"

    jar cvf $HINEMOS_WAR -C ${HINEMOS_WEB_HOME}/war/ . > "${HINEMOS_WEB_HOME}/sbin/jar_command.log" 
    if [ $? -ne 0 ] ; then
	echoNG
	echo "${MSG_E004}"; return 1
    fi
    echoOK
}

tomcatDeploy() {
    echo
    echo "${MSG_I100}"

    local HINEMOS_CONTEXT_XML=$HOST_CONF_DIR/hinemos.xml

    if [ -f "$HINEMOS_CONTEXT_XML" ] ; then
	backupFile $HINEMOS_CONTEXT_XML || return 1
    fi

    echo
    echo  "${MSG2_I009}"

    cp -pf ${SCRIPT_DIR}/lib/hinemos.xml $HINEMOS_CONTEXT_XML
    if [ $? -eq 0 ] ; then
	echoOK
    else
	echoNG
	printf "${MSG2_E008}\n" $HINEMOS_CONTEXT_XML
	mv ${HINEMOS_CONTEXT_XML}_${INSTALL_TIME} $HINEMOS_CONTEXT_XML || return 1
	echo "${MSG_E004}"; return 1
    fi
}

deploy() {
    if [ $APSERVER_TYPE -eq 1 ] ; then
	jbossDeploy
    else
	tomcatDeploy
    fi

    if [ $? -ne 0 ] ; then
	return 1
    fi
}

saveInstallationInfo() {
    local installationInfoFile=${HINEMOS_WEB_HOME}/_installation
    echo "APSERVER_TYPE=${APSERVER_TYPE}" > $installationInfoFile
    echo "APSERVER_PATH=${APSERVER_PATH}" >> $installationInfoFile
    if [ "x"$JBOSS_CONFIG_PATH != "x" ] ; then
	echo "JBOSS_CONFIG_PATH=${JBOSS_CONFIG_PATH}" >> $installationInfoFile
    fi
}

################################################################################
# Main
################################################################################

########################################
# Prepare
########################################

trap SIGINTExit 2

checkJavaEnv || exit 2

welcome

checkVersion || exit 3

########################################
# Interact
########################################

confirmInstall || exit 4

readAPServerType  

readAPServerDir
checkAPServerSubDir || exit 5

if [ "$APSERVER_TYPE" -eq 1 ] ; then
    readServerConfiguration || exit 6
fi

confirmModifyListenPort

if [ $MODIFY_LISTEN_PORT -eq 0 ] ;then
    checkServerXml || exit 7
fi

confirmInstallationInfo || exit 4

########################################
# Install 
########################################

copyBaseFile || exit 8

if [ "$APSERVER_TYPE" -eq 1 ] ; then
    copyJBossLib || exit 9
fi

if [ $MODIFY_LISTEN_PORT -eq 0 ] ;then
    modifyListenPort || exit 10
fi

if [ "$APSERVER_TYPE" -eq 1 ] ; then
    modifyRunConf || exit 10
fi

deploy || exit 11

saveInstallationInfo

echo
echo "${MSG_I099}"
echo "${MSG_I300}"

################################################################################
# Exit Code List
# -1	Failed to Execute source command.
#  0	Success.
#  1	Not Support Language.
#  2	Environment check failed.		(3.1.1)
#  3	Incompatible version.			(3.1.3)
#  4	User cancelled installation.		(3.1.4)
#  5	AP server missed some directories.	(3.1.6)
#  6	No server configuration in JBoss.	(3.1.7)
#  7	Server.xml doesn't exist.		(3.1.8)
#  8	Failed to copy base file.		(3.1.10)
#  9	Failed to copy JBoss library file.	(3.1.11)
# 10	Failed to modify server configuration.	(3.1.12, 3.1.13)
# 11	Failed to deploy.			(3.1.14)
################################################################################
