/*
 
Copyright (C) NEC Corporation 2012. All Rights Reserved. 
Copyright (C) NEC Soft, Ltd. 2012. All Rights Reserved. 
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.necsoft.hinemos.webclient.monitor.servlet;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.Locale;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.lang.StringEscapeUtils;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.bean.Property;
import com.clustercontrol.monitor.bean.EventTableDefine;
import com.necsoft.hinemos.webclient.common.Constants;
import com.necsoft.hinemos.webclient.exception.HinemosWebAccessException;
import com.necsoft.hinemos.webclient.exception.HinemosWebException;
import com.necsoft.hinemos.webclient.exception.HinemosWebInvalidRequestParameterException;
import com.necsoft.hinemos.webclient.exception.HinemosWebInvalidValueRequestParameterException;
import com.necsoft.hinemos.webclient.exception.HinemosWebRemoteException;
import com.necsoft.hinemos.webclient.json.CommonJson;
import com.necsoft.hinemos.webclient.json.GridDataJson;
import com.necsoft.hinemos.webclient.monitor.MonitorConstants;
import com.necsoft.hinemos.webclient.monitor.business.ConfirmEventBusiness;
import com.necsoft.hinemos.webclient.monitor.business.GetEventListBusiness;
import com.necsoft.hinemos.webclient.monitor.form.MonitorEventForm;
import com.necsoft.hinemos.webclient.monitor.util.MonitorMappingUtil;
import com.necsoft.hinemos.webclient.servlet.CommonServlet;
import com.necsoft.hinemos.webclient.util.ExceptionHandlerUtils;
import com.necsoft.hinemos.webclient.util.LogUtils;
import com.necsoft.hinemos.webclient.util.PopulateBeanUtils;

/**
 * イベント情報データをリフレッシュするクラス
 * 
 * @version 1.0.0
 * @since 1.0.0
 **/
public class GetEventListServlet extends CommonServlet {
    private static final long serialVersionUID = 996012390849870607L;

    private static LogUtils logUtils = LogUtils
            .getLogUtils(GetEventListServlet.class);

    public CommonJson process(HttpServletRequest request,
            HttpServletResponse response) throws ServletException, IOException,
            HinemosWebAccessException {

        CommonJson result = new CommonJson();

        try {
            // NammingContextを取得する
            NamingContext namingContext = doLoadNamingContext(request);

            // Get data form
            MonitorEventForm form = getDataForm(request);

            // Action form
            doAction(form, request, namingContext);

            // 中間変数を宣言
            ArrayList<ArrayList<Object>> listInfoData = doLoadListInfoData(
                    request, form, namingContext);

            // Create JSON
            result = doConvertToJson(listInfoData, request.getLocale());

        } catch (HinemosWebAccessException ex) {
            logUtils.info(ex.getMessage());
            throw new HinemosWebAccessException();
        } catch (HinemosWebRemoteException ex) {
            logUtils.error(ex.getMessage(), ex.getCause());
            result = ExceptionHandlerUtils.doBuildCommonJson(
                    ExceptionHandlerUtils.LEVEL_WARNING, ex.getMessage());

        } catch (Exception ex) {
            logUtils.error(ex.getMessage(), ex.getCause());
            result = ExceptionHandlerUtils.doBuildCommonJson(
                    ExceptionHandlerUtils.LEVEL_ERROR, ex.getMessage());
        }
        return result;
    }

    /**
     * @param request
     * @return
     */
    private MonitorEventForm getDataForm(HttpServletRequest request) {
        MonitorEventForm form = new MonitorEventForm();
        PopulateBeanUtils.populateBean(form, request);
        return form;
    }

    /**
     * @param form
     * @param request
     * @param namingContext
     * @throws HinemosWebException
     */
    private void doAction(MonitorEventForm form, HttpServletRequest request,
            NamingContext namingContext) throws HinemosWebException {
        String action;
        // Check action
        if (form.getAction() != null) {
            action = form.getAction();
            if (action.equals("confirmed")) {
                ArrayList<ArrayList<Object>> list = doGetConfirmedData(request);
                if (list.size() > 0) {
                    doConfirmed(list, namingContext);
                }
            }
        }

    }

    /**
     * @param list
     * @param namingContext
     * @throws HinemosWebException
     */
    private void doConfirmed(ArrayList<ArrayList<Object>> list, NamingContext namingContext)
            throws HinemosWebException {
        ConfirmEventBusiness eventBusiness = new ConfirmEventBusiness();
        eventBusiness.confirm(list, namingContext);
    }

    private ArrayList<ArrayList<Object>> doLoadListInfoData(
            HttpServletRequest request, MonitorEventForm form,
            NamingContext namingContext) throws HinemosWebException {
        ArrayList<ArrayList<Object>> listLogData;

        // Check filter
        if (form.isFilter()) {
            Property condition = doLoadCondition(request);
            GetEventListBusiness business = new GetEventListBusiness();
            listLogData = business.getEventList(form, condition, namingContext);
        } else {
            GetEventListBusiness business = new GetEventListBusiness();
            listLogData = business.getEventList(form, null, namingContext);
        }

        return listLogData;
    }

    /**
     * @param request
     * @return
     * @throws HinemosWebException
     */
    private Property doLoadCondition(HttpServletRequest request)
            throws HinemosWebException {
        HttpSession session = request.getSession(true);
        if (session.getAttribute(MonitorConstants.EVENT_FILTER_PROPERTIES) == null) {
            throw new HinemosWebException(
                    "exception.monitor.event.filter.session");
        }

        Property condition = (Property) session
                .getAttribute(MonitorConstants.EVENT_FILTER_PROPERTIES);
        convertCondition(request, condition);
        return condition;
    }

    private GridDataJson doConvertToJson(
            ArrayList<ArrayList<Object>> listInfoData, Locale locale)
            throws HinemosWebException {
        GetEventListBusiness business = new GetEventListBusiness();
        return business.doConvertToJson(listInfoData, locale);
    }

    /**
     * @param request
     * @return
     * @throws HinemosWebException
     */
    private ArrayList<ArrayList<Object>> doGetConfirmedData(
            HttpServletRequest request) throws HinemosWebException {
        ArrayList<ArrayList<Object>> list = new ArrayList<ArrayList<Object>>();
        String actionData = Constants.ACTION_DATA_REQUEST;
        int numData;

        if (request.getParameter("numData") == null) {
            throw new HinemosWebInvalidRequestParameterException(
                    "exception.monitor.event.confirm.numdata");
        } else {
            numData = Integer.valueOf(request.getParameter("numData"));
            if (numData <= 0) {
                throw new HinemosWebInvalidRequestParameterException(
                        "exception.monitor.event.confirm.numdata");
            } else if (numData > Constants.EVENT_ROW_NUM) {
                numData = Constants.EVENT_ROW_NUM;
            }
        }

        for (int count = 0; count < numData; count++) {
            ArrayList<Object> row = new ArrayList<Object>();
            String[] row_data = new String[MonitorConstants.NUMBER_HEADER_EVENT];
            for (int cnt_col = 0; cnt_col < MonitorConstants.NUMBER_HEADER_EVENT; cnt_col++) {
                String row_name = getNameRequestParam(actionData, count,
                        MonitorMappingUtil.getEventName(cnt_col));
                if (request.getParameter(row_name) == null) {
                    throw new HinemosWebInvalidValueRequestParameterException(
                            "exception.monitor.event.confirm.field.null");
                } else {
                    row_data[cnt_col] = request.getParameter(row_name);
                }
            }
            ;

            long receiveTime;
            long outputDate;
            try {
                receiveTime = Long
                        .parseLong(row_data[EventTableDefine.RECEIVE_TIME]);
                outputDate = Long
                        .parseLong(row_data[EventTableDefine.OUTPUT_DATE]);
            } catch (Exception e) {
                throw new HinemosWebInvalidValueRequestParameterException(
                        "exception.monitor.event.confirm.longdate", e);
            }
            ;

            row.add(EventTableDefine.PRIORITY,
                    row_data[EventTableDefine.PRIORITY]);
            row.add(EventTableDefine.RECEIVE_TIME, new Date(receiveTime));
            row.add(EventTableDefine.OUTPUT_DATE, new Date(outputDate));
            row.add(EventTableDefine.PLUGIN_ID, StringEscapeUtils
                    .unescapeHtml(row_data[EventTableDefine.PLUGIN_ID]));
            row.add(EventTableDefine.MONITOR_ID, StringEscapeUtils
                    .unescapeHtml(row_data[EventTableDefine.MONITOR_ID]));
            row.add(EventTableDefine.FACILITY_ID, StringEscapeUtils
                    .unescapeHtml(row_data[EventTableDefine.FACILITY_ID]));
            row.add(EventTableDefine.SCOPE, StringEscapeUtils
                    .unescapeHtml(row_data[EventTableDefine.SCOPE]));
            row.add(EventTableDefine.APPLICATION, StringEscapeUtils
                    .unescapeHtml(row_data[EventTableDefine.APPLICATION]));
            row.add(EventTableDefine.MESSAGE_ID, StringEscapeUtils
                    .unescapeHtml(row_data[EventTableDefine.MESSAGE_ID]));
            row.add(EventTableDefine.MESSAGE, StringEscapeUtils
                    .unescapeHtml(row_data[EventTableDefine.MESSAGE]));
            row.add(EventTableDefine.CONFIRMED,
                    row_data[EventTableDefine.CONFIRMED]);
            row.add(EventTableDefine.CONFIRM_USER,
                    row_data[EventTableDefine.CONFIRM_USER]);

            list.add(row);
        }

        return list;
    }

    /**
     * @param prefixName
     * @param count
     * @return
     */
    private String getNameRequestParam(String prefixName, int count,
            String colName) {
        StringBuffer names = new StringBuffer();
        names.append(prefixName);
        names.append("[");
        names.append(count);
        names.append("][");
        names.append(colName);
        names.append("]");

        return names.toString();
    }

}