/*

 Copyright (C) 2009 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.util;

import java.sql.Timestamp;

import javax.persistence.EntityExistsException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.entity.MonitorStatusPK;
import com.clustercontrol.notify.model.MonitorStatusEntity;
import com.clustercontrol.repository.model.FacilityEntity;
import com.clustercontrol.repository.session.RepositoryControllerBean;

public class MonitorResultStatusUpdater {
	/** ログ出力のインスタンス */
	private static Log m_log = LogFactory.getLog( MonitorResultStatusUpdater.class );

	/**
	 * 同一重要度カウンタの最大値（この値を超えた場合は、以降は更新されない）
	 */
	public static int MAX_INITIAL_NOTIFY_COUNT = 10;

	static{
		// プロパティファイルの読み込み
		try {
			MAX_INITIAL_NOTIFY_COUNT = Integer.parseInt(
					HinemosProperties.getProperty("notify.initial.count.max",
							String.valueOf(MAX_INITIAL_NOTIFY_COUNT)));
		} catch (Exception e) {
			m_log.warn("static() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
		}
	}

	// このメソッドを利用する場合は、getLockを使用すること。
	public static Long getCounter (MonitorStatusPK pk) throws NotifyNotFound {
		Long count = null;
		MonitorStatusEntity monitorStatus;
		monitorStatus = QueryUtil.getMonitorStatusPK(
				pk.getFacilityId(),
				pk.getPluginId(),
				pk.getMonitorId(),
				pk.getSubKey());
		count = monitorStatus.getCounter();
		if (count == null) {
			m_log.info("getCounter() counter is null. pk=" + pk);
		}
		return count;
	}

	// このメソッドを利用する場合は、getLockを使用すること。
	public static Integer getPriority (MonitorStatusPK pk) throws NotifyNotFound {
		Integer priority = null;
		MonitorStatusEntity monitorStatus;
		monitorStatus = QueryUtil.getMonitorStatusPK(
				pk.getFacilityId(),
				pk.getPluginId(),
				pk.getMonitorId(),
				pk.getSubKey());
		priority = monitorStatus.getPriority();
		if (priority == null) {
			m_log.info("getPriority() priority is null. pk=" + pk);
		}
		return priority;
	}

	/**
	 * 直前の監視結果と現在の監視結果の重要度を比較し、変更がある場合は、DBで保持している同一重要度カウンタをリセット。
	 * 戻り値として、trueを返す。
	 * 重要度変化がない場合は、同一重要度カウンタをカウントアップし、DB情報を更新。
	 * 戻り値として、falseを返す。
	 * 
	 * @param facilityId ファシリティID
	 * @param pluginId プラグインID
	 * @param monitorId 監視項目ID
	 * @param generateDate 監視結果生成時刻
	 * @param currentPriority 現在の監視結果の重要度
	 * @return 重要度変化の有無（有:true / 無:false）
	 */
	private static boolean update(String facilityId, String pluginId, String monitorId,
			String subkey, Long generateDate, int currentPriority) {
		m_log.debug("update() facilityId = " + facilityId +
				", pluginId = " + pluginId +
				", monitorId = " + monitorId +
				", subkey = " + subkey +
				", generateDate = " + generateDate +
				", currentPriority = " + currentPriority);

		JpaTransactionManager jtm = new JpaTransactionManager();

		MonitorStatusEntity monitorStatus = null;

		MonitorStatusPK pk = new MonitorStatusPK(facilityId, pluginId, monitorId, subkey);

		try {
			// 重要度が変化しているか確認
			if(currentPriority != getPriority(pk)){
				// 重要度が変化している場合
				if (monitorStatus == null) {
					monitorStatus = QueryUtil.getMonitorStatusPK(
							pk.getFacilityId(),
							pk.getPluginId(),
							pk.getMonitorId(),
							pk.getSubKey());
				}
				if(m_log.isDebugEnabled()){
					m_log.debug("prioityChangeFlag = true. " + pk + " ," +
							monitorStatus.getPriority() + " to " +
							currentPriority);
				}

				// 重要度を更新
				monitorStatus.setPriority(currentPriority);

				// 同一重要度カウンタを1にリセット
				monitorStatus.setCounter(1l);

				return true;
			} else {
				// 重要度は変化していない
				// 50行ぐらい下の箇所で、カウンタをアップする。
			}
		} catch (NotifyNotFound e) {
			// 検索の結果該当タプルが存在しない場合
			m_log.debug("create new entity. " + pk);

			try {
				FacilityEntity facilityEntity
				= new RepositoryControllerBean().getFacilityEntityByPK(facilityId);
			} catch (Exception e1) {
				// 出力先のファシリティIDが存在しない場合
				// 未登録のファシリティIDで通知される場合もある
				// その場合ここを通るため、エラーログは出さない。
				return true;
			}

			// 新規タプルを生成
			try {
				// 同一重要度カウンタは1で生成
				// インスタンス生成
				monitorStatus = new MonitorStatusEntity(facilityId, pluginId, monitorId, subkey);
				// 重複チェック
				jtm.checkEntityExists(MonitorStatusEntity.class, monitorStatus.getId());
				monitorStatus.setPriority(currentPriority);
				monitorStatus.setLastUpdate(new Timestamp(generateDate));
				monitorStatus.setCounter(1l);
				return true;
			} catch (EntityExistsException e1) {
				m_log.info("update() : "
						+ e.getClass().getSimpleName() + ", " + e.getMessage());
				return true;
			}
		}

		// 同一重要度カウンタをアップ
		try {
			long oldCount = getCounter(pk);
			// 最大カウント数を超えると、それ以上は増やさない（DBへのupdateを減らすための方策）
			if(oldCount <= MAX_INITIAL_NOTIFY_COUNT){
				if (monitorStatus == null) {
					try {
						monitorStatus = QueryUtil.getMonitorStatusPK(
								pk.getFacilityId(),
								pk.getPluginId(),
								pk.getMonitorId(),
								pk.getSubKey());
					} catch (Exception e) {
						// ここは通らないはず。
						m_log.warn("update() : "
								+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
					}
				}
				if (monitorStatus == null) {
					// ここは通らないはず。
					m_log.info("update() monitorStatus == null " +
							getCounter(pk) + ", " + getPriority(pk));
				} else {
					monitorStatus.setCounter(oldCount+1);
					monitorStatus.setLastUpdate(new Timestamp(generateDate));
				}
			}
		} catch (NotifyNotFound e) {
			// ここには到達しないはず。
			m_log.warn("update() : " + e.getMessage() + ", pk=" + pk);
		}
		return false;
	}

	public static boolean update(OutputBasicInfo output) {
		if(output.getSubKey() == null){
			m_log.info("SubKey is null. PluginId = " + output.getPluginId() +
					", MonitorId = " + output.getMonitorId() +
					", FacilityId = " + output.getFacilityId());
			output.setSubKey("");
		}

		return update(output.getFacilityId(), output.getPluginId(), output.getMonitorId(),
				output.getSubKey(), output.getGenerationDate(), output.getPriority());
	}
}
