/*

Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.factory;

import java.sql.Timestamp;
import java.util.Date;
import java.util.List;

import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyInfoDetail;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.bean.NotifyTypeConstant;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.bean.RenotifyTypeConstant;
import com.clustercontrol.notify.entity.MonitorStatusPK;
import com.clustercontrol.notify.model.NotifyHistoryEntity;
import com.clustercontrol.notify.model.NotifyHistoryEntityPK;
import com.clustercontrol.notify.model.NotifyInfoEntity;
import com.clustercontrol.notify.util.MonitorResultStatusUpdater;
import com.clustercontrol.notify.util.NotifyCache;
import com.clustercontrol.notify.util.QueryUtil;
import com.clustercontrol.plugin.impl.AsyncWorkerPlugin;
import com.clustercontrol.util.XMLUtil;

/**
 * 通知を通知ID毎のキューに振り分ける処理を行うユーティリティクラス
 */
public class NotifyDispatcher {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( NotifyDispatcher.class );

	public static final String MODE_VERBOSE = "verbose";
	public static final String MODE_NORMAL = "normal";

	private static final String NOTIFY_MODE_KEY = "common.notify.mode";

	private static String _mode = "normal";

	static {
		_mode = HinemosProperties.getProperty(NOTIFY_MODE_KEY, MODE_NORMAL);
		m_log.info("static() notify mode : " + _mode);
	}

	/**
	 * 引数で指定された情報を各種通知に出力します。<BR>
	 * イベント通知、ステータス通知、メール通知、ジョブ通知、ログエスカレーション通知を行います。
	 * 
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyEventBean#onMessage(javax.jms.Message)
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyStatusBean#onMessage(javax.jms.Message)
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyMailBean#onMessage(javax.jms.Message)
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyJobBean#onMessage(javax.jms.Message)
	 * @see com.clustercontrol.notify.ejb.mdb.NotifyLogEscalationBean#onMessage(javax.jms.Message)
	 * 
	 * @param info 通知出力情報
	 * @param notifyIdList 通知対象の通知ID
	 * @param queueDeliveryMode JMSの永続化モード指定
	 * @throws HinemosUnknown 
	 */
	public static void notifyAction(OutputBasicInfo info, List<String> notifyIdList, boolean persist) throws HinemosUnknown {
		if(info == null || notifyIdList == null) {
			m_log.info("notifyAction() invalid argument. info = " + info
					+ ", notifyIdList = " + notifyIdList
					+ ", persist = " + persist);
			return;
		}

		// 監視結果重要度ステータスを更新
		boolean prioityChangeFlag = MonitorResultStatusUpdater.update(info);

		Timestamp outputDate = new Timestamp(System.currentTimeMillis());

		// 通知対象のリスト分の通知種別を振り分ける。
		if(notifyIdList != null && notifyIdList.size() > 0) {
			for (String notifyId : notifyIdList) {
				NotifyInfo notifyInfo = NotifyCache.getNotifyInfo(notifyId);
				Integer notifyType = notifyInfo.getNotifyType();
				Integer notifyValid = notifyInfo.getValidFlg();
				if (notifyType == null || notifyValid == null) {
					// 該当の通知IDの設定が見つからない場合はエラーログを出力し次の通知IDの処理を継続する
					m_log.info("notifyAction() : notifyId = " + notifyId +" not found.");
					continue;
				}
				// 通知情報が「通知する」となっている場合
				if (notifyValid == ValidConstant.TYPE_VALID) {
					// デフォルトでは抑制せずに通知する（予期せぬエラーの場合は通知を行う）
					boolean isNotify = notifyCheck(
							info.getFacilityId(),
							info.getPluginId(),
							info.getMonitorId(),
							info.getSubKey(),
							notifyId,
							info.getPriority(),
							outputDate,
							prioityChangeFlag,
							_mode);

					if(isNotify){
						// 通知出力情報をディープコピーする（AsyncWorkerPlugin.addTaskのため）
						OutputBasicInfo clonedInfo = info.clone();
						
						// Ignore Invalid XML Chars
						clonedInfo.setMessage(XMLUtil.ignoreInvalidString(clonedInfo.getMessage()));
						clonedInfo.setMessageOrg(XMLUtil.ignoreInvalidString(clonedInfo.getMessageOrg()));
						
						// 通知IDは設定されていないため、ここで設定する
						// TODO queueDeliveryMode
						NotifyRequestMessage msg = new NotifyRequestMessage(
								clonedInfo,
								notifyId,
								outputDate,
								prioityChangeFlag
								);
						
						m_log.debug("add notify task for sending message"
								+ " : notifyId=" + notifyId
								+ ", pluginId=" + clonedInfo.getPluginId()
								+ ", priority=" + clonedInfo.getPriority()
								+ ", generationDate=" + clonedInfo.getGenerationDate()
								+ ", monitorId=" + clonedInfo.getMonitorId()
								+ ", facilityId=" + clonedInfo.getFacilityId());

						switch (notifyType) {
						case NotifyTypeConstant.TYPE_STATUS :
							AsyncWorkerPlugin.addTask(NotifyStatusTaskFactory.class.getSimpleName(), msg, persist);
							break;
						case NotifyTypeConstant.TYPE_EVENT :
							AsyncWorkerPlugin.addTask(NotifyEventTaskFactory.class.getSimpleName(), msg, persist);
							break;
						case NotifyTypeConstant.TYPE_MAIL :
							AsyncWorkerPlugin.addTask(NotifyMailTaskFactory.class.getSimpleName(), msg, persist);
							break;
						case NotifyTypeConstant.TYPE_COMMAND :
							AsyncWorkerPlugin.addTask(NotifyCommandTaskFactory.class.getSimpleName(), msg, persist);
							break;
						case NotifyTypeConstant.TYPE_LOG_ESCALATE :
							AsyncWorkerPlugin.addTask(NotifyLogEscalationTaskFactory.class.getSimpleName(), msg, persist);
							break;
						case NotifyTypeConstant.TYPE_JOB :
							AsyncWorkerPlugin.addTask(NotifyJobTaskFactory.class.getSimpleName(), msg, persist);
							break;
						default :
							m_log.warn("notify type is invalid. (notifyType = " + notifyType + ")");
						}
					}
				} else {
					// TODO:通知しない場合は、キャッシュ情報をアップデートしたほうが良い
				}
			}
		}
	}

	/**
	 * 抑制条件を確認し現時点で通知すべきか確認する。
	 * 
	 * @param facilityId ファシリティID
	 * @param pluginId プラグインID
	 * @param monitorId 監視ID
	 * @param subkey 通知抑制用サブキー
	 * @param notifyId 通知ID
	 * @param priority 重要度
	 * @param outputDate 出力日時
	 * @param priorityChangeFlag 重要度変更フラグ
	 * @param mode 抑制モード（VERBOSE:重要度が変化した場合は出力する, NORMAL:重要度が変化しても前回通知の重要度と同じ場合は通知しない）
	 * @return 通知する場合は true
	 */
	private static boolean notifyCheck(String facilityId, String pluginId, String monitorId, String subkey, String notifyId, int priority, Date outputDate, boolean priorityChangeFlag, String mode){

		JpaTransactionManager jtm = new JpaTransactionManager();
		EntityManager em = jtm.getEntityManager();

		if(m_log.isDebugEnabled()){
			m_log.debug("notifyCheck() " +
					"facilityId=" + facilityId +
					", pluginId=" +  pluginId +
					", monitorId=" + monitorId +
					", subkey=" + subkey +
					", notifyId=" + notifyId +
					", priority=" + priority +
					", outputDate=" + outputDate +
					", priorityChangeFlag=" + priorityChangeFlag +
					", mode="+ mode);
		}

		MonitorStatusPK mspk = new MonitorStatusPK(facilityId, pluginId, monitorId, subkey);

		// チェック処理を（ファシリティID, プラグインID, 監視項目ID, サブキー）の粒度で排他
		NotifyHistoryEntityPK entityPk = new NotifyHistoryEntityPK(facilityId, pluginId, monitorId, notifyId, subkey);
		String pkStr = "facilityId = " + facilityId
				+ ", pluginId  " + pluginId
				+ ", monitorId  " + monitorId
				+ ", notifyId  " + notifyId
				+ ", subkey  " + subkey;
		try {
			boolean isNotifyPriority = true;

			if(MODE_VERBOSE.equals(mode)){
				// 重要度変化があった場合
				if(priorityChangeFlag == true){
					m_log.debug("priorityChangeFlag == true. remove entity. pk = " + pkStr);

					// 通知履歴の該当タプルを削除する
					try {
						NotifyHistoryEntity entity = null;
						try {
							entity = QueryUtil.getNotifyHistoryPK(entityPk);
						} catch (NotifyNotFound e) {
						}
						if (entity != null) {
							em.remove(entity);
						}
					} catch (Exception e) {
						m_log.warn("notifyCheck() : "
								+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
					}
				}
			}

			// 重要度単位の通知フラグを確認する
			NotifyInfo notifyInfo = NotifyCache.getNotifyInfo(notifyId);
			NotifyInfoDetail notifyDetail = NotifyCache.getNotifyInfoDetail(notifyId, priority);

			// 重要度単位の有効無効を確認
			if(notifyDetail.getValidFlg().intValue() == ValidConstant.TYPE_INVALID){
				// 無効の場合は通知しない
				m_log.debug("ValidFlg is invalid. " + pkStr + ", priority = " + priority);
				m_log.debug("notify NG. (VALIDFLAG IS INVALID)." + pkStr);
				isNotifyPriority = false;
			}

			// 通知条件である同一重要度カウンタ数を満たしているか確認
			// 同一重要度カウンタを保持する監視結果ステータス情報を検索
			Long counter = null;
			try {
				// MonitorStatusの更新タイミングと時間差が出る可能性はほぼ少ないため、本処理箇所にてカウンタを取得する
				counter = MonitorResultStatusUpdater.getCounter(mspk);
			} catch (NotifyNotFound e) {
				// ジョブからの通知はここを通る。
				m_log.debug("notify OK. (MONITOR STATUS NOT FOUND)." + pkStr);
				return (true && isNotifyPriority);
			}

			if(counter >= notifyInfo.getInitialCount()){
				// カウンタが条件を満たしている場合
				m_log.debug("counter check. " + counter + " >= " + notifyInfo.getInitialCount() + "  " + pkStr);

				try{
					NotifyHistoryEntity history = null;
					if(!MODE_VERBOSE.equals(mode)){
						// 通知履歴の該当タプルを検索
						history = QueryUtil.getNotifyHistoryPK(entityPk);
						// 現在の監視結果の重要度と最終通知時の重要度が異なる場合は通知する
						if (priority != history.getPriority()) {
							m_log.debug("update notify history." + pkStr);
							history.setLastNotify(new Timestamp(outputDate.getTime()));
							history.setPriority(priority);
							m_log.debug("notify OK. (PRIORITY CHANGE)." + pkStr);
							return (true && isNotifyPriority);
						}
					}

					// 該当のタプルが存在する場合
					// 再通知種別を確認
					if(notifyInfo.getRenotifyType() == RenotifyTypeConstant.TYPE_NO_NOTIFY){
						// 再通知なしの場合
						m_log.debug("notify NG. (RENOTIFY NO)." + pkStr);
						return false;
					} else if(notifyInfo.getRenotifyType() == RenotifyTypeConstant.TYPE_ALWAYS_NOTIFY){
						// 常に再通知の場合
						// history.setLastNotify(new Timestamp(outputDate.getTime())); 常に通知するため更新の必要がない
						// history.setPriority(priority); 常に通知するため更新の必要がない
						m_log.debug("notify OK. (RENOTIFY ALWAYS)." + pkStr);
						return (true && isNotifyPriority);
					} else {
						if (history == null) {
							// 通知履歴の該当タプルを検索
							history = QueryUtil.getNotifyHistoryPK(entityPk);
						}
						if(outputDate != null && outputDate.getTime() >=
								(history.getLastNotify().getTime() + (notifyInfo.getRenotifyPeriod() * 60 * 1000l))){
							m_log.debug("update notify history." + pkStr);
							// 通知時刻が抑制期間を超えている場合
							history.setLastNotify(new Timestamp(outputDate.getTime()));
							history.setPriority(priority);
							m_log.debug("notify OK. (RENOTIFY PERIOD)." + pkStr);
							return (true && isNotifyPriority);
						} else {
							m_log.debug("notify NG. (RENOTIFY PERIOD)." + pkStr);
							return false;
						}
					}
				} catch (NotifyNotFound e) {
					// 該当のタプルが存在しない場合
					// 初回通知
					m_log.debug("first notify. " + pkStr + ", priority = " + priority);

					// 新規に通知履歴を作成する
					NotifyInfoEntity notifyInfoEntity = null;
					try {
						notifyInfoEntity = QueryUtil.getNotifyInfoPK(notifyId);
					} catch (NotifyNotFound e1) {
					} catch (InvalidRole e1) {
					}
					NotifyHistoryEntity newHistoryEntity
					= new NotifyHistoryEntity(notifyInfoEntity, facilityId, pluginId, monitorId, subkey);
					// 重複チェック
					jtm.checkEntityExists(NotifyHistoryEntity.class, newHistoryEntity.getId());
					newHistoryEntity.setLastNotify(outputDate==null?null:new Timestamp(outputDate.getTime()));
					newHistoryEntity.setPriority(priority);
					m_log.debug("notify OK. (NEW)." + pkStr);
					return (true && isNotifyPriority);
				}
			} else {
				m_log.debug("notify NG. (PRIORITY CHANGE)." + pkStr);
				return false;
			}
		} catch (EntityExistsException e) {
			m_log.info("notifyCheck() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage());
		} catch (Exception e) {
			m_log.warn("notifyCheck() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
		}

		m_log.info("notifyCheck() notify OK. (cause unexpected errors)" + pkStr);
		return true;
	}
}
