/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.collectiverun.factory;

import java.util.Date;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.collectiverun.message.CollectiveRunInstructionInfo;
import com.clustercontrol.collectiverun.session.CollectiveRunControllerBean;
import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.commons.util.JpaTransactionManager;
import com.clustercontrol.fault.CollectiveRunNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.jobmanagement.bean.RunResultInfo;
import com.clustercontrol.jobmanagement.bean.RunStatusConstant;
import com.clustercontrol.plugin.api.AsyncTaskFactory;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.session.RepositoryControllerBean;

public class CollectiveRunExecuteTaskFactory implements AsyncTaskFactory {

	public static final Log log = LogFactory.getLog(CollectiveRunExecuteTaskFactory.class);

	private static final String RSH = "rsh";
	private static final String SSH = "ssh";

	@Override
	public Runnable createTask(Object param) {
		return new CollectiveRunExecuteTask(param);
	}

	public class CollectiveRunExecuteTask implements Runnable {

		private final CollectiveRunInstructionInfo msg;

		public CollectiveRunExecuteTask(Object param) {
			if (param instanceof CollectiveRunInstructionInfo) {
				msg = (CollectiveRunInstructionInfo)param;
			} else {
				msg = null;
			}
		}

		@Override
		public void run() {

			if (msg == null) {
				log.warn("message is not assigned.");
				return;
			}

			if (log.isDebugEnabled()) {
				log.debug("message received. : SessionID = " + msg.getSessionId() + ", FacilityID = " + msg.getFacilityId());
			}
			
			JpaTransactionManager jtm = null;

			try {
				jtm = new JpaTransactionManager();
				jtm.begin();
				
				try {
					//コマンド実行
					exec(msg);
				} catch (Exception e) {
					//コマンド実行にて、想定外のExceptionが発生した場合、実行失敗とする

					log.warn("onMessage() : "
							+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);

					//メッセージ作成
					RunResultInfo resultInfo = null;
					resultInfo = new RunResultInfo();
					resultInfo.setSessionId(msg.getSessionId());
					resultInfo.setFacilityId(msg.getFacilityId());
					resultInfo.setCommandType(msg.getCommandType());
					resultInfo.setStatus(RunStatusConstant.START);
					resultInfo.setTime(new Date().getTime());

					// 送信
					try {
						new CollectiveRunControllerBean().endNode(resultInfo);
					} catch (HinemosUnknown | CollectiveRunNotFound e2) {
						log.warn(e2.getMessage(), e2);
					}

					//メッセージ作成
					resultInfo = new RunResultInfo();
					resultInfo.setSessionId(msg.getSessionId());
					resultInfo.setFacilityId(msg.getFacilityId());
					resultInfo.setCommandType(msg.getCommandType());
					resultInfo.setStatus(RunStatusConstant.ERROR);
					resultInfo.setTime(new Date().getTime());
					resultInfo.setErrorMessage(e.getMessage());

					//送信
					try {
						new CollectiveRunControllerBean().endNode(resultInfo);
					} catch (HinemosUnknown | CollectiveRunNotFound e2) {
						log.warn(e2.getMessage(), e2);
					}
				}
				
				jtm.commit();
			} catch (Exception e) {
				jtm.rollback();
			} finally {
				jtm.close();
			}
		}

		@Override
		public String toString() {
			return this.getClass().getSimpleName() + "[" + msg.toString() + "]";
		}

	}

	public void exec(CollectiveRunInstructionInfo instructionInfo) {
		log.debug("run start :" + instructionInfo.getFacilityId());

		//---------------------------
		//-- 開始メッセージ送信
		//---------------------------

		//メッセージ作成
		RunResultInfo resultInfo = null;
		resultInfo = new RunResultInfo();
		resultInfo.setSessionId(instructionInfo.getSessionId());
		resultInfo.setJobId(instructionInfo.getJobId());
		resultInfo.setFacilityId(instructionInfo.getFacilityId());
		resultInfo.setCommandType(instructionInfo.getCommandType());
		resultInfo.setStatus(RunStatusConstant.START);
		resultInfo.setTime(new Date().getTime());

		log.debug("run SessionID=" + instructionInfo.getSessionId() + ", JobID="
				+ instructionInfo.getJobId());

		//送信
		try {
			new CollectiveRunControllerBean().endNode(resultInfo);
		} catch (HinemosUnknown | CollectiveRunNotFound e) {
			log.warn(e.getMessage(), e);
		}

		//---------------------------
		//-- コマンド実行
		//---------------------------

		//リモート実行ホスト名を取得
		String host = "";
		try {
			host = getRemoteHostName(instructionInfo.getFacilityId());
		} catch (Exception e) {
			//終了メッセージの種別をセット
			resultInfo = new RunResultInfo();
			resultInfo.setSessionId(instructionInfo.getSessionId());
			resultInfo.setJobId(instructionInfo.getJobId());
			resultInfo.setFacilityId(instructionInfo.getFacilityId());
			resultInfo.setCommandType(instructionInfo.getCommandType());
			resultInfo.setStatus(RunStatusConstant.ERROR);
			resultInfo.setTime(new Date().getTime());
			resultInfo.setErrorMessage(e.getMessage());

			//送信
			try {
				new CollectiveRunControllerBean().endNode(resultInfo);
			} catch (HinemosUnknown | CollectiveRunNotFound e1) {
				log.warn(e1.getMessage(), e1);
			}
			return;
		}

		//コマンド終了値取得用文字列「セッションID:ret=」
		String endValueStr = instructionInfo.getSessionId() + ":ret=";

		//実行コマンド組み立て
		String cmd[] = new String[5];
		cmd[0] = getRemoteExec();
		cmd[1] = host;
		cmd[2] = "-l";
		cmd[3] = getRemoteUser();
		//        cmd[4] = instructionInfo.getCommand() + ";echo " + endValueStr + "$?;sleep 1";
		cmd[4] = instructionInfo.getCommand() + ";echo " + endValueStr + "$?;read ANS";
		//      sleep はコマンド終了時に標準出力が切れてしまうので暫定

		CommandExecutor executor = new CommandExecutor();

		//        boolean ret = executor.exec(cmd, instructionInfo.getInputFile());
		boolean ret = executor.exec(cmd, instructionInfo.getInputFile(), endValueStr);
		int exitValue = executor.getExitValue();
		String resultMsg = executor.getResultMsg();

		//---------------------------
		//-- 終了メッセージ送信
		//---------------------------
		boolean errFlg = true;
		if (ret) {
			//リモートシェル自体の実行ステータスをチェック
			if (exitValue == 0) {
				//リモートシェルが正常ならば、コマンドの終了値を標準出力から取得する
				int idx = resultMsg.lastIndexOf(endValueStr);
				if (idx != -1) {
					//コマンドの終了値が取得できれば、終了値をコマンド終了値にする
					String endValue = resultMsg.substring(idx
							+ endValueStr.length(), resultMsg.length() - 1);
					log.debug(endValue);
					exitValue = Integer.parseInt(endValue);
					log.debug("int:" + exitValue);
					resultMsg = resultMsg.substring(0, idx);
					errFlg = false;
				}
			}
		}

		//終了メッセージの種別をセット
		resultInfo = new RunResultInfo();
		resultInfo.setSessionId(instructionInfo.getSessionId());
		resultInfo.setJobId(instructionInfo.getJobId());
		resultInfo.setFacilityId(instructionInfo.getFacilityId());
		resultInfo.setCommandType(instructionInfo.getCommandType());
		if (errFlg) {
			resultInfo.setStatus(RunStatusConstant.ERROR);
		} else {
			resultInfo.setStatus(RunStatusConstant.END);
			resultInfo.setEndValue(exitValue);
		}
		//終了を送信
		resultInfo.setTime(new Date().getTime());
		resultInfo.setErrorMessage(executor.getErrMsg());
		resultInfo.setMessage(resultMsg);

		//送信
		try {
			new CollectiveRunControllerBean().endNode(resultInfo);
		} catch (HinemosUnknown | CollectiveRunNotFound e) {
			log.warn(e.getMessage(), e);
		}

		log.debug("run end :" + instructionInfo.getFacilityId());
	}

	/**
	 * リモートホスト取得. ファシリティIDからIPアドレスまたは、ホスト名を返す。
	 * 
	 * @param facilityID
	 *            ファシリティID
	 * @return IPアドレスまたはホスト名
	 * @since
	 */
	private String getRemoteHostName(String facilityID) throws Exception {
		String host = null;
		//RepositoryControllerを取得
		try {
			NodeInfo info = new RepositoryControllerBean().getNode(facilityID);
			//IPが取得できれば、IP、取得できなければ、ホスト名(ノード名)を返す
			host = info.getAvailableIpAddress();
			if (host == null || host.length() == 0) {
				host = info.getNodeName();;
				if (host == null) {
					host = "";
				}
			}

			if (log.isDebugEnabled()) {
				log.debug("getRemoteHostName() : HostName=" + host);
			}

		} catch (Exception e) {
			log.warn("getRemoteHostName() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			throw e;
		}

		return host;
	}

	/**
	 * 実行シェルをプロパティファイルから取得
	 * @return remoteExec
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	private String getRemoteExec() {
		String remoteExec = RSH;
		//FIXME RuntimeのExceptionが発生する恐れがあるためtry-catchとする
		try {
			// プロパティファイルからキーと値のリストを読み込みます
			remoteExec = HinemosProperties.getProperty("collective.run.shell", "");
			if(remoteExec.equalsIgnoreCase(RSH)){
				remoteExec = RSH;
			}
			else if(remoteExec.equalsIgnoreCase(SSH)){
				remoteExec = SSH;
			}
		} catch (Exception e) {
			log.debug("getRemoteExec() : " + e.getMessage());
		}
		log.debug("getRemoteExec() remoteExec = " + remoteExec);
		return remoteExec;
	}

	/**
	 * リモートシェル実行時ユーザをプロパティファイルから取得
	 * @return userName
	 */
	private String getRemoteUser(){
		String userName = HinemosProperties.getProperty("collective.run.user", "root");
		log.debug("getRemoteUser() userName = " + userName);
		return userName;
	}

}
