/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.repository.dialog;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import com.clustercontrol.accesscontrol.util.LocalSession;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.repository.composite.NodeFilterComposite;
import com.clustercontrol.repository.util.RepositoryEndpointWrapper;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.repository.InvalidRole_Exception;

/**
 * スコープへのノード割り当てダイアログクラス<BR>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class NodeAssignDialog extends CommonDialog {

	// 後でpackするためsizeXはダミーの値。
	private final int sizeX = 500;
	private final int sizeY = 500;

	// ----- instance フィールド ----- //

	/** 割り当て対象スコープのファシリティID */
	private String facilityId = "";

	/** ノード一覧テーブル */
	private NodeFilterComposite nodeList = null;

	/** 選択されたアイテム */
	private List<String> filterItems = null;

	// ----- コンストラクタ ----- //

	/**
	 * 指定した形式のダイアログのインスタンスを返します。
	 * 
	 * @param parent
	 *            親のシェルオブジェクト
	 * @param facilityId
	 *            初期表示するスコープのファシリティID
	 */
	public NodeAssignDialog(Shell parent, String facilityId) {
		super(parent);

		this.facilityId = facilityId;
	}

	// ----- instance メソッド ----- //

	/**
	 * ダイアログの初期サイズを返します。
	 * 
	 * @return 初期サイズ
	 */
	@Override
	protected Point getInitialSize() {
		return new Point(500, 500);
	}

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent
	 *            親のインスタンス
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		Shell shell = this.getShell();

		// タイトル
		shell.setText(Messages.getString("dialog.repository.select.nodes"));

		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		parent.setLayout(layout);

		/*
		 * ノード一覧
		 */

		// テーブル
		this.nodeList = new NodeFilterComposite(parent, SWT.NONE, facilityId, true);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 1;
		this.nodeList.setLayoutData(gridData);

		this.nodeList.update();

		// ラインを引く
		Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 1;
		line.setLayoutData(gridData);

		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);

		//ダイアログのサイズ調整（pack:resize to be its preferred size）
		shell.pack();
		shell.setSize(new Point(shell.getSize().x, sizeY ));

	}

	/**
	 * 既存のボタンに加え、検索ボタンを追加します。
	 * 
	 * @param parent
	 *            ボタンバーコンポジット
	 */
	@Override
	protected void createButtonsForButtonBar(Composite parent) {
		// 検索ボタン
		this.createButton(parent, IDialogConstants.OPEN_ID, Messages
				.getString("filter"), false);
		this.getButton(IDialogConstants.OPEN_ID).addSelectionListener(
				new SelectionAdapter() {
					@Override
					public void widgetSelected(SelectionEvent e) {
						// 検索処理
						NodeFilterDialog dialog = new NodeFilterDialog(
								getShell());
						if (dialog.open() == IDialogConstants.OK_ID) {
							nodeList.update(dialog.getInputData());
						}
					}
				});

		super.createButtonsForButtonBar(parent);
	}

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
	@Override
	protected ValidateResult validate() {
		return super.validate();
	}

	/**
	 * 割り当て対象スコープのファシリティIDを返します。
	 * 
	 * @return 割り当て対象スコープのファシリティID
	 */
	public String getScopeId() {
		return this.facilityId;
	}

	/**
	 * 選択されたノードを返します。
	 * 
	 * @return ノードのファシリティID
	 */
	public List<String> getFilterItems() {
		return this.filterItems;
	}

	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		return Messages.getString("assign");
	}

	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}

	/**
	 * OKボタンが押下された場合の動作。
	 */
	@Override
	protected void okPressed() {

		// 画面情報を取得
		StructuredSelection selection = (StructuredSelection) this.nodeList
				.getTableViewer().getSelection();

		Object[] items = selection.toArray();
		if (items != null) {
			int size = items.length;
			this.filterItems = new ArrayList<String>();
			for (int i = 0; i < size; i++) {
				this.filterItems.add((String) ((ArrayList) items[i]).get(0));
			}
		}

		// 登録
		try {
			RepositoryEndpointWrapper.assignNodeScope(facilityId, this.filterItems);

			// リポジトリキャッシュの更新
			LocalSession.doCheck();

			// 成功報告ダイアログを生成
			MessageDialog.openInformation(
					null,
					Messages.getString("successful"),
					Messages.getString("message.repository.6"));

			super.okPressed();

		} catch (Exception e) {
			String errMessage = "";
			if (e instanceof InvalidRole_Exception) {
				// アクセス権なしの場合、エラーダイアログを表示する
				MessageDialog.openInformation(null, Messages.getString("message"),
						Messages.getString("message.accesscontrol.16"));
			} else {
				errMessage = ", " + e.getMessage();
			}

			// 失敗報告ダイアログを生成
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.repository.7") + errMessage);
		}
	}
}