/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */
package com.clustercontrol.performance.view;

import java.util.HashMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.jfree.chart.JFreeChart;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.performance.action.GetPerformanceDataSettings;
import com.clustercontrol.performance.composite.PerformanceGraphComposite;
import com.clustercontrol.performance.preference.PerformancePreferencePage;
import com.clustercontrol.util.Messages;
import com.clustercontrol.view.ScopeListBaseView;
import com.clustercontrol.ws.collector.PerformanceDataSettings;
import com.clustercontrol.ws.repository.FacilityTreeItem;

/**
 * 性能[グラフ]ビュー
 * 
 * ビュー名は性能[監視ID]として表示する
 * 
 * @version 4.0.0
 * @since 4.0.0
 * 
 */
public class PerformanceGraphView extends ScopeListBaseView {
	private static Log log = LogFactory.getLog(PerformanceGraphView.class);

	public static final String ID = "com.clustercontrol.performance.view.PerformanceGraphView";
	public static int GRAPH_MAX; // グラフ表示線の最大数

	private String monitorId = null;	// ビュータイトルに表示する監視ID
	private PerformanceGraphComposite graphComposite;
	private static HashMap<String, PerformanceDataSettings> settingMap = new HashMap<String, PerformanceDataSettings>();

	static {
		GRAPH_MAX =
				ClusterControlPlugin.getDefault().getPreferenceStore().getInt(
						PerformancePreferencePage.P_GRAPH_MAX);
		log.info("PerformanceGraphView : GRAPH_MAX = " + GRAPH_MAX);
	}

	/**
	 * デフォルトコンストラクタ
	 */
	public PerformanceGraphView() {
		// ノードを含め、INTERNAL, UNREGISTEREDを含めず, TOPICを受信しないスコープツリーとする
		super(false, false, false, false);
	}

	/**
	 * コンポーネントの作成(初期起動)
	 */
	@Override
	protected Composite createListContents(Composite parent) {
		log.debug("createListContents()");

		super.getScopeTreeComposite().setScopeTree(null);

		// セカンダリIDとして登録した監視IDを元に、ビュー名をセット
		this.monitorId = this.getViewSite().getSecondaryId();
		String args[] = {this.monitorId};
		this.setPartName(Messages.getString("view.performance.graph", args));

		// マネージャから情報を取得
		PerformanceDataSettings perfDataSettings = getPerformanceDataSettings(this.monitorId);

		// FacilityTree情報をセット
		super.getScopeTreeComposite().setScopeTree(perfDataSettings.getFacilityTreeItem());

		// ビューのレイアウト設定
		GridLayout layout = new GridLayout(1, true);
		layout.marginHeight = 0;
		layout.marginWidth = 0;
		parent.setLayout(layout);

		// ビューに張るコンポジットの初期化
		this.graphComposite = new PerformanceGraphComposite(perfDataSettings, parent, SWT.NONE);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		this.graphComposite.setLayoutData(gridData);

		// 完了
		return this.graphComposite;
	}


	/**
	 * ビュー更新
	 */
	@Override
	public void update() {
		log.debug("update()");
		this.graphComposite.update();
	}


	/**
	 * ファシリティツリーを選択した場合にコールされるメソッド
	 */
	@Override
	protected void doSelectTreeItem(FacilityTreeItem item) {
		super.doSelectTreeItem(item);
		log.debug("doSelectTreeItem()");
		// ファシリティの更新によるグラフの更新
		this.graphComposite.updateGraph(item);
	}

	/**
	 * ビューの自動更新開始
	 */
	@Override
	public void startAutoReload() {
		log.debug("startAutoReload() title = " + this.getTitle());
		super.startAutoReload();
	}

	/**
	 * ビューの自動更新停止
	 */
	@Override
	public void stopAutoReload() {
		log.debug("stopAutoReload() title = " + this.getTitle());
		super.stopAutoReload();
	}

	/**
	 * ビューの自動更新間隔の設定
	 */
	@Override
	public void setInterval(int interval) {
		log.debug("setInterval() interval = " + interval);
		super.setInterval(interval);
	}

	/**
	 * チャートの取得
	 */
	public JFreeChart getChart() {
		log.debug("getChart()");
		return this.graphComposite.getChart();
	}

	/**
	 * グラフの表示本数の最大値を設定する
	 * @param max
	 */
	public static void setGraphMax(int max){
		log.info("setGraphMax() max = " + max);
		GRAPH_MAX = max;
	}

	/**
	 * 取得済みのグラフヘッダ情報を取得する
	 * 
	 * @param moniorId
	 * @return
	 */
	private PerformanceDataSettings getPerformanceDataSettings(String moniorId){
		PerformanceDataSettings perfDataSettings = settingMap.get(moniorId);
		if(perfDataSettings == null){
			perfDataSettings = GetPerformanceDataSettings.getPerformanceGraphInfo(this.monitorId);
		}
		return perfDataSettings;
	}

	/**
	 * 取得済みのグラフヘッダ情報を設定する
	 * 
	 * @param monitorId
	 * @param perfDataSettings
	 */
	public static void setPerformanceDataSettings(String monitorId, PerformanceDataSettings perfDataSettings){
		settingMap.put(monitorId, perfDataSettings);
	}
}
