/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.composite;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.accesscontrol.util.LocalSession;
import com.clustercontrol.bean.Property;
import com.clustercontrol.jobmanagement.action.GetPlanTableDefine;
import com.clustercontrol.jobmanagement.composite.action.SessionJobDoubleClickListener;
import com.clustercontrol.jobmanagement.preference.JobManagementPreferencePage;
import com.clustercontrol.jobmanagement.util.JobEndpointWrapper;
import com.clustercontrol.jobmanagement.util.JobPropertyUtil;
import com.clustercontrol.util.Messages;
import com.clustercontrol.viewer.CommonTableViewer;
import com.clustercontrol.ws.jobmanagement.InvalidRole_Exception;
import com.clustercontrol.ws.jobmanagement.JobPlan;
import com.clustercontrol.ws.jobmanagement.JobPlanFilter;
import com.sun.xml.internal.ws.client.ClientTransportException;

/**
 * ジョブ[スケジュール予定]ビュー用のコンポジットクラスです。
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class JobPlanComposite extends Composite {

	// ログ
	private static Log m_log = LogFactory.getLog( JobPlanComposite.class );

	/** テーブルビューア */
	private CommonTableViewer m_viewer = null;
	/** ヘッダ用ラベル */
	private Label m_labelType = null;
	/** 件数用ラベル */
	private Label m_labelCount = null;

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親のコンポジット
	 * @param style スタイル
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public JobPlanComposite(Composite parent, int style) {
		super(parent, style);
		initialize();
	}

	/**
	 * コンポジットを配置します。
	 */
	private void initialize() {
		GridLayout layout = new GridLayout(1, true);
		this.setLayout(layout);
		layout.marginHeight = 0;
		layout.marginWidth = 0;

		m_labelType = new Label(this, SWT.LEFT);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		m_labelType.setLayoutData(gridData);

		Table table = new Table(this, SWT.H_SCROLL | SWT.V_SCROLL
				| SWT.FULL_SELECTION | SWT.SINGLE);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);

		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 1;
		table.setLayoutData(gridData);

		m_labelCount = new Label(this, SWT.RIGHT);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		m_labelCount.setLayoutData(gridData);

		m_viewer = new CommonTableViewer(table);
		m_viewer.createTableColumn(GetPlanTableDefine.get(),
				GetPlanTableDefine.SORT_COLUMN_INDEX,
				GetPlanTableDefine.SORT_ORDER);

		m_viewer.addDoubleClickListener(
				new SessionJobDoubleClickListener(this));
	}

	/**
	 * テーブルビューアを更新します。<BR>
	 * ジョブ[スケジュール予定]一覧情報を取得し、テーブルビューアにセットします。
	 * <p>
	 * <ol>
	 * <li>ジョブ管理のプレファレンスページより、ジョブ[スケジュール予定]ビューの表示件数を取得します。</li>
	 * <li>ジョブ[スケジュール予定]一覧情報を、表示数分取得します。</li>
	 * <li>テーブルビューアにジョブ[スケジュール予定]一覧情報をセットします。</li>
	 * </ol>
	 * 
	 */
	@Override
	public void update() {
		update(null);
	}

	/**
	 * テーブルビューアを更新します。<BR>
	 * 引数で指定された条件に一致するジョブ[スケジュール予定]情報を取得し、テーブルビューアにセットします。
	 * <p>
	 * <ol>
	 * <li>ジョブ管理のプレファレンスページより、ジョブ[スケジュール予定]ビューの表示数を取得します。</li>
	 * <li>引数で指定された条件に一致するジョブ[スケジュール予定]一覧情報を、表示数分取得します。</li>
	 * <li>テーブルビューアにジョブ[スケジュール予定]一覧情報をセットします。</li>
	 * </ol>
	 * 
	 * @param condition 検索条件
	 * 
	 */
	public void update(Property condition) {
		List<JobPlan> planList = new ArrayList<JobPlan>();
		//ジョブ[スケジュール予定]情報取得
		int plans = ClusterControlPlugin.getDefault().getPreferenceStore().getInt(
				JobManagementPreferencePage.P_PLAN_MAX_SCHEDULE);
		try {
			if (condition == null) {
				planList = JobEndpointWrapper.getPlanList(null,plans);
			} else {
				planList = JobEndpointWrapper.getPlanList(
						JobPropertyUtil.property2jobPlanFilter(condition), plans);

				JobPlanFilter filter = new JobPlanFilter();
				filter = JobPropertyUtil.property2jobPlanFilter(condition);
				filter.toString();
			}
		} catch (InvalidRole_Exception e) {
			if(LocalSession.isDialogFlag()){
				LocalSession.setDialogFlag(false);
				// アクセス権なしの場合、エラーダイアログを表示する
				MessageDialog.openInformation(null, Messages.getString("message"),
						Messages.getString("message.accesscontrol.16"));
				LocalSession.setDialogFlag(true);
			}
		} catch (Exception e) {
			if(LocalSession.isDialogFlag()){
				LocalSession.setDialogFlag(false);
				if (e instanceof ClientTransportException) {
					m_log.warn("update() getPlanList, " + e.getMessage());
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.hinemos.failure.transfer") + ", " + e.getMessage());
				} else {
					m_log.warn("update() getPlanList, " + e.getMessage(), e);
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
				}
				LocalSession.setDialogFlag(true);
			}
		}
		ArrayList<Object> listInput = new ArrayList<Object>();
		SimpleDateFormat sdfYmd = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss");

		for (JobPlan plan : planList) {
			ArrayList<Object> a = new ArrayList<Object>();
			String date = sdfYmd.format(plan.getDate());
			a.add(date);
			a.add(plan.getJobKickId());
			a.add(plan.getJobKickName());
			a.add(plan.getJobunitId());
			a.add(plan.getJobId());
			a.add(plan.getJobName());

			listInput.add(a);
		}
		m_viewer.setInput(listInput);

		//selectPlan(listInput);

		if (condition != null) {
			m_labelType.setText(Messages.getString("filtered.list"));
			Object[] args = { new Integer(planList.size()) };
			m_labelCount.setText(Messages.getString("filtered.records", args));
		}
		else {
			// 表示件数をセット(最大件数以上に達しているか否かの分岐)
			m_labelType.setText("");
			Object[] args = null;
			if(plans > planList.size()){
				args = new Object[]{ planList.size() };
			} else {
				args = new Object[]{ plans };
			}
			m_labelCount.setText(Messages.getString("records", args));
		}
	}
}
