/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.collectiverun.util;

import java.rmi.AccessException;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;

import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.PropertyDefineConstant;
import com.clustercontrol.collectiverun.bean.EditorTypeConstant;
import com.clustercontrol.collectiverun.bean.ParameterConstant;
import com.clustercontrol.collectiverun.bean.ParameterTypeConstant;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.PropertyUtil;
import com.clustercontrol.ws.collectiverun.CommandInfo;
import com.clustercontrol.ws.collectiverun.CommandParameterInfo;
import com.clustercontrol.ws.collectiverun.CommandTypeInfo;
import com.clustercontrol.ws.collectiverun.InvalidRole_Exception;
import com.clustercontrol.ws.collectiverun.ParameterInfo;
import com.clustercontrol.ws.collectiverun.ParameterSelectInfo;
import com.clustercontrol.ws.collectiverun.SessionInfo;
import com.clustercontrol.ws.collectiverun.SessionParameterInfo;
import com.clustercontrol.ws.repository.FacilityInfo;
import com.clustercontrol.ws.repository.FacilityTreeItem;

/**
 * 一括制御パラメータ情報のDTOとプロパティを相互変換するためのユーティリティクラスです。
 * @since 0.1
 */
public class ParameterPropertyUtil {

	// ログ
	private static Log m_log = LogFactory.getLog( ParameterPropertyUtil.class );

	private static final Integer FACILITY_TYPE_DEFAULT = 1;
	private static final Integer DISPLAY_SORT_ORDER_DEFAULT = 100;

	/**
	 * プロパティを返します。
	 * 
	 * @param local
	 * @return プロパティ
	 * @throws FinderException
	 * @throws NamingException
	 * @throws RemoteException
	 */
	public static Property getPropertyBySessionId(String sessionId, Locale locale) throws InvalidRole_Exception {

		//マネージャからセッション情報とコマンドマスタを取得する

		SessionInfo sessionInfo = null;
		CommandInfo commandMst = null;

		try {
			sessionInfo = CollectiveRunEndpointWrapper.getSessionInfo(sessionId);
			commandMst = CollectiveRunEndpointWrapper.getCommandMst(sessionInfo.getCommandId());
		} catch (InvalidRole_Exception e) {
			throw e;
		} catch (Exception e) {
			m_log.warn("getPropertyBySessionId(), " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}

		//種別IDからプロパティを取得
		Property property = null;
		try {
			property = getParameterProperty(commandMst.getCommandTypeId(), locale);
		} catch (InvalidRole_Exception e) {
			throw e;
		}
		if(property instanceof Property){

			//名前プロパティの子プロパティを切り替える
			selectNameProperty(property,
					Messages.getString(commandMst.getName(), locale), locale);

			//パラメータプロパティに値を設定する
			setPropertyValue(property, sessionInfo, locale);
		}

		return property;
	}

	/**
	 * コマンドパラメータ用プロパティ取得処理
	 * 
	 * @param cmdParam
	 * @return コマンドパラメータ用プロパティ
	 * @throws FinderException
	 * @throws NamingException
	 */
	private static Property getCommandParameterProperty(CommandParameterInfo cmdParam, Locale locale) throws NamingException {
		Property parameter = null;

		if(cmdParam != null){
			ParameterInfo param = cmdParam.getParamMst();
			int paramType = param.getParamType();

			//パラメータ用プロパティ作成
			parameter = new Property(
					param.getId(),
					Messages.getString(param.getName(), locale),
					EditorTypeConstant.paramTypeToEditor(paramType));
			//パラメータ用プロパティの入力制限を設定
			if(paramType == ParameterTypeConstant.TYPE_STRING ||
					paramType == ParameterTypeConstant.TYPE_PASSWORD){
				//文字列及びパスワードの場合、DBの上限で制限
				parameter.setStringUpperValue(DataRangeConstant.VARCHAR_4096);
			}
			parameter.setModify(PropertyDefineConstant.MODIFY_OK);

			//値の初期化
			parameter.setValue("");
			if(paramType == ParameterTypeConstant.TYPE_SELECT){
				List<ParameterSelectInfo> paramSelectList = param.getSelectMsts();
				if(paramSelectList != null && paramSelectList.size() > 0){
					//値を初期化
					Object selectValues[][] = {new Object[paramSelectList.size()], new Object[paramSelectList.size()]};

					for(ParameterSelectInfo paramSelect : paramSelectList){
						int index = paramSelect.getOrderNo();

						selectValues[PropertyDefineConstant.SELECT_VALUE][index] = Messages.getString(paramSelect.getName(), locale);
						selectValues[PropertyDefineConstant.SELECT_DISP_TEXT][index] = Messages.getString(paramSelect.getName(), locale);
					}

					parameter.setSelectValues(selectValues);
					parameter.setValue(selectValues[PropertyDefineConstant.SELECT_DISP_TEXT][0]);
				}
			}
		}

		return parameter;
	}

	/**
	 * パラメータ用プロパティを返します。
	 * 
	 * @param typeId
	 * @param locale
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 * @throws AccessException
	 */
	@SuppressWarnings("unchecked")
	public static Property getParameterProperty(String typeId, Locale locale) throws InvalidRole_Exception {
		//スコープ
		Property scope =
				new Property(ParameterConstant.SCOPE, Messages.getString("scope", locale), PropertyDefineConstant.EDITOR_FACILITY);
		scope.setModify(PropertyDefineConstant.MODIFY_OK);
		//名前
		Property name =
				new Property(ParameterConstant.NAME, Messages.getString("name", locale), PropertyDefineConstant.EDITOR_SELECT);
		name.setModify(PropertyDefineConstant.MODIFY_OK);

		//マネージャからコマンド種別マスタを取得する
		CommandTypeInfo cmdTypeMst = null;
		try {
			cmdTypeMst = CollectiveRunEndpointWrapper.getCommandTypeMst(typeId);
		} catch (InvalidRole_Exception e) {
			throw e;
		} catch (Exception e) {
			m_log.warn("getParameterProperty(), " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}

		//コマンドパラメータ用のプロパティを格納するHashMap
		HashMap propertyMap = new HashMap();

		//種別に紐づく全コマンドマスタ取得
		List<CommandInfo> cmdList = cmdTypeMst.getCmdMsts();
		if(cmdList != null && cmdList.size() > 0){
			//値を初期化
			Object nameValues[][] = {new Object[cmdList.size()], new Object[cmdList.size()]};
			for(CommandInfo cmd : cmdList){
				int index = cmd.getOrderNo();

				ArrayList list = new ArrayList();
				//コマンドに紐づく全コマンドパラメータを取得
				List<CommandParameterInfo> cmdParamList = null;
				try {
					cmdParamList = CollectiveRunEndpointWrapper.getCommandParameterMstList(cmd.getId());
				} catch (InvalidRole_Exception e) {
					throw e;
				} catch (Exception e) {
					m_log.warn("getParameterProperty() getCommandParameterMstList, " + e.getMessage(), e);
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
				}

				if(cmdParamList != null && cmdParamList.size() > 0){
					for(CommandParameterInfo cmdParam : cmdParamList){
						//コマンドパラメータ用プロパティを取得
						Property parameter = null;
						try {
							parameter = getCommandParameterProperty(cmdParam, locale);
						} catch (NamingException e) {
						}

						//既にコマンドパラメータ用プロパティが格納済みかチェック
						if(propertyMap.get(cmdParam.getId()) == null){
							//未格納の場合
							list.add(parameter);
							propertyMap.put(cmdParam.getId(), parameter);
						}
						else{
							//格納済み
							list.add(propertyMap.get(cmdParam.getId()));
						}
					}
				}

				nameValues[PropertyDefineConstant.SELECT_DISP_TEXT][index] = Messages.getString(cmd.getName(), locale);
				HashMap map = new HashMap();
				map.put(PropertyDefineConstant.MAP_VALUE, Messages.getString(cmd.getName(), locale));
				map.put(PropertyDefineConstant.MAP_PROPERTY, list);
				nameValues[PropertyDefineConstant.SELECT_VALUE][index] = map;
			}

			name.setSelectValues(nameValues);
			name.setValue(nameValues[PropertyDefineConstant.SELECT_VALUE][0]);
		}

		Property property = new Property(null, null, "");

		// 初期表示ツリーを構成。
		property.removeChildren();
		property.addChildren(scope);
		property.addChildren(name);

		//名前プロパティの子プロパティを選択
		Object nameValues[][] = name.getSelectValues();
		selectNameProperty(property,
				(String)nameValues[PropertyDefineConstant.SELECT_DISP_TEXT][0],
				locale);

		return property;
	}

	/**
	 * 名前パラメータ切替処理
	 * 
	 * 表示文字列により名前パラメータの子プロパティを切替する。
	 * 
	 * @param property
	 * @param dispText
	 * @param locale
	 */
	private static void selectNameProperty(Property property, String dispText, Locale locale) {
		int index = 0;

		//名前プロパティを取得
		Property name = null;
		ArrayList<Property> properties = null;
		properties = PropertyUtil.getProperty(property, ParameterConstant.NAME);
		if(properties != null){
			name = properties.get(0);
		}

		if(name instanceof Property){
			Object nameValues[][] = name.getSelectValues();

			if(dispText != null){
				for(int i = 0; i < nameValues[PropertyDefineConstant.SELECT_DISP_TEXT].length; i++){
					if(dispText.compareTo((String)nameValues[PropertyDefineConstant.SELECT_DISP_TEXT][i]) == 0){
						index = i;
						break;
					}
				}
			}

			if(nameValues[PropertyDefineConstant.SELECT_VALUE][index] instanceof HashMap){
				HashMap map = (HashMap)nameValues[PropertyDefineConstant.SELECT_VALUE][index];
				name.setValue(map.get(PropertyDefineConstant.MAP_VALUE));

				ArrayList<Property> propertyList = (ArrayList<Property>)map.get(PropertyDefineConstant.MAP_PROPERTY);
				if(propertyList != null){
					name.removeChildren();
					for(int i = 0; i < propertyList.size(); i++){
						name.addChildren(propertyList.get(i));
					}
				}
			}
			else{
				name.removeChildren();
				Object select = nameValues[PropertyDefineConstant.SELECT_VALUE][index];
				name.setValue(select);
			}
		}
	}

	/**
	 * プロパティ値設定処理
	 * 
	 * facilityId, scopeText
	 * 
	 * @param property
	 * @param sessionId
	 * @param locale
	 */
	private static void setPropertyValue(Property property, SessionInfo session, Locale locale) throws InvalidRole_Exception {
		//パラメータIDからプロパティを取得
		ArrayList<Property> properties = PropertyUtil.getProperty(property, ParameterConstant.SCOPE);
		((Property)properties.get(0)).setValue("");

		//スコープを設定
		FacilityTreeItem item = null;
		FacilityInfo info = new FacilityInfo();
		info.setValid(true);
		info.setCreateDatetime(System.currentTimeMillis());
		info.setModifyDatetime(System.currentTimeMillis());
		info.setBuiltInFlg(false);
		info.setFacilityType(FACILITY_TYPE_DEFAULT);
		info.setDisplaySortOrder(DISPLAY_SORT_ORDER_DEFAULT);
		info.setFacilityId(session.getFacilityId());
		info.setFacilityName(session.getScopeText());
		item = new FacilityTreeItem();
		item.setParent(null);
		item.setData(info);

		((Property)properties.get(0)).setValue(item);

		//セッション(実行履歴)からセッションパラメータ(実行履歴パラメータ)を取得
		List<SessionParameterInfo> paramList = session.getParams();

		if(paramList != null && paramList.size() > 0){
			for(SessionParameterInfo paramInfo : paramList){
				//パラメータIDからプロパティを取得
				properties =
						PropertyUtil.getProperty(property, paramInfo.getId());

				((Property)properties.get(0)).setValue("");

				try {
					Integer paramType = CollectiveRunEndpointWrapper.getParameterType(paramInfo.getId());

					if(paramInfo.getValue() != null){
						if(paramType.intValue() == ParameterTypeConstant.TYPE_SELECT){
							((Property)properties.get(0)).setValue(getParamValue(paramInfo.getId(), paramInfo.getValue(), locale));
						}
						else{
							((Property)properties.get(0)).setValue(paramInfo.getValue());
						}
					}
				} catch (InvalidRole_Exception e) {
					throw e;
				} catch (Exception e) {
					m_log.warn("setPropertyValue(), " + e.getMessage(), e);
					MessageDialog.openError(
							null,
							Messages.getString("failed"),
							Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
				}
			}
		}
	}

	/**
	 * パラメータ表示文字列取得
	 * 
	 * @param paramId パラメータID
	 * @param value パラメータ選択値
	 * @param locale ロケール
	 * @return 表示文字列
	 * @throws FinderException
	 * @throws NamingException
	 */
	private static String getParamValue(String paramId, String value, Locale locale) throws InvalidRole_Exception {

		String name = null;

		//パラメータIDからパラメータ選択マスタを取得
		List<ParameterSelectInfo> paramSelectList = null;

		try {
			paramSelectList = CollectiveRunEndpointWrapper.getParameterSelectMstList(paramId);
		} catch (InvalidRole_Exception e) {
			throw e;
		} catch (Exception e) {
			m_log.warn("getParamValue(), " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}

		if(paramSelectList != null && paramSelectList.size() > 0){
			for(ParameterSelectInfo paramSelect : paramSelectList){
				//値が一致したら表示文字列を返す
				if(paramSelect.getValue().equals(value)){
					name = Messages.getString(paramSelect.getName(), locale);
					break;
				}
			}
		}

		return name;
	}
}
