/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.accesscontrol.composite;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.ui.PlatformUI;

import com.clustercontrol.accesscontrol.bean.RoleSettingTreeConstant;
import com.clustercontrol.accesscontrol.dialog.RoleSettingDialog;
import com.clustercontrol.accesscontrol.util.AccessEndpointWrapper;
import com.clustercontrol.accesscontrol.view.RoleSettingTreeView;
import com.clustercontrol.accesscontrol.viewer.RoleSettingTreeContentProvider;
import com.clustercontrol.accesscontrol.viewer.RoleSettingTreeLabelProvider;
import com.clustercontrol.accesscontrol.viewer.RoleSettingTreeViewer;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.access.InvalidRole_Exception;
import com.clustercontrol.ws.access.RoleInfo;
import com.clustercontrol.ws.accesscontrol.RoleTreeItem;

/**
 * ロールツリー用のコンポジットクラスです。
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class RoleSettingTreeComposite extends Composite {

	// ログ
	private static Log m_log = LogFactory.getLog( RoleSettingTreeComposite.class );

	/** ツリービューア */
	private RoleSettingTreeViewer m_viewer = null;
	/** ツリービュ */
	private RoleSettingTreeView m_view = null;
	/** 選択ジョブツリーアイテム */
	private RoleTreeItem m_selectItem = null;

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親コンポジット
	 * @param style スタイル
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public RoleSettingTreeComposite(RoleSettingTreeView view , Composite parent, int style) {
		super(parent, style);
		this.m_view = view;
		initialize();
	}

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親コンポジット
	 * @param style スタイル
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public RoleSettingTreeComposite(Composite parent, int style) {
		super(parent, style);

		initialize();
	}

	/**
	 * コンストラクタ
	 * 
	 * @param parent 親コンポジット
	 * @param style スタイル
	 * @param selectItem 選択したノード
	 * 
	 * @see org.eclipse.swt.SWT
	 * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
	 * @see #initialize()
	 */
	public RoleSettingTreeComposite(Composite parent, int style, RoleTreeItem selectItem) {
		super(parent, style);

		m_selectItem = selectItem;
		initialize();
	}

	/**
	 * コンポジットを構築します。
	 */
	private void initialize() {
		GridLayout layout = new GridLayout(1, true);
		this.setLayout(layout);
		layout.marginHeight = 0;
		layout.marginWidth = 0;

		Tree tree = new Tree(this, SWT.SINGLE | SWT.BORDER);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		tree.setLayoutData(gridData);

		m_viewer = new RoleSettingTreeViewer(tree);
		m_viewer.setContentProvider(new RoleSettingTreeContentProvider());
		m_viewer.setLabelProvider(new RoleSettingTreeLabelProvider());

		// 選択アイテム取得イベント定義
		m_viewer.addSelectionChangedListener(new ISelectionChangedListener() {
			@Override
			public void selectionChanged(SelectionChangedEvent event) {
				StructuredSelection selection = (StructuredSelection) event.getSelection();
				m_selectItem = (RoleTreeItem) selection.getFirstElement();
			}
		});

		// ダブルクリックしたらジョブを開く
		m_viewer.addDoubleClickListener(
				new IDoubleClickListener() {
					@Override
					public void doubleClick(DoubleClickEvent event) {
						StructuredSelection selection = (StructuredSelection) event.getSelection();
						RoleTreeItem item = (RoleTreeItem) selection.getFirstElement();
						Object data = item.getData();
						if (data instanceof RoleInfo
								&& !((RoleInfo)data).getId().equals(RoleSettingTreeConstant.ROOT_ID)) {
							RoleSettingDialog dialog = new RoleSettingDialog(
									PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell(),
									((RoleInfo)data).getId());
							//ダイアログ表示
							if (dialog.open() == IDialogConstants.OK_ID) {
								m_view.update();
							}
						} else {
							return;
						}
					}
				});

		update();
	}

	/**
	 * このコンポジットが利用するツリービューアを返します。
	 * 
	 * @return ツリービューア
	 */
	public RoleSettingTreeViewer getTreeViewer() {
		return m_viewer;
	}

	/**
	 * このコンポジットが利用するツリーを返します。
	 * 
	 * @return ツリー
	 */
	public Tree getTree() {
		return m_viewer.getTree();
	}

	/**
	 * ツリービューアーを更新します。<BR>
	 * ツリー情報を取得し、ツリービューアーにセットします。
	 * <p>
	 * <ol>
	 * <li>ロールツリー情報を取得します。</li>
	 * <li>ツリービューアーにロールツリー情報をセットします。</li>
	 * </ol>
	 * 
	 */
	@Override
	public void update() {
		RoleTreeItem tree = null;

		//　ロール一覧情報取得
		try {
			tree = AccessEndpointWrapper.getRoleTree();
		} catch (InvalidRole_Exception e) {
			// アクセス権なしの場合、エラーダイアログを表示する
			MessageDialog.openInformation(
					null,
					Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
		} catch (Exception e) {
			m_log.warn("update() getJobTree, " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}

		m_selectItem = null;

		m_viewer.setInput(tree);
	}

	/**
	 * 選択ロールツリーアイテムを返します。
	 * 
	 * @return ロールツリーアイテム
	 */
	public RoleTreeItem getSelectItem() {
		return m_selectItem;
	}

	/**
	 * 選択ロールツリーアイテムを設定
	 * 
	 * @param item ロールツリーアイテム
	 */
	public void setSelectItem(RoleTreeItem item) {
		m_selectItem = item;
	}
}