/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.util;

import java.net.Authenticator;
import java.net.PasswordAuthentication;
import java.net.URL;
import java.rmi.AccessException;
import java.sql.Timestamp;

import javax.naming.CommunicationException;
import javax.naming.NamingException;
import javax.security.auth.login.LoginException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.action.IContributionItem;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.action.StatusLineContributionItem;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.internal.WorkbenchWindow;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.accesscontrol.dialog.LoginDialog;
import com.clustercontrol.accesscontrol.etc.action.LogoutAction;
import com.clustercontrol.accesscontrol.util.AccessEndpointWrapper;
import com.clustercontrol.accesscontrol.util.LocalSession;
import com.clustercontrol.jobmanagement.util.JobEditStateUtil;
import com.clustercontrol.jobmanagement.util.JobEndpointWrapper;
import com.clustercontrol.preference.ClusterControlCorePreferencePage;
import com.clustercontrol.ws.access.InvalidUserPass_Exception;
import com.clustercontrol.ws.jobmanagement.JobInfo;

/**
 * ログインマネージャクラス<BR>
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class LoginManager {
	private static Log m_log = LogFactory.getLog( LoginManager.class );

	public static final String KEY_EJB_URL = "ejbUrl";
	public static final String KEY_EJB_URL_NUM = "numOfUrlHistory";
	public static final String VALUE_EJB_URL = "http://localhost:8080/HinemosWS/";
	public static final String KEY_EJB_UID = "ejbUid";
	public static final String VALUE_EJB_UID = "hinemos";
	public static final String KEY_INTERVAL = "managerPollingInterval";
	public static final int VALUE_INTERVAL = 60;

	public static final String USER_OBJECT_ID = "com.clustercontrol.accesscontrol.user";
	public static final String URL_OBJECT_ID = "com.clustercontrol.accesscontrol.url";

	public static final String ACTION_SET_ID    = "com.clustercontrol.accesscontrol.ActionSet";
	public static final String ACTION_ID_LOGIN  = "com.clustercontrol.accesscontrol.etc.action.LoginAction";
	public static final String ACTION_ID_LOGOUT = "com.clustercontrol.accesscontrol.etc.action.LogoutAction";

	public static final String KEY_HTTP_CONNECT_TIMEOUT = "httpConnectTimeout";	// Utilityオプションからも使用されています。

	public static final int VALUE_HTTP_CONNECT_TIMEOUT = 10;
	public static final String KEY_HTTP_REQUEST_TIMEOUT = "httpRequestTimeout";	// Utilityオプションからも使用されています。
	public static final int VALUE_HTTP_REQUEST_TIMEOUT = 60;

	public static final String KEY_HTTPS_KEYSTORE_PATH = "httpsKeystorePath";
	public static final String VALUE_HTTPS_KEYSTORE_PATH = "";
	public static final String KEY_HTTPS_KEYSTORE_PASSWORD = "httpsKeystorePassword";
	public static final String VALUE_HTTPS_KEYSTORE_PASSWORD = "";

	public static final String KEY_PROXY_HOST = "proxyHost";
	public static final String VALUE_PROXY_HOST = "";
	public static final String KEY_PROXY_PORT = "proxyPort";
	public static final int VALUE_PROXY_PORT = 8080;
	public static final String KEY_PROXY_USER = "proxyUser";
	public static final String VALUE_PROXY_USER = "";
	public static final String KEY_PROXY_PASSWORD = "proxyPassword";
	public static final String VALUE_PROXY_PASSWORD = "";

	private static String m_url = null;
	private static String m_uid = null;
	private static String m_password = null;

	private static Display m_dislpay = null;
	private static IStatusLineManager m_lineManager = null;

	/** ダイアログ表示制御用 */
	private static boolean openInformation = false;

	/**
	 * コンストラクタ
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	private LoginManager() {
	}

	/**
	 * ログイン(接続先URLあり)
	 * 
	 * @param uid
	 * @param password
	 * @throws NamingException
	 * @throws LoginException
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	private synchronized static void login(String uid, String password, String url) throws Exception {
		m_uid = uid;
		m_password = password;
		m_url = url;


		IPreferenceStore store = ClusterControlPlugin.getDefault().getPreferenceStore();
		int httpConnectTimeout = store.getInt(KEY_HTTP_CONNECT_TIMEOUT);
		int httpRequestTimeout = store.getInt(KEY_HTTP_REQUEST_TIMEOUT);
		m_log.info("connect.timeout=" + httpConnectTimeout + ", request.timeout=" + httpRequestTimeout);

		// HTTPS設定
		URL u = new URL(url);
		String keyStorePath = store.getString(KEY_HTTPS_KEYSTORE_PATH);
		String keyStorePassword = store.getString(KEY_HTTPS_KEYSTORE_PASSWORD);
		if("https".equals(u.getProtocol())){
			System.setProperty("javax.net.ssl.trustStore", keyStorePath);
			System.setProperty("javax.net.ssl.trustStorePassword", keyStorePassword);
			m_log.info("keystore.path=" + System.getProperty("javax.net.ssl.trustStore"));
		}

		// プロキシ設定
		String proxyHost = store.getString(KEY_PROXY_HOST);
		int proxyPort = store.getInt(KEY_PROXY_PORT);
		String proxyUser = store.getString(KEY_PROXY_USER);
		String proxyPassword = store.getString(KEY_PROXY_PASSWORD);
		if(!"".equals(proxyHost)){
			System.setProperty(u.getProtocol() + ".proxyHost", proxyHost);
			System.setProperty(u.getProtocol() + ".proxyPort", Integer.toString(proxyPort));
			BasicAuth basicAuth = new BasicAuth(proxyUser, proxyPassword);
			Authenticator.setDefault(basicAuth);
			m_log.info("proxy.host=" + System.getProperty(u.getProtocol() + ".proxyHost") +
					", proxy.port=" + System.getProperty("http.proxyPort") + ", proxy.user=" + proxyUser);
		}

		EndpointManager.init(uid, password, url, httpConnectTimeout, httpRequestTimeout);
		try {
			//ログインチェック
			AccessEndpointWrapper.checkLogin();

			// ステータスバーへユーザIDを登録
			IWorkbench workbench = PlatformUI.getWorkbench();
			WorkbenchWindow workbenchWindow = (WorkbenchWindow)workbench.getActiveWorkbenchWindow();
			IActionBars bars = workbenchWindow.getActionBars();
			m_lineManager = bars.getStatusLineManager();
			m_dislpay = workbench.getDisplay();

			StatusLineContributionItem statusLine = new StatusLineContributionItem(USER_OBJECT_ID,100);
			statusLine.setText(Messages.getString("hinemos.user") + " : " + m_uid +
					"     " +
					Messages.getString("connection.url") + " : " + m_url);

			m_lineManager.add(statusLine);
			m_lineManager.update(true);

			//UIDと接続先をプレファレンスに書き戻す。
			//セッションに接続先URLを格納
			store.setValue(ClusterControlCorePreferencePage.EJB_URL,url);
			store.setValue(ClusterControlCorePreferencePage.EJB_UID,uid);

			//今回の接続先URLがURL履歴に存在するかチェックする。
			int numOfUrlHistory = store.getInt(LoginManager.KEY_EJB_URL_NUM);
			boolean urlExist = false;
			for(int i=0; i<numOfUrlHistory; i++){
				String histUrl = store.getString(LoginManager.KEY_EJB_URL + "_" + i);
				if(url.equals(histUrl)){
					//TODO 存在する場合、対象のURLを履歴の先頭に移動させ、他のURLを一つずつ後方にずらす
					urlExist = true;
					break;
				}
			}
			//存在しない場合、URL履歴の末尾に追加
			if(!urlExist && !url.equals("")){
				store.setValue(LoginManager.KEY_EJB_URL + "_" + numOfUrlHistory ,url);
				numOfUrlHistory++;
				store.setValue(LoginManager.KEY_EJB_URL_NUM, numOfUrlHistory);
			}

			// クライアント側で保持するローカル情報の削除処理
			LocalSession.clearSessionData();
			// マネージャの死活監視開始
			LocalSession.getLocalSession().startChecktask(getInterval());

		} catch (Exception e) {
			logout();
			throw e;
		}
	}


	/**
	 * ログアウト
	 * 
	 * @throws NamingException
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	public synchronized static void logout() {
		// 既にログアウトしている場合は、何もしない。
		if (!isLogin()) {
			return;
		}
		
		if (JobEditStateUtil.getLockedJobunitList().size() > 0) {
			// ロックしているジョブユニットがある場合
			for (JobInfo jobunit : JobEditStateUtil.getLockedJobunitList()) {
				try {
					JobEndpointWrapper.releaseEditLock(JobEditStateUtil.getEditSession(jobunit));
					JobEditStateUtil.removeLockedJobunit(jobunit);
				} catch (Exception e) {
					m_log.warn("dispose() : " + e.getMessage());
				}
			}
		}
		JobEditStateUtil.clearEditStateAll();
		
		PerspectiveUtil.closeAllHinemosPerspective();
		EndpointManager.logout();
		m_uid = null;
		m_url = null;

		// ステータスバーからユーザIDを削除
		if(m_lineManager != null){
			IContributionItem item = m_lineManager.find(USER_OBJECT_ID);
			m_lineManager.remove(item);
			m_lineManager.update(true);
		}

		LocalSession.clearSessionData();
		LocalSession.getLocalSession().stopChecktask();
	}

	/**
	 * ログインチェック
	 * 
	 * 本メソッドはUtilityオプションからも使用されています。
	 * 
	 * @return ログイン中か否か
	 * 
	 * @version 2.0.0
	 * @since 2.0.0
	 */
	public static boolean isLogin() {
		if (m_uid != null) {
			return true;
		}

		return false;
	}

	public synchronized static void login() {
		if(!isLogin()){
			Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
			//ログインダイアログ表示
			LoginDialog dialog = new LoginDialog(shell);
			//ダイアログ表示
			int loginRet = dialog.open();
			if (loginRet != IDialogConstants.OK_ID) {
				m_log.info("login() : cancel, " + loginRet);
				PerspectiveUtil.closeAllHinemosPerspective();
				return;
			}
			try {

				// login(dialog.getUserid(), dialog.getPassword());
				login(dialog.getUserid(), dialog.getPassword(),dialog.getUrl());

				// ログイン成功ダイアログを生成
				MessageDialog.openInformation(
						null,
						Messages.getString("successful"),
						Messages.getString("message.accesscontrol.5"));

				m_log.info("Login Success : userId = " + dialog.getUserid() + ", url = " + dialog.getUrl());
				m_log.info("Hinemos Client started");
			} catch (CommunicationException e) {
				// 接続失敗ダイアログを生成
				Status status = new Status(
						IStatus.ERROR,
						ClusterControlPlugin.getPluginId(),
						IStatus.OK,
						Messages.getString("message.accesscontrol.22"),
						e);

				ErrorDialog.openError(
						null,
						Messages.getString("failed"),
						Messages.getString("message.accesscontrol.21"),
						status);
				m_log.info("Login Fail : userId = " + dialog.getUserid() + ", url = " + dialog.getUrl());
				
				// ログイン失敗の場合は再度ログインダイアログを出す。
				login();
			} catch (AccessException e) {
				// ログイン失敗ダイアログを生成
				MessageDialog.openWarning(
						null,
						Messages.getString("failed"),
						Messages.getString("message.accesscontrol.6"));
				m_log.info("Login Fail : userId = " + dialog.getUserid() + ", url = " + dialog.getUrl());
				
				// ログイン失敗の場合は再度ログインダイアログを出す。
				login();
			} catch (InvalidUserPass_Exception e){
				// ログイン失敗ダイアログを生成
				MessageDialog.openWarning(
						null,
						Messages.getString("failed"),
						Messages.getString("message.accesscontrol.45"));
				m_log.info("Login Fail : userId = " + dialog.getUserid() + ", url = " + dialog.getUrl());
				
				// ログイン失敗の場合は再度ログインダイアログを出す。
				login();
			} catch (Exception e) {
				// 予期せぬエラーダイアログを生成
				Status status = new Status(
						IStatus.ERROR,
						ClusterControlPlugin.getPluginId(),
						IStatus.OK,
						Messages.getString("message.accesscontrol.23"),
						e);

				ErrorDialog.openError(
						null,
						Messages.getString("failed"),
						Messages.getString("message.accesscontrol.6"),
						status
						);
				m_log.info("Login Fail : userId = " + dialog.getUserid() + ", url = " + dialog.getUrl());
				// ログイン失敗の場合は再度ログインダイアログを出す。
				login();
			}
		}
		return;
	}

	/**
	 * 接続先URLを取得します。
	 * 
	 * 本メソッドはUtilityオプションから使用されています。
	 * 
	 * @return String 接続先URL
	 * @version 4.0.0
	 * @since 2.0.0
	 */
	public static String getUrl() {

		//リソースストアから接続先URLを取得
		IPreferenceStore store = ClusterControlPlugin.getDefault()
				.getPreferenceStore();
		String url = store.getString(KEY_EJB_URL);
		if (url.compareTo("") == 0) {
			url = VALUE_EJB_URL;
		}

		return url;
	}

	/**
	 * 
	 * @return
	 */
	private static int getInterval() {

		//リソースストアから接続先の死活間隔を取得
		IPreferenceStore store = ClusterControlPlugin.getDefault()
				.getPreferenceStore();
		int interval = store.getInt(KEY_INTERVAL);

		//m_log.debug("LoginManager.getInterval() interval = " + interval);
		return interval;
	}

	/**
	 * ユーザIDを取得します。
	 * 
	 * 本メソッドはUtilityオプションからも使用されています。
	 * 
	 * @return String ユーザID
	 * @version 4.0.0
	 */
	public static String getUserId() {
		return m_uid;
	}

	/**
	 * パスワードを取得します。
	 * 
	 * @return String パスワード
	 * @version 4.0.0
	 */
	public static String getPassword() {
		return m_password;
	}

	/**
	 * LocalSession.SessionCheckTaskからのダイアログ&ログアウト処理を受け付けるスレッド生成用メソッド
	 * @param r
	 * @return
	 */
	private static boolean checkAsyncExec(Runnable r){

		if(m_dislpay == null){
			//m_log.debug("LoginManager.checkAsyncExec() m_dislpay is null");
			return false;
		}

		if(!m_dislpay.isDisposed()){
			//m_log.debug("LoginManager.checkAsyncExec() is true");
			m_dislpay.asyncExec(r);
			return true;
		}
		else{
			//m_log.debug("LoginManager.checkAsyncExec() is false");
			return false;
		}
	}


	/**
	 * マネージャ切断をダイアログ表示してログアウト処理を行う
	 */
	public static void disconnectManager(){
		checkAsyncExec(new Runnable(){@Override
			public void run() {
			// ログアウトダイアログの表示
			//m_log.debug("LoginManager.disconnectManager() start openInformation = " + openInformation);
			if(!openInformation){
				//m_log.debug("LoginManager.disconnectManager() OpenDialog");

				openInformation = true;

				// ダイアログ表示
				String[] args = {new Timestamp(System.currentTimeMillis()).toString()};
				MessageDialog.openError(
						null,
						Messages.getString("failed"),
						Messages.getString("message.force.disconnect",args));

				// ログアウト処理
				(new LogoutAction()).runLogout();

				openInformation = false;
			}else{
				//m_log.debug("LoginManager.disconnectManager() Not OpenDialog");
			}
			//m_log.debug("LoginManager.disconnectManager() stop openInformation = " + openInformation);
		}}
				);

	}

	/**
	 * 定期的な監視を再実行
	 * @param interval 間隔(分)
	 */
	public static void restartChecktask(int interval){
		//m_log.debug("LoginManager.restartChecktask() interval = " + interval);
		LocalSession.getLocalSession().restartChecktask(interval);
	}
}

class BasicAuth extends Authenticator{
	private String username;
	private String password;
	protected BasicAuth(String username, String password){
		this.username = username;
		this.password = password;
	}

	@Override
	protected PasswordAuthentication getPasswordAuthentication() {
		return new PasswordAuthentication(username,password.toCharArray());
	}
}