/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.presenter;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.clustercontrol.cloud.commons.util.ControlUtil;
import com.clustercontrol.cloud.commons.util.annotation.Import;
import com.clustercontrol.ws.cloud.CloudAccountResource;
import com.clustercontrol.ws.cloud.CloudEndpoint;
import com.clustercontrol.ws.cloud.CloudRegion;

public class AccountResource extends InternalComponent<AccountResourceManager, CloudAccountResource> implements IAccountResource {
	public static class Region implements IRegion {
		private AccountResource accountResource;
		private String region;
		private String regionName;
		private CloudComputeService cloudComputeService;
		
		private Map<Class<?>, Object> pluginServicesMap = new HashMap<Class<?>, Object>();
		
		public Region(AccountResource accountResource, CloudRegion region) {
			this.accountResource = accountResource;
			this.region = region.getRegion();
			this.regionName = region.getRegionName();
		}

		@Override
		public AccountResource getAccountResource() {
			return accountResource;
		}

		@Override
		public String getRegion() {
			return region;
		}
		
		@Override
		public String getRegionName() {
			return regionName;
		}

		@SuppressWarnings("unchecked")
		@Override
		public <T> T getCloudService(Class<T> serviceClass) {
			if (serviceClass.isAssignableFrom(ICloudComputeService.class)) {
				if (cloudComputeService == null) {
					cloudComputeService = new CloudComputeService(this);
				}
				return (T)cloudComputeService;
			}
			
			if(pluginServicesMap != null){
				return (T) pluginServicesMap.get(serviceClass);
			}
			
			return null;
		}
		
		public CloudEndpoint getEndpoint() {
			return getAccountResource().getEndpoint();
		}

		@Override
		public <T> void putService(Class<T> interfaceClass,	T instance) {
			pluginServicesMap.put(interfaceClass, instance);
		}
	}


	private Map<Class<?>, Object> pluginServicesMap = new HashMap<Class<?>, Object>();

	@SuppressWarnings("unchecked")
	@Override
	public <T> T getCloudService(Class<T> serviceClass) {
		if(pluginServicesMap != null){
			return (T) pluginServicesMap.get(serviceClass);
		}
		return null;
	}
	
	
	private CloudUserManager cloudUserManager;

	private Region[] regions;

	private String accountId;
	private String accountResourceId;
	private String accountResourceName;
	private String cloudServiceId;
	private String description;
	private Boolean billingDetailCollectorFlg;
	private Integer retentionPeriod;
	private Long regDate;
	private String regUser;
	private Long updateDate;
	private String updateUser;

	private ICloudUser activeCloudUser;
	
	private CloudTemplateManager cloudTemplateManager;

	public static AccountResource createAccountResource(AccountResourceManager accountResourceManager, CloudAccountResource source) {
		AccountResource accountResource = new AccountResource(accountResourceManager);
		accountResource.overwrite(source);
		return accountResource;
	}
	
	public AccountResource(AccountResourceManager accountResourceManager){
		super(accountResourceManager);
	}

	@Override
	public String getAccountResourceId() {
		return accountResourceId;
	}

	@Override
	public String getAccountResourceName() {
		return accountResourceName;
	}

	@Override
	public String getCloudServiceId() {
		return cloudServiceId;
	}

	@Override
	public String getDescription() {
		return description;
	}

	@Override
	public Boolean getBillingDetailCollectorFlg() {
		return billingDetailCollectorFlg;
	}

	@Override
	public void setActiveUser(ICloudUser activeCloudUser) {
		this.activeCloudUser = activeCloudUser;
	}

	@Override
	public ICloudUser getActiveUser() {
		return activeCloudUser;
	}
	
	@Override
	public Long getRegDate() {
		return regDate;
	}

	@Override
	public String getRegUser() {
		return regUser;
	}

	@Override
	public Long getUpdateDate() {
		return updateDate;
	}

	@Override
	public String getUpdateUser() {
		return updateUser;
	}

	@Override
	public AccountResourceManager getAccountResourceManager() {
		return getContainer();
	}

	@Override
	public CloudUserManager getCloudUserManager() {
		if (cloudUserManager == null){
			cloudUserManager = new CloudUserManager(this);
		}
		return cloudUserManager;
	}

	@Override
	public Integer getRetentionPeriod() {
		return retentionPeriod;
	}

	@Import
	public void setAccountResourceId(String accountResourceId) {
		internalSet("accountResourceId", p2.accountResourceId, accountResourceId);
	}

	@Import
	public void setAccountResourceName(String accountResourceName) {
		internalSet("accountResourceName", p2.accountResourceName, accountResourceName);
	}

	@Import
	public void setCloudServiceId(String cloudServiceId) {
		internalSet("cloudServiceId", p2.cloudServiceId, cloudServiceId);
	}

	@Import
	public void setDescription(String description) {
		internalSet("description", p2.description, description);
	}
	
	@Import
	public void setBillingDetailCollectorFlg(Boolean billingDetailCollectorFlg) {
		internalSet("billingDetailCollectorFlg", p2.billingDetailCollectorFlg, billingDetailCollectorFlg);
	}

	@Import
	public void setRegDate(Long regDate) {
		internalSet("regDate", p2.regDate, regDate);
	}

	@Import
	public void setRegUser(String regUser) {
		internalSet("regUser", p2.regUser, regUser);
	}

	@Import
	public void setUpdateDate(Long updateDate) {
		internalSet("updateDate", p2.updateDate, updateDate);
	}

	@Import
	public void setUpdateUser(String updateUser) {
		internalSet("updateUser", p2.updateUser, updateUser);
	}

	@Import
	public void setRetentionPeriod(Integer retentionPeriod) {
		internalSet("retentionPeriod", p2.retentionPeriod, retentionPeriod);
	}

	@Override
	public Region[] getRegions() {
		if (regions == null) {
			try {
				// とりあえず、ここでリージョンを取得するが、今後、CloudService 配下にリージョン用のモデルを用意する必要がある。
				List<CloudRegion> cloudRegions = getAccountResourceManager().getCloudResourceManager().getEndpoint(CloudEndpoint.class).getCloudRegionsByService(getCloudServiceId());
				Region[] rs = new Region[cloudRegions.size()];
				for (int i  = 0; i < cloudRegions.size(); ++i) {
					rs[i] = new Region(this, cloudRegions.get(i));
				}
				regions = rs;
			}
			catch (Exception e) {
				throw new CloudModelException(e);
			}
		}
		return regions;
	}

	@Override
	public CloudAccountResource getSource() throws CloudModelException {
		return null;
	}

	@Override
	public void overwrite(CloudAccountResource source) {
		ControlUtil.importObject(this, source);
		setBillingDetailCollectorFlg(source.isBillingDetailCollectorFlg());
	}

	@Override
	public boolean equalValues(CloudAccountResource source) {
		assert source != null;
		return this.getAccountResourceId().equals(source.getAccountResourceId());
	}

	@Override
	public AccountResource modifyAccountResource(ModifyAccountResourceRequest modifyAccountResourceRequest) {
		com.clustercontrol.ws.cloud.ModifyAccountResourceRequest wsModifyAccountResourceRequest = new com.clustercontrol.ws.cloud.ModifyAccountResourceRequest();
		ControlUtil.exportObject(modifyAccountResourceRequest, wsModifyAccountResourceRequest);
		try {
			com.clustercontrol.ws.cloud.CloudAccountResource reWsAccountResource = getAccountResourceManager().getCloudResourceManager().getEndpoint(CloudEndpoint.class).modifyAccountResource(wsModifyAccountResourceRequest);
			overwrite(reWsAccountResource);
			return this;
		}
		catch (Exception e) {
			throw new CloudModelException(e);
		}
	}

	@Override
	public void editBillingDetailCollectorSetting(EditBillingDetailCollectorSettingRequet request) {
		com.clustercontrol.ws.cloud.ModifyAccountBillingDetailRequest wsRequest = new com.clustercontrol.ws.cloud.ModifyAccountBillingDetailRequest();
		ControlUtil.exportObject(request, wsRequest);
		try {
			overwrite(getEndpoint().modifyAccountBillingDetail(wsRequest));
		}
		catch (Exception e) {
			throw new CloudModelException(e);
		}
	}

	@Override
	public String getAccountId() {
		return accountId;
	}

	@Import
	public void setAccountId(String accountId) {
		internalSet("accountId", p2.accountId, accountId);
	}
	
	public CloudEndpoint getEndpoint(){
		return getAccountResourceManager().getCloudResourceManager().getEndpoint(CloudEndpoint.class);
	}

	@Override
	public Region getRegion(String region) {
		for (Region r: getRegions()) {
			if (r.getRegion().equals(region)) {
				return r;
			}
		}
		return null;
	}

	@Override
	public CloudTemplateManager getCloudTemplateManager() {
		if(cloudTemplateManager == null){
			cloudTemplateManager = new CloudTemplateManager(this);
		}
		return cloudTemplateManager;
	}

	@Override
	public <T> void putService(Class<T> interfaceClass,	T instance) {
		pluginServicesMap.put(interfaceClass, instance);
	}
}