/*
Copyright (C) 2014 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */

package com.clustercontrol.cloud.cloudn.base.ui.views.actions;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.window.Window;
import com.clustercontrol.cloud.cloudn.base.commons.util.CloudnConstants;
import com.clustercontrol.cloud.cloudn.base.commons.util.MessageManagerExt;
import com.clustercontrol.cloud.cloudn.base.presenter.JobTrackingService;
import com.clustercontrol.cloud.cloudn.base.resources.CloudResourceProviderImpl;
import com.clustercontrol.cloud.cloudn.base.ui.dialogs.CreateInstanceDialog;
import com.clustercontrol.cloud.presenter.CloudModelException;
import com.clustercontrol.cloud.presenter.CreateCloudInstanceByTemplateRequest;
import com.clustercontrol.cloud.presenter.CreateCloudInstanceRequest;
import com.clustercontrol.cloud.presenter.InvalidStateException;
import com.clustercontrol.cloud.presenter.IAccountResource.IRegion;
import com.clustercontrol.cloud.presenter.ICloudComputeService;
import com.clustercontrol.cloud.presenter.ICloudInstance;
import com.clustercontrol.cloud.presenter.ICloudInstance.StateType;
import com.clustercontrol.cloud.presenter.ICloudTemplateManager;
import com.clustercontrol.cloud.ui.views.InstanceMonitorService;
import com.clustercontrol.cloud.ui.views.InstancesView;
import com.clustercontrol.cloud.ui.views.actions.AbstractViewAction;
import com.clustercontrol.cloud.ui.views.actions.InvocationException;
import com.clustercontrol.util.Messages;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;


public class CreateInstanceAction extends AbstractViewAction implements CloudnConstants {
	public static class InstanceDetail {
		public List<String> networkIds = new ArrayList<>();
		public Boolean attachingDisk;
		public String group;
	}
	
	private MessageManagerExt messages = MessageManagerExt.getInstance("messages");
	
	private String msgCreateInstanceConfirm = messages.getString("message.confirm_some_action_to_something", new Object[]{"word.instance", "word.create"});
	private String msgCreateInstanceResult = messages.getString("message.result_some_action_to_something", new Object[]{"word.instance", "word.create"});

	
	@Override
	protected void invoke(IAction action) throws InvocationException {
		CreateInstanceDialog dialog = new CreateInstanceDialog(this.viewPart.getSite().getShell());
		
		final InstancesView view = (InstancesView)viewPart;
		IRegion region = view.getActiveRegion();
		
		if (region == null) {
			assert view.getCurrentAccountResource().getRegions().length > 0;
			region = view.getCurrentAccountResource().getRegions()[0];
		}
		
		final ICloudTemplateManager templateManager = region.getAccountResource().getCloudTemplateManager();
		dialog.setCloudTemplateManager(templateManager);
		
		dialog.setCloudInfoProvider(new CloudResourceProviderImpl(region.getAccountResource().getAccountResourceId()));
		dialog.setRegion(region.getRegion());
		
		if(dialog.open() == Window.OK){
			if (MessageDialog.openConfirm(
				null,
				Messages.getString("confirmed"),
				msgCreateInstanceConfirm)){
				
				CreateInstanceDialog.Model m = dialog.getOutput();
				
				ICloudComputeService service = region.getCloudService(ICloudComputeService.class);
				IRegion r = service.getRegion();
				
				final ICloudInstance instance;
				
				InstanceDetail detail = new InstanceDetail();
				if (m.getNetworkId() != null) detail.networkIds.add(m.getNetworkId());
				detail.group = m.getGroup();
				detail.attachingDisk = m.getAttachingDisk();
				
				ObjectMapper om = new ObjectMapper();
				ObjectWriter dw = om.writerWithType(InstanceDetail.class);
				String detailJson;
				try {
					detailJson = dw.writeValueAsString(detail);
				} catch (Exception e) {
					throw new CloudModelException(e);
				}
				
				if(m.isTemplateUsed()){
					CreateCloudInstanceByTemplateRequest request = new CreateCloudInstanceByTemplateRequest();
					
					request.setZone(m.getZone());
					
					request.setTemplateId(m.getTemplateId());
					
					request.setFlavor(m.getFlavor());
					
					request.setFacilityId(m.getFacilityId());
					request.setFacilityName(m.getFacilityName());
					request.setDescription(m.getDescription());
					request.setNodeName(m.getNodeName());
					request.setReplaceNode(m.getReplaceNode());
					
					request.setInstanceDetail(detailJson);
					
					request.setTags(m.getTags());
					request.setArguments(m.getArguments());
					
					instance = r.getAccountResource().getRegion(DEFAULT_REGION).getCloudService(ICloudComputeService.class).getInstanceManager().createCloudInstanceByTemplate(request);
				} else {
					CreateCloudInstanceRequest request = new CreateCloudInstanceRequest();
					
					request.setZone(m.getZone());
					request.setImageId(m.getImageId());
					request.setFlavor(m.getFlavor());
					
					request.setFacilityId(m.getFacilityId());
					request.setFacilityName(m.getFacilityName());
					request.setDescription(m.getDescription());
					request.setNodeName(m.getNodeName());
					request.setReplaceNode(m.getReplaceNode());
					
					request.setInstanceDetail(detailJson);
					
					request.setTags(m.getTags());
					
					instance = r.getAccountResource().getRegion(DEFAULT_REGION).getCloudService(ICloudComputeService.class).getInstanceManager().createCloudInstance(request);
				}
				
				// 成功報告ダイアログを生成
				MessageDialog.openInformation(
					null,
					Messages.getString("successful"),
					msgCreateInstanceResult);

				// Password を追跡して別ダイアログに表示する
				try {
					JobTrackingService.trackPassword(view, instance);
				} catch (InvalidStateException e) {
						throw new InvocationException(e);
				}
				
				view.getSite().getShell().getDisplay().asyncExec(new Runnable() {
					@Override
					public void run() {
						InstanceMonitorService.getInstanceMonitorService().startMonitor(instance, StateType.running, StateType.terminated, StateType.stopped);
						com.clustercontrol.action.FacilityTree.refresh();
					}
				});
			}
			else {
				return;
			}
		}
	}
	
	@Override
	public void selectionChanged(IAction action, ISelection selection) {
		action.setEnabled(true);
	}
}