/*
Copyright (C) 2014 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.factory;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.persistence.EntityExistsException;
import javax.persistence.Query;

import org.apache.log4j.Logger;

import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.ErrorCode;
import com.clustercontrol.cloud.Filter;
import com.clustercontrol.cloud.IResourceManagement;
import com.clustercontrol.cloud.IResourceManagement.Instance;
import com.clustercontrol.cloud.IResourceManagement.Instance.BlockDeviceMapping;
import com.clustercontrol.cloud.IResourceManagement.InstanceBackup;
import com.clustercontrol.cloud.MessagesHolder;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.SessionService.RolebackAction;
import com.clustercontrol.cloud.bean.CloudInstance;
import com.clustercontrol.cloud.bean.CloudInstanceBackup;
import com.clustercontrol.cloud.bean.CloudTemplate;
import com.clustercontrol.cloud.bean.CreateInstanceBackupRequest;
import com.clustercontrol.cloud.bean.CreateInstanceBackupRequest2;
import com.clustercontrol.cloud.bean.CreateInstanceByTemplateRequest;
import com.clustercontrol.cloud.bean.CreateInstanceRequest;
import com.clustercontrol.cloud.bean.Image;
import com.clustercontrol.cloud.bean.InstanceStateChange;
import com.clustercontrol.cloud.bean.InstanceStateKind;
import com.clustercontrol.cloud.bean.LoadBalancer;
import com.clustercontrol.cloud.bean.NodeDetail;
import com.clustercontrol.cloud.bean.RegistNodeRequest;
import com.clustercontrol.cloud.bean.RestoreInstanceRequest;
import com.clustercontrol.cloud.bean.RestoreStatus;
import com.clustercontrol.cloud.bean.StartInstanceRequest;
import com.clustercontrol.cloud.bean.StopInstanceRequest;
import com.clustercontrol.cloud.bean.Tag;
import com.clustercontrol.cloud.commons.CloudConstants;
import com.clustercontrol.cloud.commons.CloudPropertyConstants;
import com.clustercontrol.cloud.commons.PropertyContract;
import com.clustercontrol.cloud.dao.CloudInstanceBackupDao;
import com.clustercontrol.cloud.dao.CloudInstanceDao;
import com.clustercontrol.cloud.factory.TemplateJobUtil.JobEventListener;
import com.clustercontrol.cloud.factory.monitors.InstanceMonitorService;
import com.clustercontrol.cloud.persistence.EntityManagerEx;
import com.clustercontrol.cloud.persistence.PersistenceUtil.TransactionScope;
import com.clustercontrol.cloud.persistence.TransactionException;
import com.clustercontrol.cloud.persistence.Transactional;
import com.clustercontrol.cloud.util.AccountResourceUtil;
import com.clustercontrol.cloud.util.CloudMessageUtil;
import com.clustercontrol.cloud.util.HinemosUtil;
import com.clustercontrol.cloud.util.RepositoryControllerBeanWrapper;
import com.clustercontrol.fault.FacilityDuplicate;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.UsedFacility;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.session.RepositoryControllerBean;

@Transactional
public class InstanceOperator extends ResourceOperatorBase implements IInstanceOperator {
	
	private boolean update = CloudPropertyConstants.autoupdate_instance.match(PropertyContract.on);
	private boolean regist = CloudPropertyConstants.autoregist_instance.match(PropertyContract.on);
	private boolean relation = CloudPropertyConstants.autoregist_scope_relation.match(PropertyContract.on);

	public InstanceOperator() throws CloudManagerFault {
	}

	public InstanceOperator(boolean upate, boolean regist, boolean relation)
			throws CloudManagerFault {
		this.update = upate;
		this.regist = regist;
		this.relation = relation;
	}

	@Override
	public CloudInstance createInstance(CreateInstanceRequest request) throws CloudManagerFault, InvalidRole {
		CloudInstance instance = internalCreateInstance(
				request.getNodeDetail(), request.getFlavor(),
				request.getImageId(), request.getZone(),
				request.getInstanceDetail(), null, request.getTags());
		if (instance.getFacilityId() != null) {
			RegistAgent.asyncRegistAgent(instance.getFacilityId());
		}
		return instance;
	}

	@Override
	public CloudInstance createInstanceByTemplate(CreateInstanceByTemplateRequest request) throws CloudManagerFault,InvalidRole {
		ITemplateOperator templateOperator = getResourceOperator(ITemplateOperator.class);

		CloudTemplate ct = templateOperator.findTemplate(request.getTemplateId());
		CloudInstance ci = internalCreateInstance(request.getNodeDetail(), request.getFlavor(), ct.getImageId(), request.getZone(), request.getInstanceDetail(), ct.getTemplateId(), request.getTags());

		if (ct.getLaunchJobId() != null && !ct.getLaunchJobId().equals("")) {
			TemplateJobUtil.asyncExecute(
					ci.getFacilityId(),
					ct.getAccountResourceId(),
					ct.getLaunchJobId(),
					request.getArguments(),
					new JobEventListener() {
						@Override
						public void jobEventOccurerd(String facilityId, String jobunitId, String jobId) throws Exception {
							RegistAgent.registAgent(facilityId);
						}
					},
					null);
		}
		
		return ci;
	}

	@SuppressWarnings("unchecked")
	@Override
	public CloudInstanceBackup createInstanceBackup(CreateInstanceBackupRequest request) throws CloudManagerFault, InvalidRole {
		CreateInstanceBackupRequest2 request2 = new CreateInstanceBackupRequest2();
		request2.setName(request.getName());
		request2.setInstanceId(request.getInstanceId());
		request2.setDescription(request.getDescription());
		request2.setNoReboot(request.getNoReboot());
		request2.setStorageIds(request.getWithVolumes() ? null: (List<String>)Collections.EMPTY_LIST);
		request2.setDetail(request.getDetail());
		return createInstanceBackup2(request2);
	}
	
	@Override
	public CloudInstanceBackup createInstanceBackup2(CreateInstanceBackupRequest2 request) throws CloudManagerFault, InvalidRole {
		IResourceManagement rm = getResourceManagement();

		CloudInstance instance = getInstance(request.getInstanceId());
		NodeInfo nodeInfo = null;
		try {
			nodeInfo = new RepositoryControllerBean().getNode(instance.getFacilityId());
		}
		catch (FacilityNotFound e) {
			throw ErrorCode.CLOUDINSTANCEBACKUP_NOT_FOUND_NODE.cloudManagerFault(request.getInstanceId());
		}
		catch (HinemosUnknown e) {
			throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
		}

		InstanceBackup instanceBackup = rm.createInstanceBackup(request.getInstanceId(), request.getName(), request.getDescription(), request.getNoReboot(), request.getStorageIds(), request.getDetail());

		CloudInstanceBackupDao dao = new CloudInstanceBackupDao();
		dao.setInstanceBackupId(instanceBackup.getInstanceBackupId());
		dao.setCloudServiceId(getAccountResource().getCloudServiceId());
		dao.setRegion(getCloudRegion().getRegion());
		dao.setAccountResourceId(getAccountResource().getAccountResourceId());
		dao.setName(request.getName());
		dao.setCloudUserId(getCloudUser().getCloudUserId());
		dao.setDescription(request.getDescription());
		dao.setRestoreStatus(CloudInstanceBackupDao.RestoreStatus.valueOf(RestoreStatus.available.name()));

		CloudInstanceBackupDao.BackupedData backup = new CloudInstanceBackupDao.BackupedData();
		backup.setFacilityId(nodeInfo.getFacilityId());
		backup.setFacilityName(nodeInfo.getFacilityName());
		backup.setFacilityDescription(nodeInfo.getDescription());
		backup.setNodeName(nodeInfo.getNodeName());
		backup.setInstanceId(instance.getInstanceId());

		dao.setBackupedData(backup);

		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		em.persist(dao);

		return new CloudInstanceBackup(dao, instanceBackup);
	}

	protected CloudInstance internalCreateInstance(NodeDetail nodeDetail, String flavor, String imageId, String zone, String instanceDetail, String templateId, List<Tag> tags) throws CloudManagerFault, InvalidRole {
		Instance instance = getResourceManagement().createInstance(nodeDetail.getFacilityId(), nodeDetail.getFacilityName(), flavor, imageId, zone, instanceDetail, tags);
		CloudInstanceDao dao = createCloudInstanceDao(instance, templateId);
		CloudInstance cloudInstance = createCloudInstance(nodeDetail, dao, instance);

		InstanceMonitorService.getSingleton().startMonitor(
				instance.getInstanceId(), cloudInstance.getFacilityId(),
				SessionService.current().getContext(),
				getCloudService(), getCloudRegion(), getCloudUser(), true,
				InstanceStateKind.running, InstanceStateKind.terminated,
				InstanceStateKind.stopped);

		return cloudInstance;
	}

	@Override
	public CloudInstance restoreInstance(RestoreInstanceRequest request) throws CloudManagerFault, InvalidRole, InvalidRole {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudInstanceBackupDao backup = em.find(
				CloudInstanceBackupDao.class,
				new CloudInstanceBackupDao.CloudInstanceBackupPK(request.getInstanceBackupId(), getCloudRegion().getRegion(), getCloudUser().getAccountResourceId()));
		if (backup == null) {
			throw ErrorCode.CLOUDINSTANCEBACKUP_NOT_FOUND_NODE.cloudManagerFault(request.getInstanceBackupId());
		}

		NodeDetail nodeDetail = new NodeDetail();
		nodeDetail.setFacilityId(backup.getBackupedData().getFacilityId());
		nodeDetail.setFacilityName(backup.getBackupedData().getFacilityName());
		nodeDetail.setDescription(backup.getBackupedData().getFacilityDescription());
		nodeDetail.setNodeName(backup.getBackupedData().getNodeName());
		nodeDetail.setReplaceNode(Boolean.TRUE);

		if (request.getNodeDetail() != null) {
			if (request.getNodeDetail().getFacilityId() != null)
				nodeDetail.setFacilityId(request.getNodeDetail().getFacilityId());
			if (request.getNodeDetail().getFacilityName() != null)
				nodeDetail.setFacilityName(request.getNodeDetail().getFacilityName());
			if (request.getNodeDetail().getDescription() != null)
				nodeDetail.setDescription(request.getNodeDetail().getDescription());
			if (request.getNodeDetail().getNodeName() != null)
				nodeDetail.setNodeName(request.getNodeDetail().getNodeName());
			if (request.getNodeDetail().getReplaceNode() != null)
				nodeDetail.setReplaceNode(request.getNodeDetail().getReplaceNode());
		}

		Instance instance = getResourceManagement().restoreInstance(
				request.getNodeDetail().getFacilityId(),
				request.getInstanceBackupId(), nodeDetail.getFacilityName(),
				request.getFlavor(), request.getZone(),
				request.getInstanceDetail(), request.getTags());
		CloudInstanceDao dao = createCloudInstanceDao(instance, null);
		CloudInstance cloudInstance = createCloudInstance(nodeDetail, dao,instance);
		
		InstanceMonitorService.getSingleton().startMonitor(
				instance.getInstanceId(), cloudInstance.getFacilityId(),
				SessionService.current().getContext(),
				getCloudService(), getCloudRegion(), getCloudUser(), true,
				InstanceStateKind.running, InstanceStateKind.terminated,
				InstanceStateKind.stopped);

		return cloudInstance;
	}

	private CloudInstance createCloudInstance(NodeDetail nodeDetail, CloudInstanceDao dao, Instance instance) throws CloudManagerFault, InvalidRole {
		NodeInfo nodeInfo = null;

		RepositoryControllerBean repositoryControllerBean = RepositoryControllerBeanWrapper.bean();

		String beforeZoneScopeId = null;
		if (Boolean.TRUE.equals(nodeDetail.getReplaceNode())) {
			EntityManagerEx em = SessionService.current().getEntityManagerEx();
			List<CloudInstanceDao> cloudInstanceDaos = em.findByFilter(CloudInstanceDao.class, new Filter("facilityId", nodeDetail.getFacilityId()));
			for (CloudInstanceDao cloudInstanceDao : cloudInstanceDaos) {
				unrelateNode(cloudInstanceDao);
				beforeZoneScopeId = AccountResourceUtil.createAccountResourceZoneId(cloudInstanceDao.getAccountResourceId(), cloudInstanceDao.getZone());
			}

			try {
				nodeInfo = repositoryControllerBean.getNode(nodeDetail.getFacilityId());
			}
			catch (FacilityNotFound e) {
				//　とりあえずエラーをそのまま返す。
				throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
			}
			catch (HinemosUnknown e) {
				throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
			}

			HinemosUtil.resetNodeInfo(
					nodeInfo,
					nodeDetail.getFacilityName(),
					instance.getPlatform().label(),
					getCloudService().getCloudTypeId(),
					nodeDetail.getNodeName(),
					nodeDetail.getDescription(),
					CloudConstants.NT_Resource,
					getCloudService().getCloudServiceId(),
					getAccountResource().getAccountResourceId(),
					instance.getResourceType(),
					instance.getInstanceId(),
					getCloudRegion().getRegion(),
					instance.getZone());

			String ipAddress = instance.getIpAddress();

			if (ipAddress != null) {
				nodeInfo.setIpAddressV4(ipAddress);
			} else {
				nodeInfo.setIpAddressV4("123.123.123.123");
			}

			try (ModifiedEventNotifier<NodeInfo> notifier = new ModifiedEventNotifier<>(CloudConstants.Node_Instance_Modify, NodeInfo.class, nodeInfo)) {
				try {
					repositoryControllerBean.modifyNode(nodeInfo);
					notifier.setCompleted();
				}
				catch (InvalidSetting | HinemosUnknown e) {
					throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
				}
			}
			// CloudInstance に対するノードは、スキーマ上 ユニークなので、 facilityId をキーにバックアップ情報を取得しても問題ない。
			List<CloudInstanceBackupDao> backups = em.findByFilter(CloudInstanceBackupDao.class, new Filter("backupedData.facilityId", nodeDetail.getFacilityId()));
			for (CloudInstanceBackupDao backup: backups) {
				backup.getBackupedData().setInstanceId(dao.getInstanceId());
			}
		}

		if (nodeInfo == null) {
			nodeInfo = HinemosUtil.createNodeInfo(
					nodeDetail.getFacilityId(),
					nodeDetail.getFacilityName(),
					instance.getPlatform().label(),
					getCloudService().getCloudTypeId(),
					nodeDetail.getNodeName(),
					nodeDetail.getDescription(),
					CloudConstants.NT_Resource,
					getCloudService().getCloudServiceId(),
					getAccountResource().getAccountResourceId(),
					instance.getResourceType(),
					instance.getInstanceId(),
					getCloudRegion().getRegion(),
					instance.getZone(),
					getCloudUser().getRoleId());

			String ipAddress = instance.getIpAddress();

			if (ipAddress != null) {
				nodeInfo.setIpAddressV4(ipAddress);
			} else {
				nodeInfo.setIpAddressV4("123.123.123.123");
			}
			
			try (AddedEventNotifier<NodeInfo> notifier = new AddedEventNotifier<>(CloudConstants.Node_Instance_Add, NodeInfo.class, nodeInfo)) {
				try {
					repositoryControllerBean.addNode(nodeInfo);
					notifier.setCompleted();
				}
				catch (FacilityDuplicate e) {
					throw ErrorCode.CLOUDINSTANCE_DUPLICATE_NODE.cloudManagerFault(dao.getAccountResourceId(), dao.getInstanceId(), nodeDetail.getFacilityId());
				}
				catch (InvalidSetting | HinemosUnknown e) {
					throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
				}
			}
		}

		dao.setFacilityId(nodeDetail.getFacilityId());
		dao.setRegistStatus(CloudInstanceDao.InstanceStatus.registered);

		try {
			if (beforeZoneScopeId != null && !beforeZoneScopeId.equals(AccountResourceUtil.createAccountResourceZoneId(dao.getAccountResourceId(), dao.getZone()))) {
				repositoryControllerBean.releaseNodeScope(beforeZoneScopeId, new String[]{nodeDetail.getFacilityId()});
			}
			
			// スコープへ関連付け。
			AccountResourceUtil.attachScope(
					repositoryControllerBean,
					getCloudUser().getAccountResourceId(),
					getAccountResource().getAccountResourceName(),
					getCloudRegion().getRegion(),
					getCloudRegion().getRegionName(),
					instance.getZone(),
					nodeDetail.getFacilityId(),
					getCloudUser().getRoleId(),
					new MessagesHolder("com.clustercontrol.cloud.messages"));
		}
		catch (InvalidSetting | HinemosUnknown | FacilityDuplicate e) {
			throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
		}

		// ストレージ情報を作成。
		IStorageOperator strageOperator = getResourceOperator(IStorageOperator.class);
		strageOperator.setFlags(true, true, true);

		for (IResourceManagement.Instance.BlockDeviceMapping bd : instance.getBlockDeviceMappings()) {
			strageOperator.updateStorage(bd.getStorageId());
		}

		// インスタンスに、登録先スコープ情報があるなら、スコープへノードを登録する。
		if (relation) {
			relateExistedScope(dao.getFacilityId(), instance);
		} else {
			// 特になにもしない。
		}

		return new CloudInstance(dao, getCloudService().getCloudTypeId(), instance, nodeInfo.getFacilityName());
	}

	@Override
	public void removeInstance(String instanceId) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudInstanceDao dao = em.find(CloudInstanceDao.class, new CloudInstanceDao.CloudInstancePK(instanceId, getCloudRegion().getRegion(), getCloudUser().getAccountResourceId()));
		if (dao == null) {
			try {
				// クラウドインスタンスが存在しない場合でも、インスタンスは削除しにいく。
				getResourceManagement().deleteInstance(instanceId);
			} catch (Exception e) {
			}

			throw ErrorCode.CLOUDINSTANCE_NOT_FOUND.cloudManagerFault(instanceId);
		}
		else {
			if (dao.getRegistStatus() == CloudInstanceDao.InstanceStatus.deleted) {
				// インスタンス削除済みの場合は、ノード解除で削除。
				throw new CloudManagerFault();
			}
			// EC2 インスタンス削除。
			switch (dao.getRegistStatus()) {
			case unregistered:
			case registered:
				// 関連している インスタンスが存在するので削除。
				getResourceManagement().deleteInstance(instanceId);
				break;
			case deleted:
				// 関連している EC2 インスタンスが存在しないのでスルー。
				break;
			}

			// 削除するべきノードがあるか確認。
			final String facilityId = dao.getFacilityId();

			// クラウドインスタンス情報削除。
			em.remove(dao);

			if (facilityId != null) {
				// ノードの削除処理。
				// クラウドインスタンスを削除するスレッドとは、別のスレッドで実行するので、ノードの削除が失敗した場合でも、
				// クライアントに通知しない。
				try {
					// Hinemos ノード削除。
					RepositoryControllerBeanWrapper.bean().deleteNode(facilityId);
				} catch (UsedFacility e) {
					getLogger().warn(e.getMessage(), e);
				} catch (Exception e) {
					getLogger().error(e.getMessage(), e);
				}
			}
		}
	}

	@Override
	public InstanceStateChange startInstance(StartInstanceRequest request) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudInstanceDao instanceDao = em.find(CloudInstanceDao.class, new CloudInstanceDao.CloudInstancePK(request.getInstanceId(), getCloudRegion().getRegion(), getCloudUser().getAccountResourceId()));
		if (instanceDao == null) {
			throw ErrorCode.CLOUDINSTANCE_NOT_FOUND.cloudManagerFault(request.getInstanceId());
		}

		InstanceStateChange instanceStateChange = internalStartInstance(instanceDao);
		if (request.getRunJobFlg() && instanceDao.getFacilityId() != null) {
			String jobId = request.getJobId() == null ? (instanceDao.getTemplateId() != null ? getResourceOperator(ITemplateOperator.class).findTemplate(instanceDao.getTemplateId()).getStartJobId(): null): request.getJobId(); 
			if (jobId != null) {
				TemplateJobUtil.asyncExecute(
						instanceDao.getFacilityId(),
						instanceDao.getAccountResourceId(),
						jobId,
						request.getArguments(),
						new JobEventListener() {
							@Override
							public void jobEventOccurerd(String facilityId, String jobunitId, String jobId) throws Exception {
								RegistAgent.findAgent(facilityId);
							}
						},
						null);
			}

			return instanceStateChange;
		} else {
			return instanceStateChange;
		}
	}

	private InstanceStateChange internalStartInstance(CloudInstanceDao instanceDao) throws CloudManagerFault {
		InstanceStateChange changed = getResourceManagement().startInstance(instanceDao.getInstanceId());

		if (instanceDao.getFacilityId() != null) {
			InstanceMonitorService.getSingleton().startMonitor(
					instanceDao.getInstanceId(), instanceDao.getFacilityId(),
					SessionService.current().getContext(),
					getCloudService(), getCloudRegion(), getCloudUser(), false,
					InstanceStateKind.running, InstanceStateKind.terminated,
					InstanceStateKind.stopped);
		}

		return changed;
	}

	@Override
	public InstanceStateChange stopInstance(StopInstanceRequest request) throws CloudManagerFault {
		IResourceManagement rm = getResourceManagement();
		Instance instance = rm.getInstance(request.getInstanceId());
		if (InstanceStateKind.running != instance.getState()) {
			throw ErrorCode.CLOUDINSTANCE_NOT_RUNNING.cloudManagerFault(request.getInstanceId());
		}

		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		final CloudInstanceDao instanceDao = em.find(CloudInstanceDao.class, new CloudInstanceDao.CloudInstancePK(request.getInstanceId(), getCloudRegion().getRegion(), getCloudUser().getAccountResourceId()));
		if (instanceDao == null) {
			throw ErrorCode.CLOUDINSTANCE_NOT_FOUND.cloudManagerFault(request.getInstanceId());
		}

		if (request.getRunJobFlg() && instanceDao.getFacilityId() != null) {
			String jobId = request.getJobId() == null ? (instanceDao.getTemplateId() != null ? getResourceOperator(ITemplateOperator.class).findTemplate(instanceDao.getTemplateId()).getStopJobId(): null): request.getJobId(); 
			if (jobId != null) {
				TemplateJobUtil.asyncExecute(
						instanceDao.getFacilityId(),
						instanceDao.getAccountResourceId(),
						jobId,
						request.getArguments(),
						null,
						new JobEventListener() {
							@Override
							public void jobEventOccurerd(String facilityId, String jobunitId, String jobId) throws Exception {
//								findAgent(facilityId);
								internalStopInstance(instanceDao);
							}
						});
			}

			InstanceStateKind state = instance.getState();
			InstanceStateChange change = new InstanceStateChange();
			change.setInstanceId(request.getInstanceId());
			change.setCurrentState(state);
			change.setPreviousState(state);
			return change;
		}
		else {
			return internalStopInstance(instanceDao);
		}
	}

	private InstanceStateChange internalStopInstance(CloudInstanceDao instanceDao) throws CloudManagerFault {
		InstanceStateChange changed = getResourceManagement().stopInstance(instanceDao.getInstanceId());

		if (instanceDao.getFacilityId() != null) {
			InstanceMonitorService.getSingleton().startMonitor(
					instanceDao.getInstanceId(), instanceDao.getFacilityId(),
					SessionService.current().getContext(),
					getCloudService(), getCloudRegion(), getCloudUser(), false,
					InstanceStateKind.running, InstanceStateKind.terminated,
					InstanceStateKind.stopped);
		}

		return changed;
	}

	@Override
	public UpdateResult updateInstance(String instanceId) throws CloudManagerFault, InvalidRole {
		// 　まず、Hinemos DB からクラウド情報を取得。
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudInstanceDao cloudInstanceDao = em.find(CloudInstanceDao.class, new CloudInstanceDao.CloudInstancePK(instanceId,getCloudRegion().getRegion(), getCloudUser().getAccountResourceId()));

		Instance instance = null;
		if (cloudInstanceDao == null
				|| cloudInstanceDao.getRegistStatus() != CloudInstanceDao.InstanceStatus.deleted) {
			// AWS のボリューム情報を取得。
			instance = getResourceManagement().getInstance(instanceId);

			// 終了状態のインスタンスは無視。
			if (InstanceStateKind.shutting_down.equals(instance.getState())
					|| InstanceStateKind.terminated.equals(instance.getState())) {
				instance = null;
			}
		}

		UpdateResult result = internalUpdate(cloudInstanceDao, instance, CriticalPoint.getCriticalPoint(getAccountResource().getAccountId(), getCloudRegion().getRegion()).isEntering());

		return result;
	}

	protected UpdateResult transactionalInternalUpdate(CloudInstanceDao cloudInstanceDao, Instance instance, boolean entering) throws CloudManagerFault, InvalidRole {
		UpdateResult result = null;
		try (TransactionScope scope = new TransactionScope(Transactional.TransactionType.RequiredNew)) {
			EntityManagerEx em = SessionService.current().getEntityManagerEx();
			if (cloudInstanceDao != null) {
				//　EntityManager が切り替わっているので、再度取得。
				cloudInstanceDao = em.find(CloudInstanceDao.class, cloudInstanceDao.getId());
			}
			
			result =  internalUpdate(cloudInstanceDao, instance, entering);
			
			scope.complete();
		}
		catch (CloudManagerFault e) {
			String instanceId = cloudInstanceDao != null ? cloudInstanceDao.getInstanceId(): instance.getInstanceId();
			CloudMessageUtil.notify_AutoUpadate_Error_InstanceOperator(getAccountResource().getAccountResourceId(), instanceId, e);
		}
		
		return result;
	}

	protected UpdateResult internalUpdate(CloudInstanceDao cloudInstanceDao, Instance instance, boolean entering) throws CloudManagerFault, InvalidRole {
		try {
			if (instance != null) {
				if (cloudInstanceDao == null) {
					// 非同期で動作するインスタンス作成や別の更新処理が動作していない場合に、以下の処理を継続する。
					if (update && !entering) {
						// AWS インスタンスに対応するインスタンス情報を追加。
						try {
							cloudInstanceDao = createCloudInstanceDao(instance, null);
						}
						catch (CloudManagerFault e) {
							// CloudInstanceDao 作成済みの場合に関しては、このインスタンスへの処理をキャンセルし、
							// 処理を継続させる。
							// 非同期で動作するインスタンス作成や別の更新処理が先に作成している場合があるので、
							// エラーとはしない。
							if (ErrorCode.CLOUDINSTANCE_ALREADY_EXIST.match(e)) {
								// 別の処理が作成済みらしい。
								EntityManagerEx em = SessionService.current().getEntityManagerEx();
								cloudInstanceDao = em.find(CloudInstanceDao.class, new CloudInstanceDao.CloudInstancePK(instance.getInstanceId(), getCloudRegion().getRegion(), getCloudUser().getAccountResourceId()));
								if (cloudInstanceDao == null) {
									throw e;
								}
							}
							else {
								throw e;
							}
						}
	
						// AWS インスタンスに対するノードを自動で作成するかどうか？
						if (regist) {
							// ノードの自動作成。
							addNodeRelation(cloudInstanceDao, instance);
	
							if (relation) {
								relateExistedScope(cloudInstanceDao.getFacilityId(), instance);
							}
							else {
								// 特になにもしない。
							}
	
							if (cloudInstanceDao.getFacilityId() != null) {
								RegistAgent.asyncRegistAgent(cloudInstanceDao.getFacilityId());
							}
						} else {
							// 特になにもしない。
						}
					}
				} else {
					if (update) {
						// 登録済みの情報を更新。
						updateCloudInstanceDao(cloudInstanceDao, instance);
	
						if (cloudInstanceDao.getFacilityId() == null) {
							// AWS インスタンスに対するノードを自動で作成するかどうか？
							if (regist) {
								// ノードの自動作成。
								addNodeRelation(cloudInstanceDao, instance);
	
								if (relation) {
									relateExistedScope(cloudInstanceDao.getFacilityId(), instance);
								} else {
									// 特になにもしない。
								}
	
								if (cloudInstanceDao.getFacilityId() != null) {
									RegistAgent.asyncRegistAgent(cloudInstanceDao.getFacilityId());
								}
							} else {
								// 特になにもしない。
							}
						} else {
							if (relation) {
								relateExistedScope(cloudInstanceDao.getFacilityId(), instance);
							} else {
								// 特になにもしない。
							}
						}
					}
				}
			}
			else {
				if (cloudInstanceDao != null) {
					if (update) {
						if (cloudInstanceDao.getRegistStatus() != CloudInstanceDao.InstanceStatus.deleted) {
							cloudInstanceDao.setRegistStatus(CloudInstanceDao.InstanceStatus.deleted);
						}
	
						// 該当するノード情報を自動で削除するかどうか。
						if (regist) {
							// AWS インスタンスがないので自動削除。
							removeNodeRelation(cloudInstanceDao);
						}
					}
	
					// ノードと紐づいていないなら、削除。
					if (cloudInstanceDao.getFacilityId() == null) {
						deleteInstance(cloudInstanceDao);
						cloudInstanceDao = null;
					}
				} else {
					// どちらもないのでスルー
				}
			}
		}
		catch (CloudManagerFault e) {
			if (ErrorCode.CLOUDINSTANCE_DUPLICATE_NODE.match(e) || ErrorCode.HINEMOS_MANAGER_ERROR.match(e)) {
				CloudMessageUtil.notify_AutoUpadate_Error_InstanceOperator(getAccountResource().getAccountResourceId(), cloudInstanceDao.getInstanceId(), e);
			}
			else {
				throw e;
			}
		}
		return new UpdateResult(instance, cloudInstanceDao);
	}

	protected void relateExistedScope(String facilityId, Instance instance) {
		String scopesString = null;
		for (Tag t : instance.getTags()) {
			if (t.getKey().equals("hinemosAssignScopeId")) {
				scopesString = t.getValue();
				break;
			}
		}

		if (scopesString != null) {
			String[] scopes = scopesString.split(",");
			try {
				for (String scope : scopes) {
					try {
						if (ActionMode.isAutoDetection()) {
							getLogger().info(
									"Assign Node, Method=autoRegist, CloudUser="
											+ getCloudUser().getCloudUserId()
											+ ", ParentFacilityID=" + scope
											+ ", AssginFacilityID="
											+ facilityId);
						}
						RepositoryControllerBeanWrapper.bean().assignNodeScope(scope, new String[] { facilityId });
					}
					catch (Exception e) {
						Logger logger = Logger.getLogger(InstanceOperator.class);
						logger.error(e.getMessage(), e);
					}
				}
			}
			catch (Exception e) {
				Logger logger = Logger.getLogger(InstanceOperator.class);
				logger.error(e.getMessage(), e);
			}
		}
	}

	private CloudInstanceDao createCloudInstanceDao(Instance instance, String templateId) throws CloudManagerFault {
		try {
			CriticalPoint.getCriticalPoint(getAccountResource().getAccountResourceId(), getCloudRegion().getRegion()).enter();

			// DB に追加する情報を作成。
			CloudInstanceDao dao = new CloudInstanceDao();
			dao.setInstanceId(instance.getInstanceId());
			dao.setCloudServiceId(getAccountResource().getCloudServiceId());
			dao.setRegion(getCloudRegion().getRegion());
			dao.setAccountResourceId(getCloudUser().getAccountResourceId());
			dao.setInstanceName(instance.getName());
			dao.setZone(instance.getZone());
			dao.setCloudUserId(getCloudUser().getCloudUserId());
			dao.setRegistStatus(CloudInstanceDao.InstanceStatus.unregistered);
			dao.setTemplateId(templateId);

			EntityManagerEx em = SessionService.current().getEntityManagerEx();
			try {
				em.persist(dao);
				return dao;
			}
			catch (EntityExistsException e) {
				throw ErrorCode.CLOUDINSTANCE_ALREADY_EXIST.cloudManagerFault(dao.getInstanceId());
			}
		} finally {
			CriticalPoint.getCriticalPoint(getAccountResource().getAccountResourceId(), getCloudRegion().getRegion()).leave();
		}
	}

	private void updateCloudInstanceDao(CloudInstanceDao instanceDao, Instance instance) throws CloudManagerFault, InvalidRole {
		instanceDao.setInstanceName(instance.getName());

		// IP アドレスを更新。
		if (instanceDao.getFacilityId() != null) {
			try {
				RepositoryControllerBean repositoryLocal = RepositoryControllerBeanWrapper.bean();
				NodeInfo node = repositoryLocal.getNode(instanceDao.getFacilityId());

				String ipAddress = instance.getIpAddress();
				if (node.getIpAddressV4() == null
						|| (ipAddress == null && !"123.123.123.123".equals(node.getIpAddressV4()))
						|| (ipAddress != null && !node.getIpAddressV4().equals(ipAddress))) {
					if (ipAddress != null) {
						node.setIpAddressV4(ipAddress);
					} else {
						node.setIpAddressV4("123.123.123.123");
					}

					repositoryLocal.modifyNode(node);
				}
			} catch (FacilityNotFound | HinemosUnknown | InvalidSetting e) {
				throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
			}
		}
	}

	private void deleteInstance(CloudInstanceDao cloudInstance) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		em.remove(cloudInstance);
	}

	@Override
	@Transactional
	public UpdateAllResult updateAllInstance() throws CloudManagerFault,
			InvalidRole {
		// AWS から、必要なインスタンス情報を取得。
		IResourceManagement rm = getResourceManagement();
		List<Instance> awsInstances = new ArrayList<>(rm.getInstances());

		// Hinemos DB からインスタンス情報を取得。
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		Filter filter = new Filter("accountResourceId", getCloudUser().getAccountResourceId());
		Filter filter2 = new Filter("region", getCloudRegion().getRegion());
		List<CloudInstanceDao> CloudInstanceDaos = new ArrayList<CloudInstanceDao>(em.findByFilter(CloudInstanceDao.class, filter, filter2));

		// AWS および Hinemos DB に登録されているインスタンスの対応情報作成。
		Iterator<Instance> iIter = awsInstances.iterator();
		List<UpdateAllResult.InstanceMapping> instanceMappings = new ArrayList<>();
		while (iIter.hasNext()) {
			Instance i = iIter.next();

			// 終了状態のインスタンスは、表示しない。
			if (InstanceStateKind.shutting_down == i.getState() || InstanceStateKind.terminated == i.getState()) {
				iIter.remove();
				continue;
			}

			Iterator<CloudInstanceDao> ciIter = CloudInstanceDaos.iterator();
			while (ciIter.hasNext()) {
				CloudInstanceDao ci = ciIter.next();

				if (i.getInstanceId().equals(ci.getInstanceId())) {
					instanceMappings.add(new UpdateAllResult.InstanceMapping(i, ci));

					iIter.remove();
					ciIter.remove();
					break;
				}
			}
		}

		List<UpdateAllResult.InstanceMapping> both = new ArrayList<>();
		List<UpdateAllResult.InstanceMapping> onlyAws = new ArrayList<>();
		List<CloudInstanceDao> onlyCloud = new ArrayList<CloudInstanceDao>();

		boolean entering = CriticalPoint.getCriticalPoint(getAccountResource().getAccountResourceId(), getCloudRegion().getRegion()).isEntering();

		// AWS にも Hinemos DB にも登録されている。
		for (UpdateAllResult.InstanceMapping instanceMapping : instanceMappings) {
			UpdateResult result = transactionalInternalUpdate(instanceMapping.cloudInstance, instanceMapping.instance, entering);
			instanceMapping.cloudInstance = result.cloudInstance;
		}
		both.addAll(instanceMappings);

		// AWS のみに存在する。
		// インスタンスの自動更新フラグが有効になっているなら、自動登録処理実施。
		for (Instance i : awsInstances) {
			UpdateResult updateResult = transactionalInternalUpdate(null, i, entering);

			// ストレージの自動アタッチ。
			if (updateResult != null && updateResult.cloudInstance != null) {
				onlyAws.add(new UpdateAllResult.InstanceMapping(updateResult.awsInstance, updateResult.cloudInstance));
				getLogger().debug("AWS : " + "instanceId=" + i.getInstanceId());
			}
		}

		for (CloudInstanceDao cs : CloudInstanceDaos) {
			UpdateResult updateResult = transactionalInternalUpdate(cs, null, entering);

			if (updateResult != null && updateResult.cloudInstance != null) {
				onlyCloud.add(updateResult.cloudInstance);
			}
		}

		return new UpdateAllResult(both, onlyAws, onlyCloud);
	}

	private void addNodeRelation(CloudInstanceDao dao, Instance instance) throws CloudManagerFault, InvalidRole {
		// インスタンスの名前を取得。
		String instanceName = instance.getName() == null || instance.getName().isEmpty() ? instance.getInstanceId(): instance.getName();

		String hostName = instance.getHostName();
		if (hostName == null || hostName.length() == 0) {
			hostName = instanceName;
		}
		
		NodeInfo nodeInfo = HinemosUtil.createNodeInfo(
				instance.getInstanceId(),
				instanceName,
				instance.getPlatform().label(),
				getCloudService().getCloudTypeId(),
				hostName,
				"Hinemos Auto Regist",
				CloudConstants.NT_Resource,
				getCloudService().getCloudServiceId(),
				getAccountResource().getAccountResourceId(),
				instance.getResourceType(),
				instance.getInstanceId(),
				getCloudRegion().getRegion(),
				instance.getZone(),
				getCloudUser().getRoleId() 
				);

		String ipAddress = instance.getIpAddress();
		if (ipAddress != null) {
			nodeInfo.setIpAddressV4(ipAddress);
		} else {
			nodeInfo.setIpAddressV4("123.123.123.123");
		}
			
		RepositoryControllerBean repositoryControllerBean = RepositoryControllerBeanWrapper.bean();

		if (ActionMode.isAutoDetection()) {
			Logger logger = Logger.getLogger(this.getClass());
			logger.info("Add Node, Method=autoRegist, CloudUser="
					+ getCloudUser().getCloudUserId() + ", FacilityID="
					+ nodeInfo.getFacilityId() + ", InstanceId="
					+ instance.getInstanceId());
		}

		try (AddedEventNotifier<NodeInfo> notifier = new AddedEventNotifier<>(CloudConstants.Node_Instance_Add, NodeInfo.class, nodeInfo)) {
			try {
				repositoryControllerBean.addNode(nodeInfo);
				notifier.setCompleted();
			}
			catch (FacilityDuplicate e) {
				throw ErrorCode.CLOUDINSTANCE_DUPLICATE_NODE.cloudManagerFault(dao.getAccountResourceId(), dao.getInstanceId(), nodeInfo.getFacilityId());
			}
			catch (InvalidSetting | HinemosUnknown e) {
				// 既に作成済みのため、とりあえず戻す。
				// 例外を上位に返すと、例えば、全体の更新がこの例外のため途中で終了してしまう。
				// 本当は、各インスタンスに更新失敗のフラグがほしい。
//					getLogger().warn(e.getClass().getSimpleName() + ":" + e.getFacilityId());
//					return;
				throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
			}
		}

		try {
			// スコープへ関連付け。
			AccountResourceUtil.attachScope(
					repositoryControllerBean,
					getCloudUser().getAccountResourceId(),
					getAccountResource().getAccountResourceName(),
					getCloudRegion().getRegion(),
					getCloudRegion().getRegionName(),
					instance.getZone(),
					instance.getInstanceId(),
					getCloudUser().getRoleId(),
					new MessagesHolder("com.clustercontrol.cloud.messages"));
		}
		catch (InvalidSetting | HinemosUnknown | FacilityDuplicate e) {
			throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
		}

		dao.setFacilityId(instance.getInstanceId());
		dao.setRegistStatus(CloudInstanceDao.InstanceStatus.registered);

		// ストレージ情報を作成。
		IStorageOperator storageOperator = getResourceOperator(IStorageOperator.class);
		storageOperator.setFlags(true, true, true);
		for (BlockDeviceMapping bd : instance.getBlockDeviceMappings()) {
			storageOperator.updateStorage(bd.getStorageId());
		}
	}

	private void removeNodeRelation(CloudInstanceDao cloudInstanceDao) throws CloudManagerFault {
		if (cloudInstanceDao.getFacilityId() == null) {
			return;
		}

		final String facilityId = cloudInstanceDao.getFacilityId();
		final String instanceId = cloudInstanceDao.getInstanceId();
		final boolean autoRegist = ActionMode.isAutoDetection();

		try {
			if (autoRegist) {
				Logger.getLogger(this.getClass());
				getLogger().info(
						"Delete Node, Method=autoRegist, CloudUser="
								+ getCloudUser().getCloudUserId()
								+ ", FacilityID=" + facilityId
								+ ", InstanceId=" + instanceId);
			}

			// Hinemos ノード削除。
			RepositoryControllerBeanWrapper.bean().deleteNode(facilityId);
		}
		catch (Exception e) {
			throw ErrorCode.AUTOUPDATE_NOT_DELETE_FACILITY.cloudManagerFault(e, cloudInstanceDao.getAccountResourceId(), cloudInstanceDao.getFacilityId(), CloudManagerFault.getMessage(e));
		}

		unrelateNode(cloudInstanceDao);
	}

	private void unrelateNode(CloudInstanceDao cloudInstanceDao) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		cloudInstanceDao.setFacilityId(null);
		cloudInstanceDao.setRegistStatus(CloudInstanceDao.InstanceStatus.unregistered);
		em.flush();
	}

	@Override
	public CloudInstance registNode(final RegistNodeRequest request) throws CloudManagerFault, InvalidRole {
		Instance instance = getResourceManagement().getInstance(request.getInstanceId());
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudInstanceDao dao = em.find(CloudInstanceDao.class, new CloudInstanceDao.CloudInstancePK(request.getInstanceId(), getCloudRegion().getRegion(), getCloudUser().getAccountResourceId()));
		if (dao == null) {
			throw ErrorCode.CLOUDINSTANCE_NOT_FOUND.cloudManagerFault(request.getInstanceId());
		}
		return createCloudInstance(request.getNodeDetail(), dao, instance);
	}

	@Override
	public void unregistNode(String instanceId) throws CloudManagerFault, InvalidRole {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudInstanceDao dao = em.find(CloudInstanceDao.class, new CloudInstanceDao.CloudInstancePK(instanceId, getCloudRegion().getRegion(), getCloudUser().getAccountResourceId()));
		if (dao == null) {
			throw ErrorCode.CLOUDINSTANCE_NOT_FOUND.cloudManagerFault(instanceId);
		}

		if (dao.getRegistStatus() != CloudInstanceDao.InstanceStatus.deleted) {
			try {
				getResourceManagement().getInstance(instanceId);
			} catch (CloudManagerFault e) {
				// 既にインスタンスが存在しない場合は、無視。
				if (IResourceManagement.ErrorCode.Resource_InvalidInstanceID_NotFound.name().equals(e.getErrorCode())) {
					dao.setRegistStatus(CloudInstanceDao.InstanceStatus.deleted);
				} else {
					throw e;
				}
			}
		}
		// クラウドインスタンス情報削除。
		em.remove(dao);

		// クラウド管理として追加した情報をノードから削除。
		if (dao.getFacilityId() != null) {
			try {
				RepositoryControllerBean repositoryBean = RepositoryControllerBeanWrapper.bean();
				NodeInfo nodeInfo = repositoryBean.getNode(dao.getFacilityId());
				HinemosUtil.clearNodeInfo(nodeInfo);
				try (ModifiedEventNotifier<NodeInfo> notifier = new ModifiedEventNotifier<>(CloudConstants.Node_Instance_Clear, NodeInfo.class, nodeInfo)) {
					repositoryBean.modifyNode(nodeInfo);
					notifier.setCompleted();
				}
			} catch (FacilityNotFound | HinemosUnknown | InvalidSetting e) {
				throw ErrorCode.HINEMOS_MANAGER_ERROR.cloudManagerFault(e);
			}
		}
	}

	@Override
	@Transactional(Transactional.TransactionType.Supported)
	public void setFlags(boolean update, boolean regist, boolean relation) {
		this.update = update;
		this.regist = regist;
		this.relation = relation;
	}

	@Override
	public CloudInstance getInstance(String instanceId) throws CloudManagerFault, InvalidRole {
		CloudInstance ci = updateInstance(instanceId).getCloudInstance();
		if (ci == null) {
			throw ErrorCode.CLOUDINSTANCE_NOT_FOUND.cloudManagerFault(instanceId);
		}
		return ci;
	}

	@Override
	public CloudInstance getInstanceByFacilityId(String facilityId) throws CloudManagerFault, InvalidRole {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		List<CloudInstanceDao> instanceDaos = em.findByFilter(CloudInstanceDao.class, new Filter("facilityId", facilityId));
		if (instanceDaos.isEmpty()) {
			throw ErrorCode.CLOUDINSTANCE_NOT_FOUND_BY_FACILITY.cloudManagerFault(facilityId);
		}
		return updateInstance(instanceDaos.get(0).getInstanceId()).getCloudInstance();
	}

	@Override
	public List<CloudInstance> getAllInstances() throws CloudManagerFault,
			InvalidRole {
		return updateAllInstance().getCloudInstances();
	}

	@Override
	public void removeInstanceBackup(final String instanceBackupId) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudInstanceBackupDao dao = em.find(CloudInstanceBackupDao.class, new CloudInstanceBackupDao.CloudInstanceBackupPK(instanceBackupId, getCloudRegion().getRegion(), getCloudUser().getAccountResourceId()));
		if (dao == null) {
			// クラウドインスタンスが存在しない場合でも、インスタンスは削除しにいく。
			getResourceManagement().deleteInstanceBackup(instanceBackupId);

			throw ErrorCode.CLOUDINSTANCEBACKUP_NOT_FOUND_NODE.cloudManagerFault(instanceBackupId);
		}
		else {
			// クラウドインスタンス情報削除。
			em.remove(dao);

			// コミット後にイメージを削除する。
			switch (dao.getRestoreStatus()) {
			case available:
				SessionService.current().addPreCommitAction(
						new SessionService.PreCommitAction() {
							@Override
							public RolebackAction preCommit() throws TransactionException {
								try {
									getResourceManagement().deleteInstanceBackup(instanceBackupId);
								}
								catch (CloudManagerFault e) {
									throw new TransactionException(e);
								}
								// 削除した後は、復活できないのでロールバック処理はなし。
								return null;
							}
						});
				break;
			case unavailable:
				break;
			}
		}
	}

	@Override
	public CloudInstanceBackup getInstanceBackup(String instanceBackupId) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		CloudInstanceBackupDao dao = em.find(CloudInstanceBackupDao.class, new CloudInstanceBackupDao.CloudInstanceBackupPK(instanceBackupId, getCloudRegion().getRegion(), getCloudUser().getAccountResourceId()));
		if (dao == null) {
			throw ErrorCode.CLOUDINSTANCEBACKUP_NOT_FOUND_NODE.cloudManagerFault(instanceBackupId);
		}

		InstanceBackup backup = getResourceManagement().getInstanceBackup(instanceBackupId);
		if (dao.getRestoreStatus() == CloudInstanceBackupDao.RestoreStatus.available) {
			if (backup == null) {
				dao.setRestoreStatus(CloudInstanceBackupDao.RestoreStatus.unavailable);
			}
		}

		return new CloudInstanceBackup(dao, backup);
	}

	@Override
	public List<CloudInstanceBackup> getAllInstanceBackup() throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		Filter filter = new Filter("accountResourceId", getCloudUser().getAccountResourceId());
		Filter filter2 = new Filter("region", getCloudRegion().getRegion());
		List<CloudInstanceBackupDao> daos = new ArrayList<>(em.findByFilter(CloudInstanceBackupDao.class, filter, filter2));

		List<String> backupIdsIds = new ArrayList<>();
		for (CloudInstanceBackupDao backupDao : daos) {
			if (backupDao.getRestoreStatus() == CloudInstanceBackupDao.RestoreStatus.available) {
				backupIdsIds.add(backupDao.getInstanceBackupId());
			}
		}

		Map<String, InstanceBackup> instanceBackupsMap = new HashMap<>();
		if (!backupIdsIds.isEmpty()) {
			List<InstanceBackup> instanceBackups = getResourceManagement().getInstanceBackups(backupIdsIds);
			for (InstanceBackup instanceBackup : instanceBackups) {
				instanceBackupsMap.put(instanceBackup.getInstanceBackupId(), instanceBackup);
			}
		}

		List<CloudInstanceBackup> backups = new ArrayList<>();
		for (CloudInstanceBackupDao backupDao : daos) {
			InstanceBackup instanceBackup = instanceBackupsMap.get(backupDao.getInstanceBackupId());
			if (backupDao.getRestoreStatus() == CloudInstanceBackupDao.RestoreStatus.available) {
				if (instanceBackup == null) {
					backupDao.setRestoreStatus(CloudInstanceBackupDao.RestoreStatus.unavailable);
				}
			}
			backups.add(new CloudInstanceBackup(backupDao, instanceBackup));
		}

		return backups;
	}

	@Override
	@Transactional(Transactional.TransactionType.Supported)
	public List<Image> getImagesWithFilter(List<Filter> filters) throws CloudManagerFault {
		IResourceManagement rm = getResourceManagement();
		return rm.getImagesWithFilter(filters);
	}

	@Override
	public void updateAllInstanceBackup() throws CloudManagerFault, InvalidRole {
		getAllInstanceBackup();
	}

	@Override
	public List<LoadBalancer> getLoadBalancers() throws CloudManagerFault {
		List<IResourceManagement.LoadBalancer> lbs = getResourceManagement().getLoadBalancers();
		List<LoadBalancer> loadBalancers = new ArrayList<>();
		for (IResourceManagement.LoadBalancer lb: lbs) {
			LoadBalancer loadBalancer = new LoadBalancer();
			loadBalancer.setLbId(lb.getLbId());
			loadBalancer.setRegisterdInstanceIds(lb.getRegisterdInstanceIds());
			loadBalancer.setAvailableInstanceIds(lb.getAvailableInstanceIds());
			loadBalancers.add(loadBalancer);
		}
		return loadBalancers;
	}

	@Override
	public void registerInstanceToLoadBalancer(String lbId, String instanceId) throws CloudManagerFault {
		getResourceManagement().registerInstanceToLoadBalancer(lbId, instanceId);
	}

	@Override
	public void unregisterInstanceToLoadBalancer(String lbId, String instanceId) throws CloudManagerFault {
		getResourceManagement().unregisterInstanceToLoadBalancer(lbId, instanceId);
	}

	@Override
	public List<CloudInstanceBackup> getInstanceBackupByInstanceId(String instanceId) throws CloudManagerFault {
		EntityManagerEx em = SessionService.current().getEntityManagerEx();
		Query query = em.createQuery("SELECT b FROM CloudInstanceBackupDao b WHERE b.backupedData.instanceId = :instanceId");
		query.setParameter("instanceId", instanceId);

		@SuppressWarnings("unchecked")
		List<CloudInstanceBackupDao> daos = query.getResultList();
		
		List<CloudInstanceBackup> backups = new ArrayList<>();

		List<String> backupIds = new ArrayList<>();
		for (CloudInstanceBackupDao dao: daos) {
			backupIds.add(dao.getInstanceBackupId());
		}
		List<InstanceBackup> cloudBackups = getResourceManagement().getInstanceBackups(backupIds);
		
		List<CloudInstanceBackupDao> tempBackupDaos = new ArrayList<>(daos);
		List<InstanceBackup> tempBackups = new ArrayList<>(cloudBackups);
		
		Iterator<CloudInstanceBackupDao> i = tempBackupDaos.iterator();
		while (i.hasNext()) {
			CloudInstanceBackupDao dao = i.next();
			Iterator<InstanceBackup> j = tempBackups.iterator();
			while (j.hasNext()) {
				InstanceBackup backup = j.next();
				if (backup.getInstanceBackupId().equals(dao.getInstanceBackupId())) {
					backups.add(new CloudInstanceBackup(dao, backup));
					i.remove();
					j.remove();
					break;
				}
			}
		}
		
		for (CloudInstanceBackupDao dao: tempBackupDaos) {
			if (dao.getRestoreStatus() == CloudInstanceBackupDao.RestoreStatus.available) {
				dao.setRestoreStatus(CloudInstanceBackupDao.RestoreStatus.unavailable);
			}
		}
		return backups;
	}

	// @Override
	// public List<String> getFlavors() throws CloudManagerFault {
	// return getResourceManagement().getInstanceFlavors();
	// }
}