/*
Copyright (C) 2014 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.bean;

import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.ErrorCode;
import com.clustercontrol.cloud.IMessagesHolder;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.factory.CloudAccessorUtil;
import com.clustercontrol.cloud.util.AuthorizingUtil;
import com.clustercontrol.cloud.validation.CustomEntityValidator;
import com.clustercontrol.cloud.validation.EntityValidator.EntityValidationContext;
import com.clustercontrol.cloud.validation.ValidationConstants;
import com.clustercontrol.cloud.validation.ValidationFault;
import com.clustercontrol.cloud.validation.annotation.CustomEntityValidation;
import com.clustercontrol.cloud.validation.annotation.ElementId;
import com.clustercontrol.cloud.validation.annotation.Identity;
import com.clustercontrol.cloud.validation.annotation.NotNull;
import com.clustercontrol.cloud.validation.annotation.Size;

/**
 * クラウドユーザ作成要求に必要な情報を保持するクラス。 
 * {@link com.clustercontrol.ws.cloud.CloudEndpoint#addCloudUser(CreateCloudUserRequest) addCloudUser 関数} にて使用される。
 *
 */
@CustomEntityValidation(CreateCloudUserRequest.CloudUserValidator.class)
public class CreateCloudUserRequest {
	public static class CloudUserValidator implements CustomEntityValidator<CreateCloudUserRequest>, ValidationConstants {
		@Override
		public void validate(final CreateCloudUserRequest entity, String group, IMessagesHolder messages, EntityValidationContext context) throws ValidationFault {
			try {
				if (
					!AuthorizingUtil.checkHinemousUser_administrators_account(SessionService.current().getHinemosCredential().getUserId(), entity.getAccountResourceId())
					) {
					throw ErrorCode.NEED_ADMINISTRATORS_ROLE_OR_ACCOUNT_USER.cloudManagerFault(SessionService.current().getHinemosCredential().getUserId(), entity.getAccountResourceId());
				}
				
				// クラウド側にユーザー情報が存在するか確認。
				CloudAccessorUtil.getUser(entity.getAccountResourceId(), entity.getAccessKey(), entity.getSecretKey());
			}
			catch (CloudManagerFault e) {
				throw new ValidationFault(e.getMessage(), e.getErrorCode(), e);
			}
		}
	}

	private String cloudUserId;
	private String cloudUserName;
	private String accountResourceId;
	private String description;
	private String accessKey;
	private String secretKey;
	private String roleId;
	
	/**
	 * クラウドユーザー ID を取得する。
	 * 
	 * @return クラウドユーザー ID。
	 */
	@ElementId("cloudUserId")
	@Identity
	public String getCloudUserId() {
		return cloudUserId;
	}
	/**
	 * クラウドユーザー ID を設定する。
	 * 
	 * @param cloudUserId クラウドユーザー ID。
	 */
	public void setCloudUserId(String cloudUserId) {
		this.cloudUserId = cloudUserId;
	}
	/**
	 * クラウドユーザー名を取得する。
	 * 
	 * @return クラウドユーザー名。
	 */
	@ElementId("cloudUserName")
	@NotNull
	@Size(max=128)
	public String getCloudUserName() {
		return cloudUserName;
	}
	/**
	 * クラウドユーザー名を設定する。
	 * 
	 * @param cloudUserName クラウドユーザー名。
	 */
	public void setCloudUserName(String cloudUserName) {
		this.cloudUserName = cloudUserName;
	}
	/**
	 * アカウントリソース ID を取得する。
	 * 
	 * @return アカウントリソース ID。
	 */
	@ElementId("accountResourceId")
	@Identity
	public String getAccountResourceId() {
		return accountResourceId;
	}
	/**
	 * アカウントリソース ID を設定する。
	 * 
	 * @param accountResourceId アカウントリソース ID。
	 */
	public void setAccountResourceId(String accountResourceId) {
		this.accountResourceId = accountResourceId;
	}
	/**
	 * クラウドユーザーの説明文を取得する。
	 * 
	 * @return クラウドユーザーの説明文。
	 */
	@ElementId("description")
	@Size(max = 256)
	@NotNull
	public String getDescription() {
		return description;
	}
	/**
	 * クラウドユーザーの説明文を設定する。
	 * 
	 * @param description クラウドユーザーの説明文。
	 */
	public void setDescription(String description) {
		this.description = description;
	}
	/**
	 * クラウドのアクセスキーを取得する。
	 * 
	 * @return クラウドのアクセスキー。
	 */
	@ElementId("accessKey")
	@Size(max = 64)
	@NotNull
	public String getAccessKey() {
		return accessKey;
	}
	/**
	 * クラウドのアクセスキーを設定する。
	 * 
	 * @param accessKey クラウドのアクセスキー。
	 */
	public void setAccessKey(String accessKey) {
		this.accessKey = accessKey;
	}
	/**
	 * クラウドのシークレットキーを取得する。
	 * 
	 * @return クラウドのシークレットキー。
	 */
	@ElementId("secretKey")
	@Size(max = 64)
	@NotNull
	public String getSecretKey() {
		return secretKey;
	}
	/**
	 * クラウドのシークレットキーを設定する。
	 * 
	 * @param secretKey クラウドのシークレットキー。
	 */
	public void setSecretKey(String secretKey) {
		this.secretKey = secretKey;
	}
	/**
	 * クラウドユーザーに紐づくロール ID を取得する。
	 * 
	 * @return クラウドユーザーに紐づくロール ID。
	 */
	@ElementId("roleId")
	@Identity
	public String getRoleId() {
		return roleId;
	}
	/**
	 * クラウドユーザーに紐づくロール ID を設定する。
	 * 
	 * @param roleId クラウドユーザーに紐づくロール ID。
	 */
	public void setRoleId(String roleId) {
		this.roleId = roleId;
	}
}