/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.aws.util;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.log4j.Logger;

import com.amazonaws.AmazonServiceException;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.Protocol;
import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.services.cloudwatch.AmazonCloudWatch;
import com.amazonaws.services.cloudwatch.AmazonCloudWatchClient;
import com.amazonaws.services.ec2.AmazonEC2;
import com.amazonaws.services.ec2.AmazonEC2Client;
import com.amazonaws.services.ec2.model.CreateTagsRequest;
import com.amazonaws.services.ec2.model.Tag;
import com.amazonaws.services.elasticloadbalancing.AmazonElasticLoadBalancing;
import com.amazonaws.services.elasticloadbalancing.AmazonElasticLoadBalancingClient;
import com.amazonaws.services.identitymanagement.AmazonIdentityManagement;
import com.amazonaws.services.identitymanagement.AmazonIdentityManagementClient;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3Client;
import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.IResourceManagement.ICredential;
import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.aws.AWSOptionPropertyConstants;

public class AWSUtil {
	public static class AMIArn {
		public static final String root = "root";
		public static final String userPrefix = "user/";
		
		public String arn;
		public String accountId;
		public String userName;
				
		public AMIArn(String arn) {
			Pattern p = Pattern.compile("^arn:aws:iam::([0-9]+):(.+)$");
			Matcher m = p.matcher(arn);
			if (m.matches()) {
				accountId = m.group(1);
				userName = m.group(2);
				if (!"root".equals(userName)) {
					if (userName.startsWith("user/")) {
						userName = userName.substring("user/".length());
					}
					else {
						throw new IllegalStateException();
					}
				}
			}
			else {
				throw new IllegalStateException();
			}
		}
		
		public boolean isRoot() {
			return root.equals(userName);
		}
	}

	public static String getAccountIdFromIAM(String arn) {
		return new AMIArn(arn).accountId;
	}
	
	public static void addTag(AmazonEC2 ec2, String resourceId, String key, String value) {
		addTags(ec2, resourceId, new Tag[]{new Tag(key, value)});
	}

	public static void addTags(AmazonEC2 ec2, String resourceId, List<Tag> tags) {
		addTags(ec2, resourceId, tags.toArray(new Tag[0]));
	}
	
	public static void addTags(AmazonEC2 ec2, String resourceId, Tag... tags) {
		// Name タグをつける際、作成直後のリソースが見つからずに失敗する場合があったので、リトライをする。
		int count = 0;
		int interval = 1000;
		for (;;) {
			try {
				CreateTagsRequest createTagsRequest = new CreateTagsRequest();
				createTagsRequest.withTags(tags).withResources(resourceId);
				ec2.createTags(createTagsRequest);
				break;
			}
			catch (AmazonServiceException e) {
				++count;
	
				Logger logger = Logger.getLogger(AWSUtil.class);
				logger.debug(e.getMessage(), e);
				
				if (count == 5) {
					throw e;
				}
				
				try {
					Thread.sleep(interval);
					interval += interval;
				} catch (InterruptedException e1) {
				}
			}
		}
	}
	
	public static ClientConfiguration createClientConfiguration() {
		ClientConfiguration conf = new ClientConfiguration();
		conf.setConnectionTimeout(Integer.valueOf(AWSOptionPropertyConstants.aws_client_config_connectionTimeout.value()));
		conf.setMaxConnections(Integer.valueOf(AWSOptionPropertyConstants.aws_client_config_maxConnections.value()));
		conf.setMaxErrorRetry(Integer.valueOf(AWSOptionPropertyConstants.aws_client_config_maxErrorRetry.value()));
		conf.setProtocol(Protocol.HTTP.toString().equals(AWSOptionPropertyConstants.aws_client_config_protocol.value()) ? Protocol.HTTP: Protocol.HTTPS);
		conf.setProxyDomain(AWSOptionPropertyConstants.aws_client_config_proxyDomain.value());
		conf.setProxyHost(AWSOptionPropertyConstants.aws_client_config_proxyHost.value());
		conf.setProxyPassword(AWSOptionPropertyConstants.aws_client_config_proxyPassword.value());
		conf.setProxyPort(Integer.valueOf(AWSOptionPropertyConstants.aws_client_config_proxyPort.value()));
		conf.setProxyUsername(AWSOptionPropertyConstants.aws_client_config_proxyUsername.value());
		conf.setProxyWorkstation(AWSOptionPropertyConstants.aws_client_config_proxyWorkstation.value());
		conf.setSocketTimeout(Integer.valueOf(AWSOptionPropertyConstants.aws_client_config_socketTimeout.value()));
		return conf;
	}
	
	public static AmazonEC2 createAmazonEC2(ICredential credential, String location) throws CloudManagerFault {
		AmazonEC2 endpoint =  createAWSEndpoint(AmazonEC2.class, AmazonEC2Client.class, credential);
		if (location != null) {
			endpoint.setEndpoint(location);
		}
		return endpoint;
	}

	public static AmazonElasticLoadBalancing createAmazonElasticLoadBalancing(ICredential credential, String location) throws CloudManagerFault {
		AmazonElasticLoadBalancing endpoint =  createAWSEndpoint(AmazonElasticLoadBalancing.class, AmazonElasticLoadBalancingClient.class, credential);
		if (location != null) {
			endpoint.setEndpoint(location);
		}
		return endpoint;
	}

	public static AmazonCloudWatch createCloudWatch(ICredential credential, String location) throws CloudManagerFault {
		AmazonCloudWatch endpoint =  createAWSEndpoint(AmazonCloudWatch.class, AmazonCloudWatchClient.class, credential);
		if (location != null) {
			endpoint.setEndpoint(location);
		}
		return endpoint;
	}

	public static AmazonS3 createS3(ICredential credential) throws CloudManagerFault {
		AmazonS3 endpoint =  createAWSEndpoint(AmazonS3.class, AmazonS3Client.class, credential);
		return endpoint;
	}

	public static AmazonIdentityManagement createIAM(ICredential credential) throws CloudManagerFault {
		AmazonIdentityManagement endpoint =  createAWSEndpoint(AmazonIdentityManagement.class, AmazonIdentityManagementClient.class, credential);
		return endpoint;
	}

	public static <T> T createAWSEndpoint(Class<T> interfaceClass, Class<? extends T> endpointClazz, ICredential credential) {
		try {
			BasicAWSCredentials awsCledential = new BasicAWSCredentials(credential.getAccessKey(), credential.getSecretKey());
			final Object endpoint = endpointClazz.getConstructor(AWSCredentials.class, ClientConfiguration.class).newInstance(awsCledential, AWSUtil.createClientConfiguration());

			InvocationHandler h = new InvocationHandler() {
				@Override
				public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
					try {
						long before = System.currentTimeMillis();
						Object result = method.invoke(endpoint, args);
						long after = System.currentTimeMillis();
						
						StringBuilder sb = new StringBuilder();
						sb.append("Called ").append(method.getName()).append(" : ").append("elapsedTime=").append(after - before).append("ms");
						if (args != null) {
							for (Object arg: args) {
								sb.append(", arg=").append(arg.toString());
							}
						}

						Logger.getLogger(this.getClass()).debug(sb.toString());
						
						return result;
					}
					catch (InvocationTargetException e) {
						throw e.getCause();
					}
				}
			};

			return interfaceClass.cast(Proxy.newProxyInstance(endpointClazz.getClassLoader(), new Class<?>[]{interfaceClass}, h));
		}
		catch (RuntimeException e) {
			throw e;
		}
		catch (Exception e) {
			throw new InternalManagerError(e);
		}
	}
}
