/*
Copyright (C) 2014 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
*/
package com.clustercontrol.cloud.cloudn;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.persistence.Query;

import org.apache.log4j.Logger;
import org.eclipse.jdt.annotation.Nullable;

import com.clustercontrol.cloud.CloudManagerBaseService;
import com.clustercontrol.cloud.CloudManagerFault;
import com.clustercontrol.cloud.CloudPlugin;
import com.clustercontrol.cloud.IResourceManagement;
import com.clustercontrol.cloud.IResourceManagement.Instance;
import com.clustercontrol.cloud.IUserManagement;
import com.clustercontrol.cloud.InternalManagerError;
import com.clustercontrol.cloud.PluginFault;
import com.clustercontrol.cloud.SessionService;
import com.clustercontrol.cloud.bean.CloudEndpoint;
import com.clustercontrol.cloud.bean.CloudRegion;
import com.clustercontrol.cloud.bean.CloudStorage;
import com.clustercontrol.cloud.bean.CloudUser;
import com.clustercontrol.cloud.cloudn.factory.CloudnResourceManagement_FLAT;
import com.clustercontrol.cloud.cloudn.factory.CloudnResourceManagement_VPC;
import com.clustercontrol.cloud.cloudn.factory.CloudnUserManagement;
import com.clustercontrol.cloud.cloudn.factory.MigrationProcesser;
import com.clustercontrol.cloud.cloudn.util.CloudnConstants;
import com.clustercontrol.cloud.dao.CloudUserDao;
import com.clustercontrol.cloud.factory.ICloudServiceOperator;
import com.clustercontrol.cloud.factory.monitors.AutoDetectionManager;
import com.clustercontrol.cloud.factory.monitors.InstanceMonitorService;
import com.clustercontrol.cloud.persistence.EntityManagerEx;
import com.clustercontrol.cloud.persistence.PersistenceUtil.TransactionScope;
import com.clustercontrol.cloud.persistence.Transactional;
import com.clustercontrol.cloud.registry.AbstractObjectChangedListener;
import com.clustercontrol.cloud.registry.IObjectChangedService;
import com.clustercontrol.cloud.registry.ObjectRegistryService;
import com.clustercontrol.cloud.util.proxy.IProxyStore;
import com.clustercontrol.ws.cloudn.CloudnOptionEndpointImpl;
import com.google.common.base.Optional;

public class CloudnManagerBaseService extends CloudPlugin {
	public final static String id = CloudnManagerBaseService.class.getName();
	
	private static Optional<CloudnManagerBaseService> singleton = Optional.absent();
	
	/**
	 * CloudServiceのコンストラクタ
	 */
	public CloudnManagerBaseService() {
		super();
		setSingleton(this);
	}

	@Override
	public Set<String> getDependency() {
		Set<String> set = new HashSet<>();
		set.add(CloudManagerBaseService.class.getName());
		return set;
	}

	@Override
	public void create() {
		Logger logger = Logger.getLogger(this.getClass());

		// メイン処理
		logger.info("creating " + getClass().getSimpleName() + "...");
		
		//　プロパティ一覧の表示
		for (CloudnOptionPropertyConstants value: CloudnOptionPropertyConstants.values()) {
			logger.info(value.id + " = " + value.value() + " (" + value.defaultValue() + ")");
		}
		
		logger.info("successful in creating " + getClass().getSimpleName() + "...");
	}

	@Override
	public void activate() {
		Logger logger = Logger.getLogger(this.getClass());

		// メイン処理
		logger.info("starting " + getClass().getSimpleName() + "...");
		
		logger.info("successful in starting " + getClass().getSimpleName() + "...");
	}

	@Override
	public void deactivate() {
		Logger logger = Logger.getLogger(this.getClass());

		// メイン処理
		logger.info("stopping " + getClass().getSimpleName() + "...");
		
		logger.info("successful in stopping " + getClass().getSimpleName() + "...");
	}

	@Override
	public void destroy() {
		Logger logger = Logger.getLogger(this.getClass());

		// メイン処理
		logger.info("destroying " + getClass().getSimpleName() + "...");
		
		logger.info("successful in destroying " + getClass().getSimpleName() + "...");
	}
	
	@Nullable public static CloudnManagerBaseService getSingleton() {
		return singleton.isPresent() ? singleton.get(): null;
	}

	private static void setSingleton(CloudnManagerBaseService service) {
		singleton = Optional.of(service);
	}

	@Override
	public String getPluginId() {
		return id;
	}

	@Override
	public void initialize() {
		Logger logger = Logger.getLogger(this.getClass());

		// メイン処理
		logger.info("initializing " + getClass().getSimpleName() + "...");
		
		ObjectRegistryService.registry().put(IUserManagement.class, CloudnConstants.TYPE_VPC_ID, CloudnUserManagement.class);
		ObjectRegistryService.registry().put(IUserManagement.class, CloudnConstants.TYPE_FLAT_ID, CloudnUserManagement.class);
		ObjectRegistryService.registry().put(IResourceManagement.class, CloudnConstants.TYPE_VPC_ID, CloudnResourceManagement_VPC.class);
		ObjectRegistryService.registry().put(IResourceManagement.class, CloudnConstants.TYPE_FLAT_ID, CloudnResourceManagement_FLAT.class);
		
		CloudManagerBaseService.getSingleton().publish("/HinemosWS/CloudnOptionEndpoint", new CloudnOptionEndpointImpl());
		String proxyHost = CloudnOptionPropertyConstants.cloudn_client_config_proxyHost.value();
		int proxyPort = Integer.valueOf(CloudnOptionPropertyConstants.cloudn_client_config_proxyPort.value());
		if (proxyHost != null && !proxyHost.isEmpty() && proxyPort > 0) {
			Pattern pattern = Pattern.compile("^([^/]+).*$");

			String proxyUsername = CloudnOptionPropertyConstants.cloudn_client_config_proxyUsername.value();
			String proxyPassword = CloudnOptionPropertyConstants.cloudn_client_config_proxyPassword.value();
			IProxyStore proxyStore = ObjectRegistryService.registry().get(IProxyStore.class);
			
			ICloudServiceOperator service = ObjectRegistryService.registry().get(ICloudServiceOperator.class);
			List<CloudRegion> regions;
			List<String> destinations = new ArrayList<>();
			try {
				regions = service.findCloudRegionsByService(CloudnConstants.SERVICE_VPC_ID);
			}
			catch (CloudManagerFault e) {
				throw new InternalManagerError(e);
			}
			for (CloudRegion region: regions) {
				CloudEndpoint endpoint = region.getEndpoint(CloudnConstants.EP_TYPE_COMPUTE_VPC);
				if (endpoint != null) {
					Matcher m = pattern.matcher(endpoint.getLocation());
					if (m.matches()) {
						destinations.add(m.group(1));
					}
					else {
						throw new InternalManagerError();
					}
				}
			}
			try {
				regions = service.findCloudRegionsByService(CloudnConstants.SERVICE_FLAT_ID);
			}
			catch (CloudManagerFault e) {
				throw new InternalManagerError(e);
			}
			for (CloudRegion region: regions) {
				CloudEndpoint endpoint = region.getEndpoint(CloudnConstants.EP_TYPE_COMPUTE_FLAT);
				if (endpoint != null) {
					Matcher m = pattern.matcher(endpoint.getLocation());
					if (m.matches()) {
						destinations.add(m.group(1));
					}
					else {
						throw new InternalManagerError();
					}
				}
			}
			proxyStore.add(
				proxyHost,
				proxyPort,
				proxyUsername,
				proxyPassword,
				destinations.toArray(new String[destinations.size()])
				);
		}
		
		IObjectChangedService service = ObjectRegistryService.registry().get(IObjectChangedService.class);
		service.addObjectChangedListener(null, CloudUser.class, new AbstractObjectChangedListener<CloudUser>() {
			@Override
			public void postAdded(String type, CloudUser object) throws PluginFault {
				if (!CloudUserDao.CloudUserType.user.equals(object.getCloudUserType())) return;

				String serviceId = object.getTableData().getAccountResource().getCloudServiceId();
				if (!CloudnConstants.SERVICE_VPC_ID.equals(serviceId) &&
					!CloudnConstants.SERVICE_FLAT_ID.equals(serviceId)) return;
				
				CloudUserDao account = object.getTableData().getAccountResource().getAccount();
				if (!account.getAccessKey().equals(object.getAccessKey()) ||
					!account.getSecretKey().equals(object.getSecretKey())) throw ErrorCode.CLOUDUSER_MUST_USE_SAME_PASSWORD_OF_ACCOUNT.cloudManagerFault(object.getCloudUserName());
			}
		});
		service.addObjectChangedListener(null, CloudStorage.class, new MigrationProcesser());
		
		AutoDetectionManager.getService().addListener(new AutoDetectionManager.AutoDetectionListner() {
			@Override
			public void onPreUpdateBranch() {
			}
			@Override
			public void onPostUpdateBranch() {
			}
			@Override
			public void onPreUpdateRoot() {
			}
			@Override
			public void onPostUpdateRoot() {
				// cc_cloud_resource_storage のごみ掃除
				try (TransactionScope t = new TransactionScope(Transactional.TransactionType.RequiredNew)) {
					EntityManagerEx em = SessionService.current().getEntityManagerEx();
					Query query = em.createQuery("DELETE FROM CloudResourceStoreDao r WHERE " +
							"r.resourceType = :resourceType AND r.cloudServiceId IN :cloudServiceId AND " +
							"NOT EXISTS (SELECT r FROM CloudInstanceBackupDao b WHERE " + 
								"r.resourceId = b.instanceBackupId AND r.accountResourceId = b.accountResourceId AND r.region = b.region)");
					query.setParameter("resourceType", CloudnResourceManagement_VPC.RT_InstanceBackup);
					query.setParameter("cloudServiceId", Arrays.asList(CloudnConstants.SERVICE_FLAT_ID, CloudnConstants.SERVICE_VPC_ID));
					query.executeUpdate();
					t.complete();
				}
				try (TransactionScope t = new TransactionScope(Transactional.TransactionType.RequiredNew)) {
					EntityManagerEx em = SessionService.current().getEntityManagerEx();
					Query query = em.createQuery("DELETE FROM CloudResourceStoreDao r WHERE " +
							"r.resourceType = :resourceType AND r.cloudServiceId IN :cloudServiceId AND " +
							"NOT EXISTS (SELECT r FROM CloudStorageBackupDao b WHERE " +
								"r.resourceId = b.storageBackupId AND r.accountResourceId = b.accountResourceId AND r.region = b.region) ");
					query.setParameter("resourceType", CloudnResourceManagement_VPC.RT_StorageBackup);
					query.setParameter("cloudServiceId", Arrays.asList(CloudnConstants.SERVICE_FLAT_ID, CloudnConstants.SERVICE_VPC_ID));
					query.executeUpdate();
					t.complete();
				}
			}
		});
		
		InstanceMonitorService.getSingleton().addListener(new InstanceMonitorService.InstanceTraceListner() {
			@Override
			public void onNotify(String accountResourceId, Instance instance) {
				
			}
			@Override
			public void onError(String accountResourceId, String instanceId, Exception e) {
				
			}
		});
		logger.info("successful in initializing " + getClass().getSimpleName() + "...");
	}
}
