/*

 Copyright (C) 2012 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.snmptrap.factory;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.opennms.protocols.snmp.SnmpObjectId;
import org.opennms.protocols.snmp.SnmpOctetString;
import org.opennms.protocols.snmp.SnmpPduPacket;
import org.opennms.protocols.snmp.SnmpPduRequest;
import org.opennms.protocols.snmp.SnmpPduTrap;
import org.opennms.protocols.snmp.SnmpSyntax;
import org.opennms.protocols.snmp.SnmpTimeTicks;
import org.opennms.protocols.snmp.SnmpTrapHandler;
import org.opennms.protocols.snmp.SnmpTrapSession;
import org.opennms.protocols.snmp.SnmpVarBind;

import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.snmptrap.bean.SnmpTrapV1;

/**
 * SNMPTRAPを受信し、その情報をV1形式に変換したのち、フィルタリングクラスに委譲するクラス
 * 
 * @since 4.0
 */
public class SnmpTrapReceiver implements SnmpTrapHandler {

	private static final Log _log = LogFactory.getLog(SnmpTrapReceiver.class);

	private final SnmpTrapSession session;
	private final SnmpTrapFilter filter;

	private volatile long counter;
	private static final int _statsInterval;

	public static final String _snmpSysuptimeOid = ".1.3.6.1.2.1.1.3.0";
	public static final String _snmpSysuptimePrefixOid = ".1.3.6.1.2.1.1.3";

	public static final String _snmpTrapOid = ".1.3.6.1.6.3.1.1.4.1.0";
	public static final String _snmpTrapEnterpriseOid = ".1.3.6.1.6.3.1.1.4.3.0";
	public static final String _snmpTrapsOid = ".1.3.6.1.6.3.1.1.5";

	public static final List<SnmpObjectId> _genericTraps;

	public static final String _genericTrapColdStartOid = "1.3.6.1.6.3.1.1.5.1";
	public static final String _genericTrapWarnStartOid = "1.3.6.1.6.3.1.1.5.2";
	public static final String _genericTrapLinkDownOid = "1.3.6.1.6.3.1.1.5.3";
	public static final String _genericTrapLinkUpOid = "1.3.6.1.6.3.1.1.5.4";
	public static final String _genericTrapAuthFailure = "1.3.6.1.6.3.1.1.5.5";
	public static final String _genericTrapEgpNeighborLoss = "1.3.6.1.6.3.1.1.5.6";

	static {
		// initialize generic traps
		_genericTraps = new ArrayList<SnmpObjectId>();
		_genericTraps.add(new SnmpObjectId(_genericTrapColdStartOid));
		_genericTraps.add(new SnmpObjectId(_genericTrapWarnStartOid));
		_genericTraps.add(new SnmpObjectId(_genericTrapLinkDownOid));
		_genericTraps.add(new SnmpObjectId(_genericTrapLinkUpOid));
		_genericTraps.add(new SnmpObjectId(_genericTrapAuthFailure));
		_genericTraps.add(new SnmpObjectId(_genericTrapEgpNeighborLoss));

		// initialize properties
		int statsInterval = 100;
		try {
			statsInterval = Integer.parseInt(HinemosProperties.getProperty("snmptrap.stats.interval", "100"));
		} catch (Exception e) { }
		_statsInterval = statsInterval;
		_log.info("initilized SnmpTrapReceiver : statsInterval = " + _statsInterval);
	}

	public SnmpTrapReceiver(String bindAddr, int port, SnmpTrapFilter filter) throws HinemosUnknown {
		try {
			_log.info("initializing snmptrap receiver. (bindAddr = " + bindAddr + ", port = " + port + ")");

			if (bindAddr == null) {
				session = new SnmpTrapSession(this, port);
			} else {
				InetAddress addr = InetAddress.getByName(bindAddr);
				session = new SnmpTrapSession(this, port, addr);
			}
			this.filter = filter;
		} catch (Exception e) {
			throw new HinemosUnknown("snmptrap receiver initialization failure.", e);
		}

		counter = 0;
	}

	public void shutdown() {
		if (session != null) {
			_log.info("shutting down snmptrap receiver.");
			session.close();
		}
		filter.shutdown();
	}

	/**
	 * SNMP v2を受信した際のハンドラ実装
	 */
	@Override
	public void snmpReceivedTrap(SnmpTrapSession session, InetAddress addr,
			int port, SnmpOctetString community, SnmpPduPacket pdu) {

		long receivedTime = System.currentTimeMillis();
		countup();

		if (_log.isDebugEnabled()) {
			_log.debug("SnmpTrap v2 received from agent " + addr + " on port " + port);
			_log.debug("SnmpTrap v2 : community = " + community);
			_log.debug("SnmpTrap v2 PDU : command = " + pdu.getCommand());
			_log.debug("SnmpTrap v2 PDU : ID = " + pdu.getRequestId());
			_log.debug("SnmpTrap v2 PDU : Length = " + pdu.getLength());

			if (pdu instanceof SnmpPduRequest) {
				_log.debug("SnmpTrap v2 PDU : Error Status = " + ((SnmpPduRequest)pdu).getErrorStatus());
				_log.debug("SnmpTrap v2 PDU : Error Index = " + ((SnmpPduRequest)pdu).getErrorIndex());
			}
		}

		SnmpTrapV1 snmptrap = convertV2toV1(receivedTime, community.toString(), addr, pdu);
		if (snmptrap == null) {
			_log.info("dicarding unexpected SnmpTrap v2 received from host " + addr);
		} else {
			if (_log.isDebugEnabled()) {
				_log.debug("start filtering : " + snmptrap);
			}
			filter.put(snmptrap);
		}

	}

	/**
	 * SNMP v1を受信した際のハンドラ実装
	 */
	@Override
	public void snmpReceivedTrap(SnmpTrapSession session, InetAddress addr,
			int port, SnmpOctetString community, SnmpPduTrap pdu) {

		long receivedTime = System.currentTimeMillis();
		countup();

		if (_log.isDebugEnabled()) {
			_log.debug("SnmpTrap v1 received from agent " + addr + " on port " + port);
			_log.debug("SnmpTrap v1 : community = " + community);
			_log.debug("SnmpTrap v1 PDU : IP ADDRESS = " + pdu.getAgentAddress());
			_log.debug("SnmpTrap v1 PDU : Enterprise Id = " + pdu.getEnterprise());
			_log.debug("SnmpTrap v1 PDU : Generic = " + pdu.getGeneric());
			_log.debug("SnmpTrap v1 PDU : Specific = " + pdu.getSpecific());
			_log.debug("SnmpTrap v1 PDU : Timestamp = " + pdu.getTimeStamp());
			_log.debug("SnmpTrap v1 PDU : Length = " + pdu.getLength());
		}

		List<SnmpSyntax> varbinds = new ArrayList<SnmpSyntax>();
		for (int i = 0; i < pdu.getLength(); i++) {
			try {
				SnmpVarBind varbind = pdu.getVarBindAt(i);
				SnmpSyntax value = varbind.getValue();
				varbinds.add(value);

				if (_log.isDebugEnabled()) {
					_log.debug("SnmpTrap v2 PDU (varbind[" + i +  "]) : name = " + varbind.getName().toString() + ", value = " + value.toString());
				}
			} catch (Exception e) {
				_log.info("SnmpTrap v2 PDU (varbind " + i + " access failure) : Length = " + pdu.getLength()
						+ " [" + e.getMessage() + "]");
			}
		}

		SnmpTrapV1 snmptrap = new SnmpTrapV1(receivedTime, community.toString(), addr,
				pdu.getEnterprise().toString(), pdu.getGeneric(), pdu.getSpecific(),
				pdu.getTimeStamp(), varbinds);
		if (_log.isDebugEnabled()) {
			_log.debug("start filtering : " + snmptrap);
		}
		filter.put(snmptrap);

	}

	/**
	 * 受信時に異常が発生した際のハンドラ実装
	 */
	@Override
	public void snmpTrapSessionError(SnmpTrapSession session, int error, Object ref) {
		_log.warn("An error occurred in snmptrap session. (error code = " + error + ")");
		if (ref != null) {
			_log.warn("snmptrap session error : reference = " + ref.toString());
		}

	}

	/**
	 * SNMPTRAP v2をv1にマッピングさせる。
	 * 
	 * <p>
	 * 抜粋：RFC2089 ('Mapping SNMPv2 onto SNMPv1'), section 3.3 ('Processing an outgoing SNMPv2 TRAP')
	 * </p>
	 * 
	 * <p>
	 * <strong>2b </strong>
	 * <p>
	 * If the snmpTrapOID.0 value is one of the standard traps the specific-trap
	 * field is set to zero and the generic trap field is set according to this
	 * mapping:
	 * <p>
	 * 
	 * <pre>
	 * 
	 *       value of snmpTrapOID.0                generic-trap
	 *       ===============================       ============
	 *       1.3.6.1.6.3.1.1.5.1 (coldStart)                  0
	 *       1.3.6.1.6.3.1.1.5.2 (warmStart)                  1
	 *       1.3.6.1.6.3.1.1.5.3 (linkDown)                   2
	 *       1.3.6.1.6.3.1.1.5.4 (linkUp)                     3
	 *       1.3.6.1.6.3.1.1.5.5 (authenticationFailure)      4
	 *       1.3.6.1.6.3.1.1.5.6 (egpNeighborLoss)            5
	 * 
	 * </pre>
	 * 
	 * <p>
	 * The enterprise field is set to the value of snmpTrapEnterprise.0 if this
	 * varBind is present, otherwise it is set to the value snmpTraps as defined
	 * in RFC1907 [4].
	 * </p>
	 * 
	 * <p>
	 * <strong>2c. </strong>
	 * </p>
	 * <p>
	 * If the snmpTrapOID.0 value is not one of the standard traps, then the
	 * generic-trap field is set to 6 and the specific-trap field is set to the
	 * last subid of the snmpTrapOID.0 value.
	 * </p>
	 * 
	 * <p>
	 * If the next to last subid of snmpTrapOID.0 is zero, then the enterprise
	 * field is set to snmpTrapOID.0 value and the last 2 subids are truncated
	 * from that value. If the next to last subid of snmpTrapOID.0 is not zero,
	 * then the enterprise field is set to snmpTrapOID.0 value and the last 1
	 * subid is truncated from that value.
	 * </p>
	 * 
	 * <p>
	 * In any event, the snmpTrapEnterprise.0 varBind (if present) is ignored in this case.
	 * </p>
	 * 
	 * @param agent The address of the remote sender.
	 * @param pdu The decoded V2 trap pdu.
	 * @return トラップOID情報
	 */
	private SnmpTrapV1 convertV2toV1(long receivedTime, String community, InetAddress agentAddr, SnmpPduPacket pdu) {

		if (pdu.typeId() != (byte)SnmpPduPacket.V2TRAP) {
			// pduがV2でない場合はnullを返す
			_log.info("SnmpTrap v2 PDU (command must be " + SnmpPduPacket.V2TRAP + ") : Command = " + pdu.getCommand());
			return null;
		}

		if (pdu.getLength() < 2) {
			// pduのvarbind数が2より少ない場合はnullを返す
			_log.info("SnmpTrap v2 PDU (sysUpTime.0 and snmpTrapOID.0 must be contained) : Length = " + pdu.getLength());
			return null;
		}

		// 1番目はsysuptime
		SnmpVarBind varBind0 = pdu.getVarBindAt(0);

		String varBindName0 = varBind0.getName().toString();
		if (varBindName0 != null && varBindName0.startsWith(_snmpSysuptimePrefixOid)) {
			// 一部の機器(extremeなどで末尾に0が付与されていない場合がある)への対策
			varBindName0 = _snmpSysuptimeOid;
		} else {
			_log.info("SnmpTrap v2 PDU (first varbind must be sysUpTime.0) : varBindName0 = " + varBindName0);
			return null;
		}

		long timestamp = receivedTime;
		SnmpSyntax varBindValue0 = varBind0.getValue();
		if (varBindValue0 instanceof SnmpTimeTicks) {
			timestamp = ((SnmpTimeTicks)varBindValue0).getValue();
		} else {
			_log.info("SnmpTrap v2 PDU (first varbind must be timeticks) : varBindValue0 = " + varBindValue0.toString());
			return null;
		}

		// 2番目はSNMPトラップOID
		SnmpVarBind varBind1 = pdu.getVarBindAt(1);

		String varBindName1 = varBind1.getName().toString();
		if (! _snmpTrapOid.equals(varBindName1)) {
			_log.info("SnmpTrap v2 PDU (first varbind must be snmpTrapOID.0) : varBindName1 = " + varBindName1);
			return null;
		}

		SnmpObjectId oidObj = null;
		String oid = "";
		SnmpSyntax varBindValue1 = varBind1.getValue();
		if (varBindValue1 instanceof SnmpObjectId) {
			oidObj = (SnmpObjectId)varBindValue1;
			oid = oidObj.toString();
			if (! oid.startsWith(".")) {
				oid = "." + oid;
			}
		} else {
			_log.info("SnmpTrap v2 PDU (second varbind must be object id) : varBindValue1 = " + varBindValue1.toString());
			return null;
		}

		int dotIdx = oid.lastIndexOf(".");
		int index = -1;
		try {
			index = Integer.parseInt(oid.substring(dotIdx + 1));
			if (_log.isDebugEnabled()) {
				_log.debug("SnmpTrap v2 PDU (oid) : index = " + index);
			}
		} catch (Exception e) {
			_log.info("SnmpTrap v2 PDU (oid) : snmpTrapOID.0 = " + oid);
		}

		String enterpriseId = null;
		int genericId = -1;
		int specificId = -1;
		if (_genericTraps.contains(oidObj)) {
			if (_log.isDebugEnabled()) {
				_log.debug("SnmpTrap v2 PDU (this is Generic Trap) : oid = " + oid);
			}

			// スタンダードトラップの場合
			genericId = index - 1;
			specificId = 0;

			for (int i = 0; i < pdu.getLength(); i++) {
				SnmpVarBind varBind = pdu.getVarBindAt(i);
				String name = varBind.getName().toString();

				if (_snmpTrapEnterpriseOid.equals(name)) {
					SnmpSyntax value = varBind.getValue();
					if (value instanceof SnmpObjectId) {
						if (_log.isDebugEnabled()) {
							_log.debug("SnmpTrap v2 PDU (SnmpObjectId found) : enterpriseId = " + value.toString());
						}

						// SNMPTRAP_ENTERPRISE_OIDの1番目を採用する(複数定義されている可能性もある)
						enterpriseId = value.toString();
						break;
					}
				}

				if (enterpriseId == null) {
					// varbindの値にSNMPトラップ値をセット（RFC 1907）
					enterpriseId = _snmpTrapsOid + "." + oid.charAt(oid.length() - 1);
				}
			}

			if (_log.isDebugEnabled()) {
				_log.debug("SnmpTrap v2 PDU : enterpriseId = " + enterpriseId + ", genericId = " + genericId + ", specificId = " + specificId);
			}
		} else {
			if (_log.isDebugEnabled()) {
				_log.debug("SnmpTrap v2 PDU (this is not Generic Trap) : oid = " + oid);
			}

			// スタンダードでないトラップの場合
			genericId = 6;
			specificId = index;

			int nextDotIdx = oid.lastIndexOf(".", dotIdx - 1);
			String nextIndex = oid.substring(nextDotIdx + 1, dotIdx);

			if ("0".equals(nextIndex)) {
				enterpriseId = oid.substring(0, nextDotIdx);
			} else {
				enterpriseId = oid.substring(0, dotIdx);
			}

			if (_log.isDebugEnabled()) {
				_log.debug("SnmpTrap v2 PDU : enterpriseId = " + enterpriseId + ", genericId = " + genericId + ", specificId = " + specificId);
			}
		}

		List<SnmpSyntax> varbinds = new ArrayList<SnmpSyntax>();
		// varbind(v2c)の1番目はsysuptime
		// varbind(v2c)の2番目はSNMPトラップOID
		// varbind(v2c)の3番目以降をvarbinds(v1)に入れる。
		for (int i = 2; i < pdu.getLength(); i++) {
			try {
				SnmpVarBind varbind = pdu.getVarBindAt(i);
				SnmpSyntax value = varbind.getValue();
				varbinds.add(value);

				if (_log.isDebugEnabled()) {
					_log.debug("SnmpTrap v2 PDU (varbind[" + i +  "]) : name = " + varbind.getName().toString() + ", value = " + value.toString());
				}
			} catch (Exception e) {
				_log.info("SnmpTrap v2 PDU (varbind " + i + " access failure) : Length = " + pdu.getLength()
						+ " [" + e.getMessage() + "]");
			}
		}

		return new SnmpTrapV1(receivedTime, community, agentAddr, enterpriseId, genericId, specificId, timestamp, varbinds);
	}

	/**
	 * 受信したSNMPTRAP数のカウンタ実装
	 */
	private synchronized void countup() {
		if (counter < Long.MAX_VALUE) {
			counter++;
			if (_statsInterval != 0 && counter % _statsInterval == 0) {
				_log.info("The number of received snmptrap : " + counter);
			}
		} else {
			counter = 0;
			_log.info("snmptrap counter is reseted.");
		}
	}

}
