/*

Copyright (C) since 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.repository.factory;

import java.io.IOException;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.commons.util.Ipv6Util;
import com.clustercontrol.commons.util.ObjectValidator;
import com.clustercontrol.repository.bean.FacilityConstant;
import com.clustercontrol.repository.bean.FacilityInfo;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.bean.FacilityTreeItem;
import com.clustercontrol.repository.bean.NodeConstant;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.ejb.entity.FacilityLocal;
import com.clustercontrol.repository.ejb.entity.FacilityRelationLocal;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;
import com.clustercontrol.repository.ejb.entity.NodeHostnameLocal;
import com.clustercontrol.repository.ejb.entity.NodeHostnameUtil;
import com.clustercontrol.repository.ejb.entity.NodeLocal;
import com.clustercontrol.repository.ejb.entity.NodeUtil;
import com.clustercontrol.repository.ejb.session.RepositoryControllerBean;
import com.clustercontrol.repository.util.FacilityValidator;
import com.clustercontrol.repository.util.ListSorter;
import com.clustercontrol.util.Messages;

public class FacilitySelector {

	private static Log m_log = LogFactory.getLog(FacilitySelector.class);

	public static final String SEPARATOR = ">";

	/** ----- ファシリティの木情報のキャッシュ ----- */
	/*
	 * TODO
	 * ・複数のキャッシュの統合。
	 * ・HashMapからConcurrentHashMapへの移行
	 * などを検討すること。
	 */
	private static volatile HashMap<Locale, HashMap<Boolean, FacilityTreeItem>> cachedFacilityTree = new HashMap<Locale, HashMap<Boolean, FacilityTreeItem>>();
	private static volatile HashMap<Locale, HashMap<Boolean, FacilityTreeItem>> cachedFacilityTreeScope = new HashMap<Locale, HashMap<Boolean, FacilityTreeItem>>();

	// getFacilityIdList(hostname, ipAddress)のためのキャッシュ
	private static ConcurrentHashMap<String, ArrayList<String>> nodeIpFacilityIdMap =
		new ConcurrentHashMap<String, ArrayList<String>>();
	private static final Object nodeIpFacilityIdMapLock = new Object();

	// <小文字ノード名, ファシリティIDのセット>（監視対象外となっているものは含まない。）
	private static HashMap<String, HashSet<String>> nodenameFacilityIdMap;
	private static final Object nodenameFacilityIdMapLock = new Object();

	// <IPアドレス, ファシリティIDのセット>（監視対象外となっているものは含まない。）
	private static HashMap<InetAddress, HashSet<String>> inetAddressFacilityIdMap;
	private static final Object inetAddressFacilityIdMapLock = new Object();

	// <ホスト名, ファシリティIDのセット>（監視対象外となっているものは含まない。）
	private static HashMap<String, HashSet<String>> hostnameFacilityIdMap;
	private static final Object hostnameFacilityIdMapLock = new Object();
	
	// containsFacilityIdメソッドのためのキャッシュ
	private static ConcurrentHashMap<String, Boolean> containsFacilityIdMap;
	private static final Object containsFacilityIdMapLock = new Object();

	static {
		initCacheFacilityTree();
	}

	/**
	 * ファシリティの木情報のキャッシュを初期化する
	 */
	public static void initCacheFacilityTree() {
		m_log.info("initializing a facility's tree to cache.");
		synchronized (cachedFacilityTree) {
			cachedFacilityTree = new HashMap<Locale, HashMap<Boolean, FacilityTreeItem>>();
		}
		synchronized (cachedFacilityTreeScope) {
			cachedFacilityTreeScope = new HashMap<Locale, HashMap<Boolean, FacilityTreeItem>>();
		}
		synchronized(nodeIpFacilityIdMapLock) {
			nodeIpFacilityIdMap.clear();
		}
		synchronized (nodenameFacilityIdMapLock) {
			nodenameFacilityIdMap = null;
		}
		synchronized (inetAddressFacilityIdMapLock) {
			inetAddressFacilityIdMap = null;
		}
		synchronized (hostnameFacilityIdMapLock) {
			hostnameFacilityIdMap = null;
		}
		synchronized (containsFacilityIdMapLock) {
			containsFacilityIdMap = null;
		}
	}

	/**
	 * ノード情報の配列を取得する。<BR>
	 * <PRE>
	 * {
	 *    {facilityId1, facilityName1, description1},
	 *    {facilityId2, facilityName2, description2},
	 *    ...
	 * }
	 * </PRE>
	 * 
	 * @return ノード情報の配列
	 */
	public static ArrayList<NodeInfo> getNodeList() {
		/** ローカル変数 */
		ArrayList<NodeInfo> nodeList = null;
		Collection nodes = null;

		/** メイン処理 */
		m_log.debug("getting a list of all nodes...");

		try {
			nodeList = new ArrayList<NodeInfo>();
			nodes = NodeUtil.getLocalHome().findAll();
			if (nodes != null) {
				for (NodeLocal node : (Collection<NodeLocal>)nodes) {
					nodeList.add(FacilityValidator.nodeToArrayList(node));
				}
			}
		} catch (FinderException e) {
			m_log.debug("no node is found.");
		} catch (Exception e) {
			m_log.info("failure to get a list of all nodes...");
			throw new EJBException("failure to get a list of all nodes...");
		}

		m_log.debug("successful in getting a list of all nodes.");
		return nodeList;
	}

	/**
	 * 検索条件にマッチしたノード情報の配列を取得する。<BR>
	 * <PRE>
	 * {
	 *    {facilityId1, facilityName1, description1},
	 *    {facilityId2, facilityName2, description2},
	 *    ...
	 * }
	 * </PRE>
	 * 
	 * @param property 検索条件
	 * @return ノード情報配列
	 */
	public static ArrayList<NodeInfo> getFilterNodeList(NodeInfo property) {
		/** ローカル変数 */
		ArrayList<NodeInfo> nodeList = null;
		String facilityId = null;
		String facilityName = null;
		String description = null;
		String ipAddressV4 = null;
		String ipAddressV6 = null;
		String osName = null;
		String osRelease = null;
		String administrator = null;
		String contact = null;

		boolean facilityIdFlg = false;
		boolean facilityNameFlg = false;
		boolean descriptionFlg = false;
		boolean ipAddressV4Flg = false;
		boolean ipAddressV6Flg = false;
		boolean osNameFlg = false;
		boolean osReleaseFlg = false;
		boolean administratorFlg = false;
		boolean contactFlg = false;

		Collection nodeAll = null;
		FacilityLocal facility = null;

		/** メイン処理 */
		m_log.debug("getting a list of nodes by using filter...");

		// 検索条件の入力値の取得
		facilityId = property.getFacilityId();

		facilityName = property.getFacilityName();

		description = property.getDescription();

		ipAddressV4 = property.getIpAddressV4();

		ipAddressV6 = property.getIpAddressV6();

		osName = property.getOsName();

		osRelease = property.getOsRelease();

		administrator = property.getAdministrator();

		contact = property.getContact();

		// 検索条件の有効確認
		if (! ObjectValidator.isEmptyString(facilityId)) {
			facilityIdFlg = true;
		}
		if (! ObjectValidator.isEmptyString(facilityName)) {
			facilityNameFlg = true;
		}
		if (! ObjectValidator.isEmptyString(description)) {
			descriptionFlg = true;
		}
		if (! ObjectValidator.isEmptyString(ipAddressV4)) {
			ipAddressV4Flg = true;
		}
		if (! ObjectValidator.isEmptyString(ipAddressV6)) {
			ipAddressV6Flg = true;
		}
		if (! ObjectValidator.isEmptyString(osName)) {
			osNameFlg = true;
		}
		if (! ObjectValidator.isEmptyString(osRelease)) {
			osReleaseFlg = true;
		}
		if (! ObjectValidator.isEmptyString(administrator)) {
			administratorFlg = true;
		}
		if (! ObjectValidator.isEmptyString(contact)) {
			contactFlg = true;
		}

		try {
			nodeAll = NodeUtil.getLocalHome().findAll();
		} catch (FinderException e) {
			m_log.debug("no node is found.");
		} catch (Exception e) {
			m_log.info("failure to get a list of nodes by using filter.", e);
			throw new EJBException("failure to get a list of nodes by using filter.", e);
		}

		nodeList = new ArrayList<NodeInfo>();
		if (nodeAll != null) {
			// 文字列が部分一致した場合、マッチしたノードとする
			for (NodeLocal node : (Collection<NodeLocal>)nodeAll) {
				facility = node.getFacility();

				if (facilityIdFlg && facility.getFacilityId().indexOf(facilityId) == -1) {
					continue;
				}
				if (facilityNameFlg && facility.getFacilityName().indexOf(facilityName) == -1) {
					continue;
				}
				if (descriptionFlg && facility.getDescription().indexOf(description) == -1) {
					continue;
				}
				if (ipAddressV4Flg && node.getIpAddressV4().indexOf(ipAddressV4) == -1) {
					continue;
				}
				if (ipAddressV6Flg && node.getIpAddressV6().indexOf(ipAddressV6) == -1) {
					continue;
				}
				if (osNameFlg && node.getOsName().indexOf(osName) == -1) {
					continue;
				}
				if (osReleaseFlg && node.getOsRelease().indexOf(osRelease) == -1) {
					continue;
				}
				if (administratorFlg && node.getAdministrator().indexOf(administrator) == -1) {
					continue;
				}
				if (contactFlg && node.getContact().indexOf(contact) == -1) {
					continue;
				}

				nodeList.add(FacilityValidator.nodeToArrayList(node));
			}
		}

		m_log.debug("successful in getting a list of nodes by using filter.");
		return nodeList;
	}

	/**
	 * スコープ配下にあるノード情報の一覧を取得する。<BR>
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param level 取得する階層数
	 * @return ノード情報の一覧
	 */
	public static ArrayList<NodeInfo> getNodeList(String parentFacilityId, int level) {
		/** ローカル変数 */
		ArrayList<NodeInfo> nodes = null;
		ArrayList<FacilityLocal> facilities = null;
		NodeLocal node = null;

		/** メイン処理 */
		facilities = getFacilityList(parentFacilityId, level);

		nodes = new ArrayList<NodeInfo>();
		if (facilities != null) {
			for (FacilityLocal facility : facilities) {
				// ノードの場合、配列に追加
				if (facility.isNode()) {
					node = facility.getNode();
					nodes.add(FacilityValidator.nodeToArrayList(node));
				}
			}
		}

		return nodes;
	}

	/**
	 * スコープ配下にあるファシリティIDの一覧を取得してソートする。<BR>
	 * parentFacilityIdがノードの場合は、そのノードのファシリティIDを返す。
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param level 取得する階層数
	 * @param sort ファシリティIDをソートする場合はtrue, ソートしない場合はfalse
	 * @return ファシリティIDの配列
	 */
	public static ArrayList<String> getFacilityIdList(String parentFacilityId, int level, boolean sort) {
		/** ローカル変数 */
		ArrayList<String> facilityIds = null;
		ArrayList<FacilityLocal> facilities = null;

		/** メイン処理 */
		facilities = getFacilityList(parentFacilityId, level);

		facilityIds = new ArrayList<String>();
		if (facilities != null) {
			for (FacilityLocal facility : facilities) {
				facilityIds.add(facility.getFacilityId());
			}
		}

		if (sort) {
			ListSorter sorter = new ListSorter();
			sorter.sort(facilityIds);
		}

		return facilityIds;
	}

	/**
	 * スコープ配下にあるファシリティIDの一覧を有効/無効を指定して取得してソートする。<BR>
	 * parentFacilityIdが有効ノードの場合は、ノードのファシリティIDを返す。
	 * parentFacilityIdが無効ノードの場合は、空のArrayListを返す。
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param level 取得する階層数
	 * @param sort ファシリティIDをソートする場合はtrue, ソートしない場合はfalse
	 * @param validFlg 有効/無効の指定
	 * @return ファシリティIDの配列
	 */
	public static ArrayList<String> getNodeFacilityIdList(String parentFacilityId, int level, boolean sort, Boolean validFlg) {
		/** ローカル変数 */
		ArrayList<String> facilityIdList = null;
		ArrayList<FacilityLocal> facilityList = null;

		/** メイン処理 */
		facilityList = getFacilityList(parentFacilityId, level);

		facilityIdList = new ArrayList<String>();
		if (facilityList != null) {
			for (FacilityLocal facility : facilityList) {
				if (facility.isNode()) {
					if (validFlg == null) {
						facilityIdList.add(facility.getFacilityId());
					} else if (validFlg == facility.isValid()) {
						facilityIdList.add(facility.getFacilityId());
					}
				}
			}
		}

		if (sort) {
			ListSorter sorter = new ListSorter();
			sorter.sort(facilityIdList);
		}

		return facilityIdList;
	}

	/**
	 * 全てのノードのファシリティIDの一覧を取得する。<BR>
	 * 
	 * @param sort ファシリティIDをソートする場合はtrue, ソートしない場合はfalse
	 * @return ファシリティIDの配列
	 */
	public static ArrayList<String> getNodeFacilityIdList(boolean sort) {
		/** ローカル変数 */
		ArrayList<String> facilityIdList = null;
		Collection nodes = null;

		/** メイン処理 */
		try {
			facilityIdList = new ArrayList<String>();

			nodes = NodeUtil.getLocalHome().findAll();

			if (nodes != null) {
				for (NodeLocal node : (Collection<NodeLocal>)nodes) {
					facilityIdList.add(node.getFacilityId());
				}
			}

			if (sort) {
				ListSorter sorter = new ListSorter();
				sorter.sort(facilityIdList);
			}
		} catch (FinderException e) {
			m_log.debug("no node is found.");
		} catch (Exception e) {
			m_log.info("failure to get a list of all node.", e);
			throw new EJBException("failure to get a list of all node.", e);
		}

		return facilityIdList;
	}

	/**
	 * IPアドレスから該当するノードのファシリティID一覧を取得する。
	 * @param ipaddr IPアドレス(Inet4Address or Inet6Address)
	 * @return ファシリティIDのリスト
	 * @throws HinemosUnknown
	 */
	public static List<String> getFacilityIdByIpAddress(InetAddress ipaddr) throws HinemosUnknown {
		List<String> ret = new ArrayList<String>();

		if (ipaddr == null) {
			return ret;
		}

		try {
			String ipaddrStr = ipaddr.getHostAddress();
			if (m_log.isDebugEnabled()) {
				m_log.debug("finding node by ipaddress. (ipaddr = " + ipaddr + ")");
			}

			Collection<NodeLocal> nodes = null;
			if (ipaddr instanceof Inet4Address) {
				nodes = NodeUtil.getLocalHome().findByIpAddressV4(ipaddrStr);
			}

			if (ipaddr instanceof Inet6Address) {
				nodes = NodeUtil.getLocalHome().findByIpAddressV6(Ipv6Util.expand(ipaddrStr));
			}

			if (nodes != null) {
				for (NodeLocal node : nodes) {
					ret.add(node.getFacilityId());
				}
			}
		} catch (Exception e) {
			m_log.warn("unexpected internal error. (ipaddr = " + ipaddr + ")", e);
			throw new HinemosUnknown("unexpected internal error. (ipaddr = " + ipaddr + ")", e);
		}

		return ret;
	}

	/**
	 * ホスト名とIPv4アドレスを指定して、該当するノードのファシリティIDの一覧を取得する。<BR>
	 * 
	 * @param hostname ホスト名
	 * @param ipAddressV4 IPv4アドレス
	 * @return ファシリティIDの配列
	 */
	public static ArrayList<String> getFacilityIdList(String hostname, String ipAddress) {

		m_log.debug("getFacilityIdList() start : hostname = " + hostname +
				", ipAddress = " + ipAddress);

		/** ローカル変数 */
		String key = hostname + "-," + ipAddress;
		ArrayList<String> facilityIds = nodeIpFacilityIdMap.get(key);
		if (facilityIds != null) {
			return facilityIds;
		}
		facilityIds = new ArrayList<String>();
		Collection<NodeLocal> nodes = null;

		if (ipAddress == null || "".equals(ipAddress) ||
				hostname == null || "".equals(hostname)) {
			return facilityIds;
		}
		/** メイン処理 */
		synchronized(nodeIpFacilityIdMapLock) {
			try {
				// hostname変数のNodeプロパティのnodename(必須項目)をLowerCaseで検索
				nodes = NodeUtil.getLocalHome().findByNodename(hostname.toLowerCase());
				if (nodes != null){
					for (NodeLocal node : nodes){

						m_log.debug("getFacilityIdList() List " +
								" FacilityId = " + node.getFacilityId() +
								" NodeName = " + node.getNodeName() +
								" IpAddressV4 = " + node.getIpAddressV4() +
								" IpAddressV6 = " + node.getIpAddressV6());

						// IPv6とマッチ
						if(node.getIpAddressVersion() == 6) {
							if(Ipv6Util.expand(ipAddress).equals(
									Ipv6Util.expand(node.getIpAddressV6()))){
								m_log.debug("getFacilityIdList() hit facilityId = " + node.getFacilityId());
								facilityIds.add(node.getFacilityId());
							}
						} else {
							if(ipAddress.equals(node.getIpAddressV4())){
								m_log.debug("getFacilityIdList() hit facilityId = " + node.getFacilityId());
								facilityIds.add(node.getFacilityId());
							}
						}
					}
				}
			} catch (FinderException e) {
				m_log.debug("no node is found.");
			} catch (Exception e) {
				m_log.info("failure to get a list of nodes by using hostname and IPv4 address filter.", e);
				throw new EJBException("failure to get a list of nodes by using hostname and IPv4 address filter.", e);
			}

			// Debugが有効の場合のみ取得したIPアドレスを表示させる
			if(m_log.isDebugEnabled()){
				for (Iterator iter = facilityIds.iterator(); iter.hasNext();) {
					String facilityId = (String) iter.next();
					m_log.debug("getFacilityIdList() hostname = " + hostname
							+ ", ipAddress = " + ipAddress + " has " + facilityId);
				}
			}
			nodeIpFacilityIdMap.put(key, facilityIds);
		}
		return facilityIds;
	}


	/**
	 * 検索条件にマッチするノードのファシリティIDの一覧を取得する。<BR>
	 * 
	 * @param condition 検索条件
	 * @return ファシリティIDの配列
	 */
	public static ArrayList<String> getNodeFacilityIdListByCondition(HashMap condition) {
		/** ローカル変数 */
		ArrayList<NodeLocal> nodes = null;
		ArrayList<String> facilityIds = null;

		/** メイン処理 */
		nodes = getNodeByCondition(condition);
		facilityIds = new ArrayList<String>();
		if (nodes != null) {
			for (NodeLocal node : nodes) {
				facilityIds.add(node.getFacilityId());
			}
		}

		return facilityIds;
	}

	/**
	 * 検索条件にマッチするノードのファシリティインスタンスの一覧を取得する。<BR>
	 * 
	 * @param condition 検索条件
	 * @return ファシリティインスタンスの配列
	 */
	private static ArrayList<NodeLocal> getNodeByCondition(HashMap condition) {
		m_log.debug("getNodeByCondition() : ");

		/** ローカル変数 */
		ArrayList<NodeLocal> nodes = null;
		String value = null;
		String valueTarget = null;
		Collection nodeAll = null;

		/** メイン処理 */
		m_log.debug("getting nodes by using filter...");

		// ノードの一覧を取得する
		try {
			nodeAll = NodeUtil.getLocalHome().findAll();
		} catch (FinderException e) {
			m_log.debug("no node is found.");
		} catch (Exception e) {
			m_log.info("failure to get nodes by using filter.", e);
			throw new EJBException("failure to get nodes by using filter.", e);
		}

		// 該当するノードの一覧を生成する
		nodes = new ArrayList<NodeLocal>();
		if (nodeAll != null) {
			for (NodeLocal node : (Collection<NodeLocal>)nodeAll) {
				if (condition != null) {
					boolean matchFlg = true;

					for (String attribute : (Collection<String>)condition.keySet()) {
						if (ObjectValidator.isEmptyString(attribute)) {
							continue;
						}
						value = (String)condition.get(attribute);
						if (value == null) {
							continue;
						}

						valueTarget = null;

						// ファシリティ関連
						if (attribute.compareTo(NodeConstant.FACILITY_ID) == 0) {
							valueTarget = node.getFacilityId();
						} else if(attribute.compareTo(NodeConstant.FACILITY_NAME) == 0) {
							valueTarget = node.getFacility().getFacilityName();
						} else if(attribute.compareTo(NodeConstant.DESCRIPTION) == 0) {
							valueTarget = node.getFacility().getDescription();
						} else if(attribute.compareTo(NodeConstant.DISPLAY_SORT_ORDER) == 0){
							valueTarget = node.getFacility().getDisplaySortOrder().toString();
						} else if(attribute.compareTo(NodeConstant.VALID) == 0){
							// 旧格納データベースであるLDAPの物理格納形式に変換（VMオプション対応）
							valueTarget = node.getFacility().getValid() == ValidConstant.TYPE_VALID ? new Boolean(true).toString().toUpperCase() : new Boolean(false).toString().toUpperCase();
						} else if(attribute.compareTo(NodeConstant.CREATOR_NAME) == 0) {
							valueTarget = node.getFacility().getCreateUserId();
						} else if(attribute.compareTo(NodeConstant.CREATE_TIME) == 0) {
							valueTarget = node.getFacility().getCreateDatetime().toString();
						} else if(attribute.compareTo(NodeConstant.MODIFIER_NAME) == 0) {
							valueTarget = node.getFacility().getModifyUserId();
						} else if(attribute.compareTo(NodeConstant.MODIFY_TIME) == 0) {
							valueTarget = node.getFacility().getModifyDatetime().toString();
						}

						// HW
						else if(attribute.compareTo(NodeConstant.PLATFORM_FAMILY_NAME) == 0){
							valueTarget = node.getPlatformFamily();
						} else if(attribute.compareTo(NodeConstant.HARDWARE_TYPE) == 0){
							valueTarget = node.getHardwareType();
						} else if(attribute.compareTo(NodeConstant.ICONIMAGE) == 0){
							valueTarget = node.getIconImage();
						}

						// IPアドレス関連
						else if(attribute.compareTo(NodeConstant.IP_ADDRESS_VERSION) == 0){
							valueTarget = node.getIpAddressVersion().toString();
						} else if(attribute.compareTo(NodeConstant.IP_ADDRESS_V4) == 0){
							valueTarget = node.getIpAddressV4();
						} else if(attribute.compareTo(NodeConstant.IP_ADDRESS_V6) == 0){
							valueTarget = node.getIpAddressV6();
						}

						// OS関連
						else if(attribute.compareTo(NodeConstant.NODE_NAME) == 0){
							valueTarget = node.getNodeName();
						} else if(attribute.compareTo(NodeConstant.OS_NAME) == 0){
							valueTarget = node.getOsName();
						} else if(attribute.compareTo(NodeConstant.OS_RELEASE) == 0){
							valueTarget = node.getOsRelease();
						} else if(attribute.compareTo(NodeConstant.OS_VERSION) == 0){
							valueTarget = node.getOsVersion();
						} else if(attribute.compareTo(NodeConstant.CHARACTER_SET) == 0){
							valueTarget = node.getCharacterSet();
						}

						// SNMP関連
						else if(attribute.compareTo(NodeConstant.SNMP_PORT) == 0){
							valueTarget = node.getSnmpPort().toString();
						} else if(attribute.compareTo(NodeConstant.SNMP_COMMUNITY) == 0){
							valueTarget = node.getSnmpCommunity();
						} else if(attribute.compareTo(NodeConstant.SNMP_VERSION) == 0){
							valueTarget = node.getSnmpVersion();
						} else if(attribute.compareTo(NodeConstant.SNMPTIMEOUT) == 0){
							valueTarget = node.getSnmpTimeout().toString();
						} else if(attribute.compareTo(NodeConstant.SNMPRETRIES) == 0){
							valueTarget = node.getSnmpRetryCount().toString();
						}

						// WBEM関連
						else if(attribute.compareTo(NodeConstant.WBEM_USER) == 0){
							valueTarget = node.getWbemUser();
						} else if(attribute.compareTo(NodeConstant.WBEM_USER_PASSWORD) == 0){
							valueTarget = node.getWbemUserPassword();
						} else if(attribute.compareTo(NodeConstant.WBEM_PORT) == 0){
							valueTarget = node.getWbemPort().toString();
						} else if(attribute.compareTo(NodeConstant.WBEM_PROTOCOL) == 0){
							valueTarget = node.getWbemProtocol();
						} else if(attribute.compareTo(NodeConstant.WBEM_TIMEOUT) == 0){
							valueTarget = node.getWbemTimeout().toString();
						} else if(attribute.compareTo(NodeConstant.WBEM_RETRIES) == 0){
							valueTarget = node.getWbemRetryCount().toString();
						}

						// IPMI関連
						else if(attribute.compareTo(NodeConstant.IPMI_IP_ADDRESS) == 0){
							valueTarget = node.getIpmiIpAddress();
						} else if(attribute.compareTo(NodeConstant.IPMI_PORT) == 0){
							valueTarget = node.getIpmiPort().toString();
						} else if(attribute.compareTo(NodeConstant.IPMI_USER) == 0){
							valueTarget = node.getIpmiUser();
						} else if(attribute.compareTo(NodeConstant.IPMI_USER_PASSWORD) == 0){
							valueTarget = node.getIpmiUserPassword();
						} else if(attribute.compareTo(NodeConstant.IPMI_TIMEOUT) == 0){
							valueTarget = node.getIpmiTimeout().toString();
						} else if(attribute.compareTo(NodeConstant.IPMI_RETRIES) == 0){
							valueTarget = node.getIpmiRetryCount().toString();
						} else if(attribute.compareTo(NodeConstant.IPMI_PROTOCOL) == 0){
							valueTarget = node.getIpmiProtocol();
						} else if(attribute.compareTo(NodeConstant.IPMI_LEVEL) == 0){
							valueTarget = node.getIpmiLevel();
						}

						// デバイス関連
						// 単一項目ではないため対象外とする

						// サーバ仮想化
						else if(attribute.compareTo(NodeConstant.VIRTNODETYPE) == 0){
							valueTarget = node.getVirtualizationNodeType();
						} else if(attribute.compareTo(NodeConstant.VMMANAGEMENTNODE) == 0){
							valueTarget = node.getVmManagementNode();
						} else if(attribute.compareTo(NodeConstant.VMINDEX) == 0){
							valueTarget = node.getVmIndex().toString();
						} else if(attribute.compareTo(NodeConstant.VMNAME) == 0){
							valueTarget = node.getVmName();
						} else if(attribute.compareTo(NodeConstant.VIRTSOLUTION) == 0){
							valueTarget = node.getVirtualizationSolution();
						} else if(attribute.compareTo(NodeConstant.VMID) == 0){
							valueTarget = node.getVmId();
						} else if(attribute.compareTo(NodeConstant.VMUSER) == 0){
							valueTarget = node.getVmUser();
						} else if(attribute.compareTo(NodeConstant.VMUSERPASSWORD) == 0){
							valueTarget = node.getVmUserPassword();
						} else if(attribute.compareTo(NodeConstant.VMPROTOCOL) == 0){
							valueTarget = node.getVmProtocol();
						}

						// ネットワーク仮想化
						else if(attribute.compareTo(NodeConstant.VNETSWITCHTYPE) == 0){
							valueTarget = node.getVNetSwitchType();
						} else if(attribute.compareTo(NodeConstant.VNETHOSTNODE) == 0){
							valueTarget = node.getVNetHostNode();
						} else if(attribute.compareTo(NodeConstant.OPEN_FLOW_DATAPATH_ID) == 0){
							valueTarget = node.getOpenFlowDatapathId().toString();
						} else if(attribute.compareTo(NodeConstant.OPEN_FLOW_CTRL_IP_ADDRESS) == 0){
							valueTarget = node.getOpenFlowCtrlIpAddress();
						}

						// 管理関連
						// 単一項目ではないため対象外とする

						// 管理関連
						else if(attribute.compareTo(NodeConstant.ADMINISTRATOR) == 0){
							valueTarget = node.getAdministrator();
						} else if(attribute.compareTo(NodeConstant.CONTACT) == 0){
							valueTarget = node.getContact();
						}

						else {
							m_log.info("a filter's attribute is invalid. (attribute = " + attribute + ")");
							throw new EJBException("a filter's attribute is invalid. (attribute = " + attribute + ")");
						}

						// 文字列としての値の比較
						if (valueTarget == null || value.compareTo(valueTarget) != 0) {
							m_log.debug("a node's attribute is not equal. (attribute = " + attribute + ", value = " + valueTarget + ", compareValue = " + value + ")");
							matchFlg = false;
							break;
						}
					}

					if (matchFlg) {
						m_log.debug("a node is matched. (facilityId = " + node.getFacilityId() + ")");
						nodes.add(node);
					}
				}
			}
		}

		m_log.debug("successful in getting nodes by using filter.");
		return nodes;
	}

	/**
	 * 特定のスコープ配下の直下に属するファシリティの一覧を取得する。<BR>
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @return ファシリティインスタンスの配列
	 */
	public static ArrayList<FacilityInfo> getFacilityListAssignedScope(String parentFacilityId) {
		/** ローカル変数 */
		ArrayList<FacilityInfo> facilityList = null;
		FacilityLocal scope = null;

		/** メイン処理 */
		m_log.debug("getting a list of facilities under a scope...");

		try {
			facilityList = new ArrayList<FacilityInfo>();

			if (ObjectValidator.isEmptyString(parentFacilityId)) {
				// コンポジットアイテムが選択された場合
				for (FacilityLocal rootScope : getRootScopeList()) {
					try {
						facilityList.add(FacilityValidator.facilityToArrayList(rootScope));
					} catch (Exception e) {
						m_log.warn("facilityToArrayList : " + e.getMessage());
					}
				}
			} else {
				// スコープが選択された場合
				scope = FacilityUtil.getLocalHome().findByPrimaryKey(parentFacilityId);

				if (scope.getFacilityRelationAsParent() != null) {
					for (FacilityRelationLocal relation : (Collection<FacilityRelationLocal>)scope.getFacilityRelationAsParent()) {
						// スコープ配下に属する全てのファシリティを追加
						facilityList.add(FacilityValidator.facilityToArrayList(relation.getChildFacility()));
					}
				}
			}
		} catch (FinderException e) {
			m_log.debug("no facility is found.");
		} catch (Exception e) {
			m_log.info("failure to get a list of facilities under a scope. (parentFacilityId = " + parentFacilityId + ")");
			throw new EJBException("failure to get a list of facilities under a scope. (parentFacilityId = " + parentFacilityId + ")");
		}

		m_log.debug("successful in getting a list of facilities under a scope...");
		return facilityList;
	}

	/**
	 * ノードが属するスコープのパス名の一覧を取得する。<BR>
	 * 
	 * @param facilityId ノードのファシリティID
	 * @return スコープのパス名の配列
	 */
	public static ArrayList<String> getNodeScopeList(String facilityId	) {
		/** ローカル変数 */
		ArrayList<String> scopePathList = null;
		NodeLocal node = null;

		/** メイン処理 */
		m_log.debug("getting scope paths of a node...");

		try {
			node = NodeUtil.getLocalHome().findByPrimaryKey(facilityId);
		} catch (Exception e) {
			m_log.info("failure to get scope paths of a node. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to get scope paths of a node. (facilityId = " + facilityId + ")", e);
		}

		scopePathList = new ArrayList<String>();
		if (node.getFacility().getFacilityRelationAsChild() != null) {
			for (FacilityRelationLocal relation : (Collection<FacilityRelationLocal>)node.getFacility().getFacilityRelationAsChild()) {
				scopePathList.add(getNodeScopePathRecursive(relation.getParentFacility(), null));
			}
		}

		m_log.debug("successful in getting scope paths of a node.");
		return scopePathList;
	}

	/**
	 * ファシリティのパス名を取得する。<BR>
	 * 
	 * @param parentFacilityId 起点となるファシリティのファシリティID
	 * @param facilityId パスを取得するファシリティID
	 * @return ファシリティのパス名
	 */
	public static String getNodeScopePath(String parentFacilityId, String facilityId) {
		/** ローカル変数 */
		FacilityLocal facility = null;
		FacilityLocal parentFacility = null;
		String path = null;

		/** メイン処理 */
		m_log.debug("getting a scope path of a facility's relation...");

		try {
			path = "";

			if (! ObjectValidator.isEmptyString(facilityId)) {
				try {
					facility = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
				} catch (FinderException e) {
					// すでにファシリティが削除されていた場合はから文字を返す
					m_log.debug("this facility is already deleted. (facilityId = " + facilityId + ")");
				}

				if (facility != null) {
					if (! ObjectValidator.isEmptyString(parentFacilityId)) {
						parentFacility = FacilityUtil.getLocalHome().findByPrimaryKey(parentFacilityId);
					}
					path = getNodeScopePathRecursive(facility, parentFacility);
					if (ObjectValidator.isEmptyString(path)) {
						path = SEPARATOR;
					}
				}
			}
		} catch (Exception e) {
			m_log.info("failure to get a scope path of a facility's relation. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to get a scope path of a facility's relation. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")", e);
		}

		m_log.debug("successful in getting a scope path of a facility's relation.");
		return path;
	}

	/**
	 * ファシリティ関連インスタンスに属するファシリティの相対パス名を取得する。<BR>
	 * スコープの場合は「facilityName1>facilityName2>facilityName3」、ノードの場合は「facilityName」となる。<BR>
	 * 
	 * @param facility パスを取得するファシリティインスタンス
	 * @param parentFacility 相対パスを取得する場合は起点となるファシリティインスタンス, 絶対パスを取得する場合はnull
	 * @return ファシリティのパス名
	 */
	private static String getNodeScopePathRecursive(FacilityLocal facility, FacilityLocal parentFacility) {
		/** ローカル変数 */
		String path = null;
		String facilityId = facility.getFacilityId();
		String parentFacilityId = null;
		if (parentFacility != null) {
			parentFacilityId = parentFacility.getFacilityId();
		}
		
		/** メイン処理 */
		if (facility.isNode()) {
			// ノードの場合、ファシリティ名を返す
			return facility.getFacilityName();
		}

		if (parentFacility != null && facilityId.equals(parentFacilityId)) {
			// 起点となるファシリティと一致していた場合、区切り文字だけ返す
			return "";
		}

		Collection<FacilityRelationLocal> list = (Collection<FacilityRelationLocal>)facility.getFacilityRelationAsChild();
		if (list == null || list.size() == 0) {
			// ルートスコープの場合
			if (parentFacility == null) {
				path = facility.getFacilityName() + SEPARATOR;
			}
		} else {
			// 再帰的にスコープのパス名を生成する
			for (FacilityRelationLocal relation : list) {
				if (parentFacility == null) {
					// 絶対パス名を取得する場合
					path = getNodeScopePathRecursive(relation.getParentFacility(), parentFacility) + facility.getFacilityName() + SEPARATOR;
				} else if (!facilityId.equals(parentFacilityId)) {
					// 相対パス名を取得する場合
					path = getNodeScopePathRecursive(relation.getParentFacility(), parentFacility) + facility.getFacilityName() + SEPARATOR;
				}
				// スコープ-スコープ間の関連は単一であるため、ループを抜ける
				break;
			}
		}

		return path;
	}

	/**
	 * 指定したFacilityId配下のTreeItemを取得します
	 * 
	 * @param facilityId
	 * @param locale
	 * @param scopeOnly
	 * @param validFlg
	 * @return
	 */
	public static FacilityTreeItem getFacilityTree(String facilityId, Locale locale, boolean scopeOnly, Boolean validFlg) {
		FacilityTreeItem top = null;
		FacilityTreeItem originalFacilityTree = getFacilityTree(locale, scopeOnly, validFlg);

		// ツリーのコピーを作成する
		FacilityTreeItem facilityTree = null;
		try {
			facilityTree = originalFacilityTree.deepCopy();
		} catch (ClassNotFoundException e) {
			m_log.error(e.getMessage(), e);
		} catch (IOException e) {
			m_log.error(e.getMessage(), e);
		}

		// 指定のファシリティID以下のツリーを取得する
		FacilityTreeItem subFacilityTree = selectFacilityTreeItem(facilityTree, facilityId);

		if(subFacilityTree == null){
			return null;
		}

		//FacilityTreeの最上位インスタンスを作成
		FacilityInfo info = new FacilityInfo();
		info.setFacilityName(FacilityConstant.STRING_COMPOSITE);
		info.setFacilityType(FacilityConstant.TYPE_COMPOSITE);
		top = new FacilityTreeItem(null, info);

		// 取得したファシリティツリーをコンポジットアイテムに繋ぐ
		subFacilityTree.setParent(top);
		top.addChildren(subFacilityTree);

		return top;
	}


	/**
	 * ファシリティツリーの中で指定のファシリティIDを持つファシリティを再帰的に探します
	 * 
	 * @param facilityTree 対象のファシリティツリー
	 * @param facilityId パスを取得したいファシリティのファシリティID
	 * @return ファシリティ情報
	 */
	private static FacilityTreeItem selectFacilityTreeItem(FacilityTreeItem facilityTree, String facilityId){
		if(facilityTree.getData().getFacilityId().equals(facilityId)){
			return facilityTree;
		} else {
			for(int i=0; i<facilityTree.getChildrenArray().length; i++){
				FacilityTreeItem target = facilityTree.getChildrenArray()[i];
				FacilityTreeItem temp = selectFacilityTreeItem(target, facilityId);  // 再帰的
				if(temp != null){
					return temp;
				}
			}
		}
		return null;
	}

	/**
	 * ファシリティの木構造を取得する。<BR>
	 * 
	 * @param locale ロケール情報
	 * @param scopeOnly スコープのみの場合はtrue, 全てのファシリティの場合はfalse
	 * @param validFlg 有効/無効（nullの場合は全てのノード）
	 * @return ファシリティの木構造
	 */
	public static FacilityTreeItem getFacilityTree(Locale locale, boolean scopeOnly, Boolean validFlg) {
		/** ローカル変数 */
		FacilityTreeItem rootTree = null;
		FacilityInfo rootInfo = null;
		FacilityInfo scopeInfo = null;
		FacilityTreeItem scopeTree = null;

		/** メイン処理 */
		m_log.debug("getting tree data of facilities...");

		// キャッシュに情報が存在すれば、それを返す
		rootTree = getFacilityTreeOnCache(locale, scopeOnly, validFlg);
		if (rootTree != null) {
			return rootTree;
		}
		long startTime = System.currentTimeMillis();

		// 木構造最上位インスタンスの生成
		rootInfo = new FacilityInfo();
		rootInfo.setFacilityName(FacilityConstant.STRING_COMPOSITE);
		rootInfo.setFacilityType(FacilityConstant.TYPE_COMPOSITE);
		rootTree = new FacilityTreeItem(null, rootInfo);

		// コンポジットアイテムの生成
		scopeInfo = new FacilityInfo();
		scopeInfo.setFacilityName(Messages.getString("scope", locale));
		scopeInfo.setFacilityType(FacilityConstant.TYPE_COMPOSITE);
		scopeTree = new FacilityTreeItem(rootTree, scopeInfo);

		try {
			for (FacilityLocal facility : getRootScopeList()) {
				getFacilityTreeRecursive(scopeTree, facility, scopeOnly, validFlg);
			}
		} catch (Exception e) {
			m_log.info("failure to get a tree data of facilities.", e);
		}

		// 初期値をキャッシュに登録（次回から高速化に初期値を取得）
		m_log.info("adding a facility's tree to cache. (locale = " + locale + ", scopeOnly = " + scopeOnly + ", validFlg = " + validFlg + ")");
		if (scopeOnly) {
			synchronized (cachedFacilityTreeScope) {
				HashMap<Boolean, FacilityTreeItem> map = cachedFacilityTreeScope.get(locale);
				if (map != null) {
					map.put(validFlg, rootTree);
				}
			}
		} else {
			synchronized (cachedFacilityTree) {
				HashMap<Boolean, FacilityTreeItem> map = cachedFacilityTree.get(locale);
				if (map != null) {
					map.put(validFlg, rootTree);
				}
			}
		}

		long endTime = System.currentTimeMillis();
		m_log.info("refresh FacilityTree(Cache). " + (endTime - startTime) + "ms.");
		m_log.debug("successful in getting tree data of facilities.");
		return rootTree;
	}

	/**
	 * キャッシュ上に存在するファシリティの木構造を取得する。<BR>
	 * キャッシュ上に存在しなければ、nullを返す。
	 * 
	 * @param locale ロケール情報
	 * @param scopeOnly スコープのみの場合はtrue, 全てのファシリティの場合はfalse
	 * @param validFlg 有効/無効（nullの場合は全てのノード）
	 * @return ファシリティの木構造
	 */
	private static FacilityTreeItem getFacilityTreeOnCache(Locale locale, boolean scopeOnly, Boolean validFlg) {
		// キャッシュに情報が存在すれば、それを返す
		if (scopeOnly) {
			synchronized (cachedFacilityTreeScope) {
				if (cachedFacilityTreeScope.containsKey(locale)) {
					if (cachedFacilityTreeScope.get(locale).containsKey(validFlg)) {
						m_log.debug("using a facility's tree in cache. (locale = " + locale + ", scopeOnly = " + scopeOnly + ", validFlg = " + validFlg + ")");
						return cachedFacilityTreeScope.get(locale).get(validFlg);
					}
				} else {
					m_log.info("adding a facility's tree to cache. (locale = " + locale + ")");
					cachedFacilityTreeScope.put(locale, new HashMap<Boolean, FacilityTreeItem>());
				}
			}
		} else {
			synchronized (cachedFacilityTree) {
				if (cachedFacilityTree.containsKey(locale)) {
					if (cachedFacilityTree.get(locale).containsKey(validFlg)) {
						m_log.debug("using a facility's tree in cache. (locale = " + locale + ", scopeOnly = " + scopeOnly + ", validFlg = " + validFlg + ")");
						return cachedFacilityTree.get(locale).get(validFlg);
					}
				} else {
					m_log.info("adding a facility's tree to cache. (locale = " + locale + ")");
					cachedFacilityTree.put(locale, new HashMap<Boolean, FacilityTreeItem>());
				}
			}
		}

		return null;
	}

	/**
	 * ファシリティの木構造を再帰的に取得する。<BR>
	 * 
	 * @param parentTree 親となるファシリティの木構造
	 * @param facility 格納するファシリティインスタンス
	 * @param scopeOnly スコープのみの場合はtrue, 全てのファシリティの場合はfalse
	 * @param validFlg 有効/無効（nullの場合は全てのノード）
	 */
	private static void getFacilityTreeRecursive(FacilityTreeItem parentTree, FacilityLocal facility, boolean scopeOnly, Boolean validFlg) {
		/** ローカル変数 */
		FacilityInfo facilityInfo = null;
		FacilityTreeItem childTree = null;

		/** メイン処理 */
		// 木構造への格納必要性の確認
		if (facility.isNode()) {
			// スコープのみの場合、ノードは格納しない
			if (scopeOnly) {
				return;
			}
			// 有効/無効の確認（nullならば必ず格納）
			if (validFlg != null && facility.isValid() != validFlg) {
				return;
			}
		}

		// ファシリティの格納
		facilityInfo = new FacilityInfo();
		facilityInfo.setFacilityId(facility.getFacilityId());
		facilityInfo.setFacilityName(facility.getFacilityName());
		facilityInfo.setFacilityType(facility.getFacilityType());
		facilityInfo.setDisplaySortOrder(facility.getDisplaySortOrder());
		facilityInfo.setBuiltInFlg(isBuildinScope(facility));
		facilityInfo.setValid(facility.isValid());

		childTree = new FacilityTreeItem(parentTree, facilityInfo);

		// 再帰的にファシリティを格納する
		if (facility.isScope() && facility.getFacilityRelationAsParent() != null) {
			for (FacilityRelationLocal facilityRelation : (Collection<FacilityRelationLocal>)facility.getFacilityRelationAsParent()) {
				getFacilityTreeRecursive(childTree, facilityRelation.getChildFacility(), scopeOnly, validFlg);
			}
		}
	}

	/**
	 * ルートとなるスコープの一覧を取得する。<BR>
	 * 
	 * @return ファシリティインスタンスの配列
	 * @throws FinderException
	 * @throws NamingException
	 */
	private static ArrayList<FacilityLocal> getRootScopeList() throws FinderException, NamingException {
		/** ローカル変数 */
		ArrayList<FacilityLocal> scopeList = null;
		Collection facilityAll = null;

		/** メイン処理 */
		facilityAll = FacilityUtil.getLocalHome().findAll();

		scopeList = new ArrayList<FacilityLocal>();
		if (facilityAll != null) {
			for (FacilityLocal facility : (Collection<FacilityLocal>)facilityAll) {
				if (facility.isScope()) {
					if (facility.getFacilityRelationAsChild() == null || facility.getFacilityRelationAsChild().size() == 0) {
						scopeList.add(facility);
					}
				}
			}
		}

		return scopeList;
	}

	/**
	 * スコープ配下にあるファシリティの一覧を取得する。<BR>
	 * 引数がノードの場合は、そのノードのファシリティIDを返す。
	 * 引数が登録ノード全ての場合は空リストを返す。
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param level 取得する階層数
	 * @return ファシリティの配列
	 */
	private static ArrayList<FacilityLocal> getFacilityList(String parentFacilityId, int level) {
		/** ローカル変数 */
		ArrayList<FacilityLocal> facilityList = null;
		FacilityLocal parentFacility = null;

		/** メイン処理 */
		m_log.debug("getting facilities under a scope. (scopeFacilityId = " + parentFacilityId + ")");

		try {
			if (ObjectValidator.isEmptyString(parentFacilityId)) {
				if (level == RepositoryControllerBean.ONE_LEVEL) {
					facilityList = getRootScopeList();
				} else {
					facilityList = new ArrayList<FacilityLocal>();
					for (FacilityLocal rootScope : getRootScopeList()) {
						getFacilityListRecursive(rootScope, level > 1 ? level - 1 : level, facilityList);
					}
				}
			} else {
				facilityList = new ArrayList<FacilityLocal>();
				parentFacility = FacilityUtil.getLocalHome().findByPrimaryKey(parentFacilityId);
				if (parentFacility.isNode()) {
					facilityList = new ArrayList<FacilityLocal> ();
					facilityList.add(parentFacility);
					return facilityList;
				}
				getFacilityListRecursive(parentFacility, level, facilityList);
			}
		} catch (Exception e) {
			m_log.info("failure to get facilities under a scope (scopeFacilityId = " + parentFacilityId + ")", e);
			throw new EJBException("failure to get facilities under a scope (scopeFacilityId = " + parentFacilityId + ")", e);
		}

		m_log.debug("successful in getting facilities under a scope. (scopeFacilityId = " + parentFacilityId + ")");
		return facilityList;
	}

	/**
	 * スコープ配下にあるファシリティの一覧を取得する。<BR>
	 * 
	 * @param parentFacility スコープのファシリティインスタンス
	 * @param level 取得する階層数
	 * @param facilityList 格納先となるファシリティの配列
	 * @throws FinderException
	 * @throws NamingException
	 */
	private static void getFacilityListRecursive(FacilityLocal parentFacility, int level, ArrayList<FacilityLocal> facilityList) throws FinderException, NamingException {
		/** ローカル変数 */
		FacilityLocal childFacility = null;
		boolean recursive = false;
		int nextLevel = 0;

		/** メイン処理 */
		// 階層数による再帰的処理の必要性の確認
		if (level == RepositoryControllerBean.ALL) {
			recursive = true;
			nextLevel = RepositoryControllerBean.ALL;
		} else if (level > 1) {
			recursive = true;
			nextLevel = level - 1;
		}

		// 再帰的にファシリティを配列に追加する
		if (parentFacility.getFacilityRelationAsParent() != null) {
			for (FacilityRelationLocal relation : (Collection<FacilityRelationLocal>)parentFacility.getFacilityRelationAsParent()) {
				childFacility = relation.getChildFacility();

				if (recursive && childFacility.isScope()) {
					getFacilityListRecursive(childFacility, nextLevel, facilityList);
				}
				if (! facilityList.contains(childFacility)) {
					facilityList.add(childFacility);
				}
			}
		}
	}

	/**
	 * ファシリティインスタンスが組み込みスコープであるかどうかを確認する。<BR>
	 * 
	 * @param facility ファシリティインスタンス
	 * @return 組み込みスコープの場合はtrue, それ以外はfalse
	 */
	private static boolean isBuildinScope(FacilityLocal facility) {
		/** メイン処理 */
		if (facility == null || ! facility.isScope()) {
			return false;
		}

		if (facility.getFacilityId().compareTo(FacilityTreeAttributeConstant.INTERNAL_SCOPE) == 0) {
			return true;
		} else if (facility.getFacilityId().compareTo(FacilityTreeAttributeConstant.REGISTEREFD_SCOPE) == 0) {
			return true;
		} else if (facility.getFacilityId().compareTo(FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE) == 0) {
			return true;
		} else {
			return false;
		}
	}

	/**
	 * ファシリティがノードかどうかを確認する。<BR>
	 * 
	 * @param facilityId ファシリティID
	 * @return ノードの場合はtrue, それ以外の場合はfalse
	 * @throws FacilityNotFound
	 */
	public static boolean isNode(String facilityId) throws FacilityNotFound {
		/** ローカル変数 */
		FacilityLocal facility = null;

		/** メイン処理 */
		m_log.debug("checking whether a facility is node...");
		try {
			facility = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
		} catch (FinderException e) {
			m_log.debug("facility not found. (facilityId = " + facilityId + ")");
			throw new FacilityNotFound("facility not found. (facilityId = " + facilityId + ")", e);
		} catch (Exception e) {
			m_log.debug("failure to check whether a facility is node. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to check whether a facility is node. (facilityId = " + facilityId + ")", e);
		}

		m_log.debug("successful in checking whether a facility is node or not.");
		return facility.isNode();
	}

	/**
	 * 指定のノード名で登録されているノードのファシリティIDを返す。
	 * 管理対象フラグ「無効」となっているものは含まない。
	 * よって、指定のノード名で登録されているノードがリポジトリに存在しているが、
	 * 全て管理対象フラグ「無効」の場合は、空のセットが返る。
	 * 
	 * @param nodename ノード名
	 * @return ファシリティIDのセット。存在しない場合は"UNREGISTEREFD"だけを含めたセット。
	 */
	public static Set<String> getNodeListByNodename(String nodename){
		synchronized (nodenameFacilityIdMapLock) {
			// キャッシュに乗っていない場合は全ファシリティをチェックする。結果はキャッシュに乗せる。
			if(nodenameFacilityIdMap == null){
				long startTime = System.currentTimeMillis();
				try {
					Collection<NodeLocal> allNodes = NodeUtil.getLocalHome().findAll();

					nodenameFacilityIdMap = new HashMap<String, HashSet<String>>();

					for(NodeLocal node : allNodes){
						String facilityId = node.getFacilityId();

						// ノード名からファシリティIDのセットを引けるようにする
						// ノード名は小文字に変換して処理する
						String checkNodename = node.getNodeName().toLowerCase();
						HashSet<String> nodenameFacilityIdSet = nodenameFacilityIdMap.get(checkNodename);
						if(nodenameFacilityIdSet == null){
							nodenameFacilityIdSet = new HashSet<String>();
							nodenameFacilityIdMap.put(checkNodename, nodenameFacilityIdSet);
						}

						// 管理対象フラグ「有効」のノードのみセットに追加する。管理対象フラグ「無効」のノードはセットに追加しない。
						// よって、最終的にノード名でnodenameFacilityIdSetから取得したファシリティのセットは存在するが、
						// その中にエンティティが含まれない場合は、該当ノード名のノードは存在するが、全て無効だったことなる。
						if(FacilityUtil.getLocalHome().findByPrimaryKey(facilityId).getValid() == ValidConstant.TYPE_VALID){
							nodenameFacilityIdSet.add(facilityId);
						}
					}
				} catch (FinderException e) {
					m_log.error("initCacheFacilityTree()", e);
				} catch (NamingException e) {
					m_log.error("initCacheFacilityTree()", e);
				}
				long endTime = System.currentTimeMillis();
				m_log.info("refresh nodenameFacilityIdMap(Cache). " + (endTime - startTime) + "ms.");
			}

			return nodenameFacilityIdMap.get(nodename.toLowerCase());
		}
	}

	/**
	 * 指定のIPアドレスで登録されているノードのファシリティIDのセットを返す。
	 * 管理対象フラグ「無効」となっているノードのファシリティIDは含まない。
	 * よって、指定のIPアドレスで登録されているノードがリポジトリに存在しているが、
	 * 全て管理対象フラグ「無効」の場合は、空のセットが返る。
	 * 
	 * @param ipAddress IPアドレス
	 * @return ファシリティIDのセット。存在しない場合はnullを返す。
	 * 	 */
	public static Set<String> getNodeListByIpAddress(InetAddress ipAddress){
		synchronized (inetAddressFacilityIdMapLock) {
			// キャッシュに乗っていない場合は全ファシリティをチェックする。結果はキャッシュに乗せる。
			if(inetAddressFacilityIdMap == null){
				long startTime = System.currentTimeMillis();
				try {
					Collection<NodeLocal> allNodes = NodeUtil.getLocalHome().findAll();

					inetAddressFacilityIdMap = new HashMap<InetAddress, HashSet<String>>();

					for(NodeLocal node : allNodes){
						String facilityId = node.getFacilityId();

						// 「IPアドレスのバージョン」により指定されたIPアドレスを設定する。
						Integer ipVersion = node.getIpAddressVersion();
						String ipAddressString = null;
						if(ipVersion != null && ipVersion.intValue() == 6){
							ipAddressString = node.getIpAddressV6();
						} else {
							ipAddressString = node.getIpAddressV4();
						}
						InetAddress checkIpAddress = InetAddress.getByName(ipAddressString);

						// 監視対象のノードは、IPアドレスからファシリティIDのセットを引けるようにする
						HashSet<String> inetAddressFacilityIdSet = inetAddressFacilityIdMap.get(checkIpAddress);
						if(inetAddressFacilityIdSet == null){
							inetAddressFacilityIdSet = new HashSet<String>();
							inetAddressFacilityIdMap.put(checkIpAddress, inetAddressFacilityIdSet);
						}

						// 管理対象フラグ「有効」のノードのみセットに追加する。管理対象フラグ「無効」のノードはセットに追加しない。
						// よって、最終的にIPアドレスでinetAddressFacilityIdMapから取得したファシリティのセットは存在するが、
						// その中にエンティティが含まれない場合は、該当IPアドレスのノードは存在するが、全て無効だったことなる。
						if(FacilityUtil.getLocalHome().findByPrimaryKey(facilityId).getValid() == ValidConstant.TYPE_VALID){
							inetAddressFacilityIdSet.add(facilityId);
						}
					}
					long endTime = System.currentTimeMillis();
					m_log.info("refresh inetAddressFacilityIdMap(Cache). " + (endTime - startTime) + "ms.");
				} catch (FinderException e) {
					m_log.error("initCacheFacilityTree()", e);
				} catch (NamingException e) {
					m_log.error("initCacheFacilityTree()", e);
				} catch (UnknownHostException e) {
					m_log.error("initCacheFacilityTree()", e);
				}
			}

			return inetAddressFacilityIdMap.get(ipAddress);
		}
	}

	/**
	 * 指定のホスト名（複数登録可能）で登録されているノードのファシリティIDを返す。
	 * 管理対象フラグ「無効」となっているものは含まない。
	 * よって、指定のノード名で登録されているノードがリポジトリに存在しているが、
	 * 全て管理対象フラグ「無効」の場合は、空のセットが返る。
	 * 
	 * @param hostname ホスト名
	 * @return ファシリティIDのセット。存在しない場合は"UNREGISTEREFD"だけを含めたセット。
	 */
	public static Set<String> getNodeListByHostname(String hostname){
		synchronized (hostnameFacilityIdMapLock) {
			// キャッシュに乗っていない場合は全ファシリティをチェックする。結果はキャッシュに乗せる。
			if(hostnameFacilityIdMap == null){
				long startTime = System.currentTimeMillis();
				try {
					Collection<NodeHostnameLocal> allNodes = NodeHostnameUtil.getLocalHome().findAll();

					hostnameFacilityIdMap = new HashMap<String, HashSet<String>>();

					for(NodeHostnameLocal node : allNodes){
						String facilityId = node.getFacilityId();

						// ホスト名は大文字小文字を区別する
						String checkHostname = node.getHostname();
						HashSet<String> hostnameFacilityIdSet = hostnameFacilityIdMap.get(checkHostname);
						if(hostnameFacilityIdSet == null){
							hostnameFacilityIdSet = new HashSet<String>();
							hostnameFacilityIdMap.put(checkHostname, hostnameFacilityIdSet);
						}

						// 管理対象フラグ「有効」のノードのみセットに追加する。管理対象フラグ「無効」のノードはセットに追加しない。
						// よって、最終的にホスト名でhostnameFacilityIdSetから取得したファシリティのセットは存在するが、
						// その中にエンティティが含まれない場合は、該当ホスト名のノードは存在するが、全て無効だったことなる。
						if(FacilityUtil.getLocalHome().findByPrimaryKey(facilityId).getValid() == ValidConstant.TYPE_VALID){
							hostnameFacilityIdSet.add(facilityId);
						}
					}
				} catch (FinderException e) {
					m_log.error("initCacheFacilityTree()", e);
				} catch (NamingException e) {
					m_log.error("initCacheFacilityTree()", e);
				}
				long endTime = System.currentTimeMillis();
				m_log.info("refresh hostnameFacilityIdMap(Cache). " + (endTime - startTime) + "ms.");
			}

			return hostnameFacilityIdMap.get(hostname);
		}
	}

	/**
	 * 指定のノードが、指定スコープ配下の有効なノードに含まれるかチェックする
	 * @param scopeFacilityId 確認対象のスコープ（このスコープ配下をチェックする）
	 * @param nodeFacilityId 確認対象ノードのファシリティID
	 * @return 含まれる場合はtrueを返す
	 */
	public static boolean containsFaciliyId(String scopeFacilityId, String nodeFacilityId){
		Boolean ret = null; 
		String key = scopeFacilityId + "," + nodeFacilityId;
		synchronized (containsFacilityIdMapLock) {
			if (containsFacilityIdMap == null) {
				containsFacilityIdMap = new ConcurrentHashMap<String, Boolean>();
			}
			ret = containsFacilityIdMap.get(key);
			if (ret == null) {
				ret = getNodeFacilityIdList(scopeFacilityId, RepositoryControllerBean.ALL, false, true).contains(nodeFacilityId);
				m_log.info("containsFacilityId key=" + key + ", ret=" + ret);
				containsFacilityIdMap.put(key, ret);
			}
		}
		return ret;
	}
}
