/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.util;

import java.io.IOException;
import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.SocketException;
import java.net.UnknownHostException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.HinemosProperties;
import com.clustercontrol.notify.bean.ExecFacilityConstant;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.bean.SyslogFacilityConstant;
import com.clustercontrol.notify.bean.SyslogSeverityConstant;
import com.clustercontrol.notify.ejb.entity.NotifyLogEscalateInfoData;
import com.clustercontrol.notify.ejb.entity.NotifyLogEscalateInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyLogEscalateInfoPK;
import com.clustercontrol.notify.ejb.entity.NotifyLogEscalateInfoUtil;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.ejb.entity.FacilityLocal;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.util.StringBinder;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * syslogに転送するクラス<BR>
 *
 * @version 3.0.0
 * @since 3.0.0
 */
public class SendSyslog implements Notifier {
	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( SendSyslog.class );

	/** Syslogヘッダー部分時刻フォーマット。 */
	public static final String HEADER_DATE_FORMAT = "MMM dd HH:mm:ss";

	/** Syslogヘッダー部分のHOSTNAME。 */
	private static int MODE_HOSTNAME;
	private static String HOSTNAME_DEFINED = null;
	private static String HOSTNAME_SERVER = "unknownhost";

	private static final int MODE_HOSTNAME_DEFAULT = 0;
	private static final int MODE_HOSTNAME_DEFINED = 1;
	private static final int MODE_HOSTNAME_FACILITYID = 2;
	private static final int MODE_HOSTNAME_NODENAME = 3;

	private static int LIMIT_SIZE = 1024;

	// プロパティファイルの読み込み
	static{
		try {
			String hostname = HinemosProperties.getProperty("common.manager.hostname");

			if (hostname == null) {
				// undef hostname of syglog header
				MODE_HOSTNAME = MODE_HOSTNAME_DEFAULT;
			} else if ("".equals(hostname)) {
				// invalid hostname of syslog header
				MODE_HOSTNAME = MODE_HOSTNAME_DEFAULT;
			} else if ("${FACILITY_ID}".equals(hostname)) {
				// use facility_id as hostname of syslog header
				MODE_HOSTNAME = MODE_HOSTNAME_FACILITYID;
			} else if ("${NODE}".equals(hostname)) {
				// use nodename as hostname of syslog header
				MODE_HOSTNAME = MODE_HOSTNAME_NODENAME;
			} else {
				MODE_HOSTNAME = MODE_HOSTNAME_DEFINED;
				HOSTNAME_DEFINED = hostname;
			}

			HOSTNAME_SERVER = InetAddress.getLocalHost().getHostName();

			m_log.info("initialized syslog sender. (mode = " + MODE_HOSTNAME + ", hostname=" + HOSTNAME_DEFINED + ")");

		} catch (Exception e) {
			m_log.warn(e.getMessage(), e);
		}
	}

	/**
	 * ログエスカレーション通知：
	 * NotifyLogEscalationBean.onMessage()より呼び出す通知処理
	 * 
	 */
	@Override
	public void notify(NotifyRequestMessage message)
	throws FinderException, NamingException {
		if(m_log.isDebugEnabled()){
			m_log.debug("notify() " + message);
		}
		sendlog(message.getOutputInfo(), message.getNotifyId());
	}

	/**
	 * ログエスカレーション通知：
	 * 送信先リスト(IPADDRESS)に対してsyslog送信処理をcallする
	 * 
	 * @param outputInfo
	 * @param notifyId
	 * @throws FinderException
	 * @throws NamingException
	 */
	private void sendlog(OutputBasicInfo outputInfo, String notifyId)
	throws FinderException, NamingException {
		if(m_log.isDebugEnabled()){
			m_log.debug("sendlog() " + outputInfo);
		}

		// 該当する重要度の通知情報を取得する
		NotifyLogEscalateInfoLocal logEscalateInfo = NotifyLogEscalateInfoUtil.getLocalHome().findByPrimaryKey(
				new NotifyLogEscalateInfoPK(notifyId, outputInfo.getPriority()));

		NotifyLogEscalateInfoData escalateInfoData = logEscalateInfo.getData();

		// syslogの本文を作成
		String message = getMessage(outputInfo, escalateInfoData);

		/**
		 * 実行
		 */
		List<InetAddress> ipAddresses = getIpAddresses(outputInfo, escalateInfoData);
		if(ipAddresses == null){
			String detailMsg = "IP Address is empty.";
			m_log.error(detailMsg);
			internalErrorNotify(notifyId, "007", detailMsg);
		}

		// ヘッダー部分のTIMESTAMPを生成
		SimpleDateFormat sdf = new SimpleDateFormat(HEADER_DATE_FORMAT, Locale.US);
		String headerTimeStamp = sdf.format(new Date());
		if(m_log.isDebugEnabled()){
			m_log.debug("sendlog() target message. notifyId = " + escalateInfoData.getNotifyId() + 
					", headerTimeStamp = " + headerTimeStamp + 
					", facilityId = " + outputInfo.getFacilityId() +
					", message = " + message);
		}
		
		for(InetAddress address : ipAddresses){
			try {
				this.sendAfterConvertHostname(escalateInfoData, address, outputInfo.getFacilityId(), message, headerTimeStamp);
			} catch (IOException e) {
				String detailMsg = e.getMessage() + " IP Address = " + address;
				m_log.error("sendlog():" + detailMsg, e);
				internalErrorNotify(notifyId, "007", detailMsg);
				// 続けて次のIPを処理する
			}
		}
	}

	/**
	 * 指定の通知情報をもとにsyslog送信先のノードを特定します。
	 * NotifyLogEscalateInfoData の EscalateFacilityFlg が「ExecFacilityConstant.TYPE_GENERATION」にも
	 * かかわらず、outputInfoがnullの場合は、要素のない空のリストを返す。
	 * 
	 * @param outputInfo 通知グループ情報
	 * @param logEscalateInfo ログエスカレーション設定
	 * @return 送信先ノードのIPアドレス
	 */
	private List<InetAddress> getIpAddresses(OutputBasicInfo outputInfo, NotifyLogEscalateInfoData logEscalateInfo){
		// 送信先を特定
		ArrayList<InetAddress> ipAddresses = new ArrayList<InetAddress>();
		//リポジトリ(RepositoryControllerLocal)を取得
		try {
			RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();

			String facilityId = null;

			// 固定スコープ
			if(logEscalateInfo.getEscalateFacilityFlg() == ExecFacilityConstant.TYPE_FIX) {
				// 以下の実装は、入力されているファシリティIDがノードのモノである場合
				facilityId = logEscalateInfo.getEscalateFacility();
			} else {
				// イベントが発生したノード
				if(outputInfo == null){
					// NotifyLogEscalateInfoData の EscalateFacilityFlg が「ExecFacilityConstant.TYPE_GENERATION」にも
					// かかわらず、outputInfoがnullの場合は、要素のない空のリストを返す。
					return ipAddresses;
				} else {
					facilityId = outputInfo.getFacilityId();
				}
			}

			ArrayList<String> facilityIdList = repository.getExecTargetFacilityIdList(facilityId);

			for(String targetFacilityId : facilityIdList){
				FacilityLocal facility = FacilityUtil.getLocalHome().findByPrimaryKey(targetFacilityId);
				try {
					ipAddresses.add(getInetAdress(facility));
				} catch (UnknownHostException e) {
					m_log.error(e.getMessage(), e);
					// 続けて次のノードを処理する
				}
			}

			return ipAddresses;
		} catch (Exception e) {
			m_log.error(e.getMessage(), e);
			// 例外が発生した場合は、空のリストを返す
			return null;
		}
	}

	// 文字列を置換する
	private String getMessage(OutputBasicInfo outputInfo, NotifyLogEscalateInfoData logEscalateInfo){
		// 文字列を置換する
		try {
			Map<String, String> param = NotifyUtil.createParameter(outputInfo);
			StringBinder binder = new StringBinder(param);
			return binder.bindParam(logEscalateInfo.getEscalateMessage());
		} catch (Exception e) {
			m_log.error(e.getMessage(), e);

			// 例外が発生した場合は、置換前の文字列を返す
			return logEscalateInfo.getEscalateMessage();
		}
	}

	private InetAddress getInetAdress(FacilityLocal facility) throws UnknownHostException{
		InetAddress ret = null;

		// IPアドレスの取得
		int version = 4;
		if(facility.getNode().getIpAddressVersion() != null){
			version = facility.getNode().getIpAddressVersion();
		}else{
			version = 4;
		}
		// 入力されているバージョンを比較し、対応するIPアドレスを取得する
		if(version == 4) {
			ret = InetAddress.getByName(facility.getNode().getIpAddressV4());
		}
		else {
			ret = InetAddress.getByName(facility.getNode().getIpAddressV6());
		}

		return ret;
	}

	/**
	 * 汎用syslog送信処理(ShelfCheckTask、AplLogger)：
	 * 設定ファイルに従ってホスト名をヘッダに埋め込み、syslogパケットを送信する。<br>
	 * 
	 * @param ipAddress
	 * @param port
	 * @param facility
	 * @param severity
	 * @param facilityId
	 * @param message
	 */
	public void sendAfterConvertHostname(String ipAddress, int port, String facility, String severity, String facilityId, String message) {
		// ローカル変数
		String hostname = "";

		// メイン処理
		SimpleDateFormat sdf = new SimpleDateFormat(SendSyslog.HEADER_DATE_FORMAT, Locale.US);
		String timeStamp = sdf.format(new Date());
		hostname = getSyslogHeaderHost(facilityId);
		sendAfterConvertHostname(ipAddress, port, facility, severity, hostname, message, timeStamp);
	}
	
	/**
	 * 汎用syslog送信処理(ShelfCheckTask、AplLogger)：
	 * 設定ファイルに従ってホスト名をヘッダに埋め込み、syslogパケットを送信する。<br>
	 * 
	 * @param ipAddress
	 * @param port
	 * @param facility
	 * @param severity
	 * @param facilityId
	 * @param message
	 */
	public void sendAfterConvertHostname(String ipAddress, int port, String facility, String severity, String facilityId, String message, String timeStamp) {
		// ローカル変数
		String hostname = "";

		// メイン処理
		hostname = getSyslogHeaderHost(facilityId);
		send(ipAddress, port, facility, severity, hostname, message, timeStamp);
	}

	/**
	 * 汎用syslog送信処理(ShelfCheckTask、AplLogger)：
	 * syslogにメッセージを送る
	 * 
	 * @param ipAddress 送信先のIPアドレス
	 * @param port 送信先のポート番号
	 * @param facility facility (kern, user, mail, daemon, auth...)
	 * @param hostname syslogヘッダに定義するホスト名
	 * @param severity severity (emergency, alert, critical...)
	 * @param message メッセージ文字列
	 */
	private void send(String ipAddress, int port, String facility, String severity, String hostname, String message, String timeStamp) {
		// ローカル変数
		byte[] buf = new byte[LIMIT_SIZE];
		InetAddress inetAddress = null;

		int facilityInt = -1;
		int severityInt = -1;
		int priority = -1;

		String transferMsg = null;

		DatagramSocket soc = null;
		DatagramPacket sendPacket = null;

		if (m_log.isDebugEnabled())
			m_log.debug("start syslog transfer. (ipAddresss=" + ipAddress + ", port=" + port + ", "
					+ "facility=" + facility + ", severity=" + severity + ", message=" + message + ")");

		// メイン処理
		try {
			// generate InetAddress
			inetAddress = InetAddress.getByName(ipAddress);

			// generate priority of header
			facilityInt = SyslogFacilityConstant.stringToType(facility);
			severityInt = SyslogSeverityConstant.stringToType(severity);
			if (facilityInt == -1 || severityInt == -1) {
				m_log.warn("sending syslog failure. facility(" + facility + ") or severity(" + severity + ") is invalid.");
				return;
			}
			priority = facilityInt + severityInt;

			// generate message
			transferMsg = "<"+ priority + ">" + timeStamp + " " + hostname + " " + message;
			// is over 1024 byte
			if (transferMsg.getBytes().length > LIMIT_SIZE) {
				buf = transferMsg.getBytes();
				transferMsg = new String(buf, 0, LIMIT_SIZE);
			}

			// send message
			if (m_log.isDebugEnabled())
				m_log.debug("sending syslog. (ipAddresss=" + ipAddress + ", port=" + port + ", "
						+ "facility=" + facility + ", severity=" + severity + ", message=" + transferMsg + ")");

			soc = new DatagramSocket();
			sendPacket = null;
			buf = transferMsg.getBytes();
			sendPacket = new DatagramPacket(buf, buf.length, inetAddress, port);
			soc.send(sendPacket);
			soc.close();

		} catch (UnknownHostException e) {
			m_log.warn("sending syslog failure. syslog's host(" + ipAddress + ") is invalid.", e);
		} catch (SocketException e) {
			m_log.warn("sending syslog failure.", e);
		} catch (IOException e) {
			m_log.warn("sending syslog failure.", e);
		}

		if (m_log.isDebugEnabled())
			m_log.debug("sended syslog successfully. (ipAddresss=" + ipAddress + ", port=" + port + ", "
					+ "facility=" + facility + ", severity=" + severity + ", message=" + transferMsg + ")");
	}

	/**
	 * 設定ファイルに従って、syslogヘッダのホスト名として埋め込む文字列を生成する
	 * @param facilityId 通知情報の対象ファシリティID
	 * @return
	 */
	private static String getSyslogHeaderHost(String facilityId) {
		// ローカル変数
		String hostname = "";
		int mode = MODE_HOSTNAME;
		boolean isBuildinScope = false;

		// メイン処理
		if (m_log.isDebugEnabled()) m_log.debug("converting syglog hostname for '" + facilityId + "'. (mode = " + MODE_HOSTNAME + ")");

		if (FacilityTreeAttributeConstant.isBuiltinScope(facilityId)) {
			// 組み込みスコープの場合は、マネージャのホスト名を埋め込む
			if (m_log.isDebugEnabled()) m_log.debug("facility '" + facilityId + "' is buildin scope.");
			isBuildinScope = true;
		}

		switch (mode) {
		case MODE_HOSTNAME_DEFINED :
			hostname = HOSTNAME_DEFINED;
			break;
		case MODE_HOSTNAME_FACILITYID :
			hostname = isBuildinScope ? HOSTNAME_SERVER : facilityId;
			break;
		case MODE_HOSTNAME_NODENAME :
			try {
				FacilityLocal facility = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
				if (facility.isNode()) {
					hostname = isBuildinScope ? HOSTNAME_SERVER : facility.getNode().getNodeName();
				} else {
					if (m_log.isDebugEnabled()) m_log.debug("facility '" + facilityId + "' is not node.");
					hostname = HOSTNAME_SERVER;
				}
			} catch (Exception e) {
				hostname = HOSTNAME_SERVER;
				m_log.warn("use '" + hostname + "' for hostname of syslog header. (facility not found : " + facilityId + ")");
			}
			break;
		default :
			hostname = HOSTNAME_SERVER;
		}

		if (m_log.isDebugEnabled()) m_log.debug("use syslog hostname '" + hostname + "' for '" + facilityId + "'.");
		return hostname;
	}

	/**
	 * ログエスカレーション通知：
	 * 設定ファイルに従ってホスト名をヘッダに埋め込み、syslogパケットを送信する。<br>
	 * 
	 * @param logEscalateInfo 通知情報
	 * @param ipAddress 送信先のIPアドレス
	 * @param facilityId 通知情報の対象ファシリティID
	 * @param message syslogメッセージ部
	 * @throws IOException
	 */
	private void sendAfterConvertHostname(NotifyLogEscalateInfoData logEscalateInfo, InetAddress ipAddress, String facilityId, String message,String headerTimeStamp) throws IOException {
		// ローカル変数
		String hostname = "";

		// メイン処理
		hostname = getSyslogHeaderHost(facilityId);
		send(logEscalateInfo, ipAddress, hostname, message, headerTimeStamp);
	}

	/**
	 * ログエスカレーション通知：
	 * syslogにメッセージを送る
	 * 
	 * @param logEscalateInfo	通知内容
	 * @param ipAddress	IPアドレス
	 * @throws FinderException
	 * @throws NamingException
	 * @throws IOException
	 */
	private void send(NotifyLogEscalateInfoData logEscalateInfo, InetAddress ipAddress, String hostname, String message,String headerTimeStamp) throws IOException {
		byte[] buf = new byte[LIMIT_SIZE];		// 送受信バッファ
		int len;	// 送信文字数

		// 送信メッセージの作成
		int syslogPriority = logEscalateInfo.getSyslogPriority() + logEscalateInfo.getSyslogFacility();
		String sendMessage = "<"+ syslogPriority + ">" + headerTimeStamp + " " + hostname + " " + message;

		// 1024バイトを超える場合は、1024バイトまでを送信する（文字化けは考慮しない）
		if(sendMessage.getBytes().length > LIMIT_SIZE){
			buf = sendMessage.getBytes();
			sendMessage = new String(buf, 0, LIMIT_SIZE);
		}

		// ソケットを作成してサーバに接続する。
		DatagramSocket soc = new DatagramSocket();  // データグラムソケットを開く
		DatagramPacket sendPacket = null;           // データグラムパケット設定

		buf = sendMessage.getBytes();
		len = buf.length;
		int port = logEscalateInfo.getEscalatePort();			// ポート番号
		sendPacket = new DatagramPacket(buf, len, ipAddress, port );
		soc.send(sendPacket);

		soc.close();
	}

	/**
	 * 通知失敗時の内部エラー通知を定義します
	 */
	@Override
	public void internalErrorNotify(String notifyId, String msgID, String detailMsg) {
		AplLogger apllog = new AplLogger("NOTIFY", "notify");
		String[] args = { notifyId };
		// 通知失敗メッセージを出力
		apllog.put("SYS", msgID, args, detailMsg);
	}
}
