/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.factory;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.NotifyDuplicate;
import com.clustercontrol.notify.bean.NotifyCommandInfo;
import com.clustercontrol.notify.bean.NotifyEventInfo;
import com.clustercontrol.notify.bean.NotifyStatusInfo;
import com.clustercontrol.notify.bean.NotifyMailInfo;
import com.clustercontrol.notify.bean.NotifyJobInfo;
import com.clustercontrol.notify.bean.NotifyLogEscalateInfo;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyTypeConstant;
import com.clustercontrol.notify.ejb.entity.NotifyCommandInfoData;
import com.clustercontrol.notify.ejb.entity.NotifyCommandInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyJobInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyLogEscalateInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyMailInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyStatusInfoUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 通知情報を作成するクラスです。
 *
 * @version 3.0.0
 * @since 1.0.0
 */
public class AddNotify {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( AddNotify.class );

	/**
	 * 通知情報を作成します。
	 * <p>
	 * <ol>
	 *  <li>通知情報を作成します。</li>
	 *  <li>通知イベント情報を作成し、通知情報に設定します。</li>
	 *  <li>キャッシュ更新用の通知情報を生成し、ログ出力キューへ送信します。
	 *      監視管理機能で、監視管理機能で保持している通知情報キャッシュに追加されます。</li>
	 * </ol>
	 * 
	 * @param info 作成対象の通知情報
	 * @return 作成に成功した場合、<code> true </code>
	 * @throws CreateException
	 * @throws NamingException
	 * @throws JMSException
	 * @throws NotifyDuplicate
	 * @throws InvalidSetting
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyInfoBean
	 * @see com.clustercontrol.notify.ejb.entity.NotifyEventInfoBean
	 * @see com.clustercontrol.notify.util.SendQueue
	 */
	@SuppressWarnings("unchecked")
	public boolean add(NotifyInfo info, String user) throws CreateException, NamingException, NotifyDuplicate {
		m_log.debug("add " + "NotifyID = " + info.getNotifyId());

		try {
			Timestamp now = new Timestamp(new Date().getTime());

			// 通知情報を挿入
			NotifyInfoLocal notify = NotifyInfoUtil.getLocalHome().create(
					info.getNotifyId(),
					info.getDescription(),
					info.getNotifyType(),
					info.getInitialCount(),
					info.getRenotifyType(),
					info.getRenotifyPeriod(),
					now,
					now,
					user,
					user,
					info.getValidFlg()
			);

			switch(info.getNotifyType()){
			case NotifyTypeConstant.TYPE_COMMAND:
				addNotifyCommand(info, notify);
				break;
			case NotifyTypeConstant.TYPE_EVENT:
				addNotifyEvent(info, notify);
				break;
			case NotifyTypeConstant.TYPE_JOB:
				addNotifyJob(info, notify);
				break;
			case NotifyTypeConstant.TYPE_LOG_ESCALATE:
				addNotifyLogEscalate(info, notify);
				break;
			case NotifyTypeConstant.TYPE_MAIL:
				addNotifyMail(info, notify);
				break;
			case NotifyTypeConstant.TYPE_STATUS:
				addNotifyStatus(info, notify);
				break;
			}

		} catch (CreateException e) {
			if (e instanceof DuplicateKeyException) {
				m_log.warn("add " + e.getMessage());
				throw new NotifyDuplicate(e.getMessage(),e);
			}
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { info.getNotifyId() };
			apllog.put("SYS", "001", args);
			m_log.debug("add():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			String[] args = { info.getNotifyId() };
			apllog.put("SYS", "001", args);
			m_log.debug("add():" + e.getMessage());
			throw e;
		}

		return true;
	}

	/**
	 * コマンド通知情報詳細を変更します。
	 * @param info　変更情報
	 * @param notify 変更対象Bean
	 * @return
	 * @throws CreateException
	 * @throws NamingException
	 */
	private boolean addNotifyCommand(NotifyInfo info, NotifyInfoLocal notify) throws CreateException, NamingException {

		// 通知コマンド情報を挿入
		for (NotifyCommandInfo command : info.getNotifyCommandInfo()) {
			if (command != null) {

				NotifyCommandInfoData commandData = new NotifyCommandInfoData(
						command.getNotifyId(),
						command.getPriority(),
						command.getValidFlg(),
						command.getCommand(),
						command.getEffectiveUser(),
						command.getSetEnvironment(),
						command.getTimeout());
				NotifyCommandInfoUtil.getLocalHome().create(commandData);
			}
		}
		return true;
	}

	/**
	 * イベント通知情報詳細を変更します。
	 * @param info　変更情報
	 * @param notify 変更対象Bean
	 * @return
	 * @throws CreateException
	 * @throws NamingException
	 */
	private boolean addNotifyEvent(NotifyInfo info, NotifyInfoLocal notify) throws CreateException, NamingException {
		ArrayList<NotifyEventInfo> eventList = info.getNotifyEventInfo();
		// 通知イベント情報を挿入
		NotifyEventInfo event = null;
		for (int index = 0; index < eventList.size(); index++) {
			event = eventList.get(index);
			if (event != null) {
				NotifyEventInfoUtil.getLocalHome().create(
						info.getNotifyId(), event.getPriority(),
						event.getValidFlg(), event.getEventNormalState());
			}
		}
		return true;
	}

	/**
	 * ジョブ通知情報詳細を変更します。
	 * @param info　変更情報
	 * @param notify 変更対象Bean
	 * @return
	 * @throws CreateException
	 * @throws NamingException
	 */
	private boolean addNotifyJob(NotifyInfo info, NotifyInfoLocal notify) throws CreateException, NamingException {
		ArrayList<NotifyJobInfo> eventList = info.getNotifyJobInfo();
		// 通知ジョブ情報を挿入
		NotifyJobInfo job = null;
		for (int index = 0; index < eventList.size(); index++) {
			job = eventList.get(index);
			if (job != null) {
				NotifyJobInfoUtil.getLocalHome().create(info.getNotifyId(),
						job.getPriority(), job.getValidFlg(),
						job.getJobunitId(), job.getJobId(),
						job.getJobFailurePriority(),
						job.getJobExecFacilityFlg(),
						job.getJobExecFacility());
			}
		}
		return true;
	}

	/**
	 * ログエスカレーション通知情報詳細を変更します。
	 * @param info　変更情報
	 * @param notify 変更対象Bean
	 * @return
	 * @throws CreateException
	 * @throws NamingException
	 */
	private boolean addNotifyLogEscalate(NotifyInfo info, NotifyInfoLocal notify) throws CreateException, NamingException {
		ArrayList<NotifyLogEscalateInfo> eventList = info.getNotifyLogEscalateInfo();
		// 通知ログエスカレーション情報を挿入
		NotifyLogEscalateInfo log = null;
		for (int index = 0; index < eventList.size(); index++) {
			log = eventList.get(index);
			if (log != null) {
				NotifyLogEscalateInfoUtil.getLocalHome().create(
						info.getNotifyId(), log.getPriority(),
						log.getValidFlg(),
						log.getEscalateMessage(), log.getSyslogPriority(),
						log.getSyslogFacility(),
						log.getEscalateFacilityFlg(),
						log.getEscalateFacility(), log.getEscalatePort());
			}
		}
		return true;
	}

	/**
	 * メール通知情報詳細を変更します。
	 * @param info　変更情報
	 * @param notify 変更対象Bean
	 * @return
	 * @throws CreateException
	 * @throws NamingException
	 */
	private boolean addNotifyMail(NotifyInfo info, NotifyInfoLocal notify) throws CreateException, NamingException {
		ArrayList<NotifyMailInfo> eventList = info.getNotifyMailInfo();
		// 通知メール情報を挿入
		NotifyMailInfo mail = null;
		for (int index = 0; index < eventList.size(); index++) {
			mail = eventList.get(index);
			if (mail != null) {
				NotifyMailInfoUtil.getLocalHome().create(
						info.getNotifyId(), mail.getPriority(),
						mail.getMailTemplateId(), mail.getValidFlg(),
						mail.getMailAddress());
			}
		}
		return true;
	}

	/**
	 * ステータス通知情報詳細を変更します。
	 * @param info　変更情報
	 * @param notify 変更対象Bean
	 * @return
	 * @throws CreateException
	 * @throws NamingException
	 */
	private boolean addNotifyStatus(NotifyInfo info, NotifyInfoLocal notify) throws CreateException, NamingException {
		ArrayList<NotifyStatusInfo> eventList = info.getNotifyStatusInfo();
		// 通知ステータス情報を挿入
		NotifyStatusInfo status = null;
		for (int index = 0; index < eventList.size(); index++) {
			status = eventList.get(index);
			if (status != null) {
				NotifyStatusInfoUtil.getLocalHome().create(
						info.getNotifyId(), status.getPriority(),
						status.getValidFlg(),
						status.getStatusValidPeriod(),
						status.getStatusInvalidFlg(),
						status.getStatusUpdatePriority());
			}
		}
		return true;
	}
}
