/*

Copyright (C) since 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.accesscontrol.factory;

import java.sql.Timestamp;
import java.text.ParseException;
import java.util.Collection;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.ObjectNotFoundException;
import javax.ejb.RemoveException;
import javax.management.ObjectName;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.jmx.adaptor.rmi.RMIAdaptor;

import com.clustercontrol.fault.UserDuplicate;
import com.clustercontrol.fault.UserNotFound;
import com.clustercontrol.accesscontrol.bean.RoleConstant;
import com.clustercontrol.accesscontrol.bean.UserInfo;
import com.clustercontrol.accesscontrol.bean.UserTypeConstant;
import com.clustercontrol.accesscontrol.ejb.entity.UserLocal;
import com.clustercontrol.accesscontrol.ejb.entity.UserRoleBeanPK;
import com.clustercontrol.accesscontrol.ejb.entity.UserRoleLocal;
import com.clustercontrol.accesscontrol.ejb.entity.UserRoleUtil;
import com.clustercontrol.accesscontrol.ejb.entity.UserUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * ユーザ情報を更新するファクトリクラス<BR>
 *
 * @version 1.0.0
 * @since 3.2.0
 */
public class LoginUserModifier {

	/** ログ出力のインスタンス */
	private static Log m_log = LogFactory.getLog(LoginUserModifier.class);

	/** JMXのルックアップ名 */
	private static final String LOOKUP_NAME = "jmx/invoker/RMIAdaptor";
	/** JMXのオブジェクト名 */
	private static final String OBJECT_NAME = "jboss.security:service=JaasSecurityManager";
	/** JMXのオブジェクトのメソッド名 */
	private static final String OPERATION_NAME = "flushAuthenticationCache";

	/**
	 * ログインユーザを追加する。<BR>
	 * 
	 * @param info 追加するユーザ情報
	 * @param modifyUserId 作業ユーザID
	 * @return
	 * @throws ParseException ユーザIDが不正な場合
	 * @throws UserDuplicate
	 * @throws CreateException
	 */
	public static void addUserInfo(UserInfo info, String modifyUserId) throws UserDuplicate, CreateException {
		if(info != null && modifyUserId != null && modifyUserId.compareTo("") != 0){
			String userId = null;
			try {
				// ユーザID
				userId = info.getId();

				// 現在日時を取得
				Timestamp now = new Timestamp(new Date().getTime());

				// ユーザインスタンスの作成
				UserLocal local = UserUtil.getLocalHome().create(userId);

				local.setUserName(info.getName());
				local.setUserType(UserTypeConstant.LOGIN_USER);	// ユーザ種別の格納
				local.setDescription(info.getDescription());
				local.setCreateUserId(modifyUserId);
				local.setCreateDatetime(now);
				local.setModifyUserId(modifyUserId);
				local.setModifyDatetime(now);

				// クライアントから送信されないユーザ権限の登録
				UserRoleUtil.getLocalHome().create(info.getId(), RoleConstant.HINEMOS_USER);

				// クライアントから送信されるユーザ権限の登録
				for(String role : info.getRoleList()){
					try {
						UserRoleUtil.getLocalHome().create(info.getId(), role);
					} catch (CreateException e) {
						m_log.warn("failure to add a user's role. (userId = " + info.getId() + ", userRole = " + role + ")", e);
						throw e;
					}
				}
			} catch (CreateException e) {
				if (e instanceof DuplicateKeyException) {
					String[] args = {userId};
					AplLogger apllog = new AplLogger("ACC", "acc");
					apllog.put("SYS", "006", args);
					m_log.info("failure to add a user. a user'id is duplicated. (userId = " + userId + ")", e);
					throw new UserDuplicate(e.getMessage(), e);
				} else {
					throw e;
				}
			} catch (Exception e) {
				String[] args = {userId};
				AplLogger apllog = new AplLogger("ACC", "acc");
				apllog.put("SYS", "006", args);

				m_log.warn("failure to add a user. (userId = " + userId + ")", e);
				throw new EJBException("failure to add a user. (userId = " + userId + ")", e);
			}
		}
	}

	/**
	 * ログインユーザを削除する。<BR>
	 * 
	 * @param userId 削除対象のユーザID
	 * @param modifyUserId 作業ユーザID
	 * @throws UserNotFound
	 * @throws NamingException
	 * @throws RemoveException
	 */
	public static void deleteUserInfo(String userId, String modifyUserId) throws UserNotFound, NamingException, RemoveException {
		if(userId != null && userId.compareTo("") != 0 && modifyUserId != null && modifyUserId.compareTo("") != 0){
			try {
				// 作業ユーザと削除対象のユーザが一致している場合、削除不可とする
				if (userId.compareTo(modifyUserId) == 0) {
					throw new EJBException("a user will be deleted is equal to current login user.");
				}

				// 該当するユーザを検索して取得
				UserLocal user;
				user = UserUtil.getLocalHome().findByPrimaryKey(userId);
				// ユーザを削除する（DELETE CASCADEによりユーザ権限も削除される）
				user.remove();
			} catch (FinderException e) {
				String[] args = {userId};
				AplLogger apllog = new AplLogger("ACC", "acc");
				apllog.put("SYS", "007", args);

				m_log.warn("failure to delete a user. (userId = " + userId + ")", e);
				throw new UserNotFound(e.getMessage(), e);
			} catch (NamingException e) {
				String[] args = {userId};
				AplLogger apllog = new AplLogger("ACC", "acc");
				apllog.put("SYS", "007", args);

				throw e;
			} catch (EJBException e) {
				String[] args = {userId};
				AplLogger apllog = new AplLogger("ACC", "acc");
				apllog.put("SYS", "007", args);

				throw e;
			} catch (RemoveException e) {
				String[] args = {userId};
				AplLogger apllog = new AplLogger("ACC", "acc");
				apllog.put("SYS", "007", args);

				throw e;
			}
			//認証キャッシュ更新
			flushAuthenticationCache();

			m_log.info("successful in deleting a user. (userId = " + userId + ")");
		}
	}

	/**
	 * ログインユーザを変更する。<BR>
	 * 
	 * @param info 変更するユーザ情報
	 * @param modifyUserId 作業ユーザID
	 * @throws RemoveException
	 * @throws ParseException
	 * @throws UserNotFound
	 * @throws CreateException
	 * @throws NamingException
	 */
	public static void modifyUserInfo(UserInfo info, String modifyUserId) throws RemoveException, UserNotFound, CreateException, NamingException{
		if(info != null && modifyUserId != null && modifyUserId.compareTo("") != 0){
			String userId = null;
			try {
				// ユーザID
				userId = info.getId();


				// 該当するユーザを検索して取得
				UserLocal user = UserUtil.getLocalHome().findByPrimaryKey(userId);

				// 現在日時を取得
				Timestamp now = new Timestamp(new Date().getTime());

				// ユーザインスタンスの取得
				UserLocal local = UserUtil.getLocalHome().findByPrimaryKey(userId);

				local.setUserName(info.getName());
				local.setDescription(info.getDescription());
				local.setModifyUserId(modifyUserId);
				local.setModifyDatetime(now);

				// これまでに存在した権限がクライアントから送信された権限に含まれなかった場合、権限を削除
				for(UserRoleLocal roleLocal : (Collection<UserRoleLocal>)local.getUserRole()){
					if(!info.getRoleList().contains(roleLocal.getUserRole())){
						try {
							roleLocal.remove();
						} catch (RemoveException e) {
							m_log.warn("failure to remove a user's role. (userId = " + info.getId() + ", userRole = " + roleLocal.getUserRole() + ")", e);
							throw e;
						}
					}
				}

				// クライアントから送信されないユーザ権限の登録
				UserRoleUtil.getLocalHome().create(info.getId(), RoleConstant.HINEMOS_USER);

				// クライアントから送信された権限が存在しない場合、権限を追加
				for(String role : info.getRoleList()){
					try{
						UserRoleLocal roleLocal = UserRoleUtil.getLocalHome().findByPrimaryKey(new UserRoleBeanPK(userId, role));
					}catch(ObjectNotFoundException e1){
						try {
							UserRoleUtil.getLocalHome().create(info.getId(), role);
						} catch (CreateException e2) {
							m_log.warn("failure to add a user's role. (userId = " + info.getId() + ", userRole = " + role + ")", e2);
							throw e2;
						}
					}
				}
			} catch (FinderException e) {
				String[] args = {userId};
				AplLogger apllog = new AplLogger("ACC", "acc");
				apllog.put("SYS", "008", args);

				m_log.warn("failure to modify a user. (userId = " + userId + ")", e);
				throw new UserNotFound(e.getMessage(), e);
			} catch (NamingException e) {
				String[] args = {userId};
				AplLogger apllog = new AplLogger("ACC", "acc");
				apllog.put("SYS", "008", args);

				throw e;
			} catch (CreateException e) {
				String[] args = {userId};
				AplLogger apllog = new AplLogger("ACC", "acc");
				apllog.put("SYS", "008", args);

				throw e;
			}

			//認証キャッシュ更新
			flushAuthenticationCache();
			m_log.info("successful in modifing a user. (userId = " + userId + ")");
		}
	}


	/**
	 * ログインユーザに設定されたパスワードを変更する。<BR>
	 * 
	 * @param userId ユーザID
	 * @param password 新しいパスワード文字列
	 * @throws UserNotFound
	 * @throws NamingException
	 */
	public static void modifyUserPassword(String userId, String password) throws UserNotFound, NamingException {
		if(userId != null && userId.compareTo("") != 0 && password != null && password.compareTo("") != 0){
			// 該当するユーザを検索して取得
			UserLocal user;
			try {
				user = UserUtil.getLocalHome().findByPrimaryKey(userId);
				// パスワードを反映する
				user.setPassword(password);
			} catch (FinderException e) {
				AplLogger apllog = new AplLogger("ACC", "acc");
				String[] args = {userId};
				apllog.put("SYS", "009", args);

				m_log.warn("failure to modify user's password. (userId = " + userId + ")", e);
				throw new UserNotFound(e.getMessage(), e);
			} catch (NamingException e) {
				AplLogger apllog = new AplLogger("ACC", "acc");
				String[] args = {userId};
				apllog.put("SYS", "009", args);

				throw e;
			}

			//認証キャッシュ更新
			flushAuthenticationCache();

			m_log.info("successful in modifing a user's password. (userId = " + userId + ")");
		}
	}


	/**
	 * JMX経由で、認証キャッシュを初期化する。<BR>
	 */
	private static void flushAuthenticationCache() {
		/** ローカル変数 */
		InitialContext ic = null;
		RMIAdaptor server = null;
		ObjectName name = null;

		/** メイン処理 */
		m_log.debug("clearing authentication cache...");

		try {
			ic = new InitialContext();

			// RMIAdaptorを取得
			server = (RMIAdaptor) ic.lookup(LOOKUP_NAME);

			// ObjectNameを設定
			name = new ObjectName(OBJECT_NAME);

			// ObjectNameのOperationNameのメソッドを実行
			Object[]args = {"hinemos"};
			String[] signature = {String.class.getName()};
			server.invoke(name, OPERATION_NAME, args, signature);
		} catch (Exception e) {
			m_log.warn("failure to clear authentication cache", e);
		}
	}

}
