/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.monitor.view;

import org.eclipse.jface.action.ActionContributionItem;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.bean.Property;
import com.clustercontrol.monitor.composite.StatusListComposite;
import com.clustercontrol.monitor.composite.action.StatusListSelectionChangedListener;
import com.clustercontrol.monitor.preference.MonitorPreferencePage;
import com.clustercontrol.monitor.view.action.StatusDeleteAction;
import com.clustercontrol.util.Messages;
import com.clustercontrol.view.ScopeListBaseView;
import com.clustercontrol.ws.repository.FacilityTreeItem;

/**
 * 監視[ステータス]ビュークラス<BR>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class StatusView extends ScopeListBaseView {

	/** 監視[ステータス]ビューID */
	public static final String ID = "com.clustercontrol.monitor.view.StatusView";

	/** ステータス情報一覧コンポジット */
	private StatusListComposite tableComposite = null;

	/** 検索条件 */
	private Property condition = null;


	/**
	 * ステータス情報一覧テーブルを作成・追加します。
	 * <p>
	 * <ol>
	 * <li>ステータス情報一覧テーブルを取得します。</li>
	 * <li>ステータス情報一覧テーブルに、ポップアップメニューを追加します。</li>
	 * <li>ビューを更新します。</li>
	 * <li>プレファレンスページの設定より、監視[ステータス]ビューの自動更新周期を取得し、このビューにセットします。</li>
	 * <li>プレファレンスページの設定より、監視[ステータス]ビューの自動更新フラグを取得し、このビューにセットします。</li>
	 * <li>ステータス情報一覧テーブルを返します。</li>
	 * </ol>
	 * 
	 * @param parent 親のコンポジット
	 * @return ステータス情報一覧テーブルコンポジット
	 * 
	 * @see com.clustercontrol.monitor.composite.StatusListComposite#StatusListComposite(Composite, int)
	 * @see #createContextMenu()
	 * @see #update()
	 */
	@Override
	protected Composite createListContents(Composite parent) {
		GridLayout layout = new GridLayout(1, true);
		parent.setLayout(layout);
		layout.marginHeight = 0;
		layout.marginWidth = 0;

		this.tableComposite = new StatusListComposite(parent, SWT.NONE);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		this.tableComposite.setLayoutData(gridData);

		//ポップアップメニュー作成
		createContextMenu();

		// 設定情報反映
		IPreferenceStore store = ClusterControlPlugin.getDefault()
		.getPreferenceStore();

		this.setInterval(store
				.getInt(MonitorPreferencePage.P_STATUS_UPDATE_CYCLE));

		if (store.getBoolean(MonitorPreferencePage.P_STATUS_UPDATE_FLG)) {
			this.startAutoReload();
		}

		// ボタン（アクション）を制御するリスナーを登録
		this.tableComposite.getTableViewer().addSelectionChangedListener(
				new StatusListSelectionChangedListener());

		return this.tableComposite;
	}

	/**
	 * スコープツリーでアイテムを選択した際に呼ばれ、ビューを更新します。
	 * 
	 * @param item スコープツリーアイテム
	 * 
	 * @see #update()
	 */
	@Override
	protected void doSelectTreeItem(FacilityTreeItem item) {
		this.update();
	}

	/**
	 * ポップアップメニューを作成します。
	 */
	protected void createContextMenu() {
		MenuManager menuManager = new MenuManager();
		menuManager.setRemoveAllWhenShown(true);
		menuManager.addMenuListener(new IMenuListener() {
			/*
			 * (non-Javadoc)
			 * 
			 * @see org.eclipse.jface.action.IMenuListener#menuAboutToShow(org.eclipse.jface.action.IMenuManager)
			 */
			@Override
			public void menuAboutToShow(IMenuManager manager) {
				//メニュー項目設定
				fillContextMenu(manager);
			}
		});

		Menu menu = menuManager.createContextMenu(tableComposite.getTable());
		tableComposite.getTable().setMenu(menu);
	}

	/**
	 * ポップアップメニューに項目を追加します。
	 * 
	 * @param manager ポップアップメニューオブジェクト
	 */
	protected void fillContextMenu(IMenuManager manager) {
		//ビューのアクションのインスタンスを取得
		IToolBarManager toolbar = this.getViewSite().getActionBars()
		.getToolBarManager();
		ActionContributionItem deleteAction = (ActionContributionItem) toolbar
		.find(StatusDeleteAction.ID);

		//ポップアップメニューにアクションを追加
		manager.add(deleteAction.getAction());
	}

	/**
	 * ビューを更新します。<BR>
	 * <p>
	 * <ol>
	 * <li>スコープツリーで選択されているアイテムを取得します。</li>
	 * <li>アイテムより、ファシリティIDを取得します。</li>
	 * <li>ファシリティIDをキーに、ステータス情報を更新します。
	 *     検索条件が<code>null</code>ではない場合は、検索条件も指定します。</li>
	 * </ol>
	 * 
	 * @see com.clustercontrol.composite.FacilityTreeComposite#getSelectItem()
	 * @see com.clustercontrol.bean.FacilityInfo#getFacilityId()
	 * @see com.clustercontrol.monitor.composite.StatusListComposite#update(String)
	 * @see com.clustercontrol.monitor.composite.StatusListComposite#update(String, Property)
	 */
	@Override
	public void update() {

		FacilityTreeItem item = this.getScopeTreeComposite().getSelectItem();
		String facilityId = null;
		if (item != null) {
			facilityId = item.getData().getFacilityId();
		}

		if (this.condition == null) {
			// スコープツリーでアイテムが選択されていない場合
			if (facilityId == null) {
				this.getPathLabel()
				.setText(Messages.getString("scope") + " : ");
				this.getPathLabel().pack();
			}
			this.tableComposite.update(facilityId);
		} else {
			this.getPathLabel().setText(Messages.getString("filtered.list"));
			this.getPathLabel().pack();
			this.tableComposite.update(facilityId, this.condition);
		}
	}

	/**
	 * 検索条件を設定します。
	 * <p>
	 * 
	 * 設定後のupdateは、この検索条件の結果が表示されます。
	 * 
	 * @param condition
	 *            検索条件
	 */
	public void setCondition(Property condition) {
		this.condition = condition;
	}

	/**
	 * 検索条件とビューアクションを初期化します。
	 */
	@Override
	public void dispose() {
		super.dispose();

		this.setCondition(null);
	}

	/**
	 * ビューのアクションの有効/無効を設定します。
	 * 
	 * @param num 選択イベント数
	 * @param selection ボタン（アクション）を有効にするための情報
	 */
	public void setEnabledAction(int num, ISelection selection) {
		//ビューアクションの使用可/不可を設定
		super.setEnabledAction(StatusDeleteAction.ID, selection);

		if (num > 0){
			super.setEnabledAction(StatusDeleteAction.ID, true);
		} else {
			super.setEnabledAction(StatusDeleteAction.ID, false);
		}
	}
}