/*

Copyright (C) 2007 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.maintenance.composite;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.composite.CommonTableViewer;
import com.clustercontrol.maintenance.action.GetMaintenanceListTableDefine;
import com.clustercontrol.maintenance.util.MaintenanceEndpointWrapper;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.maintenance.InvalidRole_Exception;
import com.clustercontrol.ws.maintenance.MaintenanceInfo;
import com.clustercontrol.ws.maintenance.MaintenanceTypeMst;

/**
 * メンテナンス情報一覧コンポジットクラスです。
 * 
 * @version 2.2.0
 * @since 2.2.0
 */
public class MaintenanceListComposite extends Composite {

	// ログ
	private static Log m_log = LogFactory.getLog( MaintenanceListComposite.class );

	// ----- instance フィールド ----- //

	/** テーブルビューア */
	private CommonTableViewer tableViewer = null;

	// ----- コンストラクタ ----- //

	/**
	 * インスタンスを返します。
	 * 
	 * @param parent
	 *            親のコンポジット
	 * @param style
	 *            スタイル
	 */
	public MaintenanceListComposite(Composite parent, int style) {
		super(parent, style);

		this.initialize();
	}

	// ----- instance メソッド ----- //

	/**
	 * コンポジットを生成・構築します。
	 */
	private void initialize() {
		GridLayout layout = new GridLayout(1, true);
		this.setLayout(layout);
		layout.marginHeight = 0;
		layout.marginWidth = 0;

		Table table = new Table(this, SWT.MULTI | SWT.H_SCROLL | SWT.V_SCROLL
				| SWT.FULL_SELECTION);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);

		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		table.setLayoutData(gridData);

		// テーブルビューアの作成
		this.tableViewer = new CommonTableViewer(table);
		this.tableViewer.createTableColumn(GetMaintenanceListTableDefine.get(),
				GetMaintenanceListTableDefine.SORT_COLUMN_INDEX,
				GetMaintenanceListTableDefine.SORT_ORDER);

		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;

	}

	/**
	 * tableViewerを返します。
	 * 
	 * @return tableViewer
	 */
	public CommonTableViewer getTableViewer() {
		return this.tableViewer;
	}

	/**
	 * コンポジットを更新します。
	 * <p>
	 * 
	 */
	@Override
	@SuppressWarnings("unchecked")
	public void update() {
		// データ取得
		/*
		 * メンテナンス情報とスケジュール情報が別テーブルにあるため、それぞれ取得する。
		 */
		// メンテナンス情報取得
		List<MaintenanceInfo> maintenanceList = null;
		try {
			maintenanceList = MaintenanceEndpointWrapper.getMaintenanceList();
		} catch (InvalidRole_Exception e) {
			MessageDialog.openInformation(null, Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
		} catch (Exception e) {
			m_log.warn("update() getMaintenanceList, " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}

		if(maintenanceList == null){
			maintenanceList = new ArrayList<MaintenanceInfo>();
		}

		// メンテナンスタイプの取得
		List<MaintenanceTypeMst> maintenanceTypeList = null;
		try {
			maintenanceTypeList = MaintenanceEndpointWrapper.getMaintenanceTypeList();

		} catch (InvalidRole_Exception e) {
			MessageDialog.openInformation(null, Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
		} catch (Exception e) {
			m_log.warn("update() getMaintenanceTypeList, " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}
		HashMap<String, String> typeMessageMap = new HashMap<String, String>();
		for (MaintenanceTypeMst type : maintenanceTypeList) {
			typeMessageMap.put(type.getTypeId(), Messages.getString(type.getNameId()));
		}

		// tableViewer にセットするための詰め替え
		ArrayList<Object> listInput = new ArrayList<Object>();

		for (MaintenanceInfo maintenanceInfo : maintenanceList) {
			ArrayList<Object> a = new ArrayList<Object>();

			a.add(maintenanceInfo.getMaintenanceId());
			a.add(maintenanceInfo.getDescription());
			a.add(typeMessageMap.get(maintenanceInfo.getTypeId()));
			a.add(maintenanceInfo.getDataRetentionPeriod());
			a.add(maintenanceInfo.getCalendarId());
			a.add(maintenanceInfo.getSchedule());
			a.add(maintenanceInfo.getValidFlg());
			a.add(maintenanceInfo.getRegUser());
			a.add(new Date(maintenanceInfo.getRegDate()));
			a.add(maintenanceInfo.getUpdateUser());
			a.add(new Date(maintenanceInfo.getUpdateDate()));

			listInput.add(a);
		}

		// テーブル更新
		this.tableViewer.setInput(listInput);
	}


	/**
	 * 選択された行のメンテナンスIDを取得する
	 * 
	 * @return
	 */
	public ArrayList<String> getSelectionData() {

		ArrayList<String> data = new ArrayList<String>();

		//選択されたアイテムを取得
		StructuredSelection selection =
			(StructuredSelection)tableViewer.getSelection();
		List list = selection.toList();

		if (list != null) {
			for(int index = 0; index < list.size(); index++){

				ArrayList info = (ArrayList)list.get(index);
				if (info != null && info.size() > 0) {
					String maintenanceId = (String)info.get(GetMaintenanceListTableDefine.MAINTENANCE_ID);
					data.add(maintenanceId);
				}
			}
		}

		return data;
	}
}