/*

Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.dialog;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;

import com.clustercontrol.accesscontrol.util.AccessEndpointWrapper;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.composite.CommonTableViewer;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.jobmanagement.action.GetUserTableDefine;
import com.clustercontrol.util.LoginManager;
import com.clustercontrol.util.Messages;
import com.clustercontrol.ws.access.InvalidRole_Exception;
import com.clustercontrol.ws.access.UserInfo;
import com.clustercontrol.ws.jobmanagement.JobManagementUserInfo;

/**
 * ジョブ参照可能ユーザをHinemosユーザより選択するダイアログ
 * @author Hinemos
 *
 */
public class UserDialog extends CommonDialog {

	// ログ
	private static Log m_log = LogFactory.getLog( UserDialog.class );

	/** テーブル */
	protected Table table = null;
	/** テーブルビューア */
	protected CommonTableViewer m_viewer = null;
	/** シェル */
	private Shell m_shell = null;
	/** 管理ユーザ情報 */
	private List<JobManagementUserInfo> m_userList = null;
	/** Hinemosユーザ情報 */
	private List<UserInfo> m_allUserList = null;


	/**
	 * コンストラクタ
	 * 
	 * @param parent 親シェル
	 */
	public UserDialog(List<JobManagementUserInfo> userList, Shell parent) {
		super(parent);
		m_userList = userList;
	}

	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親コンポジット
	 * 
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		m_shell = this.getShell();

		parent.getShell().setText(Messages.getString("job.management.user"));

		RowLayout layout = new RowLayout();
		layout.type = SWT.VERTICAL;
		layout.spacing = 1;
		layout.marginWidth = 5;
		layout.marginHeight = 5;
		layout.fill = true;
		parent.setLayout(layout);

		Composite composite = null;
		RowLayout rowLayout = null;

		composite = new Composite(parent, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.VERTICAL;
		composite.setLayout(rowLayout);

		// Hinemosユーザ一覧を表示するテーブル
		Label tableTitle = new Label(composite, SWT.NONE);
		tableTitle.setText(Messages.getString("list"));

		table = new Table(composite, SWT.BORDER | SWT.H_SCROLL
				| SWT.V_SCROLL | SWT.FULL_SELECTION | SWT.SINGLE);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		table.setLayoutData(new RowData(430, 80));

		composite = new Composite(composite, SWT.NONE);
		rowLayout = new RowLayout();
		rowLayout.type = SWT.HORIZONTAL;
		rowLayout.spacing = 5;
		composite.setLayout(rowLayout);

		m_viewer = new CommonTableViewer(table);
		m_viewer.createTableColumn(GetUserTableDefine.get(),
				GetUserTableDefine.SORT_COLUMN_INDEX,
				GetUserTableDefine.SORT_ORDER);
		reflectUserData();

		// テーブルのレコードを選択するリスナー
		table.addSelectionListener(new SelectionAdapter(){
			@Override
			public void widgetSelected(SelectionEvent e) {
				ArrayList al;
				//選択されたTableColumnを取得します。
				TableItem[] ti = table.getSelection();
				for (int i = 0; i<ti.length; i++){
					al = (ArrayList)ti[i].getData();
					if((Boolean)al.get(GetUserTableDefine.SELECTION)){
						//YESならNO
						al.set(GetUserTableDefine.SELECTION, YesNoConstant.BOOLEAN_NO);
					}else{
						//NOならYES
						al.set(GetUserTableDefine.SELECTION, YesNoConstant.BOOLEAN_YES);
					}
				}
				//チェックボックスが入るので、再描画。
				m_viewer.refresh();
			}
		});
	}

	/**
	 * Hinemosユーザ一覧をテーブルに反映する
	 */
	private void reflectUserData() {

		// Hinemosユーザリストをマネージャより取得
		try {
			m_allUserList = AccessEndpointWrapper.getUserInfoList(null);
		} catch (InvalidRole_Exception e) {
			MessageDialog.openInformation(null, Messages.getString("message"),
					Messages.getString("message.accesscontrol.16"));
		} catch (Exception e) {
			m_log.warn("reflectUserData() getUserInfoListAll, " + e.getMessage(), e);
			MessageDialog.openError(
					null,
					Messages.getString("failed"),
					Messages.getString("message.hinemos.failure.unexpected") + ", " + e.getMessage());
		}

		// ユーザ一覧を表に反映
		if(m_allUserList != null ) {

			ArrayList tableData = new ArrayList();
			for (int i = 0; i < m_allUserList.size(); i++) {
				UserInfo info = m_allUserList.get(i);
				ArrayList tableLineData = new ArrayList();
				if(alreadyChecked(info.getId()))
					tableLineData.add(YesNoConstant.BOOLEAN_YES);
				else
					tableLineData.add(YesNoConstant.BOOLEAN_NO);

				tableLineData.add(info.getId());
				tableLineData.add(info.getName());
				tableLineData.add(info.getDescription());

				tableData.add(tableLineData);
			}
			m_viewer.setInput(tableData);
			m_viewer.refresh();
		}
	}

	/**
	 * 既に参照可能ユーザとしてチェックされているか確認
	 * @param userId
	 * @return
	 */
	private boolean alreadyChecked(String userId) {
		boolean ret = false;
		if(m_userList != null){
			for (int i = 0; i < m_userList.size(); i++) {
				JobManagementUserInfo user = m_userList.get(i);
				if(user.getUserId().equals(userId)){
					ret = true;
					break;
				}
			}
		}
		return ret;
	}

	/**
	 * チェックがONになっているユーザ一覧を作成して返却
	 * @return
	 */
	public List<JobManagementUserInfo> getSelectionData() {


		return m_userList;
	}

	/**
	 * ＯＫボタンテキスト取得
	 * 
	 * @return ＯＫボタンのテキスト
	 * @since 2.1.0
	 */
	@Override
	protected String getOkButtonText() {
		return Messages.getString("ok");
	}

	/**
	 * キャンセルボタンテキスト取得
	 * 
	 * @return キャンセルボタンのテキスト
	 * @since 2.1.0
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#validate()
	 */
	@Override
	protected ValidateResult validate() {
		ValidateResult result = null;

		// 選択されたユーザリストを作成
		ArrayList<JobManagementUserInfo> data = new ArrayList<JobManagementUserInfo>();
		ArrayList al;
		TableItem[] ti = table.getItems();
		for (int i = 0; i<ti.length; i++){
			al = (ArrayList)ti[i].getData();
			if((Boolean)al.get(GetUserTableDefine.SELECTION)){
				JobManagementUserInfo user = new JobManagementUserInfo();
				user.setUserId((String)al.get(GetUserTableDefine.USER_ID));
				user.setUserName((String)al.get(GetUserTableDefine.USER_NAME));
				user.setDescription((String)al.get(GetUserTableDefine.DESCRIPTION));
				data.add(user);
			}
		}

		//ユーザリストが0人なら入力を促す
		if(data.size() == 0){
			result = new ValidateResult();
			result.setValid(false);
			result.setID(Messages.getString("message.hinemos.1"));
			result.setMessage(Messages.getString("message.job.63"));
			return result;
		} else {
			// 1人以上設定済みの場合はユーザリストを更新する
			m_userList = data;
		}

		//ユーザリストに自分が含まれていない場合は警告を出す
		String loginUserId = LoginManager.getContextManager().getUserId();
		boolean existFlfg = false;
		Iterator<JobManagementUserInfo> it = m_userList.iterator();
		while (it.hasNext()) {
			JobManagementUserInfo user = it.next();
			if(user.getUserId() != null && user.getUserId().equals(loginUserId)){
				existFlfg = true;
				break;
			}
		}
		if(!existFlfg){
			result = new ValidateResult();
			result.setValid(false);
			result.setID(Messages.getString("message.hinemos.1"));
			Object[] args = {loginUserId};
			result.setMessage(Messages.getString("message.job.66", args));
			return result;
		}


		return result;
	}
}
