/*
Copyright (C) 2010 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.ws.endpoint;

import java.util.ArrayList;

import javax.annotation.Resource;
import javax.ejb.CreateException;
import javax.naming.NamingException;
import javax.xml.ws.WebServiceContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.accesscontrol.bean.RoleConstant;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.snmptrap.bean.SnmpTrapMasterInfo;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapMibMasterData;
import com.clustercontrol.snmptrap.ejb.session.MonitorSnmpTrapControllerLocal;
import com.clustercontrol.snmptrap.ejb.session.MonitorSnmpTrapControllerLocalHome;

/**
 * SNMPTRAP監視用のWebAPIエンドポイント
 */
@javax.jws.WebService(targetNamespace = "http://monitor.ws.clustercontrol.com")
public class MonitorSnmpTrapEndpoint {
	@Resource
	WebServiceContext wsctx;

	private static Log m_log = LogFactory.getLog( MonitorSnmpTrapEndpoint.class );
	private static Log m_opelog = LogFactory.getLog("HinemosOperation");

	/**
	 * echo(WebサービスAPI疎通用)
	 * 
	 * 権限必要なし（ユーザ名チェックのみ実施）
	 * 
	 * @param str
	 * @return
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public String echo(String str) throws InvalidUserPass, InvalidRole, HinemosUnknown {
		ArrayList<String> roleList = new ArrayList<String>();
		HttpAuthenticator.authCheck(wsctx, roleList);

		return str + ", " + str;
	}

	/**
	 * SNMPトラップ用のSessionBeanの取得
	 * 
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidUserPass
	 */
	private MonitorSnmpTrapControllerLocal getMonitorSnmpTrapControllerLocal() throws HinemosUnknown, InvalidUserPass {
		MonitorSnmpTrapControllerLocal local = null;
		try {
			String account = HttpAuthenticator.getAccount(wsctx);
			NamingContext namingContext = HttpAuthenticator.getContext(account);
			MonitorSnmpTrapControllerLocalHome localHome =
				(MonitorSnmpTrapControllerLocalHome)
				namingContext.lookup(MonitorSnmpTrapControllerLocalHome.JNDI_NAME);
			local = localHome.create();
		} catch (CreateException e) {
			m_log.error("getMonitorSnmpTrapControllerLocal CreateException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (NamingException e) {
			m_log.error("getMonitorSnmpTrapControllerLocal NamingException : " + e,e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return local;
	}

	/**
	 * MIB一覧を取得します。<BR>
	 * 戻り値はMIBの名前(String)のArrayListです。
	 * 
	 * MonitorSettingRead権限が必要
	 * 
	 * @return MIB一覧
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<String> getMibList() throws MonitorNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getMibList : ");
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_SETTING_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Get MIB, Method=getMibList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx));

		MonitorSnmpTrapControllerLocal monitor = getMonitorSnmpTrapControllerLocal();
		return monitor.getMibList();
	}

	/**
	 * 監視マスタ情報一覧を取得します。<BR>
	 * 戻り値はSnmpTrapMasterInfoのArrayListです。
	 * 
	 * MonitorSettingRead権限が必要
	 * 
	 * @return MIB一覧
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @see com.clustercontrol.snmptrap.bean.SnmpTrapMasterInfo
	 */
	@SuppressWarnings("unchecked")
	public ArrayList<SnmpTrapMasterInfo> getMasterList(String mib) throws MonitorNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getMasterList : mib=" + mib);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_SETTING_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", MIB=");
		msg.append(mib);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Get OID, Method=getMasterList, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		MonitorSnmpTrapControllerLocal monitor = getMonitorSnmpTrapControllerLocal();
		return monitor.getMasterList(mib);
	}

	/**
	 * MIBの内容を取得します。<BR>
	 * 戻り値はMIBのDTO(SnmpTrapMibMasterData)です。
	 * 
	 * MonitorSettingRead権限が必要
	 * 
	 * @return MIB一覧
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public SnmpTrapMibMasterData getMib(String Mib) throws MonitorNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("getMib : mib=" + Mib);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_SETTING_READ);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", MIB=");
		msg.append(Mib);
		m_opelog.debug(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Get MIB, Method=getMib, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		MonitorSnmpTrapControllerLocal monitor = getMonitorSnmpTrapControllerLocal();
		return monitor.getMib(Mib);
	}

	/**
	 * MIB情報の登録
	 *
	 * MonitorSettingWrite権限が必要
	 *
	 * @param MibData
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public boolean addMibMaster(SnmpTrapMibMasterData MibData) throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("addMibMaster : snmpTrapMibMasterData=" + MibData);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_SETTING_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(MibData != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", MIB=");
			msg.append(MibData.getMib());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Add MIB, Method=addMibMaster, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		MonitorSnmpTrapControllerLocal monitor = getMonitorSnmpTrapControllerLocal();
		return monitor.addMibMaster(MibData);
	}

	/**
	 * MIB情報の変更
	 * 
	 * MonitorSettingWrite権限が必要
	 * 
	 * @param MibData
	 * @return
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public boolean modifyMibMaster(SnmpTrapMibMasterData MibData) throws MonitorNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("modifyMibMaster : snmpTrapMibMasterData=" + MibData);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_SETTING_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(MibData != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", MIB=");
			msg.append(MibData.getMib());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Change MIB, Method=modifyMibMaster, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		MonitorSnmpTrapControllerLocal monitor = getMonitorSnmpTrapControllerLocal();
		return monitor.modifyMibMaster(MibData);
	}

	/**
	 * MIB情報の削除
	 * 
	 * MonitorSettingWrite権限が必要
	 * 
	 * @param mib
	 * @return
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public boolean deleteMibMaster(String Mib) throws MonitorNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("deleteMibMaster : mib=" + Mib);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_SETTING_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		StringBuffer msg = new StringBuffer();
		msg.append(", MIB=");
		msg.append(Mib);
		m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Delete MIB, Method=deleteMibMaster, User="
				+ HttpAuthenticator.getUserAccountString(wsctx)
				+ msg.toString());

		MonitorSnmpTrapControllerLocal monitor = getMonitorSnmpTrapControllerLocal();
		return monitor.deleteMibMaster(Mib);
	}


	/**
	 * OID情報の登録
	 *
	 * MonitorSettingWrite権限が必要
	 *
	 * @param MibData
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public boolean addOidMaster(SnmpTrapMasterInfo Oid) throws HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("addOidMaster : snmpTrapMasterInfo="+ Oid);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_SETTING_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(Oid != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", MIB=");
			msg.append(Oid.getMib());
			msg.append(", Trap=");
			msg.append(Oid.getUei());
			msg.append(", OID=");
			msg.append(Oid.getTrapOid());
			msg.append(", GenericID=");
			msg.append(Oid.getGenericId());
			msg.append(", SpecificID=");
			msg.append(Oid.getSpecificId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Add OID, Method=addOidMaster, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		MonitorSnmpTrapControllerLocal monitor = getMonitorSnmpTrapControllerLocal();
		return monitor.addOidMaster(Oid);
	}

	/**
	 * OID情報の変更
	 * 
	 * MonitorSettingWrite権限が必要
	 * 
	 * @param MibData
	 * @return
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 */
	public boolean modifyOidMaster(SnmpTrapMasterInfo Oid) throws MonitorNotFound, HinemosUnknown, InvalidUserPass, InvalidRole {
		m_log.debug("modifyOidMaster : snmpTrapMasterInfo="+ Oid);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_SETTING_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(Oid != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", MIB=");
			msg.append(Oid.getMib());
			msg.append(", Trap=");
			msg.append(Oid.getUei());
			msg.append(", OID=");
			msg.append(Oid.getTrapOid());
			msg.append(", GenericID=");
			msg.append(Oid.getGenericId());
			msg.append(", SpecificID=");
			msg.append(Oid.getSpecificId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Change OID, Method=modifyOidMaster, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		MonitorSnmpTrapControllerLocal monitor = getMonitorSnmpTrapControllerLocal();
		return monitor.modifyOidMaster(Oid);
	}

	/**
	 * OID情報の削除
	 * 
	 * MonitorSettingWrite権限が必要
	 * 
	 * @param mib
	 * @return
	 * @throws HinemosUnknown
	 * @throws InvalidRole
	 * @throws InvalidUserPass
	 * @throws MonitorNotFound
	 */
	public boolean deleteOidMaster(SnmpTrapMasterInfo oid) throws InvalidUserPass, InvalidRole, HinemosUnknown, MonitorNotFound {
		m_log.debug("deleteOidMaster : snmpTrapMasterInfo="+ oid);
		ArrayList<String> roleList = new ArrayList<String>();
		roleList.add(RoleConstant.MONITOR_SETTING_WRITE);
		HttpAuthenticator.authCheck(wsctx, roleList);

		// 認証済み操作ログ
		if(oid != null){
			StringBuffer msg = new StringBuffer();
			msg.append(", MIB=");
			msg.append(oid.getMib());
			msg.append(", Trap=");
			msg.append(oid.getUei());
			msg.append(", OID=");
			msg.append(oid.getTrapOid());
			msg.append(", GenericID=");
			msg.append(oid.getGenericId());
			msg.append(", SpecificID=");
			msg.append(oid.getSpecificId());
			m_opelog.info(HinemosModuleConstant.LOG_PREFIX_MONITOR_SNMPTRAP + " Delete OID, Method=deleteOidMaster, User="
					+ HttpAuthenticator.getUserAccountString(wsctx)
					+ msg.toString());
		}

		MonitorSnmpTrapControllerLocal monitor = getMonitorSnmpTrapControllerLocal();
		return monitor.deleteOidMaster(oid);
	}
}
