/*

Copyright (C) since 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.repository.factory;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.UsedFacility;
import com.clustercontrol.bean.PluginConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.commons.util.Ipv6Util;
import com.clustercontrol.commons.util.ObjectValidator;
import com.clustercontrol.repository.bean.FacilityConstant;
import com.clustercontrol.repository.bean.FacilityInfo;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.bean.NodeCpuInfo;
import com.clustercontrol.repository.bean.NodeDeviceInfo;
import com.clustercontrol.repository.bean.NodeDiskInfo;
import com.clustercontrol.repository.bean.NodeFilesystemInfo;
import com.clustercontrol.repository.bean.NodeHostnameInfo;
import com.clustercontrol.repository.bean.NodeInfo;
import com.clustercontrol.repository.bean.NodeMemoryInfo;
import com.clustercontrol.repository.bean.NodeNetworkInterfaceInfo;
import com.clustercontrol.repository.bean.NodeNoteInfo;
import com.clustercontrol.repository.bean.NodeVariableInfo;
import com.clustercontrol.repository.bean.ScopeInfo;
import com.clustercontrol.repository.bean.TopicConstant;
import com.clustercontrol.repository.ejb.entity.FacilityLocal;
import com.clustercontrol.repository.ejb.entity.FacilityRelationLocal;
import com.clustercontrol.repository.ejb.entity.FacilityRelationUtil;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;
import com.clustercontrol.repository.ejb.entity.NodeCpuLocal;
import com.clustercontrol.repository.ejb.entity.NodeCpuUtil;
import com.clustercontrol.repository.ejb.entity.NodeDeviceLocal;
import com.clustercontrol.repository.ejb.entity.NodeDeviceUtil;
import com.clustercontrol.repository.ejb.entity.NodeDiskLocal;
import com.clustercontrol.repository.ejb.entity.NodeDiskUtil;
import com.clustercontrol.repository.ejb.entity.NodeFilesystemLocal;
import com.clustercontrol.repository.ejb.entity.NodeFilesystemUtil;
import com.clustercontrol.repository.ejb.entity.NodeHostnameLocal;
import com.clustercontrol.repository.ejb.entity.NodeHostnameUtil;
import com.clustercontrol.repository.ejb.entity.NodeLocal;
import com.clustercontrol.repository.ejb.entity.NodeMemoryLocal;
import com.clustercontrol.repository.ejb.entity.NodeMemoryUtil;
import com.clustercontrol.repository.ejb.entity.NodeNetworkInterfaceLocal;
import com.clustercontrol.repository.ejb.entity.NodeNetworkInterfaceUtil;
import com.clustercontrol.repository.ejb.entity.NodeNoteLocal;
import com.clustercontrol.repository.ejb.entity.NodeNoteUtil;
import com.clustercontrol.repository.ejb.entity.NodeUtil;
import com.clustercontrol.repository.ejb.entity.NodeVariableLocal;
import com.clustercontrol.repository.ejb.entity.NodeVariableUtil;
import com.clustercontrol.repository.util.SendTopic;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * ファシリティの更新処理を実装したクラス<BR>
 */
public class FacilityModifier {

	private static Log m_log = LogFactory.getLog(FacilityModifier.class);

	/** スコープを追加する。<BR>
	 * 
	 * @param parentFacilityId スコープを配置する親スコープのファシリティID（空文字の場合はルートスコープとなる）
	 * @param property 追加するスコープ情報
	 * @param modifyUserId 作業ユーザID
	 * @param displaySortOrder 表示ソート順位
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * @throws DuplicateKeyException
	 */
	public static void addScope(String parentFacilityId, ScopeInfo property, String modifyUserId, int displaySortOrder, boolean topicSendFlg) throws DuplicateKeyException {
		/** ローカル変数 */
		FacilityLocal parentFacility = null;
		FacilityLocal facility = null;
		String facilityId = null;

		/** メイン処理 */
		m_log.debug("adding a scope...");

		try {
			// 入力値（ファシリティID）の格納
			facilityId = property.getFacilityId();

			// 親ファシリティがスコープかどうかを確認する
			if (! ObjectValidator.isEmptyString(parentFacilityId)) {
				parentFacility = FacilityUtil.getLocalHome().findByPrimaryKey(parentFacilityId);
				if (! parentFacility.isScope()) {
					throw new EJBException("a parent's facility is not a scope. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")");
				}
			}

			// ファシリティインスタンスの生成
			facility = FacilityUtil.getLocalHome().create(facilityId);
			facility.setFacilityType(FacilityConstant.TYPE_SCOPE);
			facility.setDisplaySortOrder(displaySortOrder);
			facility.setValid(ValidConstant.TYPE_VALID);
			setFacility(facility, property, modifyUserId, false);

			// ファシリティ関連インスタンスの生成
			if (! ObjectValidator.isEmptyString(parentFacilityId)) {
				assignFacilityToScope(parentFacilityId, facilityId, false);
			}
		} catch (DuplicateKeyException e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "012", args);

			m_log.info("failure to add a scope. this scope's id is duplicated. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")", e);
			throw e;
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "012", args);

			m_log.warn("failure to add a scope. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to add a scope. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")", e);
		}

		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}

		m_log.info("successful in adding a scope . (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")");
	}

	/**
	 * スコープを変更する。<BR>
	 * 
	 * @param property 変更後のスコープ情報
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 */
	public static void modifyScope(ScopeInfo property, String modifyUserId, boolean topicSendFlg) {
		/** ローカル変数 */
		FacilityLocal facility = null;
		String facilityId = null;

		/** メイン処理 */
		m_log.debug("modifing a scope...");

		try {
			// 入力値（ファシリティID）の格納
			facilityId = property.getFacilityId();

			// ファシリティインスタンスを取得
			facility = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);

			// ファシリティがスコープかどうかを確認する
			if (! facility.isScope()) {
				throw new FinderException("this facility is not a scope. (facilityId = " + facilityId + ")");
			}

			// 変更後の値を格納する
			setFacility(facility, property, modifyUserId, false);
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "014", args);

			m_log.warn("failure to modify a scope. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to modify a scope. (facilityId = " + facilityId + ")", e);
		}

		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}

		m_log.info("successful in modifing a scope. (facilityId = " + facilityId + ")");
	}

	/**
	 * スコープを削除する。<BR>
	 * 
	 * @param facilityId 削除するスコープのファシリティID
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 */
	public static void deleteScope(String facilityId, String modifyUserId, boolean topicSendFlg) throws UsedFacility {
		/** ローカル変数 */
		FacilityLocal facility = null;

		/** メイン処理 */
		m_log.debug("deleting a scope with sub scopes...");

		try {
			// 該当するファシリティインスタンスを取得
			facility = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);

			// 関連インスタンス、ファシリティインスタンスを削除する
			deleteScopeRecursive(facility);
		} catch (UsedFacility e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "013", args);

			throw e;
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "013", args);

			throw new EJBException("failure to delete a scope with sub scopes. (facilityId = " + facilityId + ")", e);
		}

		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}

		m_log.info("successful in deleting a scope with sub scopes. (facilityId = " + facilityId + ")");
	}

	/**
	 * サブスコープを含めて、スコープを削除する。<BR>
	 * 
	 * @param scope 削除するスコープインスタンス
	 * @throws UsedFacility
	 */
	private static void deleteScopeRecursive(FacilityLocal scope) throws UsedFacility {
		/** ローカル変数 */
		String facilityId = null;

		/** メイン処理 */
		facilityId = scope.getFacilityId();

		// スコープでない場合はエラーとする
		if (! scope.isScope()) {
			throw new EJBException("this facility is not a scope. (facilityId = " + facilityId + ")");
		}

		// 直下にスコープを存在する場合、そのスコープおよびサブスコープを削除する
		if (scope.getFacilityRelationAsParent() != null) {
			for (FacilityRelationLocal relation : (Collection<FacilityRelationLocal>)scope.getFacilityRelationAsParent()) {
				if (relation.getChildFacility().isScope()) {
					deleteScopeRecursive(relation.getChildFacility());
				}
			}
		}
		try {
			// ファシリティインスタンスの削除（DELETE CASCADEにて関連するインスタンスも削除）
			scope.remove();
		} catch (RemoveException e) {
			m_log.warn("failure to delete a scope. this scope is used for other settings. (facilityId = " + facilityId + ")", e);
			throw new UsedFacility(PluginConstant.TYPE_REPOSITORY, e);
		}

		m_log.info("successful in deleting a scope. (facilityId = " + facilityId + ")");
	}

	/**
	 * ノードを追加する。<BR>
	 * 
	 * @param property 追加するノード情報
	 * @param modifyUserId 作業ユーザID
	 * @param displaySortOrder 表示ソート順位
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * @throws DuplicateKeyException
	 */
	public static void addNode(NodeInfo property, String modifyUserId, int displaySortOrder, boolean topicSendFlg) throws DuplicateKeyException {
		/** ローカル変数 */
		FacilityLocal facility = null;
		NodeLocal node = null;
		ArrayList inputValue = null;
		String facilityId = null;
		Boolean valid = true;

		/** メイン処理 */
		m_log.debug("adding a node...");

		try {
			facilityId = property.getFacilityId();

			valid = property.isValid();

			// ファシリティインスタンスの生成
			facility = FacilityUtil.getLocalHome().create(facilityId);
			facility.setFacilityType(FacilityConstant.TYPE_NODE);
			facility.setDisplaySortOrder(displaySortOrder);
			facility.setValid(valid);
			setFacility(facility, property, modifyUserId, false);

			// ノードインスタンスの生成
			node = NodeUtil.getLocalHome().create(facilityId);
			setNode(node, property, false);

			// ファシリティ関連インスタンスの生成
			assignFacilityToScope(FacilityTreeAttributeConstant.REGISTEREFD_SCOPE, facilityId, false);
		} catch (DuplicateKeyException e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "009", args);

			m_log.info("failure to add a node. a node's id is depulicated. (facilityId = " + facilityId + ")");
			throw e;
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "009", args);

			m_log.warn("failure to add a node. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to add a node. (facilityId = " + facilityId + ")", e);
		}

		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}

		m_log.info("successful in adding a node. (facilityId = " + facilityId + ")");
	}

	/**
	 * ノードを変更する。<BR>
	 * 
	 * @param property 変更後のノード情報
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 */
	public static void modifyNode(NodeInfo property, String modifyUserId, boolean topicSendFlg) {
		/** ローカル変数 */
		FacilityLocal facility = null;
		NodeLocal node = null;
		String facilityId = null;
		Boolean valid = true;

		/** メイン処理 */
		m_log.debug("modifing a node...");

		try {
			facilityId = property.getFacilityId();

			valid = property.isValid();

			// ファシリティインスタンスの生成
			facility = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
			if (valid == null) {
				throw new CreateException("node's valid is invalid . (valid = " + valid + ")");
			} else {
				facility.setValid(valid);
			}

			// ファシリティがノードかどうかを確認する
			if (! facility.isNode()) {
				throw new FinderException("this facility is not a node. (facilityId = " + facilityId + ")");
			}

			setFacility(facility, property, modifyUserId, false);

			// ノードインスタンスの取得
			node = NodeUtil.getLocalHome().findByPrimaryKey(facilityId);

			// 変更情報の反映
			setNode(node, property, false);
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "011", args);

			m_log.warn("failure to modify a node. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to modify a node. (facilityId = " + facilityId + ")", e);
		}

		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}

		m_log.info("successful in modifing a node. (facilityId = " + facilityId + ")");
	}

	/**
	 * ノードを削除する。<BR>
	 * 
	 * @param facilityId 削除するノードのファシリティID
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * @throws FacilityUsedException
	 */
	public static void deleteNode(String facilityId, String modifyUserId, boolean topicSendFlg) throws UsedFacility {
		/** ローカル変数 */
		FacilityLocal facility = null;

		/** メイン処理 */
		m_log.debug("deleting a node...");

		try {
			// 該当するファシリティインスタンスを取得
			facility = FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);

			// ノードでない場合はエラーとする
			if (! facility.isNode()) {
				throw new EJBException("this facility is not a node. (facilityId = " + facilityId + ")");
			}

			// ファシリティインスタンスを削除する（DELETE CASCADEで関連するインスタンスやも削除される）
			facility.remove();
		} catch (RemoveException e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "010", args);

			m_log.info("failure to delete a node. this node is used by other settings. (facilityId = " + facilityId + ")");
			throw new UsedFacility(PluginConstant.TYPE_REPOSITORY, e);
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId};
			apllog.put("SYS", "010", args);

			m_log.warn("failure to delete a node. (facilityId = " + facilityId + ")");
			throw new EJBException("failure to delete a node. (facilityId = " + facilityId + ")");
		}

		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}

		m_log.info("successful in deleting a node. (facilityId = " + facilityId + ")");
	}

	/**
	 * ノードインスタンスに属するホスト名をすべて削除する。<BR>
	 * 
	 * @param node ホスト名を削除するノードインスタンス
	 * @throws RemoveException
	 */
	private static void deleteNodeHostname(NodeLocal node) throws RemoveException {
		m_log.debug("deleteNodeHostname() : facilityId = " + node.getFacilityId());

		/** ローカル変数 */
		String facilityId = null;
		String hostname = null;

		/** メイン処理 */
		if (node.getNodeHostname() != null) {
			for (NodeHostnameLocal nodeHostname : (Collection<NodeHostnameLocal>)node.getNodeHostname()) {
				try {
					facilityId = nodeHostname.getFacilityId();
					hostname = nodeHostname.getHostname();
					nodeHostname.remove();
					m_log.debug("successful in deleting a node's hostname. (facilityId = " + facilityId + ", hostname = " + hostname + ")");
				} catch (RemoveException e) {
					m_log.warn("failure to delete a node's hostname. (facilityId = " + facilityId + ", hostname = " + hostname + ")");
					throw e;
				}
			}
		}
	}

	/**
	 * ノードインスタンスに属するノード変数をすべて削除する。<BR>
	 * 
	 * @param node ノード変数を削除するノードインスタンス
	 * @throws RemoveException
	 */
	private static void deleteNodeVariable(NodeLocal node) throws RemoveException {
		m_log.debug("deleteNodeVariable() : facilityId = " + node.getFacilityId());

		/** ローカル変数 */
		String facilityId = null;
		String variableName = null;

		/** メイン処理 */
		if (node.getNodeVariable() != null) {
			for (NodeVariableLocal nodeVariable : (Collection<NodeVariableLocal>)node.getNodeVariable()) {
				try {
					facilityId = nodeVariable.getFacilityId();
					variableName = nodeVariable.getNodeVariableName();

					nodeVariable.remove();
					m_log.debug("successful in deleting a node's variable. (facilityId = " + facilityId + ", variableName = " + variableName + ")");
				} catch (RemoveException e) {
					m_log.warn("failure to delete a node's variable. (facilityId = " + facilityId + ", variableName = " + variableName + ")");
					throw e;
				}
			}
		}
	}

	/**
	 * ノードインスタンスに属する備考をすべて削除する。<BR>
	 * 
	 * @param node 備考を削除するノードインスタンス
	 * @throws RemoveException
	 */
	private static void deleteNodeNote(NodeLocal node) throws RemoveException {
		m_log.debug("deleteNodeNote() : facilityId = " + node.getFacilityId());

		/** ローカル変数 */
		String facilityId = null;
		Integer noteId = null;

		/** メイン処理 */
		if (node.getNodeNote() != null) {
			for (NodeNoteLocal nodeNote : (Collection<NodeNoteLocal>)node.getNodeNote()) {
				try {
					facilityId = node.getFacilityId();
					noteId = nodeNote.getNoteId();
					nodeNote.remove();
					m_log.debug("successful in deleting a node's note. (facilityId = " + facilityId + ", noteId = " + noteId + ")");
				} catch (RemoveException e) {
					m_log.warn("failure to delete a node's note. (facilityId = " + facilityId + ", noteId = " + noteId + ")");
					throw e;
				}
			}
		}
	}

	/**
	 * ノードインスタンスに属する汎用デバイスをすべて削除する。<BR>
	 * 
	 * @param node デバイスを削除するノードインスタンス
	 * @throws RemoveException
	 */
	private static void deleteNodeDevice(NodeLocal node) throws RemoveException {
		m_log.debug("deleteNodeDevice() : facilityId = " + node.getFacilityId());

		/** ローカル変数 */
		String facilityId = null;
		String deviceType = null;
		Integer deviceIndex = null;
		String deviceName = null;

		/** メイン処理 */
		if (node.getNodeDevice() != null) {
			for (NodeDeviceLocal nodeDevice : (Collection<NodeDeviceLocal>)node.getNodeDevice()) {
				try {
					// ログ表示用
					facilityId = nodeDevice.getFacilityId();
					deviceType = nodeDevice.getDeviceType();
					deviceIndex = nodeDevice.getDeviceIndex();
					deviceName = nodeDevice.getDeviceName();

					// デバイスの削除
					nodeDevice.remove();
					m_log.debug("successful in deleting a node's device. " +
							"(facilityId = " + facilityId + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
				} catch (RemoveException e) {
					m_log.warn("failure to delete a node's device. " +
							"(facilityId = " + facilityId + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
					throw e;
				}
			}
		}
	}

	/**
	 * ノードインスタンスに属するCPUをすべて削除する。<BR>
	 * 
	 * @param node デバイスを削除するノードインスタンス
	 * @throws RemoveException
	 */
	private static void deleteNodeCpu(NodeLocal node) throws RemoveException {
		m_log.debug("deleteNodeCpu() : facilityId = " + node.getFacilityId());

		/** ローカル変数 */
		String facilityId = null;
		String deviceType = null;
		Integer deviceIndex = null;
		String deviceName = null;

		/** メイン処理 */
		if (node.getNodeCpu() != null) {
			for (NodeCpuLocal nodeCpu : (Collection<NodeCpuLocal>)node.getNodeCpu()) {
				try {
					// ログ表示用
					facilityId = nodeCpu.getFacilityId();
					deviceType = nodeCpu.getDeviceType();
					deviceIndex = nodeCpu.getDeviceIndex();
					deviceName = nodeCpu.getDeviceName();

					// デバイスの削除
					nodeCpu.remove();
					m_log.debug("successful in deleting a node's cpu. " +
							"(facilityId = " + facilityId + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
				} catch (RemoveException e) {
					m_log.warn("failure to delete a node's cpu. " +
							"(facilityId = " + facilityId + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
					throw e;
				}
			}
		}
	}

	/**
	 * ノードインスタンスに属するメモリをすべて削除する。<BR>
	 * 
	 * @param node デバイスを削除するノードインスタンス
	 * @throws RemoveException
	 */
	private static void deleteNodeMemory(NodeLocal node) throws RemoveException {
		m_log.debug("deleteNodeMemory() : facilityId = " + node.getFacilityId());

		/** ローカル変数 */
		String facilityId = null;
		String deviceType = null;
		Integer deviceIndex = null;
		String deviceName = null;

		/** メイン処理 */
		if (node.getNodeMemory() != null) {
			for (NodeMemoryLocal nodeMemory : (Collection<NodeMemoryLocal>)node.getNodeMemory()) {
				try {
					// ログ表示用
					facilityId = nodeMemory.getFacilityId();
					deviceType = nodeMemory.getDeviceType();
					deviceIndex = nodeMemory.getDeviceIndex();
					deviceName = nodeMemory.getDeviceName();

					// デバイスの削除
					nodeMemory.remove();
					m_log.debug("successful in deleting a node's memory. " +
							"(facilityId = " + facilityId + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
				} catch (RemoveException e) {
					m_log.warn("failure to delete a node's memory. " +
							"(facilityId = " + facilityId + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
					throw e;
				}
			}
		}
	}

	/**
	 * ノードインスタンスに属するNICをすべて削除する。<BR>
	 * 
	 * @param node デバイスを削除するノードインスタンス
	 * @throws RemoveException
	 */
	private static void deleteNodeNetworkInterface(NodeLocal node) throws RemoveException {
		m_log.debug("deleteNodeNetworkInterface() : facilityId = " + node.getFacilityId());

		/** ローカル変数 */
		String facilityId = null;
		String deviceType = null;
		Integer deviceIndex = null;
		String deviceName = null;

		/** メイン処理 */
		if (node.getNodeNetworkInterface() != null) {
			for (NodeNetworkInterfaceLocal nodeNetworkInterface : (Collection<NodeNetworkInterfaceLocal>)node.getNodeNetworkInterface()) {
				try {
					// ログ表示用
					facilityId = nodeNetworkInterface.getFacilityId();
					deviceType = nodeNetworkInterface.getDeviceType();
					deviceIndex = nodeNetworkInterface.getDeviceIndex();
					deviceName = nodeNetworkInterface.getDeviceName();

					// デバイスの削除
					nodeNetworkInterface.remove();
					m_log.debug("successful in deleting a node's networkInterface. " +
							"(facilityId = " + facilityId + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
				} catch (RemoveException e) {
					m_log.warn("failure to delete a node's networkInterface. " +
							"(facilityId = " + facilityId + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
					throw e;
				}
			}
		}
	}

	/**
	 * ノードインスタンスに属するDISKをすべて削除する。<BR>
	 * 
	 * @param node デバイスを削除するノードインスタンス
	 * @throws RemoveException
	 */
	private static void deleteNodeDisk(NodeLocal node) throws RemoveException {
		m_log.debug("deleteNodeDisk() : facilityId = " + node.getFacilityId());

		/** ローカル変数 */
		String facilityId = null;
		String deviceType = null;
		Integer deviceIndex = null;
		String deviceName = null;

		/** メイン処理 */
		if (node.getNodeDisk() != null) {
			for (NodeDiskLocal nodeDisk : (Collection<NodeDiskLocal>)node.getNodeDisk()) {
				try {
					// ログ表示用
					facilityId = nodeDisk.getFacilityId();
					deviceType = nodeDisk.getDeviceType();
					deviceIndex = nodeDisk.getDeviceIndex();
					deviceName = nodeDisk.getDeviceName();

					// デバイスの削除
					nodeDisk.remove();
					m_log.debug("successful in deleting a node's disk. " +
							"(facilityId = " + facilityId + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
				} catch (RemoveException e) {
					m_log.warn("failure to delete a node's disk. " +
							"(facilityId = " + facilityId + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
					throw e;
				}
			}
		}
	}

	/**
	 * ノードインスタンスに属するファイルシステムをすべて削除する。<BR>
	 * 
	 * @param node ファイルシステムを削除するノードインスタンス
	 * @throws RemoveException
	 */
	private static void deleteNodeFilesystem(NodeLocal node) throws RemoveException {
		m_log.debug("deleteNodeFilesystem() : facilityId = " + node.getFacilityId());

		/** ローカル変数 */
		String facilityId = null;
		String deviceType = null;
		Integer deviceIndex = null;
		String deviceName = null;

		/** メイン処理 */
		if (node.getNodeFilesystem() != null) {
			for (NodeFilesystemLocal nodeFilesystem : (Collection<NodeFilesystemLocal>)node.getNodeFilesystem()) {
				try {
					// ログ表示用
					facilityId = nodeFilesystem.getFacilityId();
					deviceType = nodeFilesystem.getDeviceType();
					deviceIndex = nodeFilesystem.getDeviceIndex();
					deviceName = nodeFilesystem.getDeviceName();

					// デバイスの削除
					nodeFilesystem.remove();
					m_log.debug("successful in deleting a node's filesystem. " +
							"(facilityId = " + facilityId + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
				} catch (RemoveException e) {
					m_log.warn("failure to delete a node's filesystem. " +
							"(facilityId = " + facilityId + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
					throw e;
				}
			}
		}
	}

	/**
	 * スコープに属するノードを一括変更する。<BR>
	 * 
	 * @param property 変更後のノード情報
	 * @param parentFacilityId スコープのファシリティID
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 * @throws NamingException
	 * @throws CreateException
	 * @throws FinderException
	 */
	public static ArrayList<String> modifyAllNode(NodeInfo property, String parentFacilityId, String modifyUserId, boolean topicSendFlg) {
		/** ローカル変数 */
		FacilityLocal parentFacility = null;
		FacilityLocal childFacility = null;
		String facilityId = null;
		ArrayList<String> facilityIdList = new ArrayList<String>();

		/** メイン処理 */
		m_log.debug("modifing nodes under a scope...");

		try {
			// 親ファシリティを取得
			parentFacility = FacilityUtil.getLocalHome().findByPrimaryKey(parentFacilityId);

			// 親ファシリティがスコープかどうかを確認する
			if (! parentFacility.isScope()) {
				throw new EJBException("this parent's facility is not a scope. (facilityId = " + facilityId + ")");
			}

			// 親ファシリティに属するノードをすべて変更する
			if (parentFacility.getFacilityRelationAsParent() != null) {
				for (FacilityRelationLocal facilityRelation : (Collection<FacilityRelationLocal>)parentFacility.getFacilityRelationAsParent()) {
					childFacility = facilityRelation.getChildFacility();
					facilityId = childFacility.getFacilityId();

					// 子ファシリティがノードでない場合は変更しない
					if (! childFacility.isNode()) {
						continue;
					}

					// ファシリティ情報を更新する
					setFacility(childFacility, property, modifyUserId, true);

					// ノード情報を更新する（ホスト名、備考、デバイス情報、ファイルシステム情報を除く）
					setNode(childFacility.getNode(), property, true);
					facilityIdList.add(facilityId);
				}
			}
		} catch (Exception e) {
			m_log.warn("failure to modify a node. (facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to modify a node. (facilityId = " + facilityId + ")", e);
		}

		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}

		m_log.info("successful in modifing nodes under a scope. (parentFacilityId = " + parentFacilityId + ")");
		return facilityIdList;
	}

	/**
	 * スコープにノードを割り当てる。<BR>
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param facilityId ノードのファシリティID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 */
	private static void assignFacilityToScope(String parentFacilityId, String facilityId, boolean topicSendFlg) {
		/** ローカル変数 */
		String[] facilityIds = { facilityId };

		/** メイン処理 */
		assignFacilityToScope(parentFacilityId, facilityIds, topicSendFlg);
	}

	/**
	 * スコープにノードを割り当てる。<BR>
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param facilityIds ノードのファシリティID配列
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 */
	public static void assignFacilityToScope(String parentFacilityId, String[] facilityIds, boolean topicSendFlg) {
		/** ローカル変数 */
		FacilityLocal scope = null;
		boolean relationExist = false;
		String facilityId = null;

		/** メイン処理 */
		m_log.debug("assigning facilities to a scope...");

		try {
			// スコープのファシリティインスタンスの取得
			scope = FacilityUtil.getLocalHome().findByPrimaryKey(parentFacilityId);

			if (! scope.isScope()) {
				m_log.warn("parent's facility is not a scope. (parentfacilityId = " + parentFacilityId + ")");
				throw new EJBException("parent's facility is not a scope. (facilityId = " + parentFacilityId + ")");
			}

			for (int i = 0; i < facilityIds.length; i++) {
				facilityId = facilityIds[i];

				// 割り当て済みのファシリティかどうかを確認する
				relationExist = false;
				if (scope.getFacilityRelationAsParent() != null) {
					for (FacilityRelationLocal relation : (Collection<FacilityRelationLocal>)scope.getFacilityRelationAsParent()) {
						if (relation.getChildFacilityId().compareTo(facilityId) == 0) {
							relationExist = true;
							break;
						}
					}
				}
				if (relationExist) {
					m_log.info("skipped assinging a facility to a scope. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")");
				} else {
					FacilityRelationUtil.getLocalHome().create(parentFacilityId, facilityId);
					m_log.info("successful in assinging a facility to a scope. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")");
				}
			}
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId, parentFacilityId};
			apllog.put("SYS", "015", args);

			m_log.warn("failure to assign a node to a scope. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to assign a node to a scope. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")", e);
		}

		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}

		m_log.info("successful in assigning facilities to a scope.");
	}

	/**
	 * スコープからノードの割り当てを解除する。<BR>
	 * 
	 * @param parentFacilityId スコープのファシリティID
	 * @param facilityIds ノードのファシリティID
	 * @param modifyUserId 作業ユーザID
	 * @param topicSendFlg 更新を周知する場合はtrue, 周知しない場合はfalse
	 */
	public static void releaseNodeFromScope(String parentFacilityId, String[] facilityIds, String modifyUserId, boolean topicSendFlg) {
		/** ローカル変数 */
		FacilityLocal facility = null;
		String facilityId = null;

		/** メイン処理 */
		m_log.debug("releasing nodes from a scope...");

		try {
			// 該当するファシリティインスタンスを取得
			facility = FacilityUtil.getLocalHome().findByPrimaryKey(parentFacilityId);

			if (! facility.isScope()) {
				throw new EJBException("parent's facility is not a scope. (parentFacilityId = " + parentFacilityId + ")");
			}

			if (facility.getFacilityRelationAsParent() != null) {
				for (int i = 0; i < facilityIds.length; i++) {
					facilityId = facilityIds[i];
					for (FacilityRelationLocal relation : (Collection<FacilityRelationLocal>)facility.getFacilityRelationAsParent()) {
						if (relation.getChildFacility().getFacilityId().compareTo(facilityId) == 0) {
							relation.remove();
							m_log.info("successful in releaseing a node. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")");
							break;
						}
					}
				}
			}
		} catch (Exception e) {
			AplLogger apllog = new AplLogger("REP", "rep");
			String[] args = {facilityId, parentFacilityId};
			apllog.put("SYS", "016", args);

			m_log.warn("failure to release a node. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")", e);
			throw new EJBException("failure to release a node. (parentFacilityId = " + parentFacilityId + ", facilityId = " + facilityId + ")", e);
		}

		// ファシリティの変更を通知する
		if (topicSendFlg) {
			SendTopic.sendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
		}

		m_log.info("successful in releasing nodes from a scope.");
	}

	/**
	 * Propertyインスタンスに格納された値をファシリティインスタンスに格納する。<BR>
	 * 
	 * @param facility 格納先となるファシリティインスタンス
	 * @param property 格納する情報
	 * @param modifyUserId 作業ユーザID
	 * @param skipIfEmptyFlg trueにすると、Propertyインスタンスの各格納値がnullあるいは空文字の場合に格納しない
	 */
	private static void setFacility(FacilityLocal facility, FacilityInfo property, String modifyUserId, boolean skipIfEmptyFlg) {
		/** ローカル変数 */
		ArrayList inputValue = null;
		String facilityName = null;
		String description = null;
		Timestamp now = null;

		/** メイン処理 */
		// 現在日時を取得
		now = new Timestamp(new Date().getTime());

		if (facility.isScope()) {
			// 入力値（ファシリティ名）の格納
			facilityName = property.getFacilityName();

			// 入力値（説明）の格納
			description = property.getDescription();
		} else if (facility.isNode()) {
			// 入力値（ファシリティ名）の格納
			facilityName = property.getFacilityName();

			// 入力値（説明）の格納
			description = property.getDescription();
		} else {
			throw new EJBException("this facility's type is invalid. (facilityType = " + facility.getFacilityType() + ")");
		}

		// ファシリティインスタンスへの入力値の格納
		if ( ! (skipIfEmptyFlg && ObjectValidator.isEmptyString(facilityName)) ) {
			facility.setFacilityName(facilityName);
		}
		if ( ! (skipIfEmptyFlg && ObjectValidator.isEmptyString(description)) ) {
			facility.setDescription(description);
		}
		if (ObjectValidator.isEmptyString(facility.getCreateUserId())
				&& facility.getCreateDatetime() != null
				&& facility.getCreateDatetime().getTime() == 0) {
			facility.setCreateUserId(modifyUserId);
			facility.setCreateDatetime(now);
		}
		facility.setModifyUserId(modifyUserId);
		facility.setModifyDatetime(now);
	}

	/**
	 * Propertyインスタンス上のノード情報をノードインスタンスに格納する。<BR>
	 * 
	 * @param node 格納先となるノードインスタンス
	 * @param property ノード情報
	 * @param skipIfEmptyFlg trueにすると、Propertyインスタンスの各格納値がnullあるいは空文字の場合に格納しない
	 * @throws NamingException
	 * @throws CreateException
	 * @throws RemoveException
	 */
	private static void setNode(NodeLocal node, NodeInfo property, boolean skipIfEmptyFlg) throws NamingException, CreateException, RemoveException {
		m_log.debug("setNode() : facilityId = " + property.getFacilityId());

		/** ローカル変数 */

		// HW
		String platformFamily = null;
		String hardwareType = null;
		String iconImage = null;

		// IPアドレス
		Integer ipAddressVersion = null;
		String ipAddressV4 = null;
		String ipAddressV6 = null;

		// OS
		String nodeName = null;
		String osName = null;
		String osRelease = null;
		String osVersion = null;
		String characterSet = null;

		// SNMP
		Integer snmpPort = null;
		String snmpCommunity = null;
		String snmpVersion = null;
		Integer snmpTimeout = null;
		Integer snmpRetryCount = null;

		// WBEM
		String wbemUser = null;
		String wbemUserPassword = null;
		Integer wbemPort = null;
		String wbemProtocol = null;
		Integer wbemTimeout = null;
		Integer wbemRetryCount = null;

		// IPMI
		String ipmiIpAddress = null;
		Integer ipmiPort = null;
		String ipmiUser = null;
		String ipmiUserPassword = null;
		Integer ipmiTimeout = null;
		Integer ipmiRetries = null;
		String ipmiProtocol = null;
		String ipmiLevel = null;

		// WinRM
		String winrmUser = null;
		String winrmUserPassword = null;
		String winrmVersion = null;
		Integer winrmPort = null;
		String winrmProtocol = null;
		Integer winrmTimeout = null;
		Integer winrmRetries = null;

		// デバイス(主キー項目)
		Integer deviceIndex = null;
		String deviceType = null;
		String deviceName = null;
		NodeDeviceLocal nodeDevice = null;
		NodeCpuLocal nodeCpu = null;
		NodeMemoryLocal nodeMemory = null;
		NodeNetworkInterfaceLocal nodeNetworkInterface = null;
		NodeDiskLocal nodeDisk = null;
		NodeFilesystemLocal nodeFilesystem = null;

		// サーバ仮想化
		String virtualizationNodeType = null;
		String vmManagementNode = null;
		Integer vmIndex = null;
		String vmName = null;
		String virtualizationSolution = null;
		String vmId = null;
		String vmUser = null;
		String vmUserPassword = null;
		String vmProtocol = null;

		// ネットワーク仮想化
		String vNetSwitchType = null;
		String vNetHostNode = null;
		String openFlowDataPathId = null;
		String openFlowCtrlIpAddress = null;

		// ノード変数
		String nodeVariableName = null;
		NodeVariableLocal nodeVariable = null;

		// 保守
		String administrator = null;
		String contact = null;

		// 備考
		NodeNoteLocal nodeNote = null;
		int noteId = 0;


		/** メイン処理 */

		// HW
		platformFamily = property.getPlatformFamily();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(platformFamily))) {
			node.setPlatformFamily(platformFamily);
		}
		hardwareType = property.getHardwareType();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(hardwareType))) {
			node.setHardwareType(hardwareType);
		}
		iconImage = property.getIconImage();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(iconImage))) {
			node.setIconImage(iconImage);
		}

		// IPアドレス関連
		ipAddressVersion = property.getIpAddressVersion();
		if (! (skipIfEmptyFlg && ipAddressVersion == -1)) {
			node.setIpAddressVersion(ipAddressVersion);
		}
		ipAddressV4 = property.getIpAddressV4();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(ipAddressV4))) {
			node.setIpAddressV4(ipAddressV4);
		}
		ipAddressV6 = property.getIpAddressV6();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(ipAddressV6))) {
			node.setIpAddressV6(Ipv6Util.expand(ipAddressV6));
		}
		// ホスト名(複数項目)
		if (! skipIfEmptyFlg) {
			deleteNodeHostname(node);
			if (property.getNodeHostnameInfo() != null) {
				for (NodeHostnameInfo hostname : property.getNodeHostnameInfo()) {
					try {
						NodeHostnameUtil.getLocalHome().create(node.getFacilityId(), hostname.getHostname());
					} catch (CreateException e) {
						m_log.warn("failure to add a node's hostname. (facilityId = " + node.getFacilityId() + ", hostname = " + hostname + ")", e);
						throw e;
					}
				}
			}
		}

		// OS関連
		nodeName = property.getNodeName();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(nodeName))) {
			node.setNodeName(nodeName);
		}
		osName = property.getOsName();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(osName))) {
			node.setOsName(osName);
		}
		osRelease = property.getOsRelease();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(osRelease))) {
			node.setOsRelease(osRelease);
		}
		osVersion = property.getOsVersion();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(osVersion))) {
			node.setOsVersion(osVersion);
		}
		characterSet = property.getCharacterSet();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(characterSet))) {
			node.setCharacterSet(characterSet);
		}

		// SNMP関連
		snmpPort = property.getSnmpPort();
		if (! (skipIfEmptyFlg && snmpPort == -1)) {
			node.setSnmpPort(snmpPort);
		}
		snmpCommunity = property.getSnmpCommunity();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(snmpCommunity))) {
			node.setSnmpCommunity(snmpCommunity);
		}
		snmpVersion = property.getSnmpVersion();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(snmpVersion))) {
			node.setSnmpVersion(snmpVersion);
		}
		snmpTimeout = property.getSnmpTimeout();
		if (! (skipIfEmptyFlg && snmpTimeout == -1)) {
			node.setSnmpTimeout(snmpTimeout);
		}
		snmpRetryCount = property.getSnmpRetryCount();
		if (! (skipIfEmptyFlg && snmpRetryCount == -1)) {
			node.setSnmpRetryCount(snmpRetryCount);
		}

		// WBEM関連
		wbemUser = property.getWbemUser();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(wbemUser))) {
			node.setWbemUser(wbemUser);
		}
		wbemUserPassword = property.getWbemUserPassword();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(wbemUserPassword))) {
			node.setWbemUserPassword(wbemUserPassword);
		}
		wbemPort = property.getWbemPort();
		if (! (skipIfEmptyFlg && wbemPort == -1)) {
			node.setWbemPort(wbemPort);
		}
		wbemProtocol = property.getWbemProtocol();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(wbemProtocol))) {
			node.setWbemProtocol(wbemProtocol);
		}
		wbemTimeout = property.getWbemTimeout();
		if (! (skipIfEmptyFlg && wbemTimeout == -1)) {
			node.setWbemTimeout(wbemTimeout);
		}
		wbemRetryCount = property.getWbemRetryCount();
		if (! (skipIfEmptyFlg && wbemRetryCount == -1)) {
			node.setWbemRetryCount(wbemRetryCount);
		}

		// IPMI関連
		ipmiIpAddress = property.getIpmiIpAddress();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(ipmiIpAddress))) {
			node.setIpmiIpAddress(ipmiIpAddress);
		}
		ipmiPort = property.getIpmiPort();
		if (! (skipIfEmptyFlg && ipmiPort == -1)) {
			node.setIpmiPort(ipmiPort);
		}
		ipmiUser = property.getIpmiUser();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(ipmiUser))) {
			node.setIpmiUser(ipmiUser);
		}
		ipmiUserPassword = property.getIpmiUserPassword();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(ipmiUserPassword))) {
			node.setIpmiUserPassword(ipmiUserPassword);
		}
		ipmiTimeout = property.getIpmiTimeout();
		if (! (skipIfEmptyFlg && ipmiTimeout == -1)) {
			node.setIpmiTimeout(ipmiTimeout);
		}
		ipmiRetries = property.getIpmiRetries();
		if (! (skipIfEmptyFlg && ipmiRetries == -1)) {
			node.setIpmiRetryCount(ipmiRetries);
		}
		ipmiProtocol = property.getIpmiProtocol();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(ipmiProtocol))) {
			node.setIpmiProtocol(ipmiProtocol);
		}
		ipmiLevel = property.getIpmiLevel();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(ipmiLevel))) {
			node.setIpmiLevel(ipmiLevel);
		}

		// WinRM関連
		winrmUser = property.getWinrmUser();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(winrmUser))) {
			node.setWinrmUser(winrmUser);
		}
		winrmUserPassword = property.getWinrmUserPassword();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(winrmUserPassword))) {
			node.setWinrmUserPassword(winrmUserPassword);
		}
		winrmVersion = property.getWinrmVersion();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(winrmVersion))) {
			node.setWinrmVersion(winrmVersion);
		}
		winrmPort = property.getWinrmPort();
		if (! (skipIfEmptyFlg && winrmPort == -1)) {
			node.setWinrmPort(winrmPort);
		}
		winrmProtocol = property.getWinrmProtocol();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(winrmProtocol))) {
			node.setWinrmProtocol(winrmProtocol);
		}
		winrmTimeout = property.getWinrmTimeout();
		if (! (skipIfEmptyFlg && winrmTimeout == -1)) {
			node.setWinrmTimeout(winrmTimeout);
		}
		winrmRetries = property.getWinrmRetries();
		if (! (skipIfEmptyFlg && winrmRetries == -1)) {
			node.setWinrmRetryCount(winrmRetries);
		}

		// デバイス関連
		// 汎用デバイス関連
		if (! skipIfEmptyFlg) {

			// デバイス情報の削除
			deleteNodeDevice(node);

			if (property.getNodeDeviceInfo() != null) {
				for (NodeDeviceInfo deviceProperty : property.getNodeDeviceInfo()) {

					// 主キー項目(facilityId除く)
					deviceIndex = deviceProperty.getDeviceIndex();
					deviceType = deviceProperty.getDeviceType();
					deviceName = deviceProperty.getDeviceName();

					// 入力チェック
					if (deviceIndex != -1
							&& ! ObjectValidator.isEmptyString(deviceType)
							&& ! ObjectValidator.isEmptyString(deviceName)) {

						try {
							nodeDevice = NodeDeviceUtil.getLocalHome().create(
									node.getFacilityId(),
									deviceIndex,
									deviceType,
									deviceName);

							nodeDevice.setDeviceDisplayName(deviceProperty.getDeviceDisplayName());
							nodeDevice.setDeviceSize(deviceProperty.getDeviceSize());
							nodeDevice.setDeviceSizeUnit(deviceProperty.getDeviceSizeUnit());
							nodeDevice.setDeviceDescription(deviceProperty.getDeviceDescription());

						} catch (CreateException e) {
							m_log.warn("failure to add a node's device. " +
									"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")", e);
							throw e;
						}
					} else {
						throw new CreateException("both type and index of device are required. " +
								"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
					}
				}
			}
		}

		// CPUデバイス関連
		if (! skipIfEmptyFlg) {

			// CPUデバイス情報の削除
			deleteNodeCpu(node);

			if (property.getNodeCpuInfo() != null) {
				for (NodeCpuInfo cpuProperty : property.getNodeCpuInfo()) {

					// 主キー項目(facilityId除く)
					deviceIndex = cpuProperty.getDeviceIndex();
					deviceType = cpuProperty.getDeviceType();
					deviceName = cpuProperty.getDeviceName();

					// 入力チェック
					if (deviceIndex != -1
							&& ! ObjectValidator.isEmptyString(deviceType)
							&& ! ObjectValidator.isEmptyString(deviceName)) {

						try {
							nodeCpu = NodeCpuUtil.getLocalHome().create(
									node.getFacilityId(),
									deviceIndex,
									deviceType,
									deviceName);

							nodeCpu.setDeviceDisplayName(cpuProperty.getDeviceDisplayName());
							nodeCpu.setDeviceSize(cpuProperty.getDeviceSize());
							nodeCpu.setDeviceSizeUnit(cpuProperty.getDeviceSizeUnit());
							nodeCpu.setDeviceDescription(cpuProperty.getDeviceDescription());

						} catch (CreateException e) {
							m_log.warn("failure to add a node's cpu. " +
									"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")", e);
							throw e;
						}
					} else {
						throw new CreateException("both type and index of cpu are required. " +
								"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
					}
				}
			}
		}

		// MEMデバイス関連
		if (! skipIfEmptyFlg) {

			// MEMデバイス情報の削除
			deleteNodeMemory(node);

			if (property.getNodeMemoryInfo() != null) {
				for (NodeMemoryInfo memProperty : property.getNodeMemoryInfo()) {

					// 主キー項目(facilityId除く)
					deviceIndex = memProperty.getDeviceIndex();
					deviceType = memProperty.getDeviceType();
					deviceName = memProperty.getDeviceName();

					// 入力チェック
					if (deviceIndex != -1
							&& ! ObjectValidator.isEmptyString(deviceType)
							&& ! ObjectValidator.isEmptyString(deviceName)) {

						try {
							nodeMemory = NodeMemoryUtil.getLocalHome().create(
									node.getFacilityId(),
									deviceIndex,
									deviceType,
									deviceName);

							nodeMemory.setDeviceDisplayName(memProperty.getDeviceDisplayName());
							nodeMemory.setDeviceSize(memProperty.getDeviceSize());
							nodeMemory.setDeviceSizeUnit(memProperty.getDeviceSizeUnit());
							nodeMemory.setDeviceDescription(memProperty.getDeviceDescription());

						} catch (CreateException e) {
							m_log.warn("failure to add a node's memory. " +
									"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")", e);
							throw e;
						}
					} else {
						throw new CreateException("both type and index of memory are required. " +
								"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
					}
				}
			}
		}

		// NICデバイス関連
		if (! skipIfEmptyFlg) {

			// NICデバイス情報の削除
			deleteNodeNetworkInterface(node);

			if (property.getNodeNetworkInterfaceInfo() != null) {
				for (NodeNetworkInterfaceInfo nicProperty : property.getNodeNetworkInterfaceInfo()) {

					// 主キー項目(facilityId除く)
					deviceIndex = nicProperty.getDeviceIndex();
					deviceType = nicProperty.getDeviceType();
					deviceName = nicProperty.getDeviceName();

					// 入力チェック
					if (deviceIndex != -1
							&& ! ObjectValidator.isEmptyString(deviceType)
							&& ! ObjectValidator.isEmptyString(deviceName)) {

						try {
							nodeNetworkInterface = NodeNetworkInterfaceUtil.getLocalHome().create(
									node.getFacilityId(),
									deviceIndex,
									deviceType,
									deviceName);

							nodeNetworkInterface.setDeviceDisplayName(nicProperty.getDeviceDisplayName());
							nodeNetworkInterface.setDeviceSize(nicProperty.getDeviceSize());
							nodeNetworkInterface.setDeviceSizeUnit(nicProperty.getDeviceSizeUnit());
							nodeNetworkInterface.setDeviceDescription(nicProperty.getDeviceDescription());

							nodeNetworkInterface.setDeviceNicIpAddress(nicProperty.getNicIpAddress());
							nodeNetworkInterface.setDeviceNicMacAddress(nicProperty.getNicMacAddress());

						} catch (CreateException e) {
							m_log.warn("failure to add a node's nic. " +
									"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")", e);
							throw e;
						}
					} else {
						throw new CreateException("both type and index of nic are required. " +
								"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
					}
				}
			}
		}

		// DISKデバイス関連
		if (! skipIfEmptyFlg) {

			// DISKデバイス情報の削除
			deleteNodeDisk(node);

			if (property.getNodeDiskInfo() != null) {
				for (NodeDiskInfo diskProperty : property.getNodeDiskInfo()) {

					// 主キー項目(facilityId除く)
					deviceIndex = diskProperty.getDeviceIndex();
					deviceType = diskProperty.getDeviceType();
					deviceName = diskProperty.getDeviceName();

					// 入力チェック
					if (deviceIndex != -1
							&& ! ObjectValidator.isEmptyString(deviceType)
							&& ! ObjectValidator.isEmptyString(deviceName)) {

						try {
							nodeDisk = NodeDiskUtil.getLocalHome().create(
									node.getFacilityId(),
									deviceIndex,
									deviceType,
									deviceName);

							nodeDisk.setDeviceDisplayName(diskProperty.getDeviceDisplayName());
							nodeDisk.setDeviceSize(diskProperty.getDeviceSize());
							nodeDisk.setDeviceSizeUnit(diskProperty.getDeviceSizeUnit());
							nodeDisk.setDeviceDescription(diskProperty.getDeviceDescription());

							nodeDisk.setDeviceDiskRpm(diskProperty.getDiskRpm());

						} catch (CreateException e) {
							m_log.warn("failure to add a node's disk. " +
									"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")", e);
							throw e;
						}
					} else {
						throw new CreateException("both type and index of disk are required. " +
								"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
					}
				}
			}
		}

		// ファイルシステム関連
		if (! skipIfEmptyFlg) {

			// ファイルシステムデバイス情報の削除
			deleteNodeFilesystem(node);

			if (property.getNodeFilesystemInfo() != null) {
				for (NodeFilesystemInfo filesystemProperty : property.getNodeFilesystemInfo()) {

					// 主キー項目(facilityId除く)
					deviceIndex = filesystemProperty.getDeviceIndex();
					deviceType = filesystemProperty.getDeviceType();
					deviceName = filesystemProperty.getDeviceName();

					// 入力チェック
					if (deviceIndex != -1
							&& ! ObjectValidator.isEmptyString(deviceType)
							&& ! ObjectValidator.isEmptyString(deviceName)) {

						try {
							nodeFilesystem = NodeFilesystemUtil.getLocalHome().create(
									node.getFacilityId(),
									deviceIndex,
									deviceType,
									deviceName);

							nodeFilesystem.setDeviceDisplayName(filesystemProperty.getDeviceDisplayName());
							nodeFilesystem.setDeviceSize(filesystemProperty.getDeviceSize());
							nodeFilesystem.setDeviceSizeUnit(filesystemProperty.getDeviceSizeUnit());
							nodeFilesystem.setDeviceDescription(filesystemProperty.getDeviceDescription());

							nodeFilesystem.setDeviceFilesystemType(filesystemProperty.getFilesystemType());

						} catch (CreateException e) {
							m_log.warn("failure to add a node's filesystem. " +
									"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")", e);
							throw e;
						}
					} else {
						throw new CreateException("both type and index of filesystem are required. " +
								"(facilityId = " + node.getFacilityId() + ", deviceType = " + deviceType + ", deviceIndex = " + deviceIndex + ", deviceName = " + deviceName + ")");
					}
				}
			}
		}


		// サーバ仮想化関連
		virtualizationNodeType = property.getVirtualizationNodeType();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(virtualizationNodeType))) {
			node.setVirtualizationNodeType(virtualizationNodeType);
		}
		vmManagementNode = property.getVmManagementNode();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmManagementNode))) {
			node.setVmManagementNode(vmManagementNode);
		}
		vmIndex = property.getVmIndex();
		if (! (skipIfEmptyFlg && vmIndex == -1)) {
			node.setVmIndex(vmIndex);
		}
		vmName = property.getVmName();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmName))) {
			node.setVmName(vmName);
		}
		virtualizationSolution = property.getVirtualizationSolution();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(virtualizationSolution))) {
			node.setVirtualizationSolution(virtualizationSolution);
		}
		vmId = property.getVmId();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmId))) {
			node.setVmId(vmId);
		}
		vmUser = property.getVmUser();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmUser))) {
			node.setVmUser(vmUser);
		}
		vmUserPassword = property.getVmUserPassword();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmUserPassword))) {
			node.setVmUserPassword(vmUserPassword);
		}
		vmProtocol = property.getVmProtocol();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vmProtocol))) {
			node.setVmProtocol(vmProtocol);
		}

		// ネットワーク仮想化関連
		vNetSwitchType = property.getvNetSwitchType();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vNetSwitchType))) {
			node.setVNetSwitchType(vNetSwitchType);
		}
		vNetHostNode = property.getvNetHostNode();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(vNetHostNode))) {
			node.setVNetHostNode(vNetHostNode);
		}
		openFlowDataPathId = property.getOpenFlowDataPathId();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(openFlowDataPathId))) {
			node.setOpenFlowDatapathId(openFlowDataPathId);
		}
		openFlowCtrlIpAddress = property.getOpenFlowCtrlIpAddress();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(openFlowCtrlIpAddress))) {
			node.setOpenFlowCtrlIpAddress(openFlowCtrlIpAddress);
		}

		// ノード変数
		if (! skipIfEmptyFlg) {
			deleteNodeVariable(node);
			if (property.getNodeVariableInfo() != null) {
				for (NodeVariableInfo variable : property.getNodeVariableInfo()) {
					nodeVariableName = variable.getNodeVariableName();
					try {
						nodeVariable = NodeVariableUtil.getLocalHome().create(node.getFacilityId(), nodeVariableName);
						nodeVariable.setNodeVariableValue(variable.getNodeVariableValue());
					} catch (CreateException e) {
						m_log.warn("failure to add a node's variable. (facilityId = " + node.getFacilityId() + ", nodeVariableName = " + nodeVariableName + ")", e);
						throw e;
					}
				}
			}
		}

		// 管理情報関連
		administrator = property.getAdministrator();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(administrator))) {
			node.setAdministrator(administrator);
		}
		contact = property.getContact();
		if (! (skipIfEmptyFlg && ObjectValidator.isEmptyString(contact))) {
			node.setContact(contact);
		}

		// 備考
		if (! skipIfEmptyFlg) {
			deleteNodeNote(node);
			if (property.getNodeNoteInfo() != null) {
				for (NodeNoteInfo note : property.getNodeNoteInfo()) {
					try {
						nodeNote = NodeNoteUtil.getLocalHome().create(node.getFacilityId(), note.getNoteId());
						nodeNote.setNote(note.getNote());
					} catch (CreateException e) {
						m_log.warn("failure to add a node's note. (facilityId = " + node.getFacilityId() + ", noteId = " + noteId + ")", e);
						throw e;
					}
				}
			}
		}

	}

}
