/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.notify.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.NotifyDuplicate;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.fault.UsedFacility;
import com.clustercontrol.bean.PluginConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.commons.util.JmsPersistenceConfig;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.notify.bean.OutputBasicInfo;
import com.clustercontrol.notify.ejb.entity.MonitorStatusPK;
import com.clustercontrol.notify.ejb.entity.NotifyInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyJobInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyJobInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyLogEscalateInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyLogEscalateInfoUtil;
import com.clustercontrol.notify.factory.AddNotify;
import com.clustercontrol.notify.factory.AddNotifyRelation;
import com.clustercontrol.notify.factory.DeleteNotify;
import com.clustercontrol.notify.factory.DeleteNotifyRelation;
import com.clustercontrol.notify.factory.ModifyNotify;
import com.clustercontrol.notify.factory.ModifyNotifyRelation;
import com.clustercontrol.notify.factory.NotifyDispatcher;
import com.clustercontrol.notify.factory.SelectNotify;
import com.clustercontrol.notify.factory.SelectNotifyRelation;
import com.clustercontrol.notify.util.MonitorResultStatusUpdater;
import com.clustercontrol.notify.util.NotifyRelationCache;
import com.clustercontrol.notify.util.NotifyValidator;
import com.clustercontrol.repository.ejb.entity.FacilityUtil;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;

/**
 * 通知機能の管理を行う Session Bean です。<BR>
 * クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。
 *
 * @ejb.bean name="NotifyController"
 *           jndi-name="NotifyController"
 *           type="Stateless"
 *           transaction-type="Container"
 *           view-type="local"
 * 
 * @ejb.transaction type="Required"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyCommandInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyEventInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyJobInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyLogEscalateInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyMailInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyRelationInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyStatusInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=SyslogDuplicationInfoLocal"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=EventLogLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=EventMultiInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=StatusInfoLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 */
public abstract class NotifyControllerBean implements javax.ejb.SessionBean {
	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( NotifyControllerBean.class );

	/** コンテキスト情報。 */
	@SuppressWarnings("unused")
	private SessionContext m_context;

	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean の生成時に行う処理を実装します。
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	@Override
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}

	/**
	 * 通知情報を作成します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param info 作成対象の通知情報
	 * @return 作成に成功した場合、<code> true </code>
	 * @throws HinemosUnknown
	 * @throws NotifyDuplicate
	 * 
	 * @see com.clustercontrol.notify.factory.AddNotify#add(NotifyInfo)
	 */
	public boolean addNotify(NotifyInfo info) throws HinemosUnknown, NotifyDuplicate,InvalidSetting {

		Principal pri = m_context.getCallerPrincipal();

		// 通知情報を登録
		boolean flag;
		try {
			//入力チェック
			NotifyValidator.validateNotifyInfo(info);

			AddNotify notify = new AddNotify();
			flag = notify.add(info,pri.getName());
		} catch (NotifyDuplicate e) {
			m_log.error("addNotify() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		} catch (InvalidSetting e) {
			m_log.warn("addNotify() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e){
			m_log.error("addNotify() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}

		return  flag;
	}

	/**
	 * 通知情報を変更します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param info 変更対象の通知情報
	 * @return 変更に成功した場合、<code> true </code>
	 * @throws NotifyDuplicate
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.notify.factory.ModifyNotify#modify(NotifyInfo)
	 */
	public boolean modifyNotify(NotifyInfo info) throws NotifyDuplicate, HinemosUnknown,InvalidSetting {

		Principal pri = m_context.getCallerPrincipal();
		// 通知情報を更新
		boolean flag;
		try {
			//入力チェック
			NotifyValidator.validateNotifyInfo(info);

			ModifyNotify notify = new ModifyNotify();
			flag = notify.modify(info,pri.getName());
		} catch (NotifyDuplicate e) {
			m_log.error("modifyNotify() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		} catch (InvalidSetting e) {
			m_log.warn("modifyNotify() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e){
			m_log.error("modifyNotify() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return flag;
	}

	/**
	 * 通知情報を削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param notifyId 削除対象の通知ID
	 * @return 削除に成功した場合、<code> true </code>
	 * @throws NotifyDuplicate
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.notify.factory.DeleteNotify#delete(String)
	 */
	public boolean deleteNotify(String notifyId) throws NotifyNotFound, HinemosUnknown {

		// 通知情報を削除
		DeleteNotify notify = new DeleteNotify();
		boolean flag;
		try {
			flag = notify.delete(notifyId);
		} catch(NotifyNotFound e){
			m_log.error("deleteNotify() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e){
			m_log.error("deleteNotify() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return flag;
	}

	/**
	 * 引数で指定された通知情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @param notifyId 取得対象の通知ID
	 * @return 通知情報
	 * @throws NotifyNotFound
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.notify.factory.SelectNotify#getNotify(String)
	 */
	public NotifyInfo getNotify(String notifyId) throws NotifyNotFound, HinemosUnknown {

		// 通知情報を取得
		SelectNotify notify = new SelectNotify();
		NotifyInfo info = null;
		try {
			info = notify.getNotify(notifyId);
		} catch (NotifyNotFound e){
			throw e;
		} catch (Exception e) {
			m_log.error("getNotify(): " + e.getMessage(), e);
			throw new HinemosUnknown("getNotify() " + e.getMessage(), e);
		}

		return info;
	}

	/**
	 * 通知情報一覧を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @return 通知情報一覧（Objectの2次元配列）
	 * @throws NotifyNotFound
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.notify.factory.SelectNotify#getNotifyList()
	 */
	public ArrayList<NotifyInfo> getNotifyList() throws NotifyNotFound, HinemosUnknown {

		// 通知一覧を取得
		SelectNotify notify = new SelectNotify();
		ArrayList<NotifyInfo> list = null;
		try {
			list = notify.getNotifyList();
		}catch (NotifyNotFound e){
			throw e;
		} catch (Exception e) {
			m_log.error("getNotifyList(): " + e.getMessage(), e);
			throw new HinemosUnknown("getNotifyList() " + e.getMessage(), e);
		}

		return list;
	}

	/**
	 * 通知情報マップを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return 通知情報マップ
	 * @throws NotifyNotFound
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.notify.factory.SelectNotify#getNotifyMap()
	 */
	public HashMap<String, NotifyInfo> getNotifyMap() throws NotifyNotFound, HinemosUnknown {

		// 通知一覧を取得
		SelectNotify notify = new SelectNotify();
		HashMap<String, NotifyInfo> map = null;
		try {
			map = notify.getNotifyMap();
		} catch (NotifyNotFound e){
			throw e;
		} catch (Exception e) {
			m_log.error("getNotifyMap(): " + e.getMessage(), e);
			throw new HinemosUnknown("getNotifyMap(): " + e.getMessage(), e);
		}
		return map;
	}

	/**
	 * 障害検知用通知情報一覧を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @return 通知情報一覧（Objectの2次元配列）
	 * @throws NotifyNotFound
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.notify.factory.SelectNotify#getNotifyList()
	 */
	public ArrayList<NotifyInfo> getTroubleDetectionNotifyList() throws NotifyNotFound, HinemosUnknown {

		// 通知一覧を取得
		SelectNotify notify = new SelectNotify();
		ArrayList<NotifyInfo> list = null;
		try {
			list = notify.getTroubleDetectionNotifyList();
		} catch (NotifyNotFound e){
			throw e;
		} catch (Exception e) {
			m_log.error("getTroubleDetectionNotifyList(): " + e.getMessage(), e);
			throw new HinemosUnknown("getTroubleDetectionNotifyList(): " + e.getMessage(), e);
		}
		return list;
	}

	/**
	 * 通知グループに対応する通知を取得します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *  unchecked="true"
	 *     method-intf="local"
	 * 
	 * @param notifyGroupId  通知グループID
	 * @return 通知
	 * @throws NotifyNotFound
	 * @throws HinemosUnknown
	 * 
	 */
	public  ArrayList<NotifyRelationInfo> getNotifyRelation(String notifyGroupId) throws NotifyNotFound, HinemosUnknown {

		SelectNotifyRelation notify = new SelectNotifyRelation();
		ArrayList<NotifyRelationInfo> info = null;
		try {
			info = notify.getNotifyRelation(notifyGroupId);
		} catch (NotifyNotFound e){
			throw e;
		} catch (Exception e) {
			m_log.error("getNotifyRelation(String): " + e.getMessage(), e);
			throw new HinemosUnknown("getNotifyRelation(String): " + e.getMessage(), e);
		}
		return info;
	}


	/**
	 * 通知グループを変更します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *  unchecked="true"
	 *     method-intf="local"
	 * 
	 * @param info 通知のセット
	 * @param notifyGroupId 通知グループID
	 * @return 変更に成功した場合、<code> true </code>
	 * @throws NotifyNotFound
	 * @throws HinemosUnknown
	 * 
	 */
	public boolean modifyNotifyRelation(Collection<NotifyRelationInfo> info, String notifyGroupId) throws NotifyNotFound, HinemosUnknown {

		// システム通知情報を更新
		ModifyNotifyRelation notify = new ModifyNotifyRelation();
		boolean flag;
		try {
			flag = notify.modify(info, notifyGroupId);
		} catch (NotifyNotFound e){
			m_log.error("modifyNotifyRelation() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e){
			m_log.error("modifyNotifyRelation() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return flag;
	}

	/**
	 * 通知グループを削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *  unchecked="true"
	 *     method-intf="local"
	 * 
	 * @param notifyGroupId 通知グループID
	 * @return 削除に成功した場合、<code> true </code>
	 * @throws NotifyNotFound
	 * @throws HinemosUnknown
	 * 
	 */
	public boolean deleteNotifyRelation(String notifyGroupId) throws NotifyNotFound, HinemosUnknown{

		// システム通知情報を削除
		DeleteNotifyRelation notify = new DeleteNotifyRelation();
		boolean flag;
		try {
			flag = notify.delete(notifyGroupId);
		} catch (NotifyNotFound e){
			m_log.error("deleteNotifyRelation() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e){
			m_log.error("deleteNotifyRelation() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return flag;
	}

	/**
	 * 通知グループを作成します。
	 * 
	 * @ejb.interface-method
	 *
	 * @ejb.permission
	 *  unchecked="true"
	 *     method-intf="local"
	 * 
	 * @param info 通知グループ
	 * @return 作成に成功した場合、<code> true </code>
	 * @throws HinemosUnknown
	 * 
	 */
	public boolean addNotifyRelation(Collection<NotifyRelationInfo> info) throws HinemosUnknown{

		// システム通知情報を登録

		if(info != null){
			AddNotifyRelation notify = new AddNotifyRelation();
			boolean flag;
			try {
				flag = notify.add(info);
			} catch (Exception e){
				m_log.error("addNotifyRelation() : " + e.getClass().getSimpleName() +
						", " + e.getMessage(), e);
				m_context.setRollbackOnly();
				throw new HinemosUnknown(e.getMessage(), e);
			}
			return flag;
		}else{
			return true;
		}
	}

	/**
	 *　引数で指定した通知IDを利用している通知グループIDを取得する。
	 * 
	 * @ejb.interface-method
	 *
	 * @ejb.permission
	 *  unchecked="true"
	 *     method-intf="local"
	 * 
	 * @param notifyId
	 * @return　通知グループIDのリスト
	 * @throws NotifyNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<String> checkNotifyId(String notifyId) throws NotifyNotFound, HinemosUnknown{
		ArrayList<String> ret = null;

		SelectNotifyRelation notify = new SelectNotifyRelation();
		try {
			ret = notify.getNotifyGroupIdBaseOnNotifyId(notifyId);
		} catch (NotifyNotFound e){
			m_log.error("checkNotifyId() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e){
			m_log.error("checkNotifyId() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return ret;
	}

	/**
	 *　指定した通知IDを有効化/無効化する。
	 * 
	 * @ejb.interface-method
	 *
	 * @ejb.permission
	 *  unchecked="true"
	 *     method-intf="local"
	 * 
	 * @param notifyId
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws NotifyDuplicate
	 */
	public void setNotifyStatus(String notifyId, boolean validFlag) throws HinemosUnknown, NotifyNotFound, NotifyDuplicate{

		// null check
		if(notifyId == null || "".equals(notifyId)){
			throw new HinemosUnknown("target notifyId is null or empty.");
		}

		NotifyInfo info = getNotify(notifyId);
		try{
			if (validFlag) {
				// 通知設定の有効化
				if(info.getValidFlg().intValue() != ValidConstant.TYPE_VALID){
					info.setValidFlg(ValidConstant.TYPE_VALID);
					modifyNotify(info);
				}
			} else {
				// 通知設定の無効化
				if(info.getValidFlg().intValue() != ValidConstant.TYPE_INVALID){
					info.setValidFlg(ValidConstant.TYPE_INVALID);
					modifyNotify(info);
				}
			}
		} catch (InvalidSetting e) {
			m_log.warn("Unknown InvalidSettingException",e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * 指定したファシリティIDが利用されているか確認する
	 * 
	 * @ejb.interface-method
	 *
	 * @ejb.permission
	 *  unchecked="true"
	 *     method-intf="local"
	 * 
	 */
	public void isUseFacilityId(String facilityId) throws UsedFacility {
		try {
			Collection<NotifyJobInfoLocal> listJ = NotifyJobInfoUtil.getLocalHome().findAll();
			for (NotifyJobInfoLocal local : listJ) {
				if (facilityId.equals(local.getJobExecFacility())) {
					throw new UsedFacility(PluginConstant.TYPE_NOTIFY);
				}
			}
			Collection<NotifyLogEscalateInfoLocal> listL = NotifyLogEscalateInfoUtil.getLocalHome().findAll();
			for (NotifyLogEscalateInfoLocal local : listL) {
				if (facilityId.equals(local.getEscalateFacility())) {
					throw new UsedFacility(PluginConstant.TYPE_NOTIFY);
				}
			}
		} catch (FinderException e) {
			m_log.warn("isUseFacilityId() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
		} catch (NamingException e) {
			m_log.warn("isUseFacilityId() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
		}
	}


	/**
	 * 外部から直接通知処理を実行します。
	 * 
	 * @ejb.interface-method
	 *
	 * @ejb.permission
	 *  unchecked="true"
	 *  method-intf="local"
	 * 
	 * @param pluginId プラグインID
	 * @param monitorId 監視項目ID
	 * @param facilityId ファシリティID
	 * @param subKey 抑制用のサブキー（任意の文字列）
	 * @param generationDate 出力日時（エポック秒）
	 * @param priority 重要度
	 * @param application アプリケーション
	 * @param messageId メッセージID
	 * @param message メッセージ
	 * @param messageOrg オリジナルメッセージ
	 * @param notifyIdList 通知IDのリスト
	 * @param srcId 送信元を特定するためのID
	 * @throws FacilityNotFound
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 */
	public void notify(
			String pluginId,
			String monitorId,
			String facilityId,
			String subKey,
			long generationDate,
			int priority,
			String application,
			String messageId,
			String message,
			String messageOrg,
			ArrayList<String> notifyIdList,
			String srcId) throws FacilityNotFound, HinemosUnknown, NotifyNotFound{
		m_log.info("notify() "
				+ "pluginId = " + pluginId
				+ ", monitorId = " + monitorId
				+ ", facilityId = " + facilityId
				+ ", subKey = " + subKey
				+ ", generationDate = " + generationDate
				+ ", priority = " + priority
				+ ", application = " + application
				+ ", messageId = " + messageId
				+ ", message = " + message
				+ ", messageOrg = " + messageOrg
				+ ", srcId = " + srcId
		);

		// パラメータのnullチェック
		if(pluginId == null){
			m_log.error("notify() Invalid argument. pluginId is null.");
			return;
		} else if(monitorId == null){
			m_log.error("notify() Invalid argument. monitorId is null.");
			return;
		} else if(facilityId == null){
			m_log.error("notify() Invalid argument. facilityId is null.");
			return;
		} else if(application == null){
			m_log.error("notify() Invalid argument. application is null.");
			return;
		} else if(messageId == null){
			m_log.error("notify() Invalid argument. messageId is null.");
			return;
		} else if(message == null){
			m_log.error("notify() Invalid argument. message is null.");
			return;
		} else if(messageOrg == null){
			m_log.error("notify() Invalid argument. messageOrg is null.");
			return;
		} else if(notifyIdList == null){
			m_log.error("notify() Invalid argument. notifyIdList is null.");
			return;
		}

		if(subKey == null){
			// エラーとして扱わず空文字を設定する。
			subKey = "";
		}

		// 指定のファシリティIDが存在するか確認
		try {
			FacilityUtil.getLocalHome().findByPrimaryKey(facilityId);
		} catch (FinderException e) {
			m_log.error("notify() facility not found. (facilityId = " + facilityId + ")", e);
			throw new FacilityNotFound("facility not found. (facilityId = " + facilityId + ")", e);
		} catch (NamingException e) {
			m_log.error("notify() " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

		// 指定の通知設定が存在するか確認
		ArrayList<String> confirmedNotifyIdList = new ArrayList<String>();
		for(String notifyId : notifyIdList){
			try {
				NotifyInfoUtil.getLocalHome().findByPrimaryKey(notifyId);
				confirmedNotifyIdList.add(notifyId);
			} catch (FinderException e) {
				m_log.error("notify() NotifyInfo not found. (notifyId=" + notifyId + ")", e);
				throw new NotifyNotFound("NotifyInfo not found. (notifyId=" + notifyId + ")", e);
			} catch (NamingException e) {
				m_log.error("notify() " + e.getMessage(), e);
				throw new HinemosUnknown(e.getMessage(), e);
			}
		}

		OutputBasicInfo output = new OutputBasicInfo();
		output.setApplication(application);
		output.setFacilityId(facilityId);
		try {
			RepositoryControllerLocal repositoryCtrl = RepositoryControllerUtil.getLocalHome().create();
			String facilityPath = repositoryCtrl.getFacilityPath(facilityId, null);
			output.setScopeText(facilityPath);
		} catch (CreateException e) {
			m_log.warn("notify() " + e.getMessage(), e);
			// ファシリティIDをファシリティパスとする
			output.setScopeText(facilityId);
		} catch (NamingException e) {
			m_log.warn("notify() " + e.getMessage(), e);
			// ファシリティIDをファシリティパスとする
			output.setScopeText(facilityId);
		}
		output.setGenerationDate(generationDate);
		output.setMessage(message);
		output.setMessageId(messageId);
		output.setMessageOrg(messageOrg);
		output.setMonitorId(monitorId);
		output.setMultiId(srcId);
		output.setPluginId(pluginId);
		output.setPriority(priority);
		output.setSubKey(subKey);

		try {
			notify(output, confirmedNotifyIdList);
		} catch (Throwable e){
			// 想定外のエラーが発生した場合はキャッシュをクリアする
			m_log.error(e.getMessage(), e);
			
			MonitorStatusPK pk = new MonitorStatusPK(output.getFacilityId(), output.getPluginId(), output.getMonitorId(), output.getSubKey());
			MonitorResultStatusUpdater.removeCache(pk);
		}
	}

	/**
	 * 外部から直接通知処理を実行します。
	 * 
	 * @ejb.interface-method
	 *
	 * @ejb.permission
	 *  unchecked="true"
	 *  method-intf="local"
	 * 
	 * @param notifyInfo 通知情報
	 */
	public void notify(OutputBasicInfo notifyInfo, String notifyGroupId){
		// 監視設定から通知IDのリストを取得する
		List<String> notifyIdList = NotifyRelationCache.getNotifyIdList(notifyGroupId);

		// 該当の通知IDのリストがない場合は終了する
		if(notifyIdList == null || notifyIdList.size() <= 0){
			return;
		}

		try {
			// 通知処理を行う
			notify(notifyInfo, notifyIdList);
		} catch (Throwable e){
			// 想定外のエラーが発生した場合はキャッシュをクリアする
			m_log.error(e.getMessage(), e);
			
			MonitorStatusPK pk = new MonitorStatusPK(notifyInfo.getFacilityId(), notifyInfo.getPluginId(), notifyInfo.getMonitorId(), notifyInfo.getSubKey());
			MonitorResultStatusUpdater.removeCache(pk);
		}
	}

	private void notify(OutputBasicInfo notifyInfo, List<String> notifyIdList){
		// 機能毎に設定されているJMSの永続化モードを取得
		int persistMode = JmsPersistenceConfig.getJmsPersistence(notifyInfo.getPluginId());

		// 通知キューへの登録処理を実行
		NotifyDispatcher.notifyAction(notifyInfo, notifyIdList, persistMode);
	}
}
