/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.monitor.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import javax.ejb.EJBException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.naming.InitialContext;
import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.MonitorDuplicate;
import com.clustercontrol.fault.MonitorNotFound;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.custom.factory.AddCustom;
import com.clustercontrol.custom.factory.DeleteCustom;
import com.clustercontrol.custom.factory.ModifyCustom;
import com.clustercontrol.custom.factory.SelectCustom;
import com.clustercontrol.hinemosagent.factory.AddMonitorAgent;
import com.clustercontrol.hinemosagent.factory.DeleteMonitorAgent;
import com.clustercontrol.hinemosagent.factory.ModifyMonitorAgent;
import com.clustercontrol.hinemosagent.factory.SelectMonitorAgent;
import com.clustercontrol.http.factory.AddMonitorHttp;
import com.clustercontrol.http.factory.AddMonitorHttpString;
import com.clustercontrol.http.factory.DeleteMonitorHttp;
import com.clustercontrol.http.factory.ModifyMonitorHttp;
import com.clustercontrol.http.factory.ModifyMonitorHttpString;
import com.clustercontrol.http.factory.SelectMonitorHttp;
import com.clustercontrol.logfile.ejb.session.MonitorLogfileControllerBean;
import com.clustercontrol.logfile.factory.AddMonitorLogfileString;
import com.clustercontrol.logfile.factory.DeleteMonitorLogfile;
import com.clustercontrol.logfile.factory.ModifyMonitorLogfileString;
import com.clustercontrol.logfile.factory.SelectMonitorLogfile;
import com.clustercontrol.monitor.bean.MonitorFilterInfo;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.factory.AddMonitor;
import com.clustercontrol.monitor.run.factory.DeleteMonitor;
import com.clustercontrol.monitor.run.factory.ModifyMonitor;
import com.clustercontrol.monitor.run.factory.SelectMonitor;
import com.clustercontrol.monitor.run.util.MonitorValidator;
import com.clustercontrol.performance.monitor.factory.AddMonitorPerformance;
import com.clustercontrol.performance.monitor.factory.DeleteMonitorPerformance;
import com.clustercontrol.performance.monitor.factory.ModifyMonitorPerformance;
import com.clustercontrol.performance.monitor.factory.SelectMonitorPerformance;
import com.clustercontrol.ping.factory.AddMonitorPing;
import com.clustercontrol.ping.factory.DeleteMonitorPing;
import com.clustercontrol.ping.factory.ModifyMonitorPing;
import com.clustercontrol.ping.factory.SelectMonitorPing;
import com.clustercontrol.port.factory.AddMonitorPort;
import com.clustercontrol.port.factory.DeleteMonitorPort;
import com.clustercontrol.port.factory.ModifyMonitorPort;
import com.clustercontrol.port.factory.SelectMonitorPort;
import com.clustercontrol.process.factory.AddMonitorProcess;
import com.clustercontrol.process.factory.DeleteMonitorProcess;
import com.clustercontrol.process.factory.ModifyMonitorProcess;
import com.clustercontrol.process.factory.SelectMonitorProcess;
import com.clustercontrol.snmp.factory.AddMonitorSnmp;
import com.clustercontrol.snmp.factory.AddMonitorSnmpString;
import com.clustercontrol.snmp.factory.DeleteMonitorSnmp;
import com.clustercontrol.snmp.factory.ModifyMonitorSnmp;
import com.clustercontrol.snmp.factory.ModifyMonitorSnmpString;
import com.clustercontrol.snmp.factory.SelectMonitorSnmp;
import com.clustercontrol.snmptrap.factory.AddMonitorTrap;
import com.clustercontrol.snmptrap.factory.DeleteMonitorTrap;
import com.clustercontrol.snmptrap.factory.ModifyMonitorTrap;
import com.clustercontrol.snmptrap.factory.SelectMonitorTrap;
import com.clustercontrol.sql.factory.AddMonitorSql;
import com.clustercontrol.sql.factory.AddMonitorSqlString;
import com.clustercontrol.sql.factory.DeleteMonitorSql;
import com.clustercontrol.sql.factory.ModifyMonitorSql;
import com.clustercontrol.sql.factory.ModifyMonitorSqlString;
import com.clustercontrol.sql.factory.SelectMonitorSql;
import com.clustercontrol.systemlog.factory.AddMonitorSystemlogString;
import com.clustercontrol.systemlog.factory.DeleteMonitorSystemlog;
import com.clustercontrol.systemlog.factory.ModifyMonitorSystemlogString;
import com.clustercontrol.systemlog.factory.SelectMonitorSystemlog;
import com.clustercontrol.systemlog.util.SystemlogCache;
import com.clustercontrol.winservice.factory.AddMonitorWinService;
import com.clustercontrol.winservice.factory.DeleteMonitorWinService;
import com.clustercontrol.winservice.factory.ModifyMonitorWinService;
import com.clustercontrol.winservice.factory.SelectMonitorWinService;

/**
 * 監視設定を制御するSesison Bean<BR>
 * 
 * @ejb.bean name="MonitorSettingController"
 *           jndi-name="MonitorSettingController"
 *           type="Stateless"
 *           transaction-type="Container"
 *           view-type="local"
 * 
 * @ejb.transaction type="Required"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorInfoLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 */
public abstract class MonitorSettingControllerBean implements SessionBean {

	// Logger
	private static Log m_log = LogFactory.getLog( MonitorSettingControllerBean.class );

	// SessionContext
	@SuppressWarnings("unused")
	private SessionContext m_context;

	/**
	 * SessionContextの設定
	 */
	@Override
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}

	/**
	 * 監視設定情報をマネージャに登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param info
	 * @return
	 * @throws MonitorDuplicate
	 * @throws HinemosUnknown
	 */
	public boolean addMonitor(MonitorInfo info) throws MonitorDuplicate, HinemosUnknown, InvalidSetting{
		m_log.debug("addMonitor()");

		//入力チェック
		try{
			MonitorValidator.validateMonitorInfo(info);
		} catch (InvalidSetting e) {
			m_log.warn("addMonitor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		} catch (HinemosUnknown e) {
			m_log.warn("addMonitor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		} catch (Exception e) {
			m_log.warn("addMonitor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}

		Principal pri = m_context.getCallerPrincipal();
		
		AddMonitor addMonitor = null;
		String monitorTypeId = info.getMonitorTypeId();
		int monitorType = info.getMonitorType();
		m_log.debug("addMonitor() monitorTypeId = " + monitorTypeId + ", monitorType = " + monitorType);

		// Hinemos エージェント監視
		if(HinemosModuleConstant.MONITOR_AGENT.equals(monitorTypeId)){
			addMonitor = new AddMonitorAgent();
		}
		// HTTP 監視
		else if(HinemosModuleConstant.MONITOR_HTTP.equals(monitorTypeId)){
			// 数値監視
			if(MonitorTypeConstant.TYPE_NUMERIC == monitorType){
				addMonitor = new AddMonitorHttp();
			}
			// 文字列
			else if(MonitorTypeConstant.TYPE_STRING == monitorType){
				addMonitor = new AddMonitorHttpString();
			}
			// それ以外
			else{
				throw new HinemosUnknown("This monitorTypeId = " + monitorTypeId + ", but unknown monitorType = " + monitorType);
			}
		}
		// ログファイル 監視
		else if(HinemosModuleConstant.MONITOR_LOGFILE.equals(monitorTypeId)){
			addMonitor =  new AddMonitorLogfileString();
		}
		// リソース 監視
		else if(HinemosModuleConstant.MONITOR_PERFORMANCE.equals(monitorTypeId)){
			addMonitor = new AddMonitorPerformance();
		}
		// ping監視
		else if(HinemosModuleConstant.MONITOR_PING.equals(monitorTypeId)){
			addMonitor = new AddMonitorPing();
		}
		// ポート監視
		else if(HinemosModuleConstant.MONITOR_PORT.equals(monitorTypeId)){
			addMonitor = new AddMonitorPort();
		}
		// プロセス監視
		else if(HinemosModuleConstant.MONITOR_PROCESS.equals(monitorTypeId)){
			addMonitor = new AddMonitorProcess();
		}
		// SNMPTRAP監視
		else if(HinemosModuleConstant.MONITOR_SNMPTRAP.equals(monitorTypeId)){
			addMonitor = new AddMonitorTrap();
		}
		// SNMP監視
		else if(HinemosModuleConstant.MONITOR_SNMP.equals(monitorTypeId)){
			// 数値監視
			if(MonitorTypeConstant.TYPE_NUMERIC == monitorType){
				addMonitor = new AddMonitorSnmp();
			}
			// 文字列
			else if(MonitorTypeConstant.TYPE_STRING == monitorType){
				addMonitor = new AddMonitorSnmpString();
			}
			// それ以外
			else{
				throw new HinemosUnknown("This monitorTypeId = " + monitorTypeId + ", but unknown monitorType = " + monitorType);
			}
		}
		// システムログ監視
		else if(HinemosModuleConstant.MONITOR_SYSTEMLOG.equals(monitorTypeId)){
			addMonitor = new AddMonitorSystemlogString();
		}
		// SQL監視
		else if(HinemosModuleConstant.MONITOR_SQL.equals(monitorTypeId)){
			// 数値監視
			if(MonitorTypeConstant.TYPE_NUMERIC == monitorType){
				addMonitor = new AddMonitorSql();
			}
			// 文字列
			else if(MonitorTypeConstant.TYPE_STRING == monitorType){
				addMonitor = new AddMonitorSqlString();
			}
			// それ以外
			else{
				throw new HinemosUnknown("This monitorTypeId = " + monitorTypeId + ", but unknown monitorType = " + monitorType);
			}
		}
		// コマンド監視
		else if(HinemosModuleConstant.MONITOR_CUSTOM.equals(monitorTypeId)){
			addMonitor = new AddCustom();
		}
		// Windowsサービス監視
		else if(HinemosModuleConstant.MONITOR_WINSERVICE.equals(monitorTypeId)){
			addMonitor = new AddMonitorWinService();
		}
		// Other
		else{
			throw new HinemosUnknown("Unknown monitorTypeId = " + HinemosModuleConstant.MONITOR_HTTP);
		}

		TransactionManager tm = null;
		boolean flag = false;
		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();
			
			flag = addMonitor.add(info, pri.getName());
		} catch (MonitorDuplicate e) {
			m_log.warn("addMonitor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		} catch (Exception e) {
			m_log.error("addMonitor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (flag) {
						tm.commit();
						if (HinemosModuleConstant.MONITOR_SYSTEMLOG.equals(monitorTypeId)) {
							SystemlogCache.clear();
						} else if (HinemosModuleConstant.MONITOR_CUSTOM.equals(monitorTypeId)) {
							SelectCustom.clearCustomCache();
						} else if (HinemosModuleConstant.MONITOR_LOGFILE.equals(monitorTypeId)) {
							MonitorLogfileControllerBean.clearLogfileCache();
						}
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new EJBException(e.getMessage(), e);
				}
			}
		}

		return flag;
	}


	/**
	 * 監視設定情報を更新します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param info
	 * @return
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public boolean modifyMonitor(MonitorInfo info) throws MonitorNotFound, HinemosUnknown, InvalidSetting {
		m_log.debug("modifyMonitor()");

		//入力チェック
		try{
			MonitorValidator.validateMonitorInfo(info);
		} catch (InvalidSetting e) {
			m_log.warn("addMonitor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		} catch (HinemosUnknown e) {
			m_log.warn("modifyMonitor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		} catch (Exception e) {
			m_log.warn("modifyMonitor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw new HinemosUnknown(e.getMessage(), e);
		}

		Principal pri = m_context.getCallerPrincipal();

		ModifyMonitor modMonitor = null;
		String monitorTypeId = info.getMonitorTypeId();
		int monitorType = info.getMonitorType();
		m_log.debug("modifyMonitor() monitorTypeId = " + monitorTypeId + ", monitorType = " + monitorType);


		// Hinemos エージェント監視
		if(HinemosModuleConstant.MONITOR_AGENT.equals(monitorTypeId)){
			modMonitor = new ModifyMonitorAgent();
		}
		// HTTP 監視
		else if(HinemosModuleConstant.MONITOR_HTTP.equals(monitorTypeId)){
			// 数値監視
			if(MonitorTypeConstant.TYPE_NUMERIC == monitorType){
				modMonitor = new ModifyMonitorHttp();
			}
			// 文字列
			else if(MonitorTypeConstant.TYPE_STRING == monitorType){
				modMonitor = new ModifyMonitorHttpString();
			}
			// それ以外
			else{
				throw new HinemosUnknown("This monitorTypeId = " + monitorTypeId + ", but unknown monitorType = " + monitorType);
			}
		}
		// ログファイル 監視
		else if(HinemosModuleConstant.MONITOR_LOGFILE.equals(monitorTypeId)){
			modMonitor =  new ModifyMonitorLogfileString();
		}
		// リソース 監視
		else if(HinemosModuleConstant.MONITOR_PERFORMANCE.equals(monitorTypeId)){
			modMonitor = new ModifyMonitorPerformance();
		}
		// ping監視
		else if(HinemosModuleConstant.MONITOR_PING.equals(monitorTypeId)){
			modMonitor = new ModifyMonitorPing();
		}
		// ポート監視
		else if(HinemosModuleConstant.MONITOR_PORT.equals(monitorTypeId)){
			modMonitor = new ModifyMonitorPort();
		}
		// プロセス監視
		else if(HinemosModuleConstant.MONITOR_PROCESS.equals(monitorTypeId)){
			modMonitor = new ModifyMonitorProcess();
		}
		// SNMPTRAP監視
		else if(HinemosModuleConstant.MONITOR_SNMPTRAP.equals(monitorTypeId)){
			modMonitor = new ModifyMonitorTrap();
		}
		// SNMP監視
		else if(HinemosModuleConstant.MONITOR_SNMP.equals(monitorTypeId)){
			// 数値監視
			if(MonitorTypeConstant.TYPE_NUMERIC == monitorType){
				modMonitor = new ModifyMonitorSnmp();
			}
			// 文字列
			else if(MonitorTypeConstant.TYPE_STRING == monitorType){
				modMonitor = new ModifyMonitorSnmpString();
			}
			// それ以外
			else{
				throw new HinemosUnknown("This monitorTypeId = " + monitorTypeId + ", but unknown monitorType = " + monitorType);
			}
		}
		// システムログ監視
		else if(HinemosModuleConstant.MONITOR_SYSTEMLOG.equals(monitorTypeId)){
			modMonitor = new ModifyMonitorSystemlogString();
		}
		// SQL監視
		else if(HinemosModuleConstant.MONITOR_SQL.equals(monitorTypeId)){
			// 数値監視
			if(MonitorTypeConstant.TYPE_NUMERIC == monitorType){
				modMonitor = new ModifyMonitorSql();
			}
			// 文字列
			else if(MonitorTypeConstant.TYPE_STRING == monitorType){
				modMonitor = new ModifyMonitorSqlString();
			}
			// それ以外
			else{
				throw new HinemosUnknown("This monitorTypeId = " + monitorTypeId + ", but unknown monitorType = " + monitorType);
			}
		}
		// コマンド監視
		else if(HinemosModuleConstant.MONITOR_CUSTOM.equals(monitorTypeId)){
			modMonitor = new ModifyCustom();
		}
		// Windowsサービス監視
		else if(HinemosModuleConstant.MONITOR_WINSERVICE.equals(monitorTypeId)){
			modMonitor = new ModifyMonitorWinService();
		}
		// Other
		else{
			throw new HinemosUnknown("Unknown monitorTypeId = " + monitorTypeId);
		}

		// 監視設定情報を更新
		TransactionManager tm = null;
		boolean flag = false;
		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();
			
			flag = modMonitor.modify(info, pri.getName());
		} catch (MonitorNotFound e) {
			m_log.warn("modifyMonitor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		} catch (Exception e) {
			m_log.error("modifyMonitor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (flag) {
						tm.commit();
						if (HinemosModuleConstant.MONITOR_SYSTEMLOG.equals(monitorTypeId)) {
							SystemlogCache.clear();
						} else if (HinemosModuleConstant.MONITOR_CUSTOM.equals(monitorTypeId)) {
							SelectCustom.clearCustomCache();
						} else if (HinemosModuleConstant.MONITOR_LOGFILE.equals(monitorTypeId)) {
							MonitorLogfileControllerBean.clearLogfileCache();
						}
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new EJBException(e.getMessage(), e);
				}
			}
		}
		return flag;
	}


	/**
	 * 
	 * 監視設定情報をマネージャから削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param monitorId
	 * @param monitorTypeId
	 * @return
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public boolean deleteMonitor(String monitorId, String monitorTypeId) throws MonitorNotFound, HinemosUnknown {
		m_log.debug("deleteMonitor() monitorId = " + monitorId + ", monitorTypeId = " + monitorTypeId);

		// null チェック
		if(monitorId == null || "".equals(monitorId)){
			throw new HinemosUnknown("monitorId is null or empty.");
		}
		if(monitorTypeId == null || "".equals(monitorTypeId)){
			throw new HinemosUnknown("monitorTypeId is null or empty.");
		}

		DeleteMonitor deleteMonitor = null;
		// Hinemos エージェント監視
		if(HinemosModuleConstant.MONITOR_AGENT.equals(monitorTypeId)){
			deleteMonitor = new DeleteMonitorAgent();
		}
		// HTTP 監視
		else if(HinemosModuleConstant.MONITOR_HTTP.equals(monitorTypeId)){
			deleteMonitor = new DeleteMonitorHttp();
		}
		// ログファイル 監視
		else if(HinemosModuleConstant.MONITOR_LOGFILE.equals(monitorTypeId)){
			deleteMonitor =  new DeleteMonitorLogfile();
		}
		// リソース 監視
		else if(HinemosModuleConstant.MONITOR_PERFORMANCE.equals(monitorTypeId)){
			deleteMonitor = new DeleteMonitorPerformance();
		}
		// ping監視
		else if(HinemosModuleConstant.MONITOR_PING.equals(monitorTypeId)){
			deleteMonitor = new DeleteMonitorPing();
		}
		// ポート監視
		else if(HinemosModuleConstant.MONITOR_PORT.equals(monitorTypeId)){
			deleteMonitor = new DeleteMonitorPort();
		}
		// プロセス監視
		else if(HinemosModuleConstant.MONITOR_PROCESS.equals(monitorTypeId)){
			deleteMonitor = new DeleteMonitorProcess();
		}
		// SNMPTRAP監視
		else if(HinemosModuleConstant.MONITOR_SNMPTRAP.equals(monitorTypeId)){
			deleteMonitor = new DeleteMonitorTrap();
		}
		// SNMP監視
		else if(HinemosModuleConstant.MONITOR_SNMP.equals(monitorTypeId)){
			deleteMonitor = new DeleteMonitorSnmp();
		}
		// システムログ監視
		else if(HinemosModuleConstant.MONITOR_SYSTEMLOG.equals(monitorTypeId)){
			deleteMonitor = new DeleteMonitorSystemlog();
		}
		// SQL監視
		else if(HinemosModuleConstant.MONITOR_SQL.equals(monitorTypeId)){
			deleteMonitor = new DeleteMonitorSql();
		}
		// コマンド監視
		else if(HinemosModuleConstant.MONITOR_CUSTOM.equals(monitorTypeId)){
			deleteMonitor = new DeleteCustom();
		}
		// Windowsサービス監視
		else if(HinemosModuleConstant.MONITOR_WINSERVICE.equals(monitorTypeId)){
			deleteMonitor = new DeleteMonitorWinService();
		}
		// Other
		else{
			throw new HinemosUnknown("Unknown monitorTypeId = " + monitorTypeId);
		}

		// 監視設定情報を削除
		TransactionManager tm = null;
		boolean flag = false;
		try {
			// begin transaction
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			tm.begin();
			
			flag = deleteMonitor.delete(monitorTypeId, monitorId);
		} catch (MonitorNotFound e) {
			m_log.warn("deleteMonitor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			throw e;
		} catch (Exception e) {
			m_log.error("deleteMonitor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (tm != null) {
				try {
					if (flag) {
						tm.commit();
						if (HinemosModuleConstant.MONITOR_SYSTEMLOG.equals(monitorTypeId)) {
							SystemlogCache.clear();
						} else if (HinemosModuleConstant.MONITOR_CUSTOM.equals(monitorTypeId)) {
							SelectCustom.clearCustomCache();
						} else if (HinemosModuleConstant.MONITOR_LOGFILE.equals(monitorTypeId)) {
							MonitorLogfileControllerBean.clearLogfileCache();
						}
					} else {
						tm.rollback();
					}
				} catch (Exception e) {
					m_log.warn("transaction failure.", e);
					throw new EJBException(e.getMessage(), e);
				}
			}
		}
		return flag;
	}


	/**
	 * 監視情報を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param monitorId
	 * @param monitorTypeId
	 * @return
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public MonitorInfo getMonitor(String monitorId, String monitorTypeId) throws MonitorNotFound, HinemosUnknown{
		m_log.debug("getMonitor() monitorId = " + monitorId + ", monitorTypeId = " + monitorTypeId);

		// null チェック
		if(monitorId == null || "".equals(monitorId)){
			throw new HinemosUnknown("monitorId is null or empty.");
		}
		if(monitorTypeId == null || "".equals(monitorTypeId)){
			throw new HinemosUnknown("monitorTypeId is null or empty.");
		}

		SelectMonitor selectMonitor = null;
		// Hinemos エージェント監視
		if(HinemosModuleConstant.MONITOR_AGENT.equals(monitorTypeId)){
			selectMonitor = new SelectMonitorAgent();
		}
		// HTTP 監視
		else if(HinemosModuleConstant.MONITOR_HTTP.equals(monitorTypeId)){
			selectMonitor = new SelectMonitorHttp();
		}
		// ログファイル 監視
		else if(HinemosModuleConstant.MONITOR_LOGFILE.equals(monitorTypeId)){
			selectMonitor =  new SelectMonitorLogfile();
		}
		// リソース 監視
		else if(HinemosModuleConstant.MONITOR_PERFORMANCE.equals(monitorTypeId)){
			selectMonitor = new SelectMonitorPerformance();
		}
		// ping監視
		else if(HinemosModuleConstant.MONITOR_PING.equals(monitorTypeId)){
			selectMonitor = new SelectMonitorPing();
		}
		// ポート監視
		else if(HinemosModuleConstant.MONITOR_PORT.equals(monitorTypeId)){
			selectMonitor = new SelectMonitorPort();
		}
		// プロセス監視
		else if(HinemosModuleConstant.MONITOR_PROCESS.equals(monitorTypeId)){
			selectMonitor = new SelectMonitorProcess();
		}
		// SNMPTRAP監視
		else if(HinemosModuleConstant.MONITOR_SNMPTRAP.equals(monitorTypeId)){
			selectMonitor = new SelectMonitorTrap();
		}
		// SNMP監視
		else if(HinemosModuleConstant.MONITOR_SNMP.equals(monitorTypeId)){
			selectMonitor = new SelectMonitorSnmp();
		}
		// システムログ監視
		else if(HinemosModuleConstant.MONITOR_SYSTEMLOG.equals(monitorTypeId)){
			selectMonitor = new SelectMonitorSystemlog();
		}
		// SQL監視
		else if(HinemosModuleConstant.MONITOR_SQL.equals(monitorTypeId)){
			selectMonitor = new SelectMonitorSql();
		}
		// コマンド監視
		else if(HinemosModuleConstant.MONITOR_CUSTOM.equals(monitorTypeId)){
			selectMonitor = new SelectCustom();
		}
		// Windowsサービス監視
		else if(HinemosModuleConstant.MONITOR_WINSERVICE.equals(monitorTypeId)){
			selectMonitor = new SelectMonitorWinService();
		}
		// Other
		else{
			throw new HinemosUnknown("Unknown monitorTypeId = " + monitorTypeId);
		}

		// 監視設定情報を取得
		MonitorInfo info = null;
		try {
			info = selectMonitor.getMonitor(monitorTypeId, monitorId);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getMonitor() " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		return info;
	}

	/**
	 * Hinemos Agent監視一覧リストを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<MonitorInfo> getAgentList() throws MonitorNotFound, HinemosUnknown {

		// Hinemos Agent監視一覧を取得
		SelectMonitorAgent agent = new SelectMonitorAgent();
		ArrayList<MonitorInfo> list = null;
		try {
			list = agent.getMonitorList(HinemosModuleConstant.MONITOR_AGENT);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getAgentList " + e.getMessage(),e);
			throw new HinemosUnknown(e.getMessage(),e);
		}

		// null check
		if(list == null){
			list = new ArrayList<MonitorInfo>();
		}
		return list;
	}

	/**
	 * HTTP監視一覧リストを取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<MonitorInfo> getHttpList() throws MonitorNotFound, HinemosUnknown {

		// HTTP監視一覧を取得
		SelectMonitorHttp http = new SelectMonitorHttp();
		ArrayList<MonitorInfo> list = null;
		try {
			list = http.getMonitorList(HinemosModuleConstant.MONITOR_HTTP);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getHttpList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

		// null check
		if(list == null){
			list = new ArrayList<MonitorInfo>();
		}
		return list;
	}

	/**
	 * ログファイル監視一覧リストを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<MonitorInfo> getLogfileList() throws MonitorNotFound, HinemosUnknown{

		SelectMonitorLogfile logfile = new SelectMonitorLogfile();
		ArrayList<MonitorInfo> list = null;
		try {
			list = logfile.getMonitorList(HinemosModuleConstant.MONITOR_LOGFILE);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getLogfileList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

		// null check
		if(list == null){
			list = new ArrayList<MonitorInfo>();
		}
		return list;
	}

	/**
	 * リソース監視一覧リストを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<MonitorInfo> getPerformanceList() throws MonitorNotFound, HinemosUnknown{

		// リソース監視一覧を取得
		SelectMonitorPerformance selectPerf = new SelectMonitorPerformance();
		ArrayList<MonitorInfo> list;
		try {
			list = selectPerf.getMonitorList(HinemosModuleConstant.MONITOR_PERFORMANCE);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getPerformanceList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

		// null check
		if(list == null){
			list = new ArrayList<MonitorInfo>();
		}
		return list;
	}

	/**
	 * ping監視一覧リストを取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<MonitorInfo> getPingList() throws MonitorNotFound, HinemosUnknown {

		// ping監視一覧を取得
		SelectMonitorPing ping = new SelectMonitorPing();
		ArrayList<MonitorInfo> list = null;
		try {
			list = ping.getMonitorList(HinemosModuleConstant.MONITOR_PING);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getPingList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

		// null check
		if(list == null){
			list = new ArrayList<MonitorInfo>();
		}
		return list;
	}

	/**
	 * port監視一覧リストを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<MonitorInfo> getPortList() throws MonitorNotFound, HinemosUnknown{

		// port監視一覧を取得
		SelectMonitorPort port = new SelectMonitorPort();
		ArrayList<MonitorInfo> list = null;
		try {
			list = port.getMonitorList(HinemosModuleConstant.MONITOR_PORT);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getPortList " + e.getMessage(), e);
			throw new HinemosUnknown (e.getMessage(), e);
		}

		// null check
		if(list == null){
			list = new ArrayList<MonitorInfo>();
		}
		return list;
	}

	/**
	 * プロセス監視一覧リストを取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<MonitorInfo> getProcessList() throws MonitorNotFound, HinemosUnknown{

		// プロセス監視一覧を取得
		SelectMonitorProcess process = new SelectMonitorProcess();
		ArrayList<MonitorInfo> list = null;
		try {
			list = process.getMonitorList(HinemosModuleConstant.MONITOR_PROCESS);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getProcessList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

		// null check
		if(list == null){
			list = new ArrayList<MonitorInfo>();
		}
		return list;
	}

	/**
	 * SNMPTRAP監視一覧リストを取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return rrayList<MonitorInfo>
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<MonitorInfo> getTrapList() throws MonitorNotFound, HinemosUnknown{
		m_log.debug("getTrapList() : start");

		// SNMPTRAP監視一覧を取得
		SelectMonitorTrap trap = new SelectMonitorTrap();
		ArrayList<MonitorInfo> list = null;
		try {
			list = trap.getMonitorList(HinemosModuleConstant.MONITOR_SNMPTRAP);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (HinemosUnknown e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getMonitorList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

		m_log.debug("getTrapList() : end");
		// null check
		if(list == null){
			list = new ArrayList<MonitorInfo>();
		}
		return list;
	}

	/**
	 * SNMP監視一覧リストを取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<MonitorInfo> getSnmpList() throws MonitorNotFound, HinemosUnknown {

		// SNMP監視一覧を取得
		SelectMonitorSnmp snmp = new SelectMonitorSnmp();
		ArrayList<MonitorInfo> list = null;
		try {
			list = snmp.getMonitorList(HinemosModuleConstant.MONITOR_SNMP);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getSnmpList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

		// null check
		if(list == null){
			list = new ArrayList<MonitorInfo>();
		}
		return list;
	}

	/**
	 * SQL監視一覧リストを取得します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<MonitorInfo> getSqlList() throws MonitorNotFound, HinemosUnknown{

		// SQL監視一覧を取得
		SelectMonitorSql sql = new SelectMonitorSql();
		ArrayList<MonitorInfo> list = null;
		try {
			list = sql.getMonitorList(HinemosModuleConstant.MONITOR_SQL);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getSqlList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

		// null check
		if(list == null){
			list = new ArrayList<MonitorInfo>();
		}
		return list;
	}

	/**
	 * システムログ監視一覧リストを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<MonitorInfo> getSystemlogList() throws MonitorNotFound, HinemosUnknown{
		ArrayList<MonitorInfo> list = null;
		try {
			list = SystemlogCache.getSystemlogList();
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getSystemlogList " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}
		// null check
		if(list == null){
			list = new ArrayList<MonitorInfo>();
		}
		return list;
	
	}

	/**
	 * 既存のコマンド監視の一覧を返す
	 * @return コマンド監視の設定情報一覧
	 * @throws MonitorNotFound 一覧にIDが存在するが、詳細情報が存在しなかった場合
	 * @throws HinemosUnknown 予期せぬエラーが発生した場合
	 * 
	 * @ejb.interface-method
	 */
	public ArrayList<MonitorInfo> getCustomList() throws MonitorNotFound, HinemosUnknown {
		SelectCustom selector = new SelectCustom();
		ArrayList<MonitorInfo> list = null;

		try {
			list = selector.getMonitorList(HinemosModuleConstant.MONITOR_CUSTOM);
		} catch (MonitorNotFound e) {
			m_log.warn("monitor configuration is not found. (monitorTypeId = " + HinemosModuleConstant.MONITOR_CUSTOM + ")", e);
			throw e;
		} catch (Exception e) {
			m_log.error("monitor configuration is not found. (monitorTypeId = " + HinemosModuleConstant.MONITOR_CUSTOM + ")", e);
			throw new HinemosUnknown("monitor configuration is not found. (monitorTypeId = " + HinemosModuleConstant.MONITOR_CUSTOM + ")", e);
		}

		// null check
		if(list == null){
			list = new ArrayList<MonitorInfo>();
		}
		return list;
	}


	/**
	 * Windowsサービス監視一覧リストを取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return MonitorInfoのリスト
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<MonitorInfo> getWinServiceList() throws MonitorNotFound, HinemosUnknown{

		// Windowsサービス監視一覧を取得
		SelectMonitorWinService winService = new SelectMonitorWinService();
		ArrayList<MonitorInfo> list = null;
		try {
			list = winService.getMonitorList(HinemosModuleConstant.MONITOR_WINSERVICE);
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getWinServiceList() " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		}

		// null check
		if(list == null){
			list = new ArrayList<MonitorInfo>();
		}
		return list;
	}

	/**
	 * 
	 * 監視設定の監視を有効化/無効化します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param monitorId
	 * @param monitorTypeId
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 */
	public void setStatusMonitor(String monitorId, String monitorTypeId, boolean validFlag) throws HinemosUnknown, MonitorNotFound{
		// null check
		if(monitorId == null || "".equals(monitorId)){
			throw new HinemosUnknown("target monitorId is null or empty.");
		}
		if(monitorTypeId == null || "".equals(monitorTypeId)){
			throw new HinemosUnknown("target monitorTypeId is null or empty.");
		}

		MonitorInfo info = getMonitor(monitorId, monitorTypeId);
		try{
			if (validFlag) {
				if(info.getMonitorFlg() != ValidConstant.TYPE_VALID){
					info.setMonitorFlg(ValidConstant.TYPE_VALID);
					modifyMonitor(info);
				}
			} else {
				if(info.getMonitorFlg() != ValidConstant.TYPE_INVALID){
					info.setMonitorFlg(ValidConstant.TYPE_INVALID);
					modifyMonitor(info);
				}
			}
		} catch (InvalidSetting e) {
			m_log.warn("setStatusMonitor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * 
	 * 監視設定の収集を有効化/無効化します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction type="NotSupported"
	 * 
	 * @param monitorId
	 * @param monitorTypeId
	 * @throws HinemosUnknown
	 * @throws MonitorNotFound
	 */
	public void setStatusCollector(String monitorId, String monitorTypeId, boolean validFlag) throws HinemosUnknown, MonitorNotFound{
		// null check
		if(monitorId == null || "".equals(monitorId)){
			throw new HinemosUnknown("target monitorId is null or empty.");
		}
		if(monitorTypeId == null || "".equals(monitorTypeId)){
			throw new HinemosUnknown("target monitorTypeId is null or empty.");
		}

		MonitorInfo info = getMonitor(monitorId, monitorTypeId);
		try{
			if (validFlag) {
				if(info.getMonitorType() != MonitorTypeConstant.TYPE_NUMERIC){
					m_log.debug("setStatusMonitor() : monitorId = " + monitorId + " is not numeric.");
					return;
				}

				if(info.getCollectorFlg() != ValidConstant.TYPE_VALID){
					info.setCollectorFlg(ValidConstant.TYPE_VALID);
					modifyMonitor(info);
				}
			} else {
				if(info.getCollectorFlg() != ValidConstant.TYPE_INVALID){
					info.setCollectorFlg(ValidConstant.TYPE_INVALID);
					modifyMonitor(info);
				}
			}
		} catch (InvalidSetting e) {
			m_log.warn("setStatusMonitor() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw new HinemosUnknown(e.getMessage(), e);
		}
	}

	/**
	 * 監視設定一覧を取得する
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return
	 * @throws MonitorNotFound
	 * @throws HinemosUnknown
	 */
	public ArrayList<MonitorInfo> getMonitorList() throws MonitorNotFound, HinemosUnknown{
		m_log.debug("getMonitorList() : start");
		ArrayList<MonitorInfo> list = null;
		try {
			list = new SelectMonitor().getMonitorList();
		} catch (MonitorNotFound e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getMonitorList " + e.getMessage(),e);
			throw new HinemosUnknown(e.getMessage(),e);
		}
		m_log.debug("getMonitorList() : end");
		return list;
	}

	/**
	 * 監視設定一覧の取得
	 * 
	 * @ejb.interface-method
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 *     
	 * @param condition フィルタ条件
	 * @return 監視設定一覧
	 * @throws HinemosUnknown
	 */
	public ArrayList<MonitorInfo> getMonitorList(MonitorFilterInfo condition) throws HinemosUnknown{
		m_log.debug("getMonitorList(MonitorFilterInfo) : start");

		ArrayList<MonitorInfo> list = null;
		try {

			list = new SelectMonitor().getMonitorList(condition);

		} catch (Exception e) {
			m_log.error("getMonitorList() " + e.getMessage(),e);
			throw new HinemosUnknown(e.getMessage(),e);
		}

		m_log.debug("getMonitorList(MonitorFilterInfo) : end");
		return list;
	}
}
