/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.factory;

import java.sql.Time;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.JobInvalid;
import com.clustercontrol.fault.JobMasterNotFound;
import com.clustercontrol.fault.NotifyNotFound;
import com.clustercontrol.jobmanagement.bean.JobCommandInfo;
import com.clustercontrol.jobmanagement.bean.JobConstant;
import com.clustercontrol.jobmanagement.bean.JobEndStatusInfo;
import com.clustercontrol.jobmanagement.bean.JobFileInfo;
import com.clustercontrol.jobmanagement.bean.JobInfo;
import com.clustercontrol.jobmanagement.bean.JobManagementUserInfo;
import com.clustercontrol.jobmanagement.bean.JobNotificationsInfo;
import com.clustercontrol.jobmanagement.bean.JobObjectInfo;
import com.clustercontrol.jobmanagement.bean.JobParameterInfo;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.bean.JobWaitRuleInfo;
import com.clustercontrol.jobmanagement.bean.JudgmentObjectConstant;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobEndMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobFileMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobNoticeMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobNoticeMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobParamMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartJobMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobUserRelationMasterUtil;
import com.clustercontrol.jobmanagement.util.ManagementUserUtil;
import com.clustercontrol.notify.ejb.session.NotifyControllerLocal;
import com.clustercontrol.notify.ejb.session.NotifyControllerUtil;

/**
 * ジョブツリー情報を登録するクラスです。
 *
 * @version 3.0.0
 * @since 1.0.0
 */
public class ModifyJob {
	/** ログ出力のインスタンス */
	private static Log m_log = LogFactory.getLog( ModifyJob.class );

	/**
	 * ジョブツリー情報を登録します。
	 * <p>
	 * <ol>
	 * <li>全ジョブマスタ情報を取得します。</li>
	 * <li>取得したジョブマスタ情報の数だけ以下の処理を行います</li>
	 *   <ol>
	 *   <li>ジョブリレーションマスタを削除します。</li>
	 *   <li>ジョブコマンドマスタを削除します。</li>
	 *   <li>ジョブファイル転送マスタを削除します。</li>
	 *     <ol>
	 *     <li>ジョブ待ち条件ジョブマスタを削除します。</li>
	 *     <li>ジョブ待ち条件時刻マスタを削除します。</li>
	 *     <li>ジョブ待ち条件マスタを削除します。</li>
	 *     </ol>
	 *   <li>ジョブ通知マスタを削除します。</li>
	 *   <li>ジョブ終了状態マスタを削除します。</li>
	 *   <li>ジョブ変数マスタを削除します。</li>
	 *   <li>ジョブマスタを削除します。</li>
	 *   </ol>
	 *   <li>ジョブツリー情報を基に、ジョブマスタを作成します。</li>
	 * </ol>
	 * 
	 * @param item ジョブツリー情報
	 * @param userId ユーザID
	 * @throws FinderException
	 * @throws NamingException
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * @throws RemoveException
	 * @throws CreateException
	 * @throws HinemosUnknown
	 * @throws JobMasterNotFound
	 * @see com.clustercontrol.jobmanagement.factory.ModifyJob#createJobMaster(JobTreeItem, String)
	 */
	public void registerJobunit(JobTreeItem jobunit, String userId) throws CreateException, NamingException, HinemosUnknown, JobMasterNotFound, JobInvalid {

		//ジョブユニットのジョブIDを取得
		String jobunitId = jobunit.getData().getJobunitId();

		// ログインユーザで参照できない場合はreturnする
		if (!ManagementUserUtil.isReferable(jobunitId, userId)) {
			throw new JobInvalid("JobunitId " + jobunitId + " is not referable by " + userId + " now.");
		}

		int msec = Calendar.getInstance().get(Calendar.MILLISECOND);
		m_log.debug("registerJobunit(): start : " + msec);

		// ジョブマスタ作成
		createJobMaster(jobunit, jobunitId, "TOP", userId);

		m_log.debug("registerJobunit(): End : " + msec);
	}

	/**
	 * 指定されたジョブユニットを削除する
	 * 
	 * @param jobunitId 削除対象ジョブユニットのジョブID
	 * @throws FinderException
	 * @throws NamingException
	 * @throws EJBException
	 * @throws RemoveException
	 * @throws CreateException
	 * @throws HinemosUnknown
	 * @throws NotifyNotFound
	 * @throws JobMasterNotFound
	 */
	public void deleteJobunit(String jobunitId, String userId) throws NamingException, EJBException, RemoveException, CreateException, NotifyNotFound, HinemosUnknown, JobMasterNotFound, JobInvalid {

		// ログインユーザで参照できない場合はreturnする
		if (!ManagementUserUtil.isReferable(jobunitId, userId)) {
			throw new JobInvalid("JobunitId " + jobunitId + " is not referable by " + userId + " now.");
		}

		// 引数で指定されたジョブユニットIDを持つジョブを取得
		ArrayList<JobMasterLocal> jobs = null;
		try {
			jobs = (ArrayList)JobMasterUtil.getLocalHome().findByJobunitId(jobunitId);
		} catch (FinderException e) {
			m_log.warn("JobMasterUtil.getLocalHome().findByJobunitId() : " + e.getMessage());
			JobMasterNotFound je = new JobMasterNotFound(e.getMessage(), e);
			je.setJobunitId(jobunitId);
			throw je;
		}
		m_log.debug("registerJobunit() jobs : " + jobs.toString());

		if (jobs != null) {
			for (JobMasterLocal jobMasterLocal : jobs) {

				m_log.debug("registerJobunit() remove : jobunitId=" + jobMasterLocal.getJobunit_id() + ", jobId=" + jobMasterLocal.getJob_id());
				//				jobMasterLocal = JobMasterUtil.getLocalHome().findByPrimaryKey(pk);

				// 順にジョブを削除する
				if (jobMasterLocal != null) {
					// 通知を削除
					Collection collection = jobMasterLocal.getJobNoticeMaster();
					if (collection != null && collection.size() > 0) {
						Object[] noticeArray = collection.toArray();
						for (int j = 0; j < noticeArray.length; j++) {
							JobNoticeMasterLocal notice = (JobNoticeMasterLocal) noticeArray[j];
							if (notice != null) {
								// 通知情報を削除
								NotifyControllerLocal nc = NotifyControllerUtil.getLocalHome().create();
								nc.deleteNotifyRelation(notice.getNotifyGroupId());
							}
						}
					}
					// ジョブを削除
					jobMasterLocal.remove();
				}
			}
		}
		/*
		 * ジョブマスターのキャッシュを削除する。
		 */
		JobMstCache.clear();
	}


	/**
	 * ジョブマスタを作成します。 再帰呼び出しを行います。
	 * <p>
	 * <ol>
	 * <li>ジョブツリー情報を基に、ジョブマスタを作成します。</li>
	 * <li>ジョブツリー情報の配下のジョブツリー情報を取得します。</li>
	 * <li>取得した配下のジョブツリー情報の数、以下の処理を行います。</li>
	 * <ol>
	 * <li>ジョブツリー情報を基に、ジョブマスタを作成します。</li>
	 * </ol>
	 * </ol>
	 * 
	 * @param item
	 *            ジョブツリー情報
	 * @param user
	 *            ユーザID
	 * @throws CreateException
	 * @throws NamingException
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.ModifyJob#createJobMasterData(JobTreeItem,
	 *      String)
	 */
	private void createJobMaster(JobTreeItem item, String jobunitId, String parentId, String user) throws CreateException, NamingException, HinemosUnknown {

		//ジョブマスタデータ作成
		createJobMasterData(item, jobunitId, parentId, user);

		//子JobTreeItemを取得
		JobTreeItem[] childrens = item.getChildrenArray();
		for(int i = 0; i < childrens.length; i++){
			//ジョブマスタ作成
			createJobMaster(childrens[i], jobunitId, item.getData().getId(), user);
		}
	}

	/**
	 * ジョブマスタを作成します。
	 * <p>
	 * <ol>
	 * <li>ジョブツリー情報を基に、ジョブマスタを作成します。</li>
	 * <li>ジョブツリー情報を基に、ジョブリレーションマスタを作成します。</li>
	 * <li>ジョブツリー情報を基に、ジョブ待ち条件マスタを作成します。</li>
	 * <li>ジョブツリー情報を基に、ジョブ待ち条件ジョブマスタを作成します。</li>
	 * <li>ジョブツリー情報を基に、ジョブ待ち条件時刻マスタを作成します。</li>
	 * <li>ジョブツリー情報を基に、ジョブコマンドマスタを作成します。</li>
	 * <li>ジョブツリー情報を基に、ジョブファイル転送マスタを作成します。</li>
	 * <li>ジョブツリー情報を基に、ジョブ通知マスタを作成します。</li>
	 * <li>ジョブツリー情報を基に、ジョブ終了状態マスタを作成します。</li>
	 * <li>ジョブツリー情報を基に、ジョブ変数マスタを作成します。</li>
	 * </ol>
	 * 
	 * @param item ジョブツリー情報
	 * @param user ユーザID
	 * @throws CreateException
	 * @throws NamingException
	 * @throws HinemosUnknown
	 */
	private void createJobMasterData(JobTreeItem item, String jobunitId, String parentId, String user) throws CreateException, NamingException, HinemosUnknown {
		JobInfo info = item.getData();

		// ジョブユニット単位で再帰的に登録するため、親となるジョブユニットIDは変わらない
		// ただし、TOPの親ジョブユニットIDはTOPとする
		String parentJobunitId = null;
		if (parentId.equals("TOP")) {
			parentJobunitId = "ROOT";
		} else {
			parentJobunitId = jobunitId;
		}


		//判定対象ジョブのジョブユニットも同じ
		String waitJobunitId = jobunitId;

		if(info.getCreateUser() == null)
			info.setCreateUser(user);

		if(info.getCreateTime() == null)
			info.setCreateTime(new Date().getTime());

		//ジョブ作成
		JobMasterUtil.getLocalHome().create(
				jobunitId,
				info.getId(),
				info.getName(),
				info.getDescription(),
				Integer.valueOf(info.getType()),
				info.getCreateTime()==null?null:new Date(info.getCreateTime()),
						new Date(),
						info.getCreateUser(),
						user);

		//ジョブリレーション作成
		JobRelationMasterUtil.getLocalHome().create(
				jobunitId,
				info.getId(),
				parentJobunitId,
				parentId);

		//待ち条件作成
		if(info.getWaitRule() instanceof JobWaitRuleInfo){
			JobWaitRuleInfo waitRule = info.getWaitRule();
			Time startDelayTime = null;
			Time endDelayTime = null;
			if (waitRule.getStart_delay_time_value() != null) {
				startDelayTime = new Time(waitRule.getStart_delay_time_value());
			}
			if (waitRule.getEnd_delay_time_value() != null) {
				endDelayTime = new Time(waitRule.getEnd_delay_time_value());
			}

			JobStartMasterUtil.getLocalHome().create(
					jobunitId,
					info.getId(),
					Integer.valueOf(waitRule.getCondition()),
					Integer.valueOf(waitRule.getSuspend()),
					Integer.valueOf(waitRule.getSkip()),
					Integer.valueOf(waitRule.getSkipEndStatus()),
					Integer.valueOf(waitRule.getSkipEndValue()),
					Integer.valueOf(waitRule.getEndCondition()),
					Integer.valueOf(waitRule.getEndStatus()),
					Integer.valueOf(waitRule.getEndValue()),
					Integer.valueOf(waitRule.getCalendar()),
					waitRule.getCalendarId(),
					Integer.valueOf(waitRule.getCalendarEndValue()),
					Integer.valueOf(waitRule.getStart_delay()),
					Integer.valueOf(waitRule.getStart_delay_session()),
					Integer.valueOf(waitRule.getStart_delay_session_value()),
					Integer.valueOf(waitRule.getStart_delay_time()),
					startDelayTime,
					Integer.valueOf(waitRule.getStart_delay_condition_type()),
					Integer.valueOf(waitRule.getStart_delay_notify()),
					Integer.valueOf(waitRule.getStart_delay_notify_priority()),
					Integer.valueOf(waitRule.getStart_delay_operation()),
					Integer.valueOf(waitRule.getStart_delay_operation_type()),
					Integer.valueOf(waitRule.getStart_delay_operation_end_value()),
					Integer.valueOf(waitRule.getEnd_delay()),
					Integer.valueOf(waitRule.getEnd_delay_session()),
					Integer.valueOf(waitRule.getEnd_delay_session_value()),
					Integer.valueOf(waitRule.getEnd_delay_job()),
					Integer.valueOf(waitRule.getEnd_delay_job_value()),
					Integer.valueOf(waitRule.getEnd_delay_time()),
					endDelayTime,
					Integer.valueOf(waitRule.getEnd_delay_condition_type()),
					Integer.valueOf(waitRule.getEnd_delay_notify()),
					Integer.valueOf(waitRule.getEnd_delay_notify_priority()),
					Integer.valueOf(waitRule.getEnd_delay_operation()),
					Integer.valueOf(waitRule.getEnd_delay_operation_type()),
					Integer.valueOf(waitRule.getEnd_delay_operation_end_value()));

			if(waitRule.getObject() instanceof ArrayList){
				for(int i = 0; i < waitRule.getObject().size(); i++){
					if(waitRule.getObject().get(i) instanceof JobObjectInfo){
						JobObjectInfo objectInfo =
							waitRule.getObject().get(i);

						if(objectInfo.getType() == JudgmentObjectConstant.TYPE_JOB_END_STATUS){
							JobStartJobMasterUtil.getLocalHome().create(
									jobunitId,
									info.getId(),
									waitJobunitId,
									objectInfo.getJobId(),
									Integer.valueOf(JudgmentObjectConstant.TYPE_JOB_END_STATUS),
									Integer.valueOf(objectInfo.getValue()));
						}
						else if(objectInfo.getType() == JudgmentObjectConstant.TYPE_JOB_END_VALUE){
							JobStartJobMasterUtil.getLocalHome().create(
									jobunitId,
									info.getId(),
									waitJobunitId,
									objectInfo.getJobId(),
									Integer.valueOf(JudgmentObjectConstant.TYPE_JOB_END_VALUE),
									Integer.valueOf(objectInfo.getValue()));
						}
						else{
							JobStartTimeMasterUtil.getLocalHome().create(
									jobunitId,
									info.getId(),
									objectInfo.getTime()==null?null:new Time(objectInfo.getTime()));
						}
					}
				}
			}
		}

		//実行コマンド作成
		if(info.getCommand() instanceof JobCommandInfo){
			JobCommandMasterUtil.getLocalHome().create(
					jobunitId,
					info.getId(),
					info.getCommand().getFacilityID(),
					Integer.valueOf(info.getCommand().getProcessingMethod()),
					info.getCommand().getStartCommand(),
					info.getCommand().getStopCommand(),
					info.getCommand().getUser(),
					Integer.valueOf(info.getCommand().getErrorEndFlg()),
					Integer.valueOf(info.getCommand().getErrorEndValue()),
					"",
			"");
		}

		//ファイル転送作成
		if(info.getFile() instanceof JobFileInfo){
			JobFileMasterUtil.getLocalHome().create(
					jobunitId,
					info.getId(),
					Integer.valueOf(info.getFile().getProcessingMethod()),
					info.getFile().getSrcFacilityID(),
					info.getFile().getDestFacilityID(),
					info.getFile().getSrcFile(),
					info.getFile().getSrcWorkDir(),
					info.getFile().getDestDirectory(),
					info.getFile().getDestWorkDir(),
					Integer.valueOf(info.getFile().getCompressionFlg()),
					Integer.valueOf(info.getFile().getCheckFlg()),
					info.getFile().getUser());
		}

		NotifyControllerLocal nc = NotifyControllerUtil.getLocalHome().create();

		//通知メッセージを取得
		if(info.getNotifications() instanceof ArrayList){
			for(int i = 0; i < info.getNotifications().size(); i++){
				if(info.getNotifications().get(i) instanceof JobNotificationsInfo){
					JobNotificationsInfo notificationsInfo =
						info.getNotifications().get(i);

					JobNoticeMasterUtil.getLocalHome().create(
							jobunitId,
							info.getId(),
							Integer.valueOf(notificationsInfo.getType()),
							Integer.valueOf(notificationsInfo.getPriority()),
							notificationsInfo.getNotifyGroupId());

					//通知情報を投入
					nc.addNotifyRelation(notificationsInfo.getNotifyId());

				}
			}
		}

		//終了状態を取得
		if(info.getEndStatus() instanceof ArrayList){
			for(int i = 0; i < info.getEndStatus().size(); i++){
				if(info.getEndStatus().get(i) instanceof JobEndStatusInfo){
					JobEndStatusInfo endInfo =
						info.getEndStatus().get(i);

					JobEndMasterUtil.getLocalHome().create(
							jobunitId,
							info.getId(),
							Integer.valueOf(endInfo.getType()),
							Integer.valueOf(endInfo.getValue()),
							Integer.valueOf(endInfo.getStartRangeValue()),
							Integer.valueOf(endInfo.getEndRangeValue()));
				}
			}
		}

		//パラメータを取得
		if(info.getParam() instanceof ArrayList){
			for(int i = 0; i < info.getParam().size(); i++){
				if(info.getParam().get(i) instanceof JobParameterInfo){
					JobParameterInfo paramInfo =
						info.getParam().get(i);

					JobParamMasterUtil.getLocalHome().create(
							jobunitId,
							info.getId(),
							paramInfo.getParamId(),
							Integer.valueOf(paramInfo.getType()),
							paramInfo.getDescription(),
							paramInfo.getValue());
				}
			}
		}

		//管理ユーザを設定
		if (info.getType() == JobConstant.TYPE_JOBUNIT && info.getManagementUser() != null) {
			ArrayList<JobManagementUserInfo> userList = info.getManagementUser();
			for (JobManagementUserInfo managementUser : userList) {
				m_log.debug("createJobMasterData() : jobunitId=" + jobunitId + ", jobId=" + info.getId() + ", user=" + managementUser.getUserId());
				JobUserRelationMasterUtil.getLocalHome().create(jobunitId,
						info.getId(), managementUser.getUserId());
			}
		}

	}


}
