/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.jobmanagement.ejb.entity.JobScheduleBean;
import com.clustercontrol.jobmanagement.ejb.entity.JobSchedulePK;

/**
 * ジョブ情報のDAOインターフェースを実装するクラス<BR>
 * 
 * @version 2.1.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.jobmanagement.ejb.entity.JobInfoBean
 * @see com.clustercontrol.jobmanagement.dao.JobInfoDAO
 */
public class JobScheduleDAOImpl implements JobScheduleDAO {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog(JobScheduleDAOImpl.class);

	@Override
	public void init() {
	}

	/**
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobInfoDAO#load(com.clustercontrol.jobmanagement.ejb.entity.JobInfoPK, com.clustercontrol.jobmanagement.ejb.entity.JobInfoBean)
	 */
	@Override
	public void load(JobSchedulePK pk, JobScheduleBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			//SQL文の定義
			String sql = "SELECT * FROM cc_job_schedule WHERE schedule_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.getSchedule_id());

			res = stmt.executeQuery();

			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setSchedule_id(pk.getSchedule_id());
				ejb.setSchedule_name(res.getString("schedule_name"));
				ejb.setJobunit_id(res.getString("jobunit_id"));
				ejb.setJob_id(res.getString("job_id"));
				if(res.getString("calendar_id") == null){
					// 外部キー制約によりカレンダIDを空文字としてDBに登録することはできないため、NULLで登録されている
					ejb.setCalendar_id("");
				} else {
					ejb.setCalendar_id(res.getString("calendar_id"));
				}
				ejb.setSchedule_type(res.getInt("schedule_type"));

				if(res.getInt("month") >= 0) {
					ejb.setMonth(res.getInt("month"));
				}
				if(res.wasNull()) {
					ejb.setMonth(null);
				}
				if(res.getInt("day") >= 0) {
					ejb.setDay(res.getInt("day"));
				}
				if(res.wasNull()) {
					ejb.setDay(null);
				}
				if(res.getInt("week") >= 0) {
					ejb.setWeek(res.getInt("week"));
				}
				if(res.wasNull()){
					ejb.setWeek(null);
				}
				if(res.getInt("hour") >= 0) {
					ejb.setHour(res.getInt("hour"));
				}
				if(res.wasNull()) {
					ejb.setHour(null);
				}
				if(res.getInt("minute") >= 0) {
					ejb.setMinute(res.getInt("minute"));
				}
				if(res.wasNull()) {
					ejb.setMinute(null);
				}
				ejb.setValid_flg(res.getInt("valid_flg"));

				if(res.getTimestamp("reg_date") != null)
					ejb.setReg_date(new Date(res.getTimestamp("reg_date").getTime()));
				else
					ejb.setReg_date(null);

				if(res.getTimestamp("update_date") != null)
					ejb.setUpdate_date(new Date(res.getTimestamp("update_date").getTime()));
				else
					ejb.setUpdate_date(null);

				ejb.setReg_user(res.getString("reg_user"));
				ejb.setUpdate_user(res.getString("update_user"));
			} else {
				String msg = "JobInfo data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException", e);
			throw new EJBException(e.getMessage());
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage(), e1);
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}

	/**
	 * UPDATE文を発行します。
	 * 引数で指定されたEntity Beanの内容でデータベースを更新します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobInfoDAO#store(com.clustercontrol.jobmanagement.ejb.entity.JobInfoBean)
	 */
	@Override
	public void store(JobScheduleBean ejb) throws EJBException {
		m_log.debug("create() start : scheduleId=" + ejb.getSchedule_id());

		Connection conn = null;
		PreparedStatement stmt = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "UPDATE cc_job_schedule SET " +
			"schedule_name = ?, " +
			"jobunit_id = ?, " +
			"job_id = ?, " +
			"calendar_id = ?, " +
			"schedule_type = ?, " +
			"month = ?, " +
			"day = ?, " +
			"week = ?, " +
			"hour = ?, " +
			"minute = ?, " +
			"valid_flg = ?, " +
			"reg_date = ?, " +
			"update_date = ?, " +
			"reg_user = ?, " +
			"update_user = ? " +
			"WHERE schedule_id = ?";

			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, ejb.getSchedule_name());
			stmt.setString(2, ejb.getJobunit_id());
			stmt.setString(3, ejb.getJob_id());
			if("".equals(ejb.getCalendar_id())){
				// 外部キー制約により空文字を登録することは出来ない
				stmt.setString(4, null);
			} else {
				stmt.setString(4, ejb.getCalendar_id());
			}
			stmt.setInt(5, ejb.getSchedule_type());
			if(ejb.getMonth() != null)
				stmt.setInt(6, ejb.getMonth());
			else
				stmt.setNull(6, Types.INTEGER);
			if(ejb.getDay() != null)
				stmt.setInt(7, ejb.getDay());
			else
				stmt.setNull(7, Types.INTEGER);
			if(ejb.getWeek() != null)
				stmt.setInt(8, ejb.getWeek());
			else
				stmt.setNull(8, Types.INTEGER);
			if(ejb.getHour() != null)
				stmt.setInt(9, ejb.getHour());
			else
				stmt.setNull(9, Types.INTEGER);
			if(ejb.getMinute() != null)
				stmt.setInt(10, ejb.getMinute());
			else
				stmt.setNull(10, Types.INTEGER);
			stmt.setInt(11, ejb.getValid_flg());
			if(ejb.getReg_date() != null)
				stmt.setTimestamp(12, new Timestamp(ejb.getReg_date().getTime()));
			else
				stmt.setTimestamp(12, null);
			if(ejb.getUpdate_date() != null)
				stmt.setTimestamp(13, new Timestamp(ejb.getUpdate_date().getTime()));
			else
				stmt.setTimestamp(13, null);
			stmt.setString(14, ejb.getReg_user());
			stmt.setString(15, ejb.getUpdate_user());

			stmt.setString(16, ejb.getSchedule_id());

			int row = stmt.executeUpdate();

			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("store() error : scheduleId=" + ejb.getSchedule_id() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("store() error : scheduleId=" + ejb.getSchedule_id() + " SQLException " + e.getMessage(), e);
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("store() error : scheduleId=" + ejb.getSchedule_id() + " SQLException " + e1.getMessage(), e1);
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("store() end : scheduleId=" + ejb.getSchedule_id());
	}

	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobInfoDAO#remove(com.clustercontrol.jobmanagement.ejb.entity.JobInfoPK)
	 */
	@Override
	public void remove(JobSchedulePK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			//SQL文の定義
			String sql = "DELETE FROM cc_job_schedule WHERE schedule_id = ?";

			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.getSchedule_id());

			int row = stmt.executeUpdate();

			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException", e);
			throw new EJBException(e.getMessage());

		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException", e1);
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}

	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobInfoDAO#create(com.clustercontrol.jobmanagement.ejb.entity.JobInfoBean)
	 */
	@Override
	public JobSchedulePK create(JobScheduleBean ejb) throws CreateException, EJBException {
		m_log.debug("create() start : scheduleId=" + ejb.getSchedule_id());

		Connection conn = null;
		PreparedStatement stmt = null;
		JobSchedulePK pk = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "INSERT INTO cc_job_schedule (" +
			"schedule_id, " +
			"schedule_name, " +
			"jobunit_id, " +
			"job_id, " +
			"calendar_id, " +
			"schedule_type, " +
			"month, " +
			"day, " +
			"week, " +
			"hour, " +
			"minute, " +
			"valid_flg, " +
			"reg_date, " +
			"update_date, " +
			"reg_user, " +
			"update_user) " +
			"VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)";

			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, ejb.getSchedule_id());
			stmt.setString(2, ejb.getSchedule_name());
			stmt.setString(3, ejb.getJobunit_id());
			stmt.setString(4, ejb.getJob_id());
			if("".equals(ejb.getCalendar_id())){
				// 外部キー制約により空文字を登録することは出来ない
				stmt.setString(5, null);
			} else {
				stmt.setString(5, ejb.getCalendar_id());
			}
			stmt.setInt(6, ejb.getSchedule_type());
			if(ejb.getMonth() != null)
				stmt.setInt(7, ejb.getMonth());
			else
				stmt.setNull(7, Types.INTEGER);
			if(ejb.getDay() != null)
				stmt.setInt(8, ejb.getDay());
			else
				stmt.setNull(8, Types.INTEGER);
			if(ejb.getWeek() != null)
				stmt.setInt(9, ejb.getWeek());
			else
				stmt.setNull(9, Types.INTEGER);
			if(ejb.getHour() != null)
				stmt.setInt(10, ejb.getHour());
			else
				stmt.setNull(10, Types.INTEGER);
			if(ejb.getMinute() != null)
				stmt.setInt(11, ejb.getMinute());
			else
				stmt.setNull(11, Types.INTEGER);
			stmt.setInt(12, ejb.getValid_flg());
			if(ejb.getReg_date() != null)
				stmt.setTimestamp(13, new Timestamp(ejb.getReg_date().getTime()));
			else
				stmt.setTimestamp(13, null);
			if(ejb.getUpdate_date() != null)
				stmt.setTimestamp(14, new Timestamp(ejb.getUpdate_date().getTime()));
			else
				stmt.setTimestamp(14, null);
			stmt.setString(15, ejb.getReg_user());
			stmt.setString(16, ejb.getUpdate_user());

			int row = stmt.executeUpdate();
			pk = new JobSchedulePK(ejb.getSchedule_id());

			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : scheduleId=" + ejb.getSchedule_id() + " " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : scheduleId=" + ejb.getSchedule_id() + " SQLException" + "DuplicateKeyException ", e);
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : scheduleId=" + ejb.getSchedule_id() + " SQLException" + "DuplicateKeyException ", e1);
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : scheduleId=" + ejb.getSchedule_id());
		return pk;
	}

	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobInfoBean#ejbFindAll()
	 * @see com.clustercontrol.jobmanagement.dao.JobInfoDAO#findAll()
	 */
	@Override
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");

		ArrayList<JobSchedulePK> ret = new ArrayList<JobSchedulePK>();

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "SELECT schedule_id FROM cc_job_schedule";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();

			while(res.next()) {
				JobSchedulePK pk= new JobSchedulePK(res.getString("schedule_id"));
				ret.add(pk);
			}

		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException", e);
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException", e1);
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}

	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobInfoBean#ejbFindByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobInfoPK)
	 * @see com.clustercontrol.jobmanagement.dao.JobInfoDAO#findByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobInfoPK)
	 */
	@Override
	public JobSchedulePK findByPrimaryKey(JobSchedulePK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			//SQL文の定義
			String sql = "SELECT schedule_id FROM cc_job_schedule WHERE schedule_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.getSchedule_id());

			res = stmt.executeQuery();

			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException", e);
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException", e1);
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}
}
