/*

 Copyright (C) 2006 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.jobmanagement.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.StatusConstant;
import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoBean;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoPK;

/**
 * ジョブリレーション情報のDAOインターフェースを実装するクラス<BR>
 * 
 * @version 2.1.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoBean
 * @see com.clustercontrol.jobmanagement.dao.JobRelationInfoDAO
 */
public class JobRelationInfoDAOImpl implements JobRelationInfoDAO {

	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog(JobRelationInfoDAOImpl.class);

	@Override
	public void init() {
	}

	/**
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationInfoDAO#load(com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoPK, com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoBean)
	 */
	@Override
	public void load(JobRelationInfoPK pk, JobRelationInfoBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;

		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			//SQL文の定義
			String sql = "SELECT * FROM cc_job_relation_info WHERE session_id = ? AND job_id = ? AND parent_job_id = ? AND jobunit_id = ? AND parent_jobunit_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getParent_job_id());
			stmt.setString(4, pk.getJobunit_id());
			stmt.setString(5, pk.getParent_jobunit_id());

			res = stmt.executeQuery();

			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setJob_id(res.getString("job_id"));
				ejb.setParent_job_id(res.getString("parent_job_id"));
				ejb.setSession_id(res.getString("session_id"));
				ejb.setJobunit_id(res.getString("jobunit_id"));
				ejb.setParent_jobunit_id(res.getString("parent_jobunit_id"));
			} else {
				String msg = "JobRelationInfo data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}

	/**
	 * データベースの更新を行うのが基本だが、<BR>
	 * 全てのカラムが主キーになっているので、更新を行う必要はない。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationInfoDAO#store(com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoBean)
	 */
	@Override
	public void store(JobRelationInfoBean ejb) throws EJBException {
		m_log.debug("store() start : sessionId=" + ejb.getSession_id()
				+ ", jobunitId=" + ejb.getJobunit_id()
				+ ", jobId=" + ejb.getJob_id()
				+ ", parentJobunitId=" + ejb.getParent_jobunit_id()
				+ ", parentJobId=" + ejb.getParent_job_id());

		m_log.debug("store() end : sessionId=" + ejb.getSession_id()
				+ ", jobunitId=" + ejb.getJobunit_id()
				+ ", jobId=" + ejb.getJob_id()
				+ ", parentJobunitId=" + ejb.getParent_jobunit_id()
				+ ", parentJobId=" + ejb.getParent_job_id());
	}

	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationInfoDAO#remove(com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoPK)
	 */
	@Override
	public void remove(JobRelationInfoPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			//SQL文の定義
			String sql = "DELETE FROM cc_job_relation_info WHERE session_id = ? AND job_id = ? AND parent_job_id = ? AND jobunit_id = ? AND parent_jobunit_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getParent_job_id());
			stmt.setString(4, pk.getJobunit_id());
			stmt.setString(5, pk.getParent_jobunit_id());

			int row = stmt.executeUpdate();

			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());

		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}

	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationInfoDAO#create(com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoBean)
	 */
	@Override
	public JobRelationInfoPK create(JobRelationInfoBean ejb) throws CreateException, EJBException {
		m_log.debug("create() start : sessionId=" + ejb.getSession_id()
				+ ", jobunitId=" + ejb.getJobunit_id()
				+ ", jobId=" + ejb.getJob_id()
				+ ", parentJobunitId=" + ejb.getParent_jobunit_id()
				+ ", parentJobId=" + ejb.getParent_job_id());

		Connection conn = null;
		PreparedStatement stmt = null;
		JobRelationInfoPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_relation_info (");
			sql.append("session_id, ");
			sql.append("job_id, ");
			sql.append("parent_job_id, ");
			sql.append("jobunit_id, ");
			sql.append("parent_jobunit_id) ");
			sql.append("VALUES (?,?,?,?,?)");

			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			stmt.setString(1, ejb.getSession_id());
			stmt.setString(2, ejb.getJob_id());
			stmt.setString(3, ejb.getParent_job_id());
			stmt.setString(4, ejb.getJobunit_id());
			stmt.setString(5, ejb.getParent_jobunit_id());

			int row =	stmt.executeUpdate();
			pk = new JobRelationInfoPK(
					ejb.getSession_id(),
					ejb.getJobunit_id(),
					ejb.getJob_id(),
					ejb.getParent_jobunit_id(),
					ejb.getParent_job_id());

			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : sessionId=" + ejb.getSession_id()
						+ ", jobunitId=" + ejb.getJobunit_id()
						+ ", jobId=" + ejb.getJob_id()
						+ ", parentJobunitId=" + ejb.getParent_jobunit_id()
						+ ", parentJobId=" + ejb.getParent_job_id() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : sessionId=" + ejb.getSession_id()
						+ ", jobunitId=" + ejb.getJobunit_id()
						+ ", jobId=" + ejb.getJob_id()
						+ ", parentJobunitId=" + ejb.getParent_jobunit_id()
						+ ", parentJobId=" + ejb.getParent_job_id() + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : sessionId=" + ejb.getSession_id()
						+ ", jobunitId=" + ejb.getJobunit_id()
						+ ", jobId=" + ejb.getJob_id()
						+ ", parentJobunitId=" + ejb.getParent_jobunit_id()
						+ ", parentJobId=" + ejb.getParent_job_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : sessionId=" + ejb.getSession_id()
				+ ", jobunitId=" + ejb.getJobunit_id()
				+ ", jobId=" + ejb.getJob_id()
				+ ", parentJobunitId=" + ejb.getParent_jobunit_id()
				+ ", parentJobId=" + ejb.getParent_job_id());
		return pk;
	}

	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoBean#ejbFindAll()
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationInfoDAO#findAll()
	 */
	@Override
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");

		ArrayList<JobRelationInfoPK> ret = new ArrayList<JobRelationInfoPK>();

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "SELECT session_id, jobunit_id, job_id, parent_jobunit_id, parent_job_id FROM cc_job_relation_info";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();

			while(res.next()) {
				JobRelationInfoPK pk= new JobRelationInfoPK(
						res.getString("session_id"),
						res.getString("jobunit_id"),
						res.getString("job_id"),
						res.getString("parent_jobunit_id"),
						res.getString("parent_job_id")
				);
				ret.add(pk);
			}

		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}

	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoBean#ejbFindByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoPK)
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationInfoDAO#findByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoPK)
	 */
	@Override
	public JobRelationInfoPK findByPrimaryKey(JobRelationInfoPK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			//SQL文の定義
			String sql = "SELECT session_id, jobunit_id, job_id, parent_jobunit_id, parent_job_id FROM cc_job_relation_info WHERE session_id = ? AND job_id = ? AND parent_job_id = ? AND jobunit_id = ? AND parent_jobunit_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getParent_job_id());
			stmt.setString(4, pk.getJobunit_id());
			stmt.setString(5, pk.getParent_jobunit_id());

			res = stmt.executeQuery();

			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}

	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定されたセッションIDと親ジョブIDで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoBean#ejbFindByParentJobId(java.lang.String, java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationInfoDAO#findByParentJobId(java.lang.String, java.lang.String)
	 */
	@Override
	public Collection findByParentJobId(String session_id, String parent_job_id) throws FinderException {
		m_log.debug("findByParentJobId() start : " + session_id + ", " + parent_job_id);

		ArrayList<JobRelationInfoPK> ret = new ArrayList<JobRelationInfoPK>();

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			String sql = "SELECT session_id, jobunit_id, job_id, parent_jobunit_id, parent_job_id FROM cc_job_relation_info WHERE session_id = ? AND parent_job_id = ? ORDER BY job_id ASC";

			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, session_id);
			stmt.setString(2, parent_job_id);

			res = stmt.executeQuery();

			while(res.next()) {
				JobRelationInfoPK pk= new JobRelationInfoPK(
						res.getString("session_id"),
						res.getString("jobunit_id"),
						res.getString("job_id"),
						res.getString("parent_jobunit_id"),
						res.getString("parent_job_id")
				);
				ret.add(pk);
			}

		} catch (SQLException e) {
			m_log.error("findByParentJobId() error : " + session_id + ", " + parent_job_id + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByParentJobId() error : " + session_id + ", " + parent_job_id + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findByParentJobId() end : " + session_id + ", " + parent_job_id);
		return ret;
	}

	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定されたセッションIDと親ジョブIDと実行状態(待機orスキップor実行中)で、<BR>
	 * データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoBean#ejbFindByStartStatus(java.lang.String, java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationInfoDAO#findByStartStatus(java.lang.String, java.lang.String)
	 */
	@Override
	public Collection findByStartStatus(String session_id, String parent_job_id) throws FinderException {
		m_log.debug("findByStartStatus() start : " + session_id + ", " + parent_job_id);

		ArrayList<JobRelationInfoPK> ret = new ArrayList<JobRelationInfoPK>();

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("SELECT a.session_id, a.jobunit_id, a.job_id, a.parent_jobunit_id, a.parent_job_id FROM cc_job_relation_info as a, cc_job_session_job as b ");
			sql.append("WHERE a.session_id = b.session_id AND ");
			sql.append("a.job_id = b.job_id AND ");
			sql.append("a.session_id = ? AND ");
			sql.append("a.parent_job_id = ? AND ");
			sql.append("(b.status = ? OR b.status = ? OR b.status = ?)");

			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			stmt.setString(1, session_id);
			stmt.setString(2, parent_job_id);
			stmt.setInt(3, StatusConstant.TYPE_WAIT);
			stmt.setInt(4, StatusConstant.TYPE_SKIP);
			stmt.setInt(5, StatusConstant.TYPE_RUNNING);

			res = stmt.executeQuery();

			while(res.next()) {
				JobRelationInfoPK pk= new JobRelationInfoPK(
						res.getString("session_id"),
						res.getString("jobunit_id"),
						res.getString("job_id"),
						res.getString("parent_jobunit_id"),
						res.getString("parent_job_id")
				);
				ret.add(pk);
			}

		} catch (SQLException e) {
			m_log.error("findByStartStatus() error : " + session_id + ", " + parent_job_id + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByStartStatus() error : " + session_id + ", " + parent_job_id + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findByStartStatus() end : " + session_id + ", " + parent_job_id);
		return ret;
	}

	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定されたセッションIDと親ジョブIDと実行状態で、<BR>
	 * データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoBean#ejbFindByStatus(java.lang.String, java.lang.String, java.lang.Integer)
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationInfoDAO#findByStatus(java.lang.String, java.lang.String, java.lang.Integer)
	 */
	@Override
	public Collection findByStatus(String session_id, String parent_job_id, Integer status) throws FinderException {
		m_log.debug("findByStatus() start : " + session_id + ", " + parent_job_id + ", " + status);

		ArrayList<JobRelationInfoPK> ret = new ArrayList<JobRelationInfoPK>();

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("SELECT a.session_id, a.jobunit_id, a.job_id, a.parent_jobunit_id, a.parent_job_id FROM cc_job_relation_info as a, cc_job_session_job as b ");
			sql.append("WHERE a.session_id = b.session_id AND ");
			sql.append("a.job_id = b.job_id AND ");
			sql.append("a.session_id = ? AND ");
			sql.append("a.parent_job_id = ? AND ");
			sql.append("b.status = ?");

			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			stmt.setString(1, session_id);
			stmt.setString(2, parent_job_id);
			stmt.setInt(3, status);

			res = stmt.executeQuery();

			while(res.next()) {
				JobRelationInfoPK pk= new JobRelationInfoPK(
						res.getString("session_id"),
						res.getString("jobunit_id"),
						res.getString("job_id"),
						res.getString("parent_jobunit_id"),
						res.getString("parent_job_id")
				);
				ret.add(pk);
			}

		} catch (SQLException e) {
			m_log.error("findByStatus() error : " + session_id + ", " + parent_job_id + ", " + status + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByStatus() error : " + session_id + ", " + parent_job_id + ", " + status + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findByStatus() end : " + session_id + ", " + parent_job_id + ", " + status);
		return ret;
	}

	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定されたセッションIDと親ジョブIDと開始日時による昇順ソートで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoBean#ejbFindByStartDate(java.lang.String, java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationInfoDAO#findByStartDate(java.lang.String, java.lang.String)
	 */
	@Override
	public Collection findByStartDate(String session_id, String parent_job_id) throws FinderException {
		m_log.debug("findByStartDate() start : " + session_id + ", " + parent_job_id);

		ArrayList<JobRelationInfoPK> ret = new ArrayList<JobRelationInfoPK>();

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("SELECT a.session_id, a.jobunit_id, a.job_id, a.parent_jobunit_id, a.parent_job_id FROM cc_job_relation_info as a, cc_job_session_job as b ");
			sql.append("WHERE a.session_id = b.session_id AND ");
			sql.append("a.job_id = b.job_id AND ");
			sql.append("a.session_id = ? AND ");
			sql.append("a.parent_job_id = ? ");
			sql.append("ORDER BY b.start_date");

			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());

			stmt.setString(1, session_id);
			stmt.setString(2, parent_job_id);

			res = stmt.executeQuery();

			while(res.next()) {
				JobRelationInfoPK pk= new JobRelationInfoPK(
						res.getString("session_id"),
						res.getString("jobunit_id"),
						res.getString("job_id"),
						res.getString("parent_jobunit_id"),
						res.getString("parent_job_id")
				);
				ret.add(pk);
			}

		} catch (SQLException e) {
			m_log.error("findByStartDate() error : " + session_id + ", " + parent_job_id + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findByStartDate() error : " + session_id + ", " + parent_job_id + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findByStartDate() end : " + session_id + ", " + parent_job_id);
		return ret;
	}

	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたセッションIDとジョブIDで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoBean#ejbFindBySessionIdAndJobId(java.lang.String, java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobRelationInfoDAO#findBySessionIdAndJobId(java.lang.String, java.lang.String)
	 */
	@Override
	public JobRelationInfoPK findBySessionIdAndJobId(String session_id, String job_id) throws FinderException {
		m_log.debug("findBySessionIdAndJobId() start : " + session_id + ", " + job_id);

		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();

			//SQL文の定義
			String sql = "SELECT session_id, jobunit_id, job_id, parent_jobunit_id, parent_job_id FROM cc_job_relation_info WHERE session_id = ? AND job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);

			stmt.setString(1, session_id);
			stmt.setString(2, job_id);

			res = stmt.executeQuery();

			if (res.next()) {
				JobRelationInfoPK pk= new JobRelationInfoPK(
						res.getString("session_id"),
						res.getString("jobunit_id"),
						res.getString("job_id"),
						res.getString("parent_jobunit_id"),
						res.getString("parent_job_id")
				);
				return pk;
			} else {
				String msg = "id " + session_id + ", " + job_id + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("findBySessionIdAndJobId() error : " + session_id + ", " + job_id + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findBySessionIdAndJobId() error : " + session_id + ", " + job_id + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findBySessionIdAndJobId() end : " + session_id + ", " + job_id);
		}
	}

	/**
	 * ジョブリレーション情報をジョブリレーションマスタから作成します。
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobs ジョブIDのコレクション
	 * @throws CreateException
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoBean#ejbHomeMakeSession(String, Collection)
	 */
	public void makeSession(String sessionId, String jobunitId, Collection jobs) throws CreateException{
		m_log.debug("makeSession() start : " + sessionId );

		Connection conn = null;
		PreparedStatement stmt = null;
		Hashtable ht = null;


		try{
			conn = ConnectionManager.getConnectionManager().getConnection();

			//SQL文の定義(INSERT)
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_relation_info (");
			sql.append("session_id, ");
			sql.append("job_id, ");
			sql.append("parent_job_id, ");
			sql.append("jobunit_id, ");
			sql.append("parent_jobunit_id) ");
			sql.append("VALUES (?,?,?,?,?)");

			//SQL文のセット(INSERT)
			stmt = conn.prepareStatement(sql.toString());
			stmt.setString(1, sessionId);
			stmt.setString(4, jobunitId);


			Iterator it = jobs.iterator();
			while(it.hasNext()){
				//ジョブIDを取得
				ht = (Hashtable)it.next();
				m_log.debug("job relation : " + ht.toString());
				String jobId = (String)ht.get("jobId");

				//SQL文のセット(INSERT)
				stmt.setString(2, jobId);
				stmt.setString(3, (String)ht.get("parentJobId"));
				stmt.setString(5, (String)ht.get("parentJobunitId"));

				//INSERTを実行
				int row =	stmt.executeUpdate();
				if (row != 1) {
					String msg = "result row is not 1";
					m_log.error("makeSession() error : " + sessionId + ", "
							+ (String)ht.get("jobId") + ", " + (String)ht.get("parentJobId") + " SQLException");
					throw new EJBException(msg);
				}
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("makeSession() error : " +sessionId + ", "
						+ (String)ht.get("jobId") + ", " + (String)ht.get("parentJobId") + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("makeSession() error : " + sessionId + ", "
						+ (String)ht.get("jobId")  + ", " + (String)ht.get("parentJobId") + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("makeSession() end : ");
	}
}
