/*

Copyright (C) 2012 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.util;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.HinemosUnknown;

/**
 * General Command Executor Class
 * @author takahatat
 */
public class CommandExecutor {

	private static Log log = LogFactory.getLog(CommandExecutor.class);
	
	// コマンド実行時に子プロセスに引き渡す環境変数（key・value）
	private final Map<String, String> envMap = new HashMap<String, String>();

	// thread for command execution
	private final ExecutorService _commandExecutor;

	private final String[] _command;
	private final String _commandLine; //ログ出力用コマンド文字列

	private final Charset _charset;
	public static final Charset _defaultCharset = Charset.forName("UTF-8");

	private final long _timeout;
	private static final long _defaultTimeout = 30000;
	public static final long _disableTimeout = -1;

	private final int _bufferSize;
	public static final int _defaultBufferSize = 8120;

	public static final Object _runtimeExecLock = new Object();

	private Process process = null;

	public CommandExecutor(String[] command) throws HinemosUnknown {
		this(command, _defaultCharset);
	}

	public CommandExecutor(String[] command, Charset charset) throws HinemosUnknown {
		this(command, charset, _defaultTimeout);
	}

	public CommandExecutor(String[] command, long timeout) throws HinemosUnknown {
		this(command, _defaultCharset, timeout);
	}

	public CommandExecutor(String[] command, Charset charset, long timeout) throws HinemosUnknown {
		this(command, charset, timeout, _defaultBufferSize);
	}

	public CommandExecutor(String[] command, Charset charset, long timeout, int bufferSize) throws HinemosUnknown {
		this._command = command;
		this._charset = charset;
		this._timeout = timeout;
		this._bufferSize = bufferSize;

		String commandStr = "";
		for (String arg : _command) {
			commandStr += commandStr == null ? arg : " " + arg;
		}
		this._commandLine = commandStr.substring(1); //先頭の空白を取り除いて格納する

		log.debug("initializing " + this);

		if (_command == null) {
			throw new NullPointerException("command is not defined : " + this);
		}
		if (_charset == null) {
			throw new NullPointerException("charset is not defined : " + this);
		}
		
		_commandExecutor = Executors.newSingleThreadExecutor(
				new ThreadFactory() {
					private volatile int _count = 0;
					@Override
					public Thread newThread(Runnable r) {
						return new Thread(r, "CommandExecutor-" + _count++);
					}
				}
				);
	}
	@Override
	public String toString() {
		return this.getClass().getCanonicalName() + " [command = " + _command
				+ ", charset = " + _charset
				+ ", timeout = " + _timeout
				+ ", bufferSize = " + _bufferSize
				+ "]";
	}

	public Process getProcess() {
		return this.process;
	}
	
	public void addEnvironment(String key, String value) {
		envMap.put(key, value);
	}
	
	public Process execute() throws HinemosUnknown {
		// workaround for JVM(Windows) Bug
		// Runtime#exec is not thread safe on Windows.
	
		try {
			synchronized (_runtimeExecLock) {
				ProcessBuilder pb = new ProcessBuilder(_command);
				// 子プロセスには環境変数"_JAVA_OPTIONS"は渡さない
				pb.environment().remove("_JAVA_OPTIONS");
				for (String key : envMap.keySet()) {
					pb.environment().put(key, envMap.get(key));
				}
				process = pb.start();
			}
		} catch (Exception e) {
			log.warn("command executor failure. (command = " + _commandLine + ")", e);
			throw new HinemosUnknown(e.getMessage());
		}
		return process;
	}

	public CommandResult getResult() {
		CommandExecuteTask task = new CommandExecuteTask(process, _charset, _timeout, _bufferSize);
		Future<CommandResult> future = _commandExecutor.submit(task);
		log.debug("executing command. (command = " + _commandLine + ", timeout = " + _timeout + " [msec])");

		// receive result
		CommandResult ret = null;
		try {
			if (_timeout == _disableTimeout) {
				ret = future.get();
			} else {
				ret = future.get(_timeout, TimeUnit.MILLISECONDS);
			}

			log.debug("exit code : " + (ret != null ? ret.exitCode : null));
			log.debug("stdout : " + (ret != null ? ret.stdout : null));
			log.debug("stderr : " + (ret != null ? ret.stderr : null));
			log.debug("buffer discarded : " + (ret != null ? ret.bufferDiscarded : null));
		} catch (Exception e) {
			log.warn("command execution failure. (command = " + _commandLine + ")", e);
		} finally {
			// release thread pool
			log.debug("releasing command threads.");
			_commandExecutor.shutdownNow();
		}

		return ret;
	}

	/**
	 * Command Result Class
	 * @author takahatat
	 */
	public class CommandResult {
		public final Integer exitCode;			// null when command timeout
		public final String stdout;				// null when command timeout
		public final String stderr;				// null when command timeout
		public final boolean bufferDiscarded;	// true when buffer is discarded

		public CommandResult(int exitCode, String stdout, String stderr, boolean bufferDiscarded) {
			this.exitCode = exitCode;
			this.stdout = stdout;
			this.stderr = stderr;
			this.bufferDiscarded = bufferDiscarded;
		}
	}

	/**
	 * Command Execute Task
	 * @author takahatat
	 */
	public class CommandExecuteTask implements Callable<CommandResult> {

		// command timeout (receive timeout)
		private final long timeout;

		// maximun size of received string
		public final int bufferSize;

		// buffer discarded or not
		public boolean bufferDiscarded = false;

		// charset of receive string
		public final Charset charset;

		public final Process process;

		// thread for receive stdout and stderr
		private final ExecutorService _receiverService;

		public CommandExecuteTask(Process process, Charset charset, long timeout, int bufferSize) {
			this.charset = charset;
			this.timeout = timeout;
			this.bufferSize = bufferSize;
			this.process = process;

			log.debug("initializing " + this);

			_receiverService = Executors.newFixedThreadPool(
					2,
					new ThreadFactory() {
						private volatile int _count = 0;
						@Override
						public Thread newThread(Runnable r) {
							return new Thread(r, "CommendExecutor-" + _count++);
						}
					}
					);
		}

		@Override
		public String toString() {
			return this.getClass().getCanonicalName() + " [command = " + _command
					+ ", charset = " + charset
					+ ", timeout = " + timeout
					+ ", bufferSize = " + bufferSize
					+ "]";
		}

		/**
		 * execute command
		 */
		@Override
		public CommandResult call() {
			Process process = this.process;

			Future<String> stdoutTask = null;
			Future<String> stderrTask = null;

			Integer exitCode = null;
			String stdout = null;
			String stderr = null;

			try {
				log.debug("starting child process : " + this);

				StreamReader stdoutReader = new StreamReader(process.getInputStream(), "CommandStdoutReader", _bufferSize);
				StreamReader stderrReader = new StreamReader(process.getErrorStream(), "CommandStderrReader", _bufferSize);
				stdoutTask = _receiverService.submit(stdoutReader);
				stderrTask = _receiverService.submit(stderrReader);

				log.debug("waiting child process : " + _commandLine);
				exitCode = process.waitFor();
				log.debug("child process exited : " + _commandLine);

				if (timeout == _disableTimeout) {
					stdout = stdoutTask.get();
					stderr = stderrTask.get();
				} else {
					stdout = stdoutTask.get(timeout, TimeUnit.MILLISECONDS);
					stderr = stderrTask.get(timeout, TimeUnit.MILLISECONDS);
				}
			} catch (Exception e) {
				log.warn("command executor failure. (command = " + _commandLine + ")", e);
				stdout = "";
				stderr = "Internal Error : " + e.getMessage();
				exitCode = -1;
			} finally {
				log.debug("canceling stdout and stderr reader.");
				if(stdoutTask != null){
					stdoutTask.cancel(true);
				}
				if(stderrTask != null){
					stderrTask.cancel(true);
				}
				if (process != null) {
					log.debug("destroying child process.");
					process.destroy();
				}
				// release thread pool
				log.debug("releasing receiver threads.");
				_receiverService.shutdownNow();
			}

			return new CommandResult(exitCode, stdout, stderr, bufferDiscarded);
		}

		/**
		 * STDOUT and STDERR Reader Class<br/>
		 * @author takahatat
		 */
		public class StreamReader implements Callable<String> {

			// stream of stdout or stderr
			private final InputStream is;
			// receive buffer from stream
			private byte[] buf;

			private final String threadName;

			public StreamReader(InputStream is, String threadName, int bufferSize) {
				this.is = is;
				buf = new byte[bufferSize];
				this.threadName = threadName;

				log.debug("initializing " + this);
			}

			@Override
			public String toString() {
				return this.getClass().getCanonicalName() + " [threadName = " + threadName
						+ ", stream = " + is
						+ "]";
			}

			@Override
			public String call() {
				Thread.currentThread().setName(threadName);

				byte[] output = new byte[bufferSize];
				String outputStr = null;

				int size = 0;
				int used = 0;
				int total = 0;

				try {
					// read stream
					// until EOF (because some programs will be effected when stream is closed on running state)
					while ((size = is.read(buf, used, bufferSize - used)) != -1) {
						log.trace("reading " + size + "[byte] from a stream. (already received = " + total + "[byte])");

						total += size;
						if (total <= bufferSize) {
							// copy byte until buffer size
							log.debug("coping to output as stout/stderr. (offset = " + used + ", size = "  + size + "[byte])");
							System.arraycopy(buf, used, output, used, size);
						}

						if (total >= bufferSize) {
							// recycle buffer from beginning
							used = 0;
						} else {
							// offset position
							used += size;
						}
					}
					log.debug("reached end of stream.");
				} catch (IOException e) {
					log.warn("reading stream failure...", e);
				} finally {
					if (is != null) {
						try {
							log.debug("closing a stream.");
							is.close();
						} catch (IOException e) {
							log.warn("closing stream failure...", e);
						}
					}
				}

				// byte to string
				outputStr = total == 0 ? "" : new String(output, 0, total <= bufferSize ? total : bufferSize, charset);
				if (total > bufferSize) {
					bufferDiscarded = true;
					log.warn("discarding command's output. (buffer = " + bufferSize + "[byte] < total = " + total + "[byte])");
				}

				return outputStr;
			}
		}
	}
}
