/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */

package com.clustercontrol.cloud.aws.base.ui.views.actions;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.action.IAction;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.window.Window;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.cloud.aws.base.commons.util.MessageManagerExt;
import com.clustercontrol.cloud.aws.base.resources.AvailabilityZone;
import com.clustercontrol.cloud.aws.base.resources.CloudResourceProvider;
import com.clustercontrol.cloud.aws.base.resources.CloudResourceProviderImpl;
import com.clustercontrol.cloud.aws.base.resources.GroupIdentifier;
import com.clustercontrol.cloud.aws.base.resources.Image;
import com.clustercontrol.cloud.aws.base.resources.Subnet;
import com.clustercontrol.cloud.aws.base.ui.dialogs.CreateInstanceDialog;
import com.clustercontrol.cloud.presenter.CloudModelException;
import com.clustercontrol.cloud.presenter.IAccountResource.IRegion;
import com.clustercontrol.cloud.presenter.ICloudComputeService;
import com.clustercontrol.cloud.presenter.ICloudInstance;
import com.clustercontrol.cloud.presenter.ICloudInstanceBackup;
import com.clustercontrol.cloud.presenter.RestoreCloudInstanceRequest;
import com.clustercontrol.cloud.ui.views.InstanceMonitorService;
import com.clustercontrol.cloud.ui.views.actions.InvocationException;
import com.clustercontrol.cloud.ui.views.actions.SingleSelectionActionEx;
import com.clustercontrol.util.Messages;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectReader;
import com.fasterxml.jackson.databind.ObjectWriter;


public class RestoreInstanceAction extends SingleSelectionActionEx {
	private MessageManagerExt messages = MessageManagerExt.getInstance("messages");
	
	private String msgRestoreInstanceConfirm = messages.getString("message.confirm_some_action_to_something", new Object[]{"word.instance", "word.restore"});
	private String msgRestoreInstanceResult = messages.getString("message.result_some_action_to_something", new Object[]{"word.instance", "word.restore"});

	@Override
	protected void invoke(IAction action) throws InvocationException {
		CreateInstanceDialog dialog = new CreateInstanceDialog(this.viewPart.getSite().getShell());
		
		ICloudInstanceBackup backup = (ICloudInstanceBackup)((IStructuredSelection)this.viewPart.getSite().getSelectionProvider().getSelection()).getFirstElement();
		CloudResourceProvider resourceProvider = new CloudResourceProviderImpl(backup.getAccountResourceId());
		String region = backup.getRegion();
		if (region != null) {
			CreateInstanceDialog.Model m = new CreateInstanceDialog.Model();
			m.setRegionName(region);
			m.setFacilityId(backup.getBackupedData().getFacilityId());
			m.setFacilityName(backup.getBackupedData().getFacilityName());
			m.setNodeName(backup.getBackupedData().getNodeName());
			m.setReplaceNode(true);
			m.setImage(new Image());
			m.getImage().setImageId(backup.getInstanceBackupId());
			m.getImage().setName(backup.getName());
			m.setDescription(backup.getDescription());

			ObjectMapper om = new ObjectMapper();
			ObjectReader dr = om.reader(CreateInstanceDialog.InstanceDetail.class);
			try {
				CreateInstanceDialog.InstanceDetail detail = dr.readValue(backup.getBackupedData().getInstanceDetail());
				if (detail != null) {
					m.setKeyName(detail.keyName);
					m.setMonitoring(detail.monitoring);
					m.setDisableApiTermination(detail.disableApiTermination);
					m.setInstanceInitiatedShutdownBehavior(detail.instanceInitiatedShutdownBehavior);
					m.setEbsOptimized(detail.ebsOptimized);
					m.setRootBlockDevice(detail.rootBlockDevice);
					
					for (Subnet subnet: resourceProvider.getSubnetsByRegion(region)) {
						if (subnet.getSubnetId().equals(detail.subnetId)) {
							m.setSubnet(subnet);
							break;
						}
					}
					
					List<GroupIdentifier> securities = new ArrayList<>();
					for (GroupIdentifier security: resourceProvider.getSecurityGroupsByRegion(region)) {
						for (String securityId: detail.securityGroupIds) {
							if (security.getGroupId().equals(securityId)) {
								securities.add(security);
							}
						}
					}
					m.setSecurityGroups(securities);
				}
			}
			catch (IOException e) {
				throw new InvocationException(e);
			}
			
			dialog.setInput(m);
		}

		dialog.setCloudInfoProvider(resourceProvider);
		dialog.setBehavior(new CreateInstanceDialog.DialogBehavior(){
			@Override
			public void setupFacilityId(Text facilityId) {
				facilityId.setText(input.getFacilityId());
			}

			@Override
			public void setupFacilityName(Text facilityName) {
				if(input.getFacilityName() != null){
					facilityName.setText(input.getFacilityName());
				}
			}

			@Override
			public void setupReplaceNode(Button btnReplaceNode) {
				btnReplaceNode.setSelection(true);
			}

			@Override
			public void setupDescription(Text description) {
				if(input.getDescription() != null){
					description.setText(input.getDescription());
				}
			}

			@Override
			public void setupNodeName(Text nodeName) {
				nodeName.setText(input.getNodeName());
			}

			@Override
			public void setupRegion(Combo region) {
				String regionLocaleName = messages.getString(input.getRegionName());
				region.add(regionLocaleName);
				region.setData(regionLocaleName, input.getRegionName());
				region.select(0);
				region.setEnabled(false);
			}

			@Override
			public void setupAvailabilityZone(Combo availabilityZone) {
				for (AvailabilityZone zone: provider.getAvailabilityZones(input.getRegionName())){
					availabilityZone.add(zone.getZoneName());
				}
				if(input != null && input.getAvailabilityZone() != null && availabilityZone.indexOf(input.getAvailabilityZone().getZoneName()) != -1){
					availabilityZone.select(availabilityZone.indexOf(input.getAvailabilityZone().getZoneName()));
				} else if(availabilityZone.getItemCount() > 0) {
					availabilityZone.select(0);
				}
			}

			@Override
			public void setupKeyPair(Combo keyPair) {
				for(String keyName: provider.getKeyNames(input.getRegionName())){
					keyPair.add(keyName);
				}
				if(keyPair.getItemCount() > 0){
					if(input.getKeyName() != null && keyPair.indexOf(input.getKeyName()) != -1){
						keyPair.select(keyPair.indexOf(input.getKeyName()));				
					} else if(keyPair.getItemCount() > 0) {
						keyPair.select(0);
					}
					keyPair.setEnabled(true);
				} else {
					keyPair.setEnabled(false);
				}
			}

			@Override
			public void setupSelectImageFlg(Button rdoAmi, Button rdoTemplate){
				rdoAmi.setSelection(true);
				rdoAmi.setEnabled(false);
				rdoTemplate.setSelection(false);
				rdoTemplate.setEnabled(false);
			}
			
			@Override
			public void setupAmi(Text ami, Button selectAmi) {
				ami.setText("(" + input.getImage().getImageId() + ") "+ input.getImage().getName());
				ami.setData(input.getImage());
//				ami.setEnabled(false);
				ami.setEditable(false);
				selectAmi.setEnabled(false);
			}
			
			@Override
			public void setupTemplate(Button useTemplate, Combo template, Button editArgument){
				useTemplate.setEnabled(false);
				template.setEnabled(false);
				editArgument.setEnabled(false);
			}
		});
		
		if(dialog.open() == Window.OK){
			if (MessageDialog.openConfirm(
				null,
				Messages.getString("confirmed"),
				msgRestoreInstanceConfirm)){

				CreateInstanceDialog.Model m = dialog.getOutput();

				RestoreCloudInstanceRequest request = new RestoreCloudInstanceRequest();

				CreateInstanceDialog.InstanceDetail detail = new CreateInstanceDialog.InstanceDetail();

				if(m.getSubnet() != null){
					detail.subnetId = m.getSubnet().getSubnetId();
				} else {
					detail.subnetId = null;
				}
				detail.keyName = m.getKeyName();
				detail.monitoring = m.getMonitoring();
				detail.disableApiTermination = m.getDisableApiTermination();
				detail.instanceInitiatedShutdownBehavior = m.getInstanceInitiatedShutdownBehavior();
				detail.ebsOptimized = m.getEbsOptimized();
				if(m.getRootBlockDevice() != null){
					detail.rootBlockDevice = new CreateInstanceDialog.EbsBlockDevice();
					detail.rootBlockDevice.volumeSize = m.getRootBlockDevice().volumeSize;
					detail.rootBlockDevice.volumeType = m.getRootBlockDevice().volumeType;
					detail.rootBlockDevice.deleteOnTermination = m.getRootBlockDevice().deleteOnTermination;
				} else {
//					detail.rootBlockDevice = new CreateInstanceDialog.EbsBlockDevice();
//					detail.rootBlockDevice.volumeSize = 10;
//					detail.rootBlockDevice.volumeType = "standard";
//					detail.rootBlockDevice.deleteOnTermination = true;
				}
				
				if(m.getSecurityGroups() != null){
					detail.securityGroupIds = new ArrayList<>();
					for(GroupIdentifier securityGroup: m.getSecurityGroups()){
						detail.securityGroupIds.add(securityGroup.getGroupId());
					}
				} else {
//					detail.securityGroupIds = new ArrayList<>();
//					detail.securityGroupIds.add("hinemos-agent-nonVPC");
				}
				
				ObjectMapper om = new ObjectMapper();
				ObjectWriter dw = om.writerWithType(CreateInstanceDialog.InstanceDetail.class);
				String detailJson;
				try {
					detailJson = dw.writeValueAsString(detail);
				} catch (Exception e) {
					throw new CloudModelException(e);
				}

				request.setZone(m.getZone());
				if(m.getImage() != null){
					request.setInstanceBackupId(m.getImage().getImageId());
				}
				request.setFlavor(m.getFlavor());
				
				request.setFacilityId(m.getFacilityId());
				request.setFacilityName(m.getFacilityName());
				request.setDescription(m.getDescription());
				request.setNodeName(m.getNodeName());
				request.setReplaceNode(m.getReplaceNode());
				
				request.setInstanceDetail(detailJson);
				
				IRegion r = backup.getCloudInstanceBackupManager().getCloudComputeService().getRegion();
				final ICloudInstance instance;
				instance = r.getAccountResource().getRegion(m.getRegionName()).getCloudService(ICloudComputeService.class).getInstanceManager().restoreCloudInstance(request);

				// 成功報告ダイアログを生成
				MessageDialog.openInformation(
					null,
					Messages.getString("successful"),
					msgRestoreInstanceResult);
				
				viewPart.getSite().getShell().getDisplay().asyncExec(new Runnable() {
					@Override
					public void run() {
						InstanceMonitorService.getInstanceMonitorService().startMonitor(instance, ICloudInstance.StateType.running, ICloudInstance.StateType.terminated, ICloudInstance.StateType.stopped);
						com.clustercontrol.action.FacilityTree.refresh();
					}
				});
			}
			else {
				return;
			}
		}
	}
	
	@Override
	public void selectionChanged(IAction action, ISelection selection) {
		super.selectionChanged(action, selection);
		ICloudInstanceBackup backup = (ICloudInstanceBackup)((IStructuredSelection)selection).getFirstElement();
		if(backup == null || backup.getRestoreStatus() == null || backup.getRestoreStatus().equals(ICloudInstanceBackup.RestoreStatus.unavailable.name())){
			action.setEnabled(false);
		}
	}
}