/*

Copyright (C) since 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.

 */

package com.clustercontrol.accesscontrol.session;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import javax.persistence.EntityExistsException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.fault.FacilityDuplicate;
import com.clustercontrol.fault.FacilityNotFound;
import com.clustercontrol.fault.HinemosUnknown;
import com.clustercontrol.fault.InvalidRole;
import com.clustercontrol.fault.InvalidUserPass;
import com.clustercontrol.fault.InvalidSetting;
import com.clustercontrol.fault.JobMasterNotFound;
import com.clustercontrol.fault.ObjectPrivilege_InvalidRole;
import com.clustercontrol.fault.PrivilegeDuplicate;
import com.clustercontrol.fault.RoleDuplicate;
import com.clustercontrol.fault.RoleNotFound;
import com.clustercontrol.fault.UnEditableRole;
import com.clustercontrol.fault.UnEditableUser;
import com.clustercontrol.fault.UsedFacility;
import com.clustercontrol.fault.UsedObjectPrivilege;
import com.clustercontrol.fault.UsedOwnerRole;
import com.clustercontrol.fault.UsedRole;
import com.clustercontrol.fault.UsedUser;
import com.clustercontrol.fault.UserDuplicate;
import com.clustercontrol.fault.UserNotFound;
import com.clustercontrol.jobmanagement.model.JobMstEntityPK;
import com.clustercontrol.repository.bean.FacilityConstant;
import com.clustercontrol.repository.bean.FacilitySortOrderConstant;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.bean.ScopeInfo;
import com.clustercontrol.repository.factory.FacilityModifier;
import com.clustercontrol.repository.factory.FacilitySelector;
import com.clustercontrol.repository.session.RepositoryControllerBean;
import com.clustercontrol.repository.util.FacilityTreeCache;
import com.clustercontrol.repository.util.RepositoryManagerUtil;
import com.clustercontrol.repository.util.RepositoryUpdateTime;
import com.clustercontrol.repository.util.RepositoryValidator;

import com.clustercontrol.accesscontrol.bean.ObjectPrivilegeFilterInfo;
import com.clustercontrol.accesscontrol.bean.ObjectPrivilegeInfo;
import com.clustercontrol.accesscontrol.bean.PrivilegeConstant.ObjectPrivilegeMode;
import com.clustercontrol.accesscontrol.bean.RoleIdConstant;
import com.clustercontrol.accesscontrol.bean.RoleInfo;
import com.clustercontrol.accesscontrol.bean.RoleTreeItem;
import com.clustercontrol.accesscontrol.bean.SystemPrivilegeInfo;
import com.clustercontrol.accesscontrol.bean.UserInfo;
import com.clustercontrol.accesscontrol.factory.LoginUserModifier;
import com.clustercontrol.accesscontrol.factory.LoginUserSelector;
import com.clustercontrol.accesscontrol.factory.RoleModifier;
import com.clustercontrol.accesscontrol.factory.RoleSelector;
import com.clustercontrol.accesscontrol.util.ObjectPrivilegeUtil;
import com.clustercontrol.accesscontrol.util.ObjectPrivilegeValidator;
import com.clustercontrol.accesscontrol.util.RoleValidator;
import com.clustercontrol.accesscontrol.util.UserRoleCache;
import com.clustercontrol.accesscontrol.util.UserValidator;
import com.clustercontrol.accesscontrol.util.VersionUtil;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.commons.util.HinemosSessionContext;
import com.clustercontrol.commons.util.JpaTransactionManager;

/**
 * アカウント機能を実現するSession Bean<BR>
 * 
 */
public class AccessControllerBean {

	/** ログ出力のインスタンス */
	private static Log m_log = LogFactory.getLog( AccessControllerBean.class );

	/**
	 * ログインチェックの為、本メソッドを使用します。
	 * 
	 */
	public void checkLogin() {

	}

	/**
	 * ユーザ検索条件に基づき、ユーザ一覧情報を取得する。<BR>
	 * 
	 * 
	 * @return ユーザ情報のリスト
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserSelector#getUserList(Property)
	 */
	public ArrayList<UserInfo> getUserInfoList() throws HinemosUnknown {

		JpaTransactionManager jtm = null;
		ArrayList<UserInfo> userInfoList = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			userInfoList = LoginUserSelector.getUserInfoList();
			jtm.commit();
		} catch (Exception e) {
			m_log.warn("getUserInfoList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return userInfoList;
	}

	/**
	 * 自身のユーザ情報を取得する。<BR>
	 * 
	 * 
	 * @return 自身のユーザ情報
	 * @throws HinemosUnknown
	 */
	public UserInfo getOwnUserInfo() throws HinemosUnknown {
		JpaTransactionManager jtm = null;
		UserInfo userInfo = null;
		String loginUser = (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID);
		m_log.debug("getOwnUserInfo() loginUser = " + loginUser);

		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			userInfo = LoginUserSelector.getUserInfo(loginUser);
			jtm.commit();
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("getOwnUserInfo() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return userInfo;
	}


	/**
	 * ユーザ情報を取得する。<BR>
	 * 
	 * 
	 * @param userId ユーザID
	 * @param mode 取得モード
	 * @param locale ロケール情報
	 * @return ユーザ情報
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.bean.PropertyConstant
	 * @see com.clustercontrol.accesscontrol.factory.UserProperty#getProperty(String, int, Locale)
	 */
	public UserInfo getUserInfo(String userId) throws HinemosUnknown {
		JpaTransactionManager jtm = null;
		UserInfo userInfo = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			userInfo = LoginUserSelector.getUserInfo(userId);
			jtm.commit();
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			jtm.rollback();
			m_log.warn("getUserInfo() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return userInfo;
	}

	/**
	 * ユーザ認証する<BR>
	 * 
	 * 
	 * @param username ユーザ名
	 * @param password パスワード
	 * @param systemPrivilegeList システム権限情報
	 * @return ユーザ情報
	 * @throws InvalidUserPass
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public void getUserInfoByPassword(String username, String password, ArrayList<SystemPrivilegeInfo> systemPrivilegeList)
			throws InvalidUserPass, InvalidRole, HinemosUnknown {
		JpaTransactionManager jtm = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			LoginUserSelector.getUserInfoByPassword(username, password, systemPrivilegeList);
			jtm.commit();
		} catch (InvalidUserPass e) {
			jtm.rollback();
			throw e;
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			jtm.rollback();
			m_log.warn("getUserInfoByPassword() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
	}

	/**
	 * ユーザを追加する。<BR>
	 * 
	 * 
	 * @param info ユーザ情報
	 * @throws HinemosUnknown
	 * @throws UserDuplicate
	 * @throws InvalidSetting
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserModifier#addUser(Property, String, String)
	 */
	public void addUserInfo(UserInfo info) throws HinemosUnknown, UserDuplicate, InvalidSetting {
		m_log.info("user=" + HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID));

		JpaTransactionManager jtm = null;
		boolean commit = false;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			// 入力チェック
			UserValidator.validateUserInfo(info);

			/** メイン処理 */
			LoginUserModifier.addUserInfo(info, (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID));
			jtm.commit();
			commit = true;
		} catch (UserDuplicate e) {
			jtm.rollback();
			throw e;
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (InvalidSetting e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("addUserInfo() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (commit) {
				// ユーザロールキャッシュ情報を変更する
				UserRoleCache.refresh();
			}
			jtm.close();
		}
	}

	/**
	 * ユーザ情報を変更する。<BR>
	 * 
	 * 
	 * @param info ユーザ情報
	 * @throws HinemosUnknown
	 * @throws UserNotFound
	 * @throws UnEditableUser
	 * @throws InvalidSetting
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserModifier#modifyUser(Property, String)
	 */
	public void modifyUserInfo(UserInfo info) throws HinemosUnknown, UserNotFound, UnEditableUser, InvalidSetting {
		m_log.info("user=" + HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID));
		JpaTransactionManager jtm = null;

		/** メイン処理 */
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();

			//入力チェック
			UserValidator.validateUserInfo(info);

			/** メイン処理 */
			LoginUserModifier.modifyUserInfo(info, (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID));

			jtm.commit();
		} catch (UserNotFound e) {
			jtm.rollback();
			throw e;
		} catch (UnEditableUser e) {
			jtm.rollback();
			throw e;
		} catch (InvalidSetting e) {
			jtm.rollback();
			throw e;
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("modifyUserInfo() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
	}

	/**
	 * ユーザ情報を削除する。<BR>
	 * 
	 * 
	 * @param userId ユーザID
	 * @throws HinemosUnknown
	 * @throws UserNotFound
	 * @throws UsedUser
	 * @throws UnEditableUser
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserModifier#deleteUser(String, String)
	 */
	public void deleteUserInfo(String userId) throws HinemosUnknown, UserNotFound, UsedUser, UnEditableUser {
		JpaTransactionManager jtm = null;
		boolean commit = false;

		/** メイン処理 */
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();

			LoginUserModifier.deleteUserInfo(userId, (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID));

			jtm.commit();
			commit = true;
		} catch (UserNotFound e) {
			jtm.rollback();
			throw e;
		} catch (UnEditableUser e) {
			jtm.rollback();
			throw e;
		} catch (UsedUser e) {
			jtm.rollback();
			throw e;
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("deleteUserInfo() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (commit) {
				// ユーザロールキャッシュ情報を変更する
				UserRoleCache.refresh();
			}
			jtm.close();
		}
	}

	/**
	 * 自分自身のパスワードを変更する。<BR>
	 * 
	 * 
	 * @param userId ユーザID
	 * @param password パスワード
	 * @throws HinemosUnknown
	 * @throws UserNotFound
	 * 
	 */
	public void changeOwnPassword(String password) throws HinemosUnknown, UserNotFound {
		m_log.debug("changeOwnPassword() password = " + password);

		JpaTransactionManager jtm = null;
		String loginUser = (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID);
		m_log.debug("changeOwnPassword() loginUser = " + loginUser);

		try {
			jtm = new JpaTransactionManager();
			jtm.begin();

			LoginUserModifier.modifyUserPassword(loginUser, password);

			jtm.commit();
		} catch (UserNotFound e) {
			jtm.rollback();
			throw e;
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("changeOwnPassword() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
	}

	/**
	 * パスワードを変更する。<BR>
	 * 
	 * 
	 * @param userId ユーザID
	 * @param password パスワード
	 * @throws HinemosUnknown
	 * @throws UserNotFound
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserModifier#modifyUserPassword(String, String)
	 */
	public void changePassword(String userId, String password) throws HinemosUnknown, UserNotFound {
		JpaTransactionManager jtm = null;

		/** メイン処理 */
		try{
			jtm = new JpaTransactionManager();
			jtm.begin();

			LoginUserModifier.modifyUserPassword(userId, password);

			jtm.commit();
		} catch (UserNotFound e){
			jtm.rollback();
			throw e;
		} catch (HinemosUnknown e){
			jtm.rollback();
			throw e;
		} catch (Exception e){
			m_log.warn("changePassword() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
	}

	/**
	 * ログインしているユーザがADMINISTRATORSロールかHINEMOS_MODULEロールに所属しているかを確認する。<BR>
	 * 
	 * 
	 * @return ADMINISTRATORSロールに所属していればtrue, そうでなければfalse
	 * @throws HinemosUnknown
	 */
	public boolean isAdministrator() throws HinemosUnknown {
		JpaTransactionManager jtm = null;
		boolean rtn = false;

		String loginUserId = (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID);

		/** メイン処理 */
		try{
			jtm = new JpaTransactionManager();
			jtm.begin();
			// ADMINISTRATORSロールチェック
			List<String> userIdList = UserRoleCache.getUserIdList(RoleIdConstant.ADMINISTRATORS);
			if (userIdList != null && userIdList.contains(loginUserId)) {
				rtn = true;
			}
			if (!rtn) {
				// HINEMOS_MODULEロールチェック
				userIdList = UserRoleCache.getUserIdList(RoleIdConstant.HINEMOS_MODULE);
				if (userIdList != null && userIdList.contains(loginUserId)) {
					rtn = true;
				}
			}
			jtm.commit();
		} catch (Exception e){
			m_log.warn("isAdministrator() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		m_log.debug("isAdministrator() : loginUserId = " + loginUserId + ", " + rtn);
		return rtn;
	}

	/**
	 * ログインしているユーザが指定したユーザ権限を持っているかどうかを確認する。<BR>
	 * 
	 * 
	 * @param systemPrivilege システム権限情報
	 * @return ユーザ権限を保持していればtrue, そうでなければfalse
	 * @throws HinemosUnknown
	 */
	public boolean isPermission(SystemPrivilegeInfo systemPrivilege) throws HinemosUnknown {
		JpaTransactionManager jtm = null;
		boolean rtn = false;

		String loginUserId = (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID);

		/** メイン処理 */
		try{
			jtm = new JpaTransactionManager();
			jtm.begin();
			if (systemPrivilege != null
				&& UserRoleCache.isSystemPrivilege(loginUserId, systemPrivilege)) {
				rtn = true;
			}
			jtm.commit();
		} catch (Exception e){
			m_log.warn("isPermission() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return rtn;
	}

	/**
	 * ログインユーザのユーザ名を取得する。<BR>
	 * 
	 * 
	 * @return ユーザ名
	 * @throws HinemosUnknown
	 * @throws UserNotFound
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserSelector#getUserName(String)
	 */
	public String getUserName() throws HinemosUnknown, UserNotFound {
		JpaTransactionManager jtm = null;
		String userName = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			userName = LoginUserSelector.getUserName((String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID));
			jtm.commit();
		} catch (UserNotFound e) {
			jtm.rollback();
			throw e;
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("getUserName() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return userName;
	}

	/**
	 * バージョン番号を取得する。<BR>
	 * 
	 * 
	 */
	public String getVersion() {
		return VersionUtil.getVersion();
	}

	/**
	 * ログインユーザ所属するロール一覧情報を取得する。<BR>
	 * 
	 * 
	 * @return ロール情報のリスト
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserSelector#getRoleList(Property)
	 */
	public ArrayList<String> getOwnerRoleIdList() throws HinemosUnknown {
		Boolean isAdministrator = (Boolean)HinemosSessionContext.instance().getProperty(HinemosSessionContext.IS_ADMINISTRATOR);
		JpaTransactionManager jtm = null;
		ArrayList<String> roleIdList = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			String loginUser = null;
			if (isAdministrator == null || !isAdministrator) {
				loginUser = (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID);
			}
			roleIdList = RoleSelector.getOwnerRoleIdList(loginUser);
			jtm.commit();
		} catch (Exception e) {
			m_log.warn("getUserInfoList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return roleIdList;
	}

	/**
	 * ロール一覧情報を取得する。<BR>
	 * 
	 * 
	 * @return ロール情報のリスト
	 * @throws HinemosUnknown
	 * 
	 * @see com.clustercontrol.accesscontrol.factory.LoginUserSelector#getRoleList(Property)
	 */
	public ArrayList<RoleInfo> getRoleInfoList() throws HinemosUnknown {

		JpaTransactionManager jtm = null;
		ArrayList<RoleInfo> roleInfoList = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			roleInfoList = RoleSelector.getRoleInfoList();
			jtm.commit();
		} catch (Exception e) {
			m_log.warn("getUserInfoList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return roleInfoList;
	}


	/**
	 * ロール情報を取得する。<BR>
	 * 
	 * 
	 * @param roleId ロールID
	 * @return ユーザ情報
	 * @throws HinemosUnknown
	 * 
	 */
	public RoleInfo getRoleInfo(String roleId) throws HinemosUnknown {
		JpaTransactionManager jtm = null;
		RoleInfo roleInfo = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			roleInfo = RoleSelector.getRoleInfo(roleId);
			jtm.commit();
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			jtm.rollback();
			m_log.warn("getUserInfo() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return roleInfo;
	}

	/**
	 * ロールを新規に作成する。
	 * ロール作成時にロールスコープも新規に追加します。<BR>
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * 
	 * @param roleInfo
	 * @throws RoleDuplicate
	 * @throws FacilityDuplicate
	 * @throws InvalidSetting
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 */
	public void addRoleInfo(RoleInfo roleInfo)
			throws RoleDuplicate, FacilityDuplicate, InvalidSetting, InvalidRole, HinemosUnknown {
		boolean commit = false;

		JpaTransactionManager jtm = null;

		/** メイン処理 */
		try {
			// トランザクションがすでに開始されている場合は処理終了
			jtm = new JpaTransactionManager();
			jtm.begin(true);

			// ロール入力チェック
			RoleValidator.validateRoleInfo(roleInfo);

			// ロール新規作成
			RoleModifier.addRoleInfo(roleInfo, (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID));

			// ロールスコープの作成
			ScopeInfo scopeInfo = new ScopeInfo();
			scopeInfo.setFacilityId(roleInfo.getId());
			scopeInfo.setFacilityName(roleInfo.getName());
			scopeInfo.setFacilityType(FacilityConstant.TYPE_SCOPE);
			scopeInfo.setDescription(roleInfo.getName());
			scopeInfo.setValid(Boolean.TRUE);
			scopeInfo.setOwnerRoleId(roleInfo.getId());
			String parentFacilityId = FacilityTreeAttributeConstant.OWNER_SCOPE;

			// ロールスコープ入力チェック
			RepositoryValidator.validateScopeInfo(parentFacilityId, scopeInfo, false);

			// ロールスコープ新規作成
			FacilityModifier.addScope(
					parentFacilityId,
					scopeInfo,
					(String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID),
					FacilitySortOrderConstant.DEFAULT_SORT_ORDER_ROLE_SCOPE,
					false);

			jtm.commit();
			commit = true;
		} catch (RoleDuplicate e) {
			jtm.rollback();
			throw e;
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (InvalidSetting e) {
			jtm.rollback();
			throw e;
		} catch (EntityExistsException e) {
			jtm.rollback();
			throw new FacilityDuplicate(e.getMessage(), e);
		} catch (FacilityNotFound e) {
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (Exception e) {
			m_log.warn("addScope() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(),e);
		} finally {
			if (commit) {
				try {
					// ファシリティの木情報のキャッシュを初期化する
					FacilitySelector.initCacheFacilityTree();
					
					// ファシリティツリー情報を更新する
					FacilityTreeCache.refresh_ALL();
					
					// read-committedのため、commit後に外部コンポーネントに通知する
					RepositoryManagerUtil.broadcastConfigured();
				} catch (Exception e) {
					m_log.warn("addScope() transaction failure.", e);
					throw new HinemosUnknown(e.getMessage(),e);
				} finally {
					// ユーザロールキャッシュ情報を変更する
					UserRoleCache.refresh();
					jtm.close();
				}
			}
			jtm.close();
		}

		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refreshUpdateTime();
	}

	/**
	 * ロール情報を変更する。<BR>
	 * ロール変更時にロールスコープも変更します。<BR>
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * 
	 * 
	 * @param info ロール情報
	 * @throws HinemosUnknown
	 * @throws RoleNotFound
	 * @throws UnEditableRole
	 * @throws FacilityNotFound
	 * @throws InvalidSetting
	 */
	public void modifyRoleInfo(RoleInfo roleInfo) throws InvalidSetting, InvalidRole, RoleNotFound, UnEditableRole, FacilityNotFound, HinemosUnknown {
		boolean commit = false;
		JpaTransactionManager jtm = null;

		try{
			// トランザクションがすでに開始されている場合は処理終了
			jtm = new JpaTransactionManager();
			jtm.begin(true);

			// ロール入力チェック
			RoleValidator.validateRoleInfo(roleInfo);

			// ロール更新
			RoleModifier.modifyRoleInfo(roleInfo, (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID));

			// ロールスコープの作成
			ScopeInfo scopeInfo = new ScopeInfo();
			scopeInfo.setFacilityId(roleInfo.getId());
			scopeInfo.setFacilityName(roleInfo.getName());
			scopeInfo.setFacilityType(FacilityConstant.TYPE_SCOPE);
			scopeInfo.setDescription(roleInfo.getName());
			scopeInfo.setValid(Boolean.TRUE);
			scopeInfo.setOwnerRoleId(roleInfo.getId());

			// ロールスコープ入力チェック
			RepositoryValidator.validateScopeInfo(null, scopeInfo, false);

			// ロールスコープ更新
			FacilityModifier.modifyScope(scopeInfo, (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID), false);

			jtm.commit();
			commit = true;
		} catch (InvalidSetting e) {
			jtm.rollback();
			throw e;
		} catch (RoleNotFound e) {
			jtm.rollback();
			throw e;
		} catch (UnEditableRole e) {
			jtm.rollback();
			throw e;
		} catch (FacilityNotFound e) {
			jtm.rollback();
			throw e;
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (Exception e) {
			m_log.warn("modifyScope() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(),e);
		} finally {
			if (commit) {
				try {

					// ファシリティの木情報のキャッシュを初期化する
					FacilitySelector.initCacheFacilityTree();
					
					// ファシリティツリー情報を更新する
					FacilityTreeCache.refreshInfo();
					
					// read-committedのため、commit後に外部コンポーネントに通知する
					RepositoryManagerUtil.broadcastConfigured();
				} catch (Exception e) {
					m_log.warn("modifyScope() transaction failure."
							+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
					throw new HinemosUnknown(e.getMessage(),e);
				} finally {
					jtm.close();
				}
			}
			jtm.close();
		}

		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refreshUpdateTime();
	}

	/**
	 * ロール情報を削除する。<BR>
	 * ロール削除時にロールスコープも削除します。<BR>
	 * 
	 * Database commit後に実施すべきユーザ実装が存在するため、トランザクション開始はユーザが制御する。
	 * 
	 * 
	 * @param roleId ロールID
	 * @throws UsedFacility
	 * @throws RoleNotFound
	 * @throws UnEditableRole
	 * @throws UsedRole
	 * @throws UsedOwnerRole
	 * @throws FacilityNotFound
	 * @throws InvalidRole
	 * @throws HinemosUnknown
	 * 
	 */
	public void deleteRoleInfo(String roleId) throws UsedFacility, RoleNotFound, UnEditableRole, UsedRole, UsedOwnerRole, FacilityNotFound, InvalidRole, HinemosUnknown {
		boolean commit = false;
		JpaTransactionManager jtm = null;

		/** メイン処理 */
		try {
			// トランザクションがすでに開始されている場合は処理終了
			jtm = new JpaTransactionManager();
			jtm.begin(true);
			
			// ロールスコープが他機能で使用されているか確認
			new RepositoryControllerBean().checkIsUseFacility(roleId);
			// ロールスコープ削除
			FacilityModifier.deleteOwnerRoleScope(roleId, (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID), true);

			// ロールがオーナーロールとして使用されているか確認
			RoleValidator.validateDeleteRole(roleId);
			// ロール削除
			RoleModifier.deleteRoleInfo(roleId, (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID));
			
			jtm.commit();
			commit = true;
		} catch (UsedFacility e) {
			jtm.rollback();
			throw e;
		} catch (FacilityNotFound e) {
			jtm.rollback();
			throw e;
		} catch (RoleNotFound e) {
			jtm.rollback();
			throw e;
		} catch (UnEditableRole e) {
			jtm.rollback();
			throw e;
		} catch (UsedRole e) {
			jtm.rollback();
			throw e;
		} catch (InvalidRole e) {
			jtm.rollback();
			throw e;
		} catch (UsedOwnerRole e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("deleteScope() : "
					+ e.getClass().getSimpleName() +", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(),e);
		} finally {
			if (commit) {
				try {
					// ファシリティの木情報のキャッシュを初期化する
					FacilitySelector.initCacheFacilityTree();
					
					// ファシリティツリー情報を更新する
					FacilityTreeCache.refresh_ALL();

					// read-committedのため、commit後に外部コンポーネントに通知する
					RepositoryManagerUtil.broadcastConfigured();
				} catch (Exception e) {
					m_log.warn("deleteScope() transaction failure."
							+ e.getClass().getSimpleName() +", " + e.getMessage(), e);
					throw new HinemosUnknown(e.getMessage(),e);
				} finally {
					// ユーザロールキャッシュ情報を変更する
					UserRoleCache.refresh();
					jtm.close();
				}
			}
			jtm.close();
		}

		// リポジトリ更新時刻リフレッシュ
		RepositoryUpdateTime.refreshUpdateTime();
	}

	/**
	 * ロールツリー情報を取得する。<BR>
	 * 
	 * @param locale ロケール情報
	 * @return ロールツリー情報{@link com.clustercontrol.accesscontrol.bean.RoleTreeItem}の階層オブジェクト
	 * @throws HinemosUnknown
	 * @throws UserNotFound
	 */
	public RoleTreeItem getRoleTree(Locale locale) throws HinemosUnknown, UserNotFound, InvalidRole {
		m_log.debug("getRoleTree() : locale=" + locale);

		JpaTransactionManager jtm = null;
		String loginUser = (String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID);
		RoleTreeItem item = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();

			//ロールツリーを取得
			item = new RoleSelector().getRoleTree(locale, loginUser);
			jtm.commit();
		} catch (UserNotFound e) {
			jtm.rollback();
			throw e;
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (Exception e) {
			m_log.warn("getRoleTree() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return item;
	}

	/**
	 * システム権限一覧情報を取得する。<BR>
	 * 
	 * @return システム権限情報のリスト
	 * @throws HinemosUnknown
	 */
	public ArrayList<SystemPrivilegeInfo> getSystemPrivilegeInfoList() throws HinemosUnknown {

		JpaTransactionManager jtm = null;
		ArrayList<SystemPrivilegeInfo> systemPrivilegeInfoList = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			systemPrivilegeInfoList = RoleSelector.getSystemPrivilegeInfoList();
			jtm.commit();
		} catch (Exception e) {
			m_log.warn("getSystemPrivilegeInfoList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return systemPrivilegeInfoList;
	}

	/**
	 * 指定されたロールIDを条件としてシステム権限一覧情報を取得する。<BR>
	 * 
	 * @param ロールID
	 * @return システム権限情報のリスト
	 * @throws HinemosUnknown
	 */
	public ArrayList<SystemPrivilegeInfo> getSystemPrivilegeInfoListByRoleId(String roleId) throws HinemosUnknown {

		JpaTransactionManager jtm = null;
		ArrayList<SystemPrivilegeInfo> systemPrivilegeInfoList = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			systemPrivilegeInfoList = new ArrayList<SystemPrivilegeInfo>(UserRoleCache.getSystemPrivilegeList(roleId));
			jtm.commit();
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("getSystemPrivilegeInfoList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return systemPrivilegeInfoList;
	}

	/**
	 * 指定されたユーザIDを条件としてシステム権限一覧情報を取得する。<BR>
	 * 
	 * @param ユーザID
	 * @return システム権限情報のリスト
	 * @throws HinemosUnknown
	 */
	public ArrayList<SystemPrivilegeInfo> getSystemPrivilegeInfoListByUserId(String userId) throws HinemosUnknown {

		JpaTransactionManager jtm = null;
		ArrayList<SystemPrivilegeInfo> systemPrivilegeInfoList = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			systemPrivilegeInfoList = RoleSelector.getSystemPrivilegeInfoListByUserId(userId);
			jtm.commit();
		} catch (Exception e) {
			m_log.warn("getSystemPrivilegeInfoList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return systemPrivilegeInfoList;
	}

	/**
	 * ロールへのユーザの割り当てを行います<BR>
	 * 
	 * @param roleId　ユーザを割り当てるノード
	 * @param userIds 割り当てさせるユーザ(群)
	 * @throws UnEditableRole
	 * @throws HinemosUnknown
	 */
	public void assignUserRole(String roleId, String[] userIds)
			throws UnEditableRole, HinemosUnknown {
		JpaTransactionManager jtm = null;
		boolean commit = false;

		try{
			// トランザクション開始
			jtm = new JpaTransactionManager();
			jtm.begin();

			/** メイン処理 */
			RoleModifier.assignUserToRole(roleId, userIds);
			jtm.commit();
			commit = true;
		} catch (UnEditableRole e) {
			jtm.rollback();
			throw e;
		} catch (UserNotFound e) {
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("assignUserRole() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(),e);
		} finally {
			if (commit) {
				// ユーザロールキャッシュ情報を変更する
				UserRoleCache.refresh();
				// ファシリティツリーのオブジェクト権限情報をリフレッシュする
				FacilityTreeCache.refreshTreePrivilege();
			}

			jtm.close();
		}
	}

	/**
	 * ロールへのシステム権限の割り当てを行います<BR>
	 * 
	 * @param roleId　システム権限を割り当てるノード
	 * @param systemPrivileges 割り当てさせるシステム権限(群)
	 * @throws UnEditableRole
	 * @throws HinemosUnknown
	 */
	public void replaceSystemPrivilegeRole(String roleId, List<SystemPrivilegeInfo> systemPrivileges)
			throws UnEditableRole, HinemosUnknown {
		JpaTransactionManager jtm = null;
		boolean commit = false;

		try{
			// トランザクション開始
			jtm = new JpaTransactionManager();
			jtm.begin();

			/** メイン処理 */
			RoleModifier.replaceSystemPrivilegeToRole(roleId, systemPrivileges);

			jtm.commit();
			commit = true;
		} catch (UnEditableRole e) {
			jtm.rollback();
			throw e;
		} catch (RoleNotFound e) {
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("assignUserRole() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(),e);
		} finally {
			if (commit) {
				// ユーザロールキャッシュ情報を変更する
				UserRoleCache.refresh();
			}
			jtm.close();
		}
	}

	/**
	 * オブジェクト権限検索条件に基づき、オブジェクト権限一覧情報を取得する。<BR>
	 * 
	 * 
	 * @param filter オブジェクト権限検索条件
	 * @return オブジェクト権限のリスト
	 * @throws HinemosUnknown
	 * 
	 */
	public ArrayList<ObjectPrivilegeInfo> getObjectPrivilegeInfoList(ObjectPrivilegeFilterInfo filter) throws HinemosUnknown {

		JpaTransactionManager jtm = null;
		ArrayList<ObjectPrivilegeInfo> objectPrivilegeInfoList = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			objectPrivilegeInfoList = RoleSelector.getObjectPrivilegeInfoList(filter);
			jtm.commit();
		} catch (Exception e) {
			m_log.warn("getObjectPrivilegeInfoList() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return objectPrivilegeInfoList;
	}


	/**
	 * オブジェクト権限情報を取得する。<BR>
	 * 
	 * 
	 * @param objectType
	 * @param objectId
	 * @param roleId
	 * @param objectPrivilege
	 * @return ユーザ情報
	 * @throws HinemosUnknown
	 * 
	 */
	public ObjectPrivilegeInfo getObjectPrivilegeInfo(
			String objectType, String objectId, String roleId, String objectPrivilege) throws HinemosUnknown {
		JpaTransactionManager jtm = null;
		ObjectPrivilegeInfo objectPrivilegeInfo = null;
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();
			objectPrivilegeInfo = RoleSelector.getObjectPrivilegeInfo(objectType, objectId, roleId, objectPrivilege);
			jtm.commit();
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			jtm.rollback();
			m_log.warn("getObjectPrivilegeInfo() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			jtm.close();
		}
		return objectPrivilegeInfo;
	}

	/**
	 * オブジェクト種別、オブジェクトIDに紐づくオブジェクト権限情報を差し替える。<BR>
	 * 
	 * @param objectType オブジェクト種別
	 * @param objectId オブジェクトID
	 * @param info オブジェクト権限情報リスト
	 * @throws PrivilegeDuplicate
	 * @throws HinemosUnknown
	 * @throws InvalidSetting
	 * @throws InvalidRole
	 * 
	 */
	public void replaceObjectPrivilegeInfo(String objectType, String objectId, List<ObjectPrivilegeInfo> list)
			throws PrivilegeDuplicate, UsedObjectPrivilege, HinemosUnknown, InvalidSetting, InvalidRole, JobMasterNotFound {
		m_log.info("user=" + HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID));
		JpaTransactionManager jtm = null;
		boolean commit = false;
		/** メイン処理 */
		try {
			jtm = new JpaTransactionManager();
			jtm.begin();

			// JobMstEntityに対応したオブジェクト権限の編集は
			// 該当データがDBに登録されていない場合は処理終了
			if (HinemosModuleConstant.JOB.equals(objectType)) {
				com.clustercontrol.jobmanagement.util.QueryUtil.getJobMstPK_NONE(new JobMstEntityPK(objectId, objectId));
			}

			// オブジェクトWRITE権限チェック
			ObjectPrivilegeUtil.getObjectPrivilegeObject(objectType, objectId, ObjectPrivilegeMode.WRITE);

			//入力チェック
			ObjectPrivilegeValidator.validateObjectPrivilegeInfo(objectType, objectId, list);

			/** メイン処理 */
			RoleModifier.replaceObjectPrivilegeInfo(objectType, objectId, list,
					(String)HinemosSessionContext.instance().getProperty(HinemosSessionContext.LOGIN_USER_ID));

			jtm.commit();
			commit = true;
			
		} catch (InvalidSetting e) {
			jtm.rollback();
			throw e;
		} catch (UsedObjectPrivilege e) {
			jtm.rollback();
			throw e;
		} catch (PrivilegeDuplicate e) {
			jtm.rollback();
			throw e;
		} catch (ObjectPrivilege_InvalidRole e) {
			jtm.rollback();
			throw new InvalidRole(e.getMessage(), e);
		} catch (HinemosUnknown e) {
			jtm.rollback();
			throw e;
		} catch (JobMasterNotFound e) {
			jtm.rollback();
			throw e;
		} catch (Exception e) {
			m_log.warn("replaceObjectPrivilegeInfo() : "
					+ e.getClass().getSimpleName() + ", " + e.getMessage(), e);
			jtm.rollback();
			throw new HinemosUnknown(e.getMessage(), e);
		} finally {
			if (commit) {
				// スコープのオブジェクト権限処理の場合はファシリティツリー情報、最終更新時刻を更新する
				if (HinemosModuleConstant.PLATFORM_REPOSITORY.equals(objectType)) {
					RepositoryUpdateTime.refreshUpdateTime();
					FacilityTreeCache.refreshTreePrivilege();
				}
			}
			jtm.close();
		}
	}

}

