/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.commons.util;

import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.MissingResourceException;
import java.util.ResourceBundle;


/**
 * リソース取得クラス<BR>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class MessageManager {
	private static final Map<String, MessageManager> managers = new HashMap<String, MessageManager>();
	
	private String fileName;
	private ResourceBundle bundle;

	private MessageManager(String fileName) {
		this.fileName = fileName;
		bundle = ResourceBundle.getBundle(fileName);
	}

	public static MessageManager getInstance(String fileName){
		MessageManager manager = null;
		synchronized(managers){
			if(managers.containsKey(fileName)){
				manager = managers.get(fileName);
			} else { 
				manager = new MessageManager(fileName);
				managers.put(fileName, manager);
			}
		}
		return manager;
	}


	/**
	 * Returns the formatted message for the given key in the resource bundle.
	 * 
	 * @param key
	 *            the resource name
	 * @param args
	 *            the message arguments
	 * @return the string
	 */
	public String getString(String key, Object[] args) {
		MessageFormat messageFormat = new MessageFormat(getString(key));
		return messageFormat.format(args);
	}

	/**
	 * Returns the resource object with the given key in the resource bundle. If
	 * there isn't any value under the given key, the key is returned.
	 * 
	 * @param key
	 *            the resource name
	 * @return the string
	 */
	public String getString(String key) {
		try {
			return bundle.getString(key);
		} catch (MissingResourceException e) {
			return key;
		}
	}

	/**
	 * Returns the resource object with the given key in the resource bundle. If
	 * there isn't any value under the given key, the default value is returned.
	 * 
	 * @param key
	 *            the resource name
	 * @param def
	 *            the default value
	 * @return the string
	 */
	public String getString(String key, String def) {
		try {
			return bundle.getString(key);
		} catch (MissingResourceException e) {
			return def;
		}
	}

	/**
	 * Returns the formatted message for the given key in the resource bundle.
	 * 
	 * @param key
	 *            the resource name
	 * @param args
	 *            the message arguments
	 * @param locale
	 * @return the string
	 */
	public String getString(String key, Object[] args, Locale locale) {
		MessageFormat messageFormat = new MessageFormat(getString(key, locale));
		return messageFormat.format(args);
	}

	/**
	 * Returns the resource object with the given key in the resource bundle. If
	 * there isn't any value under the given key, the key is returned.
	 * 
	 * @param key
	 *            the resource name
	 * @param locale
	 * @return the string
	 */
	public String getString(String key, Locale locale) {
		ResourceBundle bundle = ResourceBundle.getBundle(fileName,
				locale);
		try {
			return bundle.getString(key);
		} catch (MissingResourceException e) {
			return key;
		}
	}

	/**
	 * Returns the resource object with the given key in the resource bundle. If
	 * there isn't any value under the given key, the default value is returned.
	 * 
	 * @param key
	 *            the resource name
	 * @param def
	 *            the default value
	 * @param locale
	 * @return the string
	 */
	public String getString(String key, String def, Locale locale) {
		ResourceBundle bundle = ResourceBundle.getBundle(fileName,
				locale);
		try {
			return bundle.getString(key);
		} catch (MissingResourceException e) {
			return def;
		}
	}

}