/*
Copyright (C) 2013 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
PURPOSE.  See the GNU General Public License for more details.
 */
package com.clustercontrol.cloud.aws.presenter;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;


public class PropChangedEventNotifier {
	public static final PropertyId2<AllPropertyObserver> allProperty = new PropertyId2<AllPropertyObserver>();

	private interface Command {
		void execute();
	}

	private Map<Object, List<?>> listenersMap = new HashMap<Object, List<?>>();
	private List<Command> commands = new ArrayList<Command>();
	protected int notifyingDepth = 0;
	
	public PropChangedEventNotifier() {
	}

	public <P, O extends PropertyObserver2<P>> void addPropertyObserver(final PropertyId2<O> pid, final O observer) {
		synchronized (listenersMap) {
			if (notifyingDepth == 0) {
				doAdd(pid, observer);
			}
			else {
				commands.add(
					new Command() {
						@Override
						public void execute() {
							doAdd(pid, observer);
						}
					});
			}
		}
	}
	
	private void doAdd(Object pid, Object observer) {
		@SuppressWarnings("unchecked")
		List<Object> list = (List<Object>)listenersMap.get(pid);
		if (list == null) {
			list = new ArrayList<Object>();
			listenersMap.put(pid, list);
		}
		list.add(observer);
	}
	
	public <P, O extends PropertyObserver2<P>> void removePropertyObserver(final PropertyId2<O> pid, final O observer) {
		synchronized (listenersMap) {
			if (notifyingDepth == 0) {
				doRemove(pid, observer);
			}
			else {
				commands.add(
					new Command() {
						@Override
						public void execute() {
							doRemove(pid, observer);
						}
					});
			}
		}
	}

	private void doRemove(Object pid, Object observer) {
		@SuppressWarnings("unchecked")
		List<Object> list = (List<Object>)listenersMap.get(pid);
		if (list != null) {
			list.remove(observer);
		}
	}
	
	protected <P, O extends PropertyObserver2<P>> void fireEvent(PropEvent<P, O> event) {
		synchronized (listenersMap) {
			++notifyingDepth;
			@SuppressWarnings("unchecked")
			List<O> list = (List<O>)listenersMap.get(event.getPropertyId());
			if (list != null) {
				for (O observer: list) {
					event.dispatch(observer);
				}
			}
			for (Command c: commands) {
				c.execute();
			}
			--notifyingDepth;
		}
	}

	public <P> void fireValueChanged(Object source, PropertyId2<ValueObserver2<P>> pid, P newValue, P oldValue) {
		ValueObserver2.ValueChangedEvent<P> event = new ValueObserver2.ValueChangedEvent<P>(source, pid, newValue, oldValue);
		fireEvent(event);
		AllPropertyObserver.ValueChangedEvent event2 = new AllPropertyObserver.ValueChangedEvent(source, allProperty, newValue, oldValue);
		fireEvent(event2);
	}

	public <P> void fireElementAdded(Object source, PropertyId2<CollectionObserver2<P>> pid, P addedValue) {
		CollectionObserver2.ElementAddedEvent<P> event = new CollectionObserver2.ElementAddedEvent<P>(source, pid, addedValue);
		fireEvent(event);
		AllPropertyObserver.ElementAddedEvent event2 = new AllPropertyObserver.ElementAddedEvent(source, allProperty, addedValue);
		fireEvent(event2);
	}

	public <P> void fireElementRemoved(Object source, PropertyId2<CollectionObserver2<P>> pid, P removedValue) {
		CollectionObserver2.ElementRemovedEvent<P> event = new CollectionObserver2.ElementRemovedEvent<P>(source, pid, removedValue);
		fireEvent(event);
		AllPropertyObserver.ElementRemovedEvent event2 = new AllPropertyObserver.ElementRemovedEvent(source, allProperty, removedValue);
		fireEvent(event2);
	}
}